<?php

use Illuminate\Auth\Events\Lockout;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use Livewire\Volt\Component;
use App\Traits\LogsActivity;
use Illuminate\View\View;

new #[Layout('components.frontend.layouts.struct ')] class extends Component {
    use LogsActivity;

    public function rendering(View $view): void
    {
        $view->title(__('frontend.tittlewelcomepage') .' | '.__('frontend.login'));
    }

    #[Validate('required|string|email')]
    public string $email = '';

    #[Validate('required|string')]
    public string $password = '';

    public bool $remember = false;
    public bool $loginDisabled = false;

    public function login(): void
{
    // أول حاجة نتأكد من الرايت ليميت قبل أي حاجة تانية
    if ($this->isRateLimited()) {
        return; // نخرج فورًا من الدالة
    }

    $this->validate();

    if (!Auth::attempt(['email' => $this->email, 'password' => $this->password], $this->remember)) {
        RateLimiter::hit($this->throttleKey());
        $this->dispatch('notify', type: 'error', message: __('auth.failed'), duration: 5000);
        return;
    }

    RateLimiter::clear($this->throttleKey());
    Session::regenerate();
    $this->logActivity('Login', 'User passed login authentication.', 'users');

    // session()->flash('toast', [
    //     'type'     => 'success',
    //     'msg'      => 'Login successful!',
    //     'duration' => 5000,
    // ]);

    // $this->redirectIntended(default: route('dashboard', absolute: false), navigate: true);
    $this->redirect(route('dashboard'));

}

protected function isRateLimited(): bool
{
    if (!RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
        return false;
    }

    event(new Lockout(request()));

    $seconds = RateLimiter::availableIn($this->throttleKey());
    $this->loginDisabled = true;

    // إشعار Frontend فقط.. من غير رسالة Error
    $this->dispatch('notify', type: 'rate-limit', message:  __('auth.throttle'), countdown: $seconds);

    return true;
}


    protected function ensureIsNotRateLimited(): void
    {
        if (!RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
            return;
        }

        event(new Lockout(request()));

        $seconds = RateLimiter::availableIn($this->throttleKey());

        $this->loginDisabled = true;
        $this->dispatch('notify', type: 'rate-limit', message: __('auth.throttle'), countdown: $seconds);
    }

    protected function throttleKey(): string
    {
        return Str::transliterate(Str::lower($this->email) . '|' . request()->ip());
    }
};
?>

<div class="min-h-screen flex items-center justify-center bg-transparent transition duration-700 overflow-hidden relative">

    <div class="z-10 max-w-md w-full space-y-6 bg-white/80 dark:bg-zinc-900/80 backdrop-blur-md p-8 rounded-xl shadow-2xl border border-zinc-300 dark:border-zinc-700">
        <div class="flex flex-col items-center gap-2 animate-fade-in-down">
            <img src="/assets/img/logo.png" alt="Logo" class="w-[300px] h-auto mb-3 drop-shadow-lg" />
            <h2 class="text-2xl font-semibold text-zinc-800 dark:text-white">{{ __('frontend.loginmsg') }}</h2>
        </div>

        <form wire:submit="login" class="space-y-3 animate-fade-in-down">
            <div>
                <label for="email" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300">{{ __('frontend.emailadress') }}</label>
                <input type="email" id="email" wire:model="email" required autofocus autocomplete="email"
                       class="mt-1 block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 ring-accent outline-none transition"
                       placeholder="email@example.com" />
            </div>

            <div class="relative">
                <label for="password" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300">{{ __('frontend.password') }}</label>
                <input type="password" id="password" wire:model="password" required autocomplete="current-password"
                       class="mt-1 block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 ring-accent outline-none transition"
                       placeholder="{{ __('frontend.epassword') }}" />

                <a href="{{ route('password.request') }}" class="absolute end-0 top-0 text-sm text-blue-500 hover:underline mt-1">
                    {{ __('frontend.forgetpassword') }}
                </a>
            </div>

            <label class="inline-flex items-center cursor-pointer group">
                <input type="checkbox" wire:model="remember" class="sr-only peer">
                <div class="w-5 h-5 border-2 border-zinc-400 dark:border-zinc-600 bg-white dark:bg-zinc-800 rounded-md flex items-center justify-center transition-colors duration-300 peer-checked:bg-indigo-500 peer-checked:border-indigo-500 peer-checked:shadow-[0_0_0_3px_rgba(99,102,241,0.3)]">
                  <svg class="w-3 h-3 text-white opacity-0 peer-checked:opacity-100 transition-opacity duration-200" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="3">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7" />
                  </svg>
                </div>
                <span class="ml-3 text-sm text-zinc-700 dark:text-zinc-300 select-none transition-colors duration-300">
                    {{ __('frontend.rememberme') }}
                </span>
              </label>



            <div class="flex items-center justify-end">
                <button id="login-btn"
                type="submit"
                class="neon-btn delay-[1000ms] neon-blue {{ $this->loginDisabled ? 'opacity-50 cursor-not-allowed grayscale text-zinc-400' : '' }}"
                {{ $this->loginDisabled ? 'disabled' : '' }}>
            <span class="btn-text">{{ __('frontend.login') }}</span>
            <span class="glow"></span>
        </button>



            </div>
        </form>

        <div class="text-center mt-4 animate-fade-in-down">
            <p class="text-sm text-zinc-600 dark:text-zinc-300">
                {{ __('frontend.dhaveaccount') }}
                <a href="{{ route('register') }}" class="text-blue-500 hover:underline">{{ __('frontend.register') }}</a>
            </p>
        </div>
    </div>
</div>
