<?php

use Livewire\Attributes\Layout;
use Livewire\Volt\Component;
use Illuminate\View\View;
use Illuminate\Support\Facades\Auth;
use Livewire\WithFileUploads;

new #[Layout('components.backend.layouts.struct')] class extends Component {
    use WithFileUploads;

    public $photo;
    public $full_name_ar;
    public $full_name_en;
    public $email;
    public array $phone = [];
    public $current_password;
    public $new_password;
    public $confirm_password;
    public $cvContent = '';
    public $activeModal = null;
    public $cv_profile_picture;
    public $cv_experience_roads = [];
    public $cv_experience_airports = [];
    public $cv_skills = [];
    public $cv_profile_picture_path = null;
    public $cv_currentStep = 1;

    public function mount()
    {
        $user = Auth::user();
        $this->full_name_ar = $user->employee?->full_name['ar'] ?? '';
        $this->full_name_en = $user->employee?->full_name['en'] ?? '';
        $this->email = $user->email;
        $this->phone = Auth::user()->employee?->phone ? json_decode(Auth::user()->employee->phone, true) : [''];
    }

    public function updateProfile()
    {
        // 1️⃣ تحقق من الحقول
        $this->validate(
            [
                'full_name_ar' => 'required|string|min:3|max:255',
                'full_name_en' => 'required|string|min:3|max:255',
                'email' => 'required|email|max:255',
                'phone' => 'required|array|min:1',
                'phone.*' => ['required', 'string', 'regex:/^(\+?20|0020)?1?\d{9}$/'],
                'current_password' => 'required|current_password',
                'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            ],
            [
                'current_password.required' => __('backend.errors.current_password_required'),
            ],
        );

        $user = Auth::user();
        $employee = $user->employee;

        $updated = false;

        // 2️⃣ تحديث اسم المستخدم
        $full_name_combined = trim($this->full_name_en);
        if ($user->name !== $full_name_combined) {
            $user->name = $full_name_combined;
            $this->dispatch('nameUpdated', $user->name);
            $updated = true;
        }

        // 3️⃣ تحديث البريد الإلكتروني
        if ($user->email !== $this->email) {
            $user->email = $this->email;
            $updated = true;
        }

        // 4️⃣ تحديث بيانات الموظف
        $newFullName = [
            'ar' => $this->full_name_ar,
            'en' => $this->full_name_en,
        ];

        if ($employee && $employee->full_name != $newFullName) {
            $employee->full_name = $newFullName;
            $updated = true;
        }

        $phoneJson = json_encode($this->phone);
        if ($employee && $employee->phone !== $phoneJson) {
            $employee->phone = $phoneJson;
            $updated = true;
        }

        // 5️⃣ معالجة الصورة
        if ($this->photo && $this->photo->isValid()) {
            try {
                $slugName = Str::slug($user->name);
                $timestamp = now()->format('Y-m-d_H-i-s');
                $extension = $this->photo->getClientOriginalExtension();
                $filename = $slugName . '_' . $timestamp . '.' . $extension;
                $directory = 'profile-photos/' . $slugName;
                Storage::disk('public')->makeDirectory($directory);

                $newImageHash = md5(file_get_contents($this->photo->getRealPath()));

                $settings = $user->settings ?? [];
                $oldAvatarPath = $settings['avatar'] ?? null;
                $oldImageHash = null;

                if ($oldAvatarPath && Storage::disk('public')->exists($oldAvatarPath)) {
                    $oldImageHash = md5(Storage::disk('public')->get($oldAvatarPath));
                }

                if ($newImageHash === $oldImageHash) {
                    $this->dispatch('notify', type: 'info', message: __('backend.avatar_same'));
                    $this->dispatch('resetLoading');
                    return;
                }

                $photoPath = $this->photo->storeAs($directory, $filename, 'public');

                if ($photoPath) {
                    $avatarPath = $directory . '/' . $filename;
                    $settings['avatar'] = $avatarPath;
                    $user->settings = $settings;

                    $this->dispatch('avatarUpdated', asset('storage/' . $avatarPath));
                    $this->dispatch('showRemoveButton', true);
                    $updated = true;
                }
            } catch (\Exception $e) {
                $this->dispatch('notify', type: 'error', message: __('backend.errors.image_compression_failed', ['error' => $e->getMessage()]));
                $this->dispatch('resetLoading');
                return;
            }
        }

        // 6️⃣ الحفظ النهائي
        if ($updated) {
            $user->save();
            if ($employee) {
                $employee->save();
            }

            $this->dispatch('notify', type: 'success', message: __('backend.profile_updated_successfully'));
            $this->dispatch('resetLoading');

            // نفضي كلمة المرور بعد النجاح
            $this->reset('current_password');
        } else {
            $this->dispatch('notify', type: 'info', message: __('backend.no_changes_made'));
            $this->dispatch('resetLoading');
        }
    }

    public function updatePassword()
    {
        $validator = Validator::make(
            [
                'current_password' => $this->current_password,
                'new_password' => $this->new_password,
                'confirm_password' => $this->confirm_password,
            ],
            [
                'current_password' => ['required', 'current_password'],
                'new_password' => ['required', 'string', 'min:12', 'regex:/[a-z]/', 'regex:/[A-Z]/', 'regex:/[0-9]/', 'same:confirm_password'],
            ],
            [
                'current_password.required' => __('backend.errors.current_password_required'),
                'new_password.required' => __('backend.errors.new_password_required'),
                'new_password.same' => __('backend.errors.password_not_confirmed'),
                'new_password.min' => __('backend.errors.password_too_short'),
                'new_password.regex' => __('backend.errors.password_not_strong'),
            ],
        );

        // ⛔ عشان نظهر كل الأخطاء مع بعض
        $validator->stopOnFirstFailure(false);

        $validator->validate();

        // ✅ بعد التحقق
        $user = Auth::user();

        if (Hash::check($this->new_password, $user->password)) {
            $this->addError('new_password', __('backend.errors.password_same_as_old'));
            return;
        }

        $user->password = bcrypt($this->new_password);
        $user->save();

        $this->reset(['current_password', 'new_password', 'confirm_password']);
        $this->dispatch('notify', type: 'success', message: __('backend.password_updated_successfully'));
    }

    public function rendering(View $view): void
    {
        $view->title(__('backend.profile') . ' | ' . __('backend.settings'));
    }

    public function removeAvatar()
    {
        $user = Auth::user();
        $settings = $user->settings ?? [];

        if (!empty($settings['avatar']) && Storage::disk('public')->exists($settings['avatar'])) {
            Storage::disk('public')->delete($settings['avatar']);
            unset($settings['avatar']);
            $user->settings = $settings;
            $user->save();

            // 🟢 نظف متغير photo (لو كان في صورة جديدة مرفوعة)
            $this->photo = null;

            // 🟢 للـ preview في الواجهة
            $this->dispatch('updateInitials', strtoupper(Str::substr($user->name, 0, 2)));
            $this->dispatch('avatarUpdated', null);
            $this->dispatch('refresh');

            // 🟢 رسالة نجاح
            $this->dispatch('notify', type: 'success', message: __('backend.avatar_removed_successfully'));
        } //  else {
        //     // ℹ️ لو مفيش صورة نبه المستخدم (اختياري)
        //     $this->dispatch('notify', type: 'info', message: __('backend.no_avatar_to_remove'));
        // }
    }

    public function editCv()
    {
        $employee = Auth::user()->employee;

        if ($employee) {
            $cv = is_array($employee->cv) ? $employee->cv : json_decode($employee->cv, true);

            $this->cv_profile_picture = null; // هنا لو عندك profile_picture مسجلة في cv
            $this->cv_experience_roads = $cv['work_experience']['roads'] ?? [];
            $this->cv_experience_airports = $cv['work_experience']['airports'] ?? [];
            $this->cv_skills = $cv['skills'] ?? [];
            $this->cv_profile_picture_path = isset($cv['profile_picture']) && $cv['profile_picture'] ? str_replace(['\\', '//'], '/', $cv['profile_picture']) : null;
            $this->cv_currentStep = 1;
            $this->activeModal = 'editCv';
        }
    }

    public function updateCv()
    {
        $employee = Auth::user()->employee;
        if ($employee) {
            $employee->cv = json_encode(json_decode($this->cvContent, true), JSON_UNESCAPED_UNICODE);
            $employee->save();

            $this->dispatch('notify', type: 'success', message: __('backend.cv_updated_successfully'));
        }

        $this->closeModal();
    }

    public function closeModal()
    {
        $this->activeModal = null;
    }

    public function nextCvStep()
    {
        if ($this->cv_currentStep < 3) {
            $this->cv_currentStep++;
        }
    }

    public function previousCvStep()
    {
        if ($this->cv_currentStep > 1) {
            $this->cv_currentStep--;
        }
    }

    public function updateCvFull()
    {
        $employee = Auth::user()->employee;

        if ($this->cv_profile_picture && $this->cv_profile_picture->isValid()) {
            $user = Auth::user();
            $slugName = Str::slug($user->name);
            $timestamp = now()->format('Y-m-d_H-i-s');
            $extension = $this->cv_profile_picture->getClientOriginalExtension();
            $filename = $slugName . '_cv_' . $timestamp . '.' . $extension;
            $directory = 'employees/' . $user->employee->employee_code;

            Storage::disk('public')->makeDirectory($directory);

            $photoPath = $this->cv_profile_picture->storeAs($directory, $filename, 'public');

            if ($photoPath) {
                $this->cv_profile_picture_path = $photoPath;
            }
        }

        if ($employee) {
            $employee->cv = json_encode(
                [
                    'work_experience' => [
                        'roads' => $this->cv_experience_roads,
                        'airports' => $this->cv_experience_airports,
                    ],
                    'skills' => $this->cv_skills,
                    // لو عندك تخزين للصورة تقدر تضيفها هنا
                   'profile_picture' => $this->cv_profile_picture_path,
                ],
                JSON_UNESCAPED_UNICODE,
            );

            $employee->save();

            $this->dispatch('notify', type: 'success', message: __('backend.cv_updated_successfully'));

            $this->closeModal();
        }
    }
};
?>
@php
    $avatarPath = Auth::user()->settings['avatar'] ?? '';
    $avatarFullUrl = $avatarPath ? asset('storage/' . $avatarPath) : '';
    $initials = strtoupper(Str::substr(Auth::user()->name, 0, 2));
@endphp

<div class="max-w-4xl mx-auto p-4 sm:p-6 lg:p-8 relative z-10" x-data="{ previewUrl: null, initials: '{{ $initials }}', disableSave: false, loading: false, showRemoveButton: '{{ $avatarFullUrl }}'.length > 5, }" x-init="avatar = '{{ $avatarFullUrl }}';
@this.on('resetLoading', () => loading = false);
@this.on('refresh', () => previewUrl = null);
@this.on('avatarUpdated', value => {
    avatar = value;
    showRemoveButton = value && value.length > 5;
});
@this.on('updateInitials', value => initials = value);">
    <div
        class="bg-white/90 dark:bg-zinc-900/80 border border-zinc-200 dark:border-zinc-700 rounded-2xl shadow-xl backdrop-blur-md overflow-hidden divide-y divide-zinc-100 dark:divide-zinc-800">

        <!-- صورة الملف الشخصي + الاسم -->
        <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-6 p-6 text-start">
            <div class="flex items-center gap-4">
                <div class="relative w-24 h-24 shrink-0">

                    <!-- لو في صورة (preview أو موجودة مسبقًا في settings['avatar']) -->
                    <template x-if="(previewUrl && previewUrl.length > 5) || (avatar && avatar.length > 5)">
                        <img :src="previewUrl || avatar" alt="profile"
                            class="w-24 h-24 rounded-full object-cover border-4 border-indigo-400 dark:border-indigo-600">
                    </template>



                    <!-- لو مفيش صورة، نعرض الحروف الأولى -->
                    <template x-if="!previewUrl && !avatar">
                        <div
                            class="w-24 h-24 rounded-full flex items-center justify-center bg-indigo-500 text-white font-bold text-xl border-4 border-indigo-400 dark:border-indigo-600">
                            <span x-text="initials"></span>
                        </div>
                    </template>

                    <!-- زر تغيير الصورة -->
                    <label for="photoInput"
                        class="absolute bottom-0 right-0 bg-indigo-600 dark:bg-indigo-400 p-1 rounded-full cursor-pointer shadow hover:scale-105 transition">
                        ✏️
                    </label>

                    <!-- input رفع صورة -->
                    <input type="file" id="photoInput" wire:model="photo" x-ref="fileInput"
                        @change="
                        const file = $refs.fileInput.files[0];
                        if (file) {
                            previewUrl = URL.createObjectURL(file);
                            disableSave = true;
                            showRemoveButton = true;
                            setTimeout(() => disableSave = false, 3000);
                        }
                    "
                        class="hidden" accept="image/*" />


                </div>
                <!-- زر إزالة الصورة (يظهر لو في صورة حالياً أو جديدة) -->
                <template x-if="showRemoveButton">
                    <div class="flex justify-center mt-3">
                        <button type="button"
                            @click="
                $wire.removeAvatar();
                previewUrl = null;
                avatar = null;
                showRemoveButton = false;
                $refs.fileInput.value = '';
                $wire.set('photo', null);
            "
                            class="px-4 py-1 bg-red-600 text-white text-sm font-semibold rounded shadow hover:bg-red-700 transition">
                            {{ __('backend.remove_avatar') }}
                        </button>
                    </div>
                </template>



                <div>
                    <h2 class="text-xl font-bold text-zinc-800 dark:text-white">{{ __('backend.update_profile') }}</h2>
                    <p class="text-sm text-zinc-600 dark:text-zinc-300">{{ __('backend.update_profile_hint') }}</p>
                </div>
            </div>
        </div>


        {{-- <div class="flex justify-between items-center mt-6 space-x-5">

            <!-- زر تعديل CV -->

            <button type="button" wire:click="editCv"
                class="px-10 py-2.5 rounded-lg bg-gradient-to-r from-cyan-500 to-blue-600 text-white font-semibold tracking-wide hover:from-cyan-600 hover:to-blue-700 transition shadow-lg hover:scale-105">
                ✏️ {{ __('frontend.edit_cv') }}
            </button>

            <!-- زر طباعة CV -->
            <a href="{{ route('cv.download', ['emp' => encrypt(Auth::user()->employee->id)]) }}" target="_blank"
                class="px-10 py-2.5 rounded-lg bg-gradient-to-r from-green-500 to-emerald-600 text-white font-semibold tracking-wide hover:from-green-600 hover:to-emerald-700 transition shadow-lg hover:scale-105">
                🖨️ {{ __('frontend.print_cv') }}
            </a>


        </div> --}}
        <!-- تعديل الاسم -->
        <form wire:submit.prevent="updateProfile" @submit="loading = true" class="p-6 space-y-4">

            <!-- اسم عربي -->
            <div class="mb-5">
                <label class="block mb-2 text-sm font-semibold text-zinc-700 dark:text-zinc-200 tracking-wide">
                    {{ __('frontend.fullname') }} (عربي)
                </label>

                <div class="relative">
                    <div
                        class="absolute inset-y-0 start-0 flex items-center ps-3 text-zinc-400 dark:text-zinc-500 text-base">
                        📝
                    </div>

                    <input type="text" wire:model.defer="full_name_ar"
                        class="block w-full ps-10 pe-4 py-2 text-sm rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-800 dark:text-white focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition duration-200 ease-in-out shadow-sm placeholder-zinc-400 dark:placeholder-zinc-500"
                        placeholder="{{ __('frontend.fullname') }} (عربي)" />
                </div>

                @error('full_name_ar')
                    <span
                        class="block mt-1 text-xs text-red-600 transition duration-150 ease-in-out">{{ $message }}</span>
                @enderror
            </div>

            <!-- اسم إنجليزي -->
            <div class="mb-5">
                <label class="block mb-2 text-sm font-semibold text-zinc-700 dark:text-zinc-200 tracking-wide">
                    {{ __('frontend.fullname') }}
                </label>

                <div class="relative">
                    <div
                        class="absolute inset-y-0 start-0 flex items-center ps-3 text-zinc-400 dark:text-zinc-500 text-base">
                        📝
                    </div>

                    <input type="text" wire:model.defer="full_name_en"
                        class="block w-full ps-10 pe-4 py-2 text-sm rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-800 dark:text-white focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition duration-200 ease-in-out shadow-sm placeholder-zinc-400 dark:placeholder-zinc-500"
                        placeholder="{{ __('frontend.fullname') }}" />
                </div>

                @error('full_name_en')
                    <span
                        class="block mt-1 text-xs text-red-600 transition duration-150 ease-in-out">{{ $message }}</span>
                @enderror
            </div>

            <!-- إيميل -->
            <div class="mb-5">
                <label class="block mb-2 text-sm font-semibold text-zinc-700 dark:text-zinc-200 tracking-wide">
                    {{ __('frontend.emailadress') }}
                </label>

                <div class="relative">
                    <div
                        class="absolute inset-y-0 start-0 flex items-center ps-3 text-zinc-400 dark:text-zinc-500 text-base">
                        📧
                    </div>

                    <input type="email" wire:model.defer="email"
                        class="block w-full ps-10 pe-4 py-2 text-sm rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-800 dark:text-white focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition duration-200 ease-in-out shadow-sm placeholder-zinc-400 dark:placeholder-zinc-500"
                        placeholder="{{ __('frontend.emailadress') }}" />
                </div>

                @error('email')
                    <span
                        class="block mt-1 text-xs text-red-600 transition duration-150 ease-in-out">{{ $message }}</span>
                @enderror
            </div>

            <div x-data="{
                phones: @entangle('phone'),
                addPhone() {
                    this.phones.push('');
                },
                removePhone(index) {
                    if (this.phones.length > 1) {
                        this.phones.splice(index, 1);
                    }
                }
            }">
                <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                    {{ __('frontend.phone') }}
                </label>

                <template x-for="(phone, index) in phones" :key="index">
                    <div class="flex items-center gap-2 mb-2">
                        <div
                            class="flex items-center gap-2 px-3 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 flex-1">
                            <!-- Icon + Prefix -->
                            <svg class="w-5 h-5 text-indigo-500 dark:text-indigo-400" fill="none"
                                stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" stroke-linecap="round"
                                stroke-linejoin="round">
                                <path
                                    d="M22 16.92V21a2 2 0 0 1-2.18 2A19.79 19.79 0 0 1 3 5.18 2 2 0 0 1 5 3h4.09a1 1 0 0 1 1 .75l1.21 4.84a1 1 0 0 1-.29 1L9.1 11.1a16 16 0 0 0 6.9 6.9l1.51-1.51a1 1 0 0 1 1-.29l4.84 1.21a1 1 0 0 1 .75 1z" />
                            </svg>
                            <span class="text-sm font-medium select-none">+20</span>
                            <input type="tel" :id="'phone_' + index" placeholder="10xxxxxxxx"
                                class="flex-1 px-2 py-1 bg-transparent outline-none text-zinc-900 dark:text-white"
                                x-model="phones[index]" />
                        </div>

                        <!-- زر الحذف -->
                        <button type="button" @click="removePhone(index)"
                            class="text-red-500 hover:text-red-700 text-lg font-bold px-2 py-1">
                            &times;
                        </button>
                    </div>
                </template>

                <!-- زر إضافة رقم -->
                <div class="flex justify-center mt-2">
                    <button type="button" @click="addPhone()"
                        class="px-3 py-1 bg-indigo-600 text-white text-sm rounded hover:bg-indigo-700 transition">
                        + {{ __('frontend.add_phone') }}
                    </button>
                </div>
            </div>




            <!-- كلمة مرور حالية لتأكيد التعديلات -->
            <div class="mb-5">
                <label class="block mb-2 text-sm font-semibold text-zinc-700 dark:text-zinc-200 tracking-wide">
                    {{ __('frontend.password') }}
                </label>

                <div class="relative">
                    <div
                        class="absolute inset-y-0 start-0 flex items-center ps-3 text-zinc-400 dark:text-zinc-500 text-base">
                        🔒
                    </div>

                    <input type="password" wire:model.defer="current_password"
                        class="block w-full ps-10 pe-4 py-2 text-sm rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-800 dark:text-white focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition duration-200 ease-in-out shadow-sm placeholder-zinc-400 dark:placeholder-zinc-500"
                        placeholder="{{ __('frontend.password') }}" />
                </div>

                @error('current_password')
                    <span
                        class="block mt-1 text-xs text-red-600 transition duration-150 ease-in-out">{{ $message }}</span>
                @enderror
            </div>

            <button type="submit" wire:loading.attr="disabled" wire:target="updateProfile"
                :disabled="disableSave || loading"
                class="px-4 py-2 rounded-md bg-indigo-600 text-white hover:bg-indigo-700 transition shadow disabled:opacity-50 disabled:cursor-not-allowed">


                <!-- الزر وقت عدم التحميل -->
                <span wire:loading.remove wire:target="updateProfile">
                    {{ __('backend.save_changes') }}
                </span>

                <!-- الزر وقت التحميل -->
                <span wire:loading wire:target="updateProfile" class="flex items-center gap-1">
                    <svg class="animate-spin h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none"
                        viewBox="0 0 24 24">
                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor"
                            stroke-width="4"></circle>
                        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8v8H4z"></path>
                    </svg>
                    <span>{{ __('backend.saving') }}</span>
                </span>
            </button>


        </form>

        <!-- تغيير كلمة المرور -->
        <form wire:submit.prevent="updatePassword" class="p-6 space-y-4">
            <h3 class="text-lg font-semibold text-zinc-800 dark:text-white">
                {{ __('backend.change_password') }}
            </h3>

            <div>
                <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-200">
                    {{ __('backend.current_password') }}
                </label>
                <input type="password" wire:model.defer="current_password"
                    class="mt-1 w-full rounded-md border-zinc-300 dark:border-zinc-600 dark:bg-zinc-800 dark:text-white shadow-sm" />
                @error('current_password')
                    <span class="text-sm text-red-600">{{ $message }}</span>
                @enderror
            </div>

            <div>
                <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-200">
                    {{ __('backend.new_password') }}
                </label>
                <input type="password" wire:model.defer="new_password"
                    class="mt-1 w-full rounded-md border-zinc-300 dark:border-zinc-600 dark:bg-zinc-800 dark:text-white shadow-sm" />
            </div>

            <div>
                <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-200">
                    {{ __('backend.confirm_password') }}
                </label>
                <input type="password" wire:model.defer="confirm_password"
                    class="mt-1 w-full rounded-md border-zinc-300 dark:border-zinc-600 dark:bg-zinc-800 dark:text-white shadow-sm" />
                @error('new_password')
                    <span class="text-sm text-red-600">{{ $message }}</span>
                @enderror
            </div>

            <button type="submit" wire:loading.attr="disabled" wire:target="updatePassword"
                class="px-4 py-2 rounded-md bg-rose-600 text-white hover:bg-rose-700 transition shadow disabled:opacity-50 disabled:cursor-not-allowed">
                <span wire:loading.remove wire:target="updatePassword">
                    {{ __('backend.update_password') }}
                </span>
                <span wire:loading wire:target="updatePassword" class="flex items-center gap-1">
                    <svg class="animate-spin h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none"
                        viewBox="0 0 24 24">
                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor"
                            stroke-width="4"></circle>
                        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8v8H4z"></path>
                    </svg>
                    <span>{{ __('backend.saving') }}</span>
                </span>
            </button>

        </form>

    </div>


    @if ($activeModal === 'editCv')
        @include('livewire.backend.user.modals.edit-cv')
    @endif
</div>
