<?php

use Livewire\Attributes\Layout;
use Livewire\Volt\Component;
use Illuminate\View\View;
use Illuminate\Validation\Rules\Password;
use Illuminate\Validation\Rule;
use App\Models\User;
use App\Models\User\UserEmployee;
use App\Models\User\UserProject;
use App\Models\System\SystemRoute;
use Livewire\WithPagination;
use Illuminate\Pagination\LengthAwarePaginator;

new #[Layout('components.backend.layouts.struct')] class extends Component {
    use WithPagination;

    public string $search = '';
    public string $sortField = 'name';
    public string $sortDirection = 'asc';
    public bool $readyToLoad = false;
    public int $perPage = 10;
    public $selectedUser;
    public $selectedUser2;
    public string $filterState = '';
    public $selectedUserForDelete = null;
    public $routes = [];
    public $cv_profile_picture;
    public $cv_profile_picture_path;
    public $cv_experience_roads = [];
    public $cv_experience_airports = [];
    public $cv_skills = [];
    public $cv_currentStep = 1;
    public $selectedUserForCvEdit = null;
    public $new = [
        'name' => '',
        'name_ar' => '',
        'email' => '',
        'password' => '',
        'settings' => [],
        'user_state' => 'active', // القيم الممكنة: active, inactive, block, rejected, suspended
        'phone' => [''],
        'pages' => [],
        'project_code' => '',
        'job_title_id' => '',
        'national_id' => '',
    ];

    public $edit = [
        'id' => '',
        'name' => '',
        'name_ar' => '',
        'email' => '',
        'password' => '',
        'settings' => [],
        'user_state' => 'active',
        'phone' => [''],
        'pages' => [],
        'project_code' => '',
        'job_title_id' => '',
        'national_id' => '',
    ];

    // متغير المودال النشط - القيم الممكنة: 'addUser', 'editState', 'details', 'privileges'
    public $activeModal = null;

    public function mount()
    {
        $this->fetchUsers();
    }

    public function rendering(View $view): void
    {
        $view->title(__('backend.department_name') . ' | ' . __('backend.users'));
    }

    public function fetchUsers()
    {
        // ─── 1) حساب الصلاحيات ─────────────────────────────
        $pagesData = Auth::user()->pages['page'] ?? [];
        $elementsData = Auth::user()->pages['element']['users.index'] ?? [];

        $isSuperAdmin = in_array('[all]', $pagesData);
        $canViewAll = $isSuperAdmin || (in_array('users.index', $pagesData) && in_array('user.view_all', $elementsData));
        $canVisitSelf = in_array('user.visit_self', $elementsData);

        // ─── 2) بناء الاستعلام الرئيسي ────────────────────
        $query = User::with('employee')
            // 2.a) تقييد العرض حسب المشروع فقط إذا ليس سوبر أدمِن ولا يملك view_all
            ->when(!$isSuperAdmin && !$canViewAll, fn($q) => $q->whereHas('employee', fn($q2) => $q2->where('project_code', Auth::user()->project_code)))
            // 2.b) استثناء السجل الذاتي إذا ليس سوبر أدمِن ولا يملك visit_self
            ->when(!$isSuperAdmin && !$canVisitSelf, fn($q) => $q->where('users.id', '!=', Auth::id()))
            // 2.c) البحث
            ->when($this->search, function ($q) {
                $term = '%' . Str::lower($this->search) . '%';
                $q->where(function ($q2) use ($term) {
                    $q2->whereRaw('LOWER(name) LIKE ?', [$term])
                        ->orWhereRaw('LOWER(email) LIKE ?', [$term])
                        ->orWhereHas(
                            'employee',
                            fn($q3) => $q3
                                ->whereRaw('LOWER(employee_code) LIKE ?', [$term])
                                ->orWhereRaw('LOWER(national_id) LIKE ?', [$term])
                                ->orWhereRaw("LOWER(JSON_VALUE(full_name, '$.ar')) LIKE ?", [$term])
                                ->orWhereRaw("LOWER(JSON_VALUE(full_name, '$.en')) LIKE ?", [$term]),
                        );
                });
            })
            // 2.d) فلترة حالة المستخدم
            ->when($this->filterState !== '', fn($q) => $q->where('user_state', $this->filterState))
            // 2.e) الترتيب
            ->orderBy($this->sortField, $this->sortDirection);

        // ─── 3) Pagination أو صفحة فارغة حتى loadUsers() ───
        if (!$this->readyToLoad) {
            return User::whereRaw('0 = 1')->paginate($this->perPage);
        }

        return $query->paginate($this->perPage);
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function loadUsers()
    {
        $this->readyToLoad = true;
    }

    // دوال فتح المودالات
    public function openAddUserModal()
    {
        $this->activeModal = 'addUser';
    }

    public function openStateEditModal($id)
    {
        // تحويل النموذج إلى مصفوفة لدعم two-way binding
        $this->selectedUser = User::findOrFail($id)->toArray();
        $this->activeModal = 'editState';
    }

    public function openDetailsModal($id)
    {
        $user = User::with(['employee.project', 'employee.department', 'employee.jobTitle'])->findOrFail($id);

        // تأكد من أن الصور مصفوفة فعلًا
        if (is_string($user->employee->id_card_image)) {
            $user->employee->id_card_image = json_decode($user->employee->id_card_image, true);
        }

        $this->selectedUser = $user;
        $this->activeModal = 'details';
    }

    public function openPrivilegesModal($id)
    {
        $user = User::with('employee')->findOrFail($id)->toArray();
        $user2 = User::with('employee')->findOrFail($id);

        // تأكد ان pages مصفوفة فعلاً
        if (is_string($user['pages'])) {
            $user['pages'] = json_decode($user['pages'], true);
        }

        // لو فاضي نحط قيم فاضية عشان الـ JS ما يضربش
        $user['pages']['page'] = $user['pages']['page'] ?? [];
        $user['pages']['element'] = $user['pages']['element'] ?? [];

        $this->selectedUser = $user;
        $this->selectedUser2 = $user2;

        $this->routes = SystemRoute::where('is_active', true)
            ->whereNotNull('order')
            ->orderBy('order')
            ->get()
            ->map(function ($route) {
                $properties = is_array($route->route_properties) ? $route->route_properties : (is_string($route->route_properties) ? json_decode($route->route_properties, true) : []);

                $elements = is_array($properties) && isset($properties['elements']) && is_array($properties['elements']) ? $properties['elements'] : [];

                return [
                    'name' => $route->name,
                    'display_name' => $route->displayName(),
                    'elements' => $elements,
                ];
            })
            ->toArray();

        $this->activeModal = 'privileges';
    }

    public function fetchRoutes()
    {
        return SystemRoute::where('is_active', true)
            ->whereNotNull('order')
            ->orderBy('order')
            ->get()
            ->map(function ($route) {
                $properties = is_array($route->route_properties) ? $route->route_properties : (is_string($route->route_properties) ? json_decode($route->route_properties, true) : []);

                $elements = is_array($properties) && isset($properties['elements']) && is_array($properties['elements']) ? $properties['elements'] : [];

                return [
                    'name' => $route->name,
                    'display_name' => $route->displayName(),
                    'elements' => $elements,
                ];
            })
            ->toArray();
    }

    public function sendEmail($id)
    {
        $user = User::findOrFail($id);
        // هنا ممكن تضيف كود إرسال الإيميل، مثال:
        // Mail::to($user->email)->send(new UserNotificationMail($user));
        $this->dispatch('notify', [
            'type' => 'success',
            'message' => __('backend.email_sent_successfully'),
        ]);
    }

    // دالة إغلاق المودال
    public function closeModal()
    {
        $this->activeModal = null;
    }

    // دوال الحفظ/التحديث
    public function storeUser()
    {
        try {
            $validated = validator($this->new, [
                'name' => [
                    'required',
                    'string',
                    'max:255',
                    function ($attribute, $value, $fail) {
                        if (str_word_count($value) < 3) {
                            $fail(__('validation.fullname_required_three_words'));
                        }
                    },
                ],
                'name_ar' => [
                    'required',
                    'string',
                    'max:255',
                    'regex:/^[\p{Arabic}\s]+$/u',
                    function ($attribute, $value, $fail) {
                        if (count(preg_split('/\s+/u', trim($value))) < 3) {
                            $fail(__('validation.fullname_required_three_words_ar'));
                        }
                    },
                ],
                'email' => ['required', 'string', 'lowercase', 'email', 'max:255', Rule::unique('users', 'email')],
                'password' => ['required', 'string', 'confirmed', Password::min(12)->mixedCase()->letters()->numbers()],
                'phone' => ['required', 'array', 'min:1'],
                'phone.*' => ['required', 'string', 'regex:/^(\+?20|0020)?1?\d{9}$/'],
                'project_code' => ['required', 'exists:users_projects,code'],
                'national_id' => [
                    'required',
                    'digits:14',
                    'unique:users_employees,national_id',
                    function ($attribute, $value, $fail) {
                        if (!preg_match('/^[23][0-9]{13}$/', $value)) {
                            return $fail(__('validation.national_id_format_invalid'));
                        }

                        $century = match (substr($value, 0, 1)) {
                            '2' => 1900,
                            '3' => 2000,
                            default => null,
                        };

                        $year = (int) substr($value, 1, 2);
                        $month = (int) substr($value, 3, 2);
                        $day = (int) substr($value, 5, 2);

                        if (!checkdate($month, $day, $century + $year)) {
                            return $fail(__('validation.national_id_birthdate_invalid'));
                        }

                        $govCode = substr($value, 7, 2);
                        $validGovCodes = ['01', '02', '03', '04', '11', '12', '13', '14', '15', '16', '17', '18', '19', '21', '22', '23', '24', '25', '26', '27', '28', '29', '31', '32', '33', '34', '35', '88'];

                        if (!in_array($govCode, $validGovCodes)) {
                            return $fail(__('validation.national_id_governorate_invalid'));
                        }
                        // ✅ تحقق النوع (ذكر/أنثى)
                        $genderDigit = substr($value, 12, 1);
                        if (!is_numeric($genderDigit)) {
                            return $fail(__('validation.national_id_gender_invalid'));
                        }
                    },
                ],
                'job_title_id' => ['required', 'exists:users_job_titles,id'],
                'user_state' => 'required|in:active,inactive,block,rejected,suspended',
            ])->validate();

            DB::transaction(function () use ($validated) {
                // توليد كود EMP-x
                $nextId = (UserEmployee::max('id') ?? 0) + 1;
                $employeeCode = 'EMP-' . str_pad((string) $nextId, 6, '0', STR_PAD_LEFT);
                while (UserEmployee::where('employee_code', $employeeCode)->exists()) {
                    $nextId++;
                    $employeeCode = 'EMP-' . str_pad((string) $nextId, 6, '0', STR_PAD_LEFT);
                }

                // تحليل الـ national_id
                $info = $this->parseNationalId($validated['national_id']);

                // إنشاء Employee
                $employee = UserEmployee::create([
                    'project_code' => $validated['project_code'],
                    'job_title_id' => $validated['job_title_id'],
                    'employee_code' => $employeeCode,
                    'full_name' => [
                        'en' => $validated['name'],
                        'ar' => $validated['name_ar'], // مفيش حقل name_ar في المودال → نستخدم نفس الاسم
                    ],
                    'phone' => json_encode($validated['phone']),
                    'national_id' => $validated['national_id'],
                    'birth_date' => $info['birth_date'],
                    'governorate' => $info['governorate'],
                    'gender' => $info['gender'],
                    'id_card_image' => json_encode([]),
                    'hiring_date' => now()->format('Y-m-d'),
                    'cv' => json_encode([], JSON_UNESCAPED_UNICODE),
                ]);

                // إنشاء User وربطه بالـ Employee
                User::create([
                    'name' => $validated['name'],
                    'email' => $validated['email'],
                    'password' => bcrypt($validated['password']),
                    'project_code' => $validated['project_code'],
                    'employee_id' => $employee->id,
                    'user_state' => $validated['user_state'],
                    'settings' => [
                        'dark_mode' => false,
                        'lang' => app()->getLocale(),
                        'is_locked' => false,
                    ],
                    'pages' => ['page' => ['dashboard', 'users', 'hr_dashboard']],
                ]);
            });

            $this->fetchUsers();
            $this->closeModal();
            $this->resetNewForm();

            $this->dispatch('notify', type: 'success', message: __('backend.user_added_successfully'));
        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('notify', type: 'error', message: $e->validator->errors()->first(), duration: 6000);
        } catch (\Throwable $e) {
            $this->dispatch('notify', type: 'error', message: __('backend.unexpected_error'), duration: 6000);
            $this->dispatch('notify', type: 'error', message: $e->getMessage(), duration: 10000);
        }
    }

    private function parseNationalId(string $id): array
    {
        $century = match (substr($id, 0, 1)) {
            '2' => 1900,
            '3' => 2000,
            default => null,
        };

        $year = (int) substr($id, 1, 2);
        $month = (int) substr($id, 3, 2);
        $day = (int) substr($id, 5, 2);

        $birthDate = $century ? sprintf('%04d-%02d-%02d', $century + $year, $month, $day) : null;

        $govCode = substr($id, 7, 2);
        $governorates = [
            '01' => ['ar' => 'القاهرة', 'en' => 'Cairo'],
            '02' => ['ar' => 'الإسكندرية', 'en' => 'Alexandria'],
            '03' => ['ar' => 'بورسعيد', 'en' => 'Port Said'],
            '04' => ['ar' => 'السويس', 'en' => 'Suez'],
            '11' => ['ar' => 'دمياط', 'en' => 'Damietta'],
            '12' => ['ar' => 'الدقهلية', 'en' => 'Dakahlia'],
            '13' => ['ar' => 'الشرقية', 'en' => 'Sharqia'],
            '14' => ['ar' => 'القليوبية', 'en' => 'Qalyubia'],
            '15' => ['ar' => 'كفر الشيخ', 'en' => 'Kafr El-Sheikh'],
            '16' => ['ar' => 'الغربية', 'en' => 'Gharbia'],
            '17' => ['ar' => 'المنوفية', 'en' => 'Monufia'],
            '18' => ['ar' => 'البحيرة', 'en' => 'Beheira'],
            '19' => ['ar' => 'الإسماعيلية', 'en' => 'Ismailia'],
            '21' => ['ar' => 'الجيزة', 'en' => 'Giza'],
            '22' => ['ar' => 'بني سويف', 'en' => 'Beni Suef'],
            '23' => ['ar' => 'الفيوم', 'en' => 'Faiyum'],
            '24' => ['ar' => 'المنيا', 'en' => 'Minya'],
            '25' => ['ar' => 'أسيوط', 'en' => 'Assiut'],
            '26' => ['ar' => 'سوهاج', 'en' => 'Sohag'],
            '27' => ['ar' => 'قنا', 'en' => 'Qena'],
            '28' => ['ar' => 'أسوان', 'en' => 'Aswan'],
            '29' => ['ar' => 'الأقصر', 'en' => 'Luxor'],
            '31' => ['ar' => 'البحر الأحمر', 'en' => 'Red Sea'],
            '32' => ['ar' => 'الوادي الجديد', 'en' => 'New Valley'],
            '33' => ['ar' => 'مطروح', 'en' => 'Matrouh'],
            '34' => ['ar' => 'شمال سيناء', 'en' => 'North Sinai'],
            '35' => ['ar' => 'جنوب سيناء', 'en' => 'South Sinai'],
            '88' => ['ar' => 'خارج الجمهورية', 'en' => 'Outside Egypt'],
        ];

        $governorate = [
            'code' => $govCode,
            'en' => $governorates[$govCode]['en'] ?? null,
            'ar' => $governorates[$govCode]['ar'] ?? null,
        ];

        $genderDigit = (int) substr($id, 12, 1);
        $gender = $genderDigit % 2 === 0 ? 'female' : 'male';

        return [
            'birth_date' => $birthDate,
            'governorate' => $governorate,
            'gender' => $gender,
        ];
    }

    public function updateUserState()
    {
        $validated = validator($this->selectedUser, [
            'id' => 'required|integer',
            'user_state' => 'required|in:active,inactive,block,rejected,suspended',
        ])->validate();

        $user = User::findOrFail($validated['id']);
        $user->update($validated);
        $this->fetchUsers();
        $this->closeModal();
        $this->dispatch('notify', type: 'success', message: __('backend.user_state_updated_successfully'));
    }

    public function updatePrivileges()
    {
        $validated = validator($this->selectedUser, [
            'id' => 'required|integer',
            'pages.page' => 'required|array',
            'pages.element' => 'present',
        ])->validate();

        $pagesData = $this->selectedUser['pages'];

        // إذا كان element من نوع stdClass، حافظ عليه
        if (isset($pagesData['element']) && $pagesData['element'] instanceof \stdClass) {
            // اجعل Laravel لا يحوّله إلى array عبر json_encode يدوياً
            $rawJson = json_encode($pagesData, JSON_UNESCAPED_UNICODE);
            User::where('id', $validated['id'])->update(['pages' => DB::raw("CAST('$rawJson' AS JSON)")]);
        } else {
            // الحالة العادية: ليس stdClass أو تريد array
            User::findOrFail($validated['id'])->update(['pages' => $pagesData]);
        }

        $this->fetchUsers();
        $this->closeModal();
        $this->dispatch('notify', type: 'success', message: __('backend.privileges_updated_successfully'));
    }

    public function render(): mixed
    {
        $users = $this->fetchUsers();

        return view('livewire.backend.user.page', [
            'users' => $users,
            'routes' => $this->routes,
            'selectedUser' => $this->selectedUser,
            'selectedUser2' => $this->selectedUser2,
        ]);
    }

    public function inlineUpdateUserState($id, $state)
    {
        $validated = validator(
            ['user_state' => $state],
            [
                'user_state' => 'required|in:active,inactive,block,rejected,suspended',
            ],
        )->validate();

        $user = User::findOrFail($id); // تأكد من أن المستخدم موجود
        $user->update(['user_state' => $validated['user_state']]); // تحديث الحالة

        // تحديث القائمة بعد التعديل
        $this->fetchUsers();

        // إرسال إشعار للمستخدم وتحديث الحالة في Alpine
        $this->dispatch('userStateUpdated', [
            'id' => $id,
            'newState' => $validated['user_state'],
            'newStateTrans' => __('backend.user_state_' . $validated['user_state']),
        ]);

        $this->dispatch('notify', type: 'success', message: __('backend.user_state_updated_successfully'));
    }

    public function inlineUpdateJobTitle($id, $jobKey)
    {
        $jobTitles = [
            'manager' => ['job_title_ar' => 'مدير', 'job_title_en' => 'Manager'],
            'supervisor' => ['job_title_ar' => 'مشرف', 'job_title_en' => 'Supervisor'],
            'staff' => ['job_title_ar' => 'موظف', 'job_title_en' => 'Staff'],
            'hr' => ['job_title_ar' => 'موظف شؤون الموارد البشرية', 'job_title_en' => 'HR Staff'],
        ];

        if (!$id || !isset($jobTitles[$jobKey])) {
            return;
        }

        $user = User::findOrFail($id);
        // تأكد إن عمود settings بيستقبل JSON بدعم الأحرف العربية (يفضل يكون collation utf8mb4_unicode_ci)
        $settings = $user->settings;
        $settings['job_title_ar'] = $jobTitles[$jobKey]['job_title_ar'];
        $settings['job_title_en'] = $jobTitles[$jobKey]['job_title_en'];
        $settings['job_title_key'] = $jobKey;
        $user->update(['settings' => $settings]);

        $this->fetchUsers();

        $this->dispatch('notify', type: 'success', message: __('backend.job_title_updated_successfully'));
    }

    public function updatedFilterState()
    {
        $this->fetchUsers();
    }

    public array $userStates = [
        'active' => 'backend.user_state_active',
        'inactive' => 'backend.user_state_inactive',
        'block' => 'backend.user_state_block',
        'rejected' => 'backend.user_state_rejected',
        'suspended' => 'backend.user_state_suspended',
    ];

    public array $userStateColors = [
        'active' => 'bg-green-500 text-white',
        'inactive' => 'bg-gray-500 text-white',
        'block' => 'bg-red-500 text-white',
        'rejected' => 'bg-purple-500 text-white',
        'suspended' => 'bg-yellow-500 text-white',
    ];

    public function resetNewForm()
    {
        $this->new = [
            'name' => '',
            'name_ar' => '',
            'email' => '',
            'password' => '',
            'settings' => [],
            'user_state' => 'active',
            'phone' => [''],
            'pages' => [],
            'project_code' => '',
            'job_title_id' => '',
            'national_id' => '',
        ];
    }

    public function confirmDeleteUser($id)
    {
        $user = User::with(['employee'])->findOrFail($id);

        // نجهز البيانات اللي هنعرضها في المودال
        $this->selectedUserForDelete = [
            'id' => $user->id,
            'name' => $user->employee?->full_name_translated ?? ($user->name ?? '-'),
            'email' => $user->email ?? '-',
        ];

        $this->activeModal = 'deleteUser';
    }

    public function deleteUserConfirmed()
    {
        $user = User::findOrFail($this->selectedUserForDelete['id']);

        $employee = $user->employee;

        // تحقق هل الموظف مدير لمشروع
        if ($employee) {
            $projectsCount = UserProject::where('manager_id', $employee->id)->count();

            if ($projectsCount > 0) {
                $this->dispatch('notify', type: 'error', message: __('backend.employee_is_manager_cannot_delete'));
                $this->selectedUserForDelete = null;
                $this->activeModal = null;
                return;
            }
        }

        // حذف اليوزر
        $user->delete();

        // لو فيه Employee مربوط → نحذفه
        if ($employee) {
            $employee->delete();
        }

        $this->selectedUserForDelete = null;
        $this->activeModal = null;

        $this->dispatch('notify', type: 'success', message: __('backend.user_deleted_successfully'));

        // تحديث الجدول بعد الحذف
        $this->fetchUsers();
    }

    public function openEditUserModal($id)
    {
        $user = User::with(['employee'])->findOrFail($id);

        $this->edit = [
            'id' => $user->id,
            'name' => $user->name,
            'name_ar' => $user->employee?->full_name['ar'] ?? '',
            'email' => $user->email,
            'password' => '', // اتركه فاضي
            'settings' => $user->settings ?? [],
            'user_state' => $user->user_state,
            'phone' => is_array($user->employee?->phone) ? $user->employee->phone : json_decode($user->employee->phone ?? '[]', true),
            'pages' => $user->pages ?? [],
            'project_code' => $user->project_code,
            'job_title_id' => $user->employee?->job_title_id ?? '',
            'national_id' => $user->employee?->national_id ?? '',
        ];

        $this->activeModal = 'editUser';
    }

    public function updateUser()
    {
        $validated = validator($this->edit, [
            'id' => 'required|integer|exists:users,id',
            'name' => [
                'required',
                'string',
                'max:255',
                function ($attribute, $value, $fail) {
                    if (str_word_count($value) < 3) {
                        $fail(__('validation.fullname_required_three_words'));
                    }
                },
            ],
            'name_ar' => [
                'required',
                'string',
                'max:255',
                'regex:/^[\p{Arabic}\s]+$/u',
                function ($attribute, $value, $fail) {
                    if (count(preg_split('/\s+/u', trim($value))) < 3) {
                        $fail(__('validation.fullname_required_three_words_ar'));
                    }
                },
            ],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', Rule::unique('users', 'email')->ignore($this->edit['id'])],
            'phone' => ['required', 'array', 'min:1'],
            'phone.*' => ['required', 'string', 'regex:/^(\+?20|0020)?1?\d{9}$/'],
            'project_code' => ['required', 'exists:users_projects,code'],
            'national_id' => ['required', 'digits:14', Rule::unique('users_employees', 'national_id')->ignore(User::findOrFail($this->edit['id'])->employee_id)],
            'job_title_id' => ['required', 'exists:users_job_titles,id'],
            'user_state' => 'required|in:active,inactive,block,rejected,suspended',
        ])->validate();

        DB::transaction(function () use ($validated) {
            $user = User::findOrFail($validated['id']);
            $employee = $user->employee;

            // تحديث User
            $user->name = $validated['name'];
            $user->email = $validated['email'];
            $user->project_code = $validated['project_code'];
            $user->user_state = $validated['user_state'];

            // ✅ لو كلمة السر اتكتب فيها حاجة → نعمل update
            if (!empty($this->edit['password'])) {
                $user->password = bcrypt($this->edit['password']);
            }

            $user->save();

            // تحديث Employee
            if ($employee) {
                $employee->project_code = $validated['project_code'];
                $employee->job_title_id = $validated['job_title_id'];
                $employee->full_name = [
                    'en' => $validated['name'],
                    'ar' => $validated['name_ar'],
                ];
                $employee->phone = json_encode($validated['phone']);
                $employee->national_id = $validated['national_id'];

                $employee->save();
            }
        });

        $this->fetchUsers();
        $this->closeModal();

        $this->dispatch('notify', type: 'success', message: __('backend.user_updated_successfully'));
    }

    public function openEditCvModal($id)
    {
        $user = User::with(['employee'])->findOrFail($id);

        $employee = $user->employee;

        $cv = is_array($employee->cv) ? $employee->cv : json_decode($employee->cv ?? '[]', true);

        $this->cv_profile_picture = null;
        $this->cv_experience_roads = $cv['work_experience']['roads'] ?? [];
        $this->cv_experience_airports = $cv['work_experience']['airports'] ?? [];
        $this->cv_skills = $cv['skills'] ?? [];
        $this->cv_profile_picture_path = isset($cv['profile_picture']) && $cv['profile_picture'] ? str_replace(['\\', '//'], '/', $cv['profile_picture']) : null;
        $this->cv_currentStep = 1;
        $this->selectedUserForCvEdit = $user->id; // ضروري لو هتعمل updateCV بعدين

        $this->activeModal = 'editUserCv';
    }

    public function nextCvStep()
    {
        if ($this->cv_currentStep < 3) {
            $this->cv_currentStep++;
        }
    }

    public function previousCvStep()
    {
        if ($this->cv_currentStep > 1) {
            $this->cv_currentStep--;
        }
    }

    public function updateCvFull()
    {
        $user = User::findOrFail($this->selectedUserForCvEdit);
        $employee = $user->employee;

        if ($this->cv_profile_picture && $this->cv_profile_picture->isValid()) {
            $slugName = Str::slug($user->name);
            $timestamp = now()->format('Y-m-d_H-i-s');
            $extension = $this->cv_profile_picture->getClientOriginalExtension();
            $filename = $slugName . '_cv_' . $timestamp . '.' . $extension;
            $directory = 'employees/' . $user->employee->employee_code;

            Storage::disk('public')->makeDirectory($directory);

            $photoPath = $this->cv_profile_picture->storeAs($directory, $filename, 'public');

            if ($photoPath) {
                $this->cv_profile_picture_path = $photoPath;
            }
        }

        if ($employee) {
            $employee->cv = json_encode(
                [
                    'work_experience' => [
                        'roads' => $this->cv_experience_roads,
                        'airports' => $this->cv_experience_airports,
                    ],
                    'skills' => $this->cv_skills,
                    // لو عندك تخزين للصورة تقدر تضيفها هنا
                    'profile_picture' => $this->cv_profile_picture_path,
                ],
                JSON_UNESCAPED_UNICODE,
            );

            $employee->save();
            $this->fetchUsers();
            $this->dispatch('notify', type: 'success', message: __('backend.cv_updated_successfully'));

            $this->closeModal();
        }
    }
};
?>


<div class="p-4 max-w-7xl mx-auto" dir="{{ app()->getLocale() === 'ar' ? 'rtl' : 'ltr' }}" wire:init="loadUsers">

    @access('users.index', 'user.search')
        <div class="mb-4 max-w-md">
            <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                {{ __('backend.search_by_code_or_name_or_email') }}
            </label>
            <div class="relative">
                <input type="text" wire:model.live="search"
                    class="w-full rounded-lg bg-white dark:bg-zinc-800 border border-zinc-300 dark:border-zinc-700 text-sm px-3 py-2"
                    placeholder="{{ __('backend.search_placeholder') }}" />
            </div>

        </div>
    @endaccess


    <!-- Header -->
    @access('users.index', 'user.add_new')
        <div class="flex flex-col sm:flex-row justify-between items-center mb-6">
            <h2 class="text-2xl font-bold text-zinc-800 dark:text-white mb-4 sm:mb-0">
                {{ __('backend.users_count') }}
            </h2>
            <button wire:click="openAddUserModal()"
                class="bg-indigo-600 hover:bg-indigo-700 text-white text-sm font-semibold px-6 py-2 rounded-md transition duration-300 transform hover:scale-105 shadow hover:shadow-lg">
                + {{ __('backend.add_new') }}
            </button>
        </div>
    @endaccess
    <!-- Users Table -->
    <div id="printable-area" class="overflow-x-auto rounded-lg border border-zinc-300 dark:border-zinc-700 shadow-sm">

        {{-- Header for print --}}
        <div class="hidden print:block fixed top-0 w-full text-center mb-6">
            <img src="{{ asset('assets/img/logo.png') }}" class="h-20 mb-2" alt="Company Logo">
            <h1 class="text-2xl font-bold text-zinc-800 dark:text-zinc-100">
                {{ __('Employees List') }}
            </h1>
        </div>
        <!-- فقط على الشاشات الصغيرة -->
        <div class="block sm:hidden mb-4 px-4">

            <select id="perPageMobile" wire:model.live="perPage"
                class="w-full text-base rounded-lg bg-gray-50 border border-gray-300
         focus:ring-2 focus:ring-indigo-200 focus:border-indigo-500
         dark:bg-zinc-800 dark:border-zinc-600 dark:text-zinc-100 transition">
                <option value="5">5</option>
                <option value="10">10</option>
                <option value="25">25</option>
                <option value="50">50</option>
                <option value="100">100</option>
            </select>
        </div>
        {{ $users->links() }}
        <table
            class="min-w-full table-auto divide-y divide-gray-300 dark:divide-zinc-700 text-sm
           print:border print:border-gray-400 print:border-collapse">

            <thead
                class="sticky top-0 z-20
         bg-gray-200 dark:bg-zinc-900
         border-b border-gray-200 dark:border-zinc-700
         print:divide-y print:divide-gray-400">
                <tr class="text-center print:text-center">
                    @access('users.index', 'user.name')
                        <th wire:click="sortBy('name')"
                            class="px-6 py-3 cursor-pointer whitespace-nowrap
                        text-sm sm:text-base lg:text-lg font-medium uppercase tracking-wide
                        text-gray-800 dark:text-zinc-200
                        hover:bg-gray-50 dark:hover:bg-zinc-800
                        transition-colors print:border print:border-gray-400">
                            {{ __('backend.name') }}</th>
                    @endaccess
                    @access('users.index', 'user.email')
                        <th wire:click="sortBy('email')"
                            class="px-6 py-3 cursor-pointer whitespace-nowrap
                        text-sm sm:text-base lg:text-lg font-medium uppercase tracking-wide
                        text-gray-800 dark:text-zinc-200
                        hover:bg-gray-50 dark:hover:bg-zinc-800
                        transition-colors print:border print:border-gray-400">
                            {{ __('frontend.emailadress') }}</th>
                    @endaccess
                    @access('users.index', 'user.status')
                        <th
                            class="px-6 py-3 cursor-pointer whitespace-nowrap
                        text-sm sm:text-base lg:text-lg font-medium uppercase tracking-wide
                        text-gray-800 dark:text-zinc-200
                        hover:bg-gray-50 dark:hover:bg-zinc-800
                        transition-colors print:border print:border-gray-400">
                            <select wire:model.lazy="filterState"
                                class="ml-2 px-4 py-2 border rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-indigo-500 bg-white dark:bg-zinc-800 dark:text-white text-sm">
                                <option value="">{{ __('backend.all') }}</option>
                                @foreach ($userStates as $value => $label)
                                    <option value="{{ $value }}">{{ __($label) }}</option>
                                @endforeach
                            </select>
                        </th>
                    @endaccess
                    <th class="hidden sm:table-cell print:hidden px-6 py-3 text-center w-[200px]">
                        <select id="perPage" wire:model.live="perPage"
                            class="w-full text-base lg:text-lg rounded-lg bg-gray-50 border border-gray-300
                          focus:ring-2 focus:ring-indigo-200 focus:border-indigo-500
                          dark:bg-zinc-800 dark:border-zinc-600 dark:text-zinc-100 transition">
                            <option value="5">5</option>
                            <option value="10">10</option>
                            <option value="25">25</option>
                            <option value="50">50</option>
                            <option value="100">100</option>
                        </select>
                    </th>
                </tr>
            </thead>
            <tbody
                class="divide-y divide-gray-100 dark:divide-zinc-700 text-zinc-900 dark:text-zinc-200
              print:divide-y print:divide-gray-300">
                @foreach ($users as $user)
                    <tr class="hover:bg-zinc-50 dark:hover:bg-zinc-800 transition text-center ">
                        <!-- Name -->
                        @access('users.index', 'user.name')
                            <td data-label="{{ __('backend.name') }}" class="px-4 py-2 print:border print:border-gray-400">
                                <div class="flex items-center gap-2">
                                    @php
                                        // مسار الإفاتار الخاص بكل user في الحلقة
                                        $avatarPath = $user->settings['avatar'] ?? null;
                                        $avatarUrl = $avatarPath ? asset('storage/' . $avatarPath) : null;
                                        // حرفين من اسم الـ user
                                        $initials = strtoupper(\Illuminate\Support\Str::substr($user->name, 0, 2));
                                    @endphp

                                    {{-- إذا كان عنده صورة --}}
                                    @if ($avatarUrl)
                                        <img src="{{ $avatarUrl }}" alt="Avatar"
                                            class="w-8 h-8 rounded-full object-cover ring-2 ring-indigo-300 dark:ring-indigo-700" />
                                    @else
                                        {{-- عرض الحروف الأولى --}}
                                        <div
                                            class="w-8 h-8 flex items-center justify-center
                                                rounded-full bg-indigo-500 text-white font-bold uppercase
                                                ring-2 ring-indigo-300 dark:ring-indigo-700">
                                            {{ $initials }}
                                        </div>
                                    @endif

                                    {{-- اسم الموظف --}}
                                    <span class="font-medium text-zinc-800 dark:text-white text-sm">
                                        {{ $user->employee?->full_name_translated ?? '-' }}
                                    </span>
                                </div>
                            </td>
                        @endaccess
                        @access('users.index', 'user.email')
                            <!-- Email -->
                            <td data-label="{{ __('frontend.emailadress') }}"
                                class="px-4 py-4 print:border print:border-gray-400">
                                {{ $user->email }}</td>
                        @endaccess
                        @access('users.index', 'user.status')
                            <!-- Account State with inline edit possibility -->
                            <td data-label="{{ __('backend.status') }}" class="px-4 py-2 print:border print:border-gray-400">
                                <div x-data="{ edit: false }" x-init="window.addEventListener('userStateUpdated', event => {
                                    if (event.detail.id === {{ $user->id }}) {
                                        $refs.stateText.textContent = event.detail.newStateTrans;
                                    }
                                });" class="inline-block">

                                    <template x-if="!edit">
                                        @php
                                            $stateColor =
                                                $userStateColors[$user->user_state] ?? 'bg-blue-500 text-white';
                                        @endphp

                                        <button @click="edit = true"
                                            class="inline-flex items-center px-3 py-1 rounded-full text-xs font-semibold transition duration-300 transform hover:scale-105 {{ $stateColor }}">
                                            <span
                                                x-ref="stateText">{{ __('backend.user_state_' . $user->user_state) }}</span>
                                        </button>
                                    </template>

                                    <template x-if="edit">
                                        <select
                                            @change="$nextTick(() => { $wire.inlineUpdateUserState({{ $user->id }}, $event.target.value); edit = false; })"
                                            @blur="edit = false"
                                            class="px-2 py-1 border rounded text-xs focus:outline-none bg-gray-500 text-white">
                                            <option value="">{{ __('backend.choosestate') }}</option>
                                            @foreach ($userStates as $value => $label)
                                                <option value="{{ $value }}">{{ __($label) }}</option>
                                            @endforeach
                                        </select>
                                    </template>

                                </div>


                            </td>
                        @endaccess
                        @php
                            $isRtl = app()->getLocale() !== 'ar';
                            $empPerms = Auth::user()->settings['element']['emp.index'] ?? [];
                            $pages = Auth::user()->pages['page'] ?? [];
                        @endphp
                        @if (in_array('[all]', $pages) ||
                                in_array('user.details', $empPerms) ||
                                in_array('user.edit', $empPerms) ||
                                in_array('user.privileges', $empPerms) ||
                                in_array('user.edit_cv', $empPerms) ||
                                in_array('user.delete', $empPerms))
                            <td class="print:hidden px-4 py-2">
                                <div x-data="{ open: false }" class="relative inline-block">
                                    <!-- الزر الرئيسي -->
                                    <button @click="open = !open"
                                        class="relative p-3 rounded-full
                                       bg-gradient-to-r from-sky-500 to-indigo-600
                                       text-white shadow-lg hover:shadow-2xl
                                       focus:outline-none focus:ring-4 focus:ring-sky-300
                                       transition-transform duration-300 ease-in-out transform"
                                        :class="{ 'rotate-90 scale-110': open }">
                                        <i class="ri-more-2-fill text-2xl"></i>
                                        <span
                                            class="absolute -top-1 -right-1 block w-2 h-2
                                         bg-red-500 rounded-full animate-ping"></span>
                                    </button>

                                    <!-- قائمة العناصر -->
                                    <div x-show="open" x-cloak @click.away="open = false"
                                        x-transition:enter="transition ease-out duration-100"
                                        x-transition:enter-start="opacity-0 scale-95"
                                        x-transition:enter-end="opacity-100 scale-100"
                                        x-transition:leave="transition ease-in duration-75"
                                        x-transition:leave-start="opacity-100 scale-100"
                                        x-transition:leave-end="opacity-0 scale-95"
                                        class="absolute top-full mt-2
                                       {{ $isRtl ? 'right-0 origin-top-right' : 'left-0 origin-top-left' }}
                                       bg-white dark:bg-zinc-900
                                       divide-y divide-gray-200 dark:divide-zinc-700
                                       rounded-lg shadow-lg ring-1 ring-black ring-opacity-5 z-20
                                       w-40 sm:w-56">
                                        <!-- التفاصيل -->
                                        @access('users.index', 'user.details')
                                            <button wire:click="openDetailsModal({{ $user->id }})"
                                                class="flex items-center gap-2 w-full px-4 py-3
                                         hover:bg-gray-100 dark:hover:bg-zinc-800
                                             border-l-4 border-yellow-500">
                                                <i class="ri-file-list-line text-yellow-500 text-xl"></i>
                                                📋 {{ __('backend.details') }}
                                            </button>
                                        @endaccess
                                        @access('users.index', 'user.edit')
                                            <button wire:click="openEditUserModal({{ $user->id }})"
                                                class="flex items-center gap-2 w-full px-4 py-3
                                                 hover:bg-gray-100 dark:hover:bg-zinc-800
                                                 border-l-4 border-orange-500">
                                                <i class="ri-edit-line text-orange-500 text-xl"></i>
                                                ✏️ {{ __('backend.edit') }}
                                            </button>
                                        @endaccess
                                        @access('users.index', 'user.privileges')
                                            <button wire:click="openPrivilegesModal({{ $user->id }})"
                                                class="flex items-center gap-2 w-full px-4 py-3
                                                 hover:bg-gray-100 dark:hover:bg-zinc-800
                                                 border-l-4 border-sky-500">
                                                <i class="ri-refresh-line text-sky-500 text-xl"></i>
                                                🔄 {{ __('backend.privileges') }}
                                            </button>
                                        @endaccess
                                        @access('users.index', 'user.edit_cv')
                                            <button wire:click="openEditCvModal({{ $user->id }})"
                                                class="flex items-center gap-2 w-full px-4 py-3
                                        hover:bg-gray-100 dark:hover:bg-zinc-800
                                        border-l-4 border-emerald-500">
                                                <i class="ri-folder-shield-line text-emerald-500 text-xl"></i>
                                                🛞 {{ __('backend.edit_cv') }}
                                            </button>
                                        @endaccess
                                        @access('users.index', 'user.delete')
                                            <button wire:click="confirmDeleteUser({{ $user->id }})"
                                                class="flex items-center gap-2 w-full px-4 py-3
                                                 hover:bg-gray-100 dark:hover:bg-zinc-800
                                                 border-l-4 border-red-500">
                                                <i class="ri-calendar-event-line text-red-500 text-xl"></i>
                                                ❌ {{ __('backend.delete') }}
                                            </button>
                                        @endaccess
                                    </div>
                                </div>
                            </td>
                        @endif
                        <!-- Actions -->

                    </tr>
                @endforeach
            </tbody>

        </table>
        {{ $users->links() }}
    </div>

    {{--
        هنا ممكن تضيف مودالات الإضافة والتعديل والتفاصيل والصلاحيات
        بناءً على قيمة $activeModal، مثال:

        @if ($activeModal === 'addUser')
            @include('livewire.backend.user.modals.add')
        @endif

        @if ($activeModal === 'editState')
            @include('livewire.backend.user.modals.edit-state')
        @endif
    --}}
    @if ($activeModal === 'addUser')
        @include('livewire.backend.user.modals.add-new')
    @endif

    @if ($activeModal === 'editUser')
        @include('livewire.backend.user.modals.edit')
    @endif

    @if ($activeModal === 'editUserCv')
        @include('livewire.backend.user.modals.edit-cv')
    @endif

    @if ($activeModal === 'details')
        @include('livewire.backend.user.modals.details')
    @endif

    @if ($activeModal === 'privileges')
        @include('livewire.backend.user.modals.privileges')
    @endif
    @if ($activeModal === 'deleteUser')
        @include('livewire.backend.user.modals.delete') {{-- أو لو هتحطه في نفس الصفحة ممكن تحط المودال هنا --}}
    @endif



</div>
