<?php

use App\Models\User\UserProject;
use App\Models\User\UserEmployee;
use Livewire\Volt\Component;
use Livewire\Attributes\Layout;
use Illuminate\View\View;
use Livewire\WithPagination;

new #[Layout('components.backend.layouts.struct')] class extends Component {
    use WithPagination;
    public string $projectSearch = '';
    public array $suggestions = [];
    public string $sortField = 'code';
    public string $sortDirection = 'asc';
    public int $perPage = 10;
    public bool $readyToLoad = true;
    public bool $canViewAll;

    public $new = [
        'code' => '',
        'name' => '',
        'description' => '',
        'manager_id' => null,
    ];
    public $edit = [
        'id' => null,
        'code' => '',
        'name' => '',
        'description' => '',
        'manager_id' => null,
    ];

    public $editMode = false;

    public $employees = [];
    public $activeModal = null;
    public $selectedProject = null;
    public function mount()
    {
        $lang = app()->getLocale();

        $this->employees = UserEmployee::get()->filter(fn($emp) => filled($emp->full_name_translated))->sortBy('full_name_translated')->values();

        // 1) الصفحات المسموحة (من عمود pages)
        $pagesData = auth()->user()->pages['page'] ?? [];

        // 2) العناصر المسموحة ضمن project.index
        $elementsData = auth()->user()->pages['element']['project.index'] ?? [];

        // 3) هل هو مدير عام ([all])؟
        $this->isSuperAdmin = in_array('[all]', $pagesData);

        // 4) يمكنه عرض كل المشاريع إذا كان مدير عام، أو إذا عنده صلاحية project.index + project.view_all
        $this->canViewAll = $this->isSuperAdmin || (in_array('project.index', $pagesData) && in_array('project.view_all', $elementsData));
    }

    public function rendering(View $view): void
    {
        $view->title(__('backend.department_name') . ' | ' . __('backend.projects'));
    }

    public function updatingProjectSearch()
    {
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
        $this->readyToLoad = true;
    }

    public function getProjects($paginate = true)
    {
        $query = UserProject::with('manager')->withCount(['employees', 'users']);

        // 🔒 لو ما ينفعش يشوف كل المشاريع، اقصرها على مشروعه فقط
        if (!$this->canViewAll) {
            $query->where('code', auth()->user()->project_code);
        }

        // بحث حر
        if ($this->projectSearch) {
            $term = '%' . strtolower($this->projectSearch) . '%';
            $query->where(function ($q) use ($term) {
                $q->whereRaw('LOWER(code) LIKE ?', [$term])
                    ->orWhereRaw('LOWER(name) LIKE ?', [$term])
                    ->orWhereHas('manager', fn($q2) => $q2->whereRaw("LOWER(JSON_VALUE(full_name,'$.ar')) LIKE ?", [$term])->orWhereRaw("LOWER(JSON_VALUE(full_name,'$.en')) LIKE ?", [$term]));
            });
        }

        return $paginate ? $query->orderBy($this->sortField, $this->sortDirection)->paginate($this->perPage) : $query->orderBy($this->sortField, $this->sortDirection)->get();
    }

    public function updatedProjectSearch($value)
    {
        $this->resetPage();
        $this->readyToLoad = true;

        if (!trim($value)) {
            $this->suggestions = [];
            return;
        }

        $term = '%' . strtolower($value) . '%';
        $base = UserProject::query()->leftJoin('users_employees', 'users_projects.manager_id', 'users_employees.id');

        // 🔒 نفس الفلتر هنا
        if (!$this->canViewAll) {
            $base->where('users_projects.code', auth()->user()->project_code);
        }

        $rows = $base
            ->where(function ($q) use ($term) {
                $q->whereRaw('LOWER(users_projects.code) LIKE ?', [$term])
                    ->orWhereRaw('LOWER(users_projects.name) LIKE ?', [$term])
                    ->orWhereRaw("LOWER(JSON_VALUE(users_employees.full_name,'$.ar')) LIKE ?", [$term]);
            })
            ->selectRaw(
                "DISTINCT TOP 10 users_projects.code AS code,
                     users_projects.name AS name,
                     JSON_VALUE(users_employees.full_name,'$.ar') AS manager",
            )
            ->get();

        $this->suggestions = $rows->flatMap(fn($r) => collect([$r->code, $r->name, $r->manager]))->unique()->values()->toArray();
    }

    public function openAddProjectModal()
    {
        $this->activeModal = 'addProject';
    }
    public function closeModal()
    {
        $this->activeModal = null;
    }

    public function storeProject()
    {
        if (UserProject::where('code', $this->new['code'])->exists()) {
            $this->dispatch('notify', type: 'error', message: __('backend.project_code_already_exists'));
            return;
        }
        if (UserProject::where('name', $this->new['name'])->exists()) {
            $this->dispatch('notify', type: 'error', message: __('backend.project_name_already_exists'));
            return;
        }

        $validated = validator($this->new, [
            'code' => 'required|string|unique:users_projects,code',
            'name' => 'required|string',
            'manager_id' => 'nullable|exists:users_employees,id',
            'description' => 'nullable|string',
        ])->validate();

        UserProject::create($validated);
        // Reset النموذج بعد الإضافة
        $this->new = [
            'code' => '',
            'name' => '',
            'description' => '',
            'manager_id' => null,
        ];

        // إغلاق المودال بعد الإضافة
        $this->closeModal();
        $this->dispatch('notify', type: 'success', message: __('backend.project_added_successfully'));
    }
    public function editProject($id)
    {
        $project = UserProject::with('manager')->findOrFail($id);
        $this->edit = $project->toArray();
        $this->activeModal = 'openEditModal';
    }

    public function updateProject()
    {
        if (UserProject::where('code', $this->edit['code'])->where('id', '!=', $this->edit['id'])->exists()) {
            $this->dispatch('notify', type: 'error', message: __('backend.project_code_already_exists'));
            return;
        }

        // التحقق من تكرار الاسم لمشروع مختلف
        if (UserProject::where('name', $this->edit['name'])->where('id', '!=', $this->edit['id'])->exists()) {
            $this->dispatch('notify', type: 'error', message: __('backend.project_name_already_exists'));
            return;
        }
        $validated = validator($this->edit, [
            'id' => 'required|exists:users_projects,id',
            'code' => 'required|string|unique:users_projects,code,' . $this->edit['id'],
            'name' => 'required|string',
            'manager_id' => 'nullable|exists:users_employees,id',
            'description' => 'nullable|string',
        ])->validate();

        UserProject::find($validated['id'])->update($validated);

        $this->new = [
            'code' => '',
            'name' => '',
            'description' => '',
            'manager_id' => null,
        ];

        // إغلاق المودال بعد الإضافة
        $this->closeModal();
        $this->dispatch('notify', type: 'success', message: __('backend.project_updated_successfully'));
    }

    public function confirmDeleteProject($id)
    {
        $this->selectedProject = UserProject::findOrFail($id)->toArray();
        $this->activeModal = 'deleteProject';
    }

    public function deleteConfirmed()
    {
        $project = UserProject::withCount(['employees', 'users'])->findOrFail($this->selectedProject['id']);

        if ($project->employees_count > 0 || $project->users_count > 0) {
            $this->dispatch('notify', type: 'error', message: __('backend.project_has_related_records_cannot_delete'));
            $this->selectedProject = null;
            $this->activeModal = null;
            return;
        }

        $project->delete();
        $this->selectedProject = null;
        $this->activeModal = null;
        $this->dispatch('notify', type: 'success', message: __('backend.project_deleted_successfully'));
    }

    public function render(): mixed
    {
        $projects = $this->readyToLoad ? $this->getProjects() : collect();

        return view('livewire.backend.project.page', [
            'projects' => $this->projects,
        ]);
    }

    public function getProjectsProperty()
    {
        $term = '%' . strtolower($this->projectSearch) . '%';
        $query = UserProject::with('manager')->withCount(['employees', 'users']);

        // 🔒 تطبيق الفلتر هنا أيضًا
        if (!$this->canViewAll) {
            $query->where('code', auth()->user()->project_code);
        }

        return $query
            ->where(function ($q) use ($term) {
                $q->whereRaw('LOWER(code) LIKE ?', [$term])
                    ->orWhereRaw('LOWER(name) LIKE ?', [$term])
                    ->orWhereHas('manager', fn($q2) => $q2->whereRaw("LOWER(JSON_VALUE(full_name,'$.ar')) LIKE ?", [$term])->orWhereRaw("LOWER(JSON_VALUE(full_name,'$.en')) LIKE ?", [$term]));
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);
    }

    public function selectSuggestion($value)
    {
        $this->projectSearch = $value;
        $this->suggestions = [];
    }

    // public array $elements = [
    //     'project.search' => 'livewire.backend.project.page-partials.search',
    //     'project.add_project' => 'livewire.backend.project.page-partials.add_project',
    //     'project.table' => 'livewire.backend.project.page-partials.table',
    // ];

    public array $elements = [
        ['permission' => 'project.search', 'partial' => 'livewire.backend.project.page-partials.search'],
        ['permission' => 'project.add_button', 'partial' => 'livewire.backend.project.page-partials.add_project']];

    public array $columns = [
        ['permission' => 'project.project_code', 'label' => 'backend.project_code', 'field' => 'code'],
        ['permission' => 'project.project_name', 'label' => 'backend.project_name', 'field' => 'name'],
        ['permission' => 'project.project_manager', 'label' => 'backend.manager', 'field' => 'manager'],
        ['permission' => 'project.employees_count', 'label' => 'backend.employees_count', 'field' => 'employees_count'],
        ['permission' => 'project.users_count', 'label' => 'backend.users_count', 'field' => 'users_count']];
};
?>

<div class="p-4 max-w-5xl mx-auto">
    <h2 class="text-xl font-bold mb-4 text-zinc-800 dark:text-white">
        {{ __('backend.projects') }}
    </h2>




    @foreach ($elements as $el)
        @access('project.index', $el['permission'])
            @include($el['partial'])
        @endaccess
    @endforeach

    <div class="bg-white dark:bg-zinc-900 shadow rounded-lg">
        <div class="mt-4">
            {{ $projects->links() }}
        </div>
        <div id="printable-area" class="overflow-x-auto rounded-lg border border-zinc-300 dark:border-zinc-700 shadow-sm">

            {{-- Header for print --}}
            <div class="hidden print:block fixed top-0 w-full text-center mb-6">
                <img src="{{ asset('assets/img/logo.png') }}" class="h-20 mb-2" alt="Company Logo">
                <h1 class="text-2xl font-bold text-zinc-800 dark:text-zinc-100">
                    {{ __('Employees List') }}
                </h1>
            </div>
            <!-- فقط على الشاشات الصغيرة -->
            <div class="block sm:hidden mb-4 px-4">

                <select id="perPageMobile" wire:model.live="perPage"
                    class="w-full text-base rounded-lg bg-gray-50 border border-gray-300
             focus:ring-2 focus:ring-indigo-200 focus:border-indigo-500
             dark:bg-zinc-800 dark:border-zinc-600 dark:text-zinc-100 transition">
                    <option value="5">5</option>
                    <option value="10">10</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
            </div>

            <table
                class="min-w-full table-auto divide-y divide-gray-300 dark:divide-zinc-700 text-sm
                    print:border print:border-gray-400 print:border-collapse">

                <thead
                    class="sticky top-0 z-20
                  bg-gray-200 dark:bg-zinc-900
                  border-b border-gray-200 dark:border-zinc-700
                  print:divide-y print:divide-gray-400">
                    <tr class="text-center print:text-center">
                        @foreach ($columns as $col)
                            @access('project.index', $col['permission'])
                                <th
                                    class="px-6 py-3 cursor-pointer whitespace-nowrap
                        text-sm sm:text-base lg:text-lg font-medium uppercase tracking-wide
                        text-gray-800 dark:text-zinc-200
                        hover:bg-gray-50 dark:hover:bg-zinc-800
                        transition-colors print:border print:border-gray-400">
                                    {{ __($col['label']) }}
                                </th>
                            @endaccess
                        @endforeach
                        <th class="hidden sm:table-cell print:hidden px-6 py-3 text-center w-[200px]">
                            <select wire:model.live="perPage"
                                class="w-full rounded-lg bg-white dark:bg-zinc-800 border border-zinc-300 dark:border-zinc-700 text-sm px-3 py-2">
                                <option value="5">5</option>
                                <option value="10">10</option>
                                <option value="25">25</option>
                                <option value="50">50</option>
                            </select>
                        </th>
                    </tr>
                </thead>
                <tbody
                    class="divide-y divide-gray-100 dark:divide-zinc-700 text-zinc-900 dark:text-zinc-200
                  print:divide-y print:divide-gray-300">
                    @foreach ($projects as $project)
                        <tr class="hover:bg-zinc-50 dark:hover:bg-zinc-800 transition text-center ">
                            @foreach ($columns as $col)
                                @access('project.index', $col['permission'])
                                    <td data-label="{{ __($col['label']) }}"
                                        class="px-4 py-2 print:border print:border-gray-400">
                                        @if ($col['field'] === 'manager')
                                            {{ $project->manager && is_array($project->manager->full_name)
                                                ? $project->manager->full_name[app()->getLocale()] ?? reset($project->manager->full_name)
                                                : '-' }}
                                        @else
                                            {{ $project->{$col['field']} }}
                                        @endif
                                    </td>
                                @endaccess
                            @endforeach
                            <td class="px-4 py-2 whitespace-nowrap text-end space-x-2">
                                @access('project.index', 'project.actions.edit')
                                    <button wire:click="editProject({{ $project->id }})"
                                        class="text-xs bg-yellow-500 hover:bg-yellow-600 text-white px-3 py-1 rounded shadow">
                                        {{ __('backend.edit') }}
                                    </button>
                                @endaccess
                                @access('project.index', 'project.actions.delete')
                                    <button wire:click="confirmDeleteProject({{ $project->id }})"
                                        class="text-xs bg-red-600 hover:bg-red-700 text-white px-3 py-1 rounded shadow">
                                        {{ __('backend.delete') }}
                                    </button>
                                @endaccess
                            </td>
                        </tr>
                    @endforeach
                </tbody>
            </table>
        </div>


    </div>



    <div class="mt-4">
        {{ $projects->links() }}
    </div>
    @access('project.index', 'project.add_button')
        @if ($activeModal === 'addProject')
            @include('livewire.backend.project.modals.add')
        @endif
    @endaccess
    @access('project.index', 'project.actions.edit')
        @if ($activeModal === 'deleteProject')
            @include('livewire.backend.project.modals.delete')
        @endif
    @endaccess
    @access('project.index', 'project.actions.delete')
        @if ($activeModal === 'openEditModal')
            @include('livewire.backend.project.modals.edit')
        @endif
    @endaccess



</div>
