<?php

use Livewire\Volt\Component;
use Livewire\Attributes\Layout;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\View\View;
use App\Models\Emp\UserEmployeeCvCode;
use PhpOffice\PhpWord\PhpWord;
use PhpOffice\PhpWord\IOFactory;
use PhpOffice\PhpWord\SimpleType\TblWidth;
use PhpOffice\PhpWord\SimpleType\JcTable;
use PhpOffice\PhpWord\SimpleType\Jc;

new #[Layout('components.backend.layouts.struct')] class extends Component {
    use WithPagination;

    /* ▼ حالة الواجهة */
    public string $search = '';
    public string $sortField = 'code';
    public string $sortDirection = 'asc';
    public int $perPage = 10;
    public bool $readyToLoad = true;
    public ?array $activeModal = null; // ['add' | 'edit' | 'delete', id?]
    public int $currentStep = 1;
    public array $newNames = [];
    public array $newPairs = [];
    public bool $canViewAll = false;
    public array $projects = [];
    /* ▼ فورمات */
    public array $new = [
        'project_code' => null, // ➊ أضف هذا
        'code' => '',
        'name' => '',
        'used' => false,
    ];
    public array $edit = [
        'project_code' => null, // ➊ أضف هذا
        'code' => '',
        'name' => '',
        'used' => false,
    ];

    public function mount(): void
    {
        $this->edit['project_code'] = Auth::user()->project_code;
        /* صلاحيات المشاهدة */
        $pagesData = Auth::user()->pages['page'] ?? [];
        $elementsData = Auth::user()->pages['element']['cv.genrate'] ?? [];
        $isSuperAdmin = in_array('[all]', $pagesData);

        $this->canViewAll = $isSuperAdmin || (in_array('cv.genrate', $pagesData) && in_array('cv.view_all', $elementsData));

        // إذا كان يُسمح له برؤية كلّ المشاريع اجلب القائمة
        if ($this->canViewAll) {
            $this->projects = \App\Models\User\UserProject::select('code', 'name')->orderBy('name')->get()->toArray();
        }

        // وإلّا ثبّت مشروعه الحالى
        else {
            $this->new['project_code'] = Auth::user()->project_code;
        }
    }
    /* ▼ إنشاء الكود التالى */
    private function nextCode(): string
    {
        do {
            // هنا نولّد 8 خانات الأحرف والأرقام (يمكن تعديل الطول حسب الحاجة)
            $code = Str::upper(Str::random(8));
        } while (UserEmployeeCvCode::where('code', $code)->exists());

        return $code;
    }

    /* ▌Lifecycle ▌ */
    public function rendering(View $view): void
    {
        $view->title(__('backend.cv_codes'));
    }

    /* ▌جلب البيانات ▌ */
    public function getCodesProperty()
    {
        $pagesData = Auth::user()->pages['page'] ?? [];
        $elementsData = Auth::user()->pages['element']['cv.genrate'] ?? [];
        $isSuperAdmin = in_array('[all]', $pagesData);

        $canViewAll = $isSuperAdmin || (in_array('cv.genrate', $pagesData) && in_array('cv.view_all', $elementsData));
        $query = UserEmployeeCvCode::query()
            ->when($this->search, fn($q) => $q->where('code', 'like', "%{$this->search}%")->orWhere('name', 'like', "%{$this->search}%"))
            ->when(!$canViewAll, fn($q) => $q->where('project_code', Auth::user()->project_code))
            ->orderBy($this->sortField, $this->sortDirection);

        return $this->readyToLoad ? $query->paginate($this->perPage) : collect();
    }

    /* ▌تبديل الفرز ▌ */
    public function sortBy(string $field): void
    {
        $this->sortDirection = $this->sortField === $field && $this->sortDirection === 'asc' ? 'desc' : 'asc';
        $this->sortField = $field;
    }

    /* ▌Modals ▌ */
    public function openAddModal(): void
    {
        $this->newNames = [];
        $this->newPairs = [];
        $this->currentStep = 1;
        $this->activeModal = ['add'];
    }

    public function openEditModal(int $id): void
    {
        $row = UserEmployeeCvCode::findOrFail($id);

        // ‼️ ➊ إذا لم يملك المستخدم صلاحية view-all وتأكّد أن الكود يخصّ مشروعه
        if (!$this->canViewAll && $row->project_code !== Auth::user()->project_code) {
            $this->dispatch('notify', type: 'error', message: __('backend.not_authorized'), duration: 4000);
            return;
        }

        $this->edit = $row->only(['id', 'project_code', 'code', 'name', 'used']);

        // ➋ إن كان يملك view-all حمِّل قائمة المشاريع (لا داعى لو كانت محمّلة سابقًا)
        if ($this->canViewAll && empty($this->projects)) {
            $this->projects = \App\Models\User\UserProject::select('code', 'name')->orderBy('name')->get()->toArray();
        }

        $this->activeModal = ['edit', $id];
    }

    public function openDeleteModal(int $id): void
    {
        $this->activeModal = ['delete', $id];
    }

    public function closeModal(): void
    {
        $this->activeModal = null;
    }

    public function downloadCvCodes()
    {
        $fileName = 'cv_codes_' . now()->format('Ymd_His') . '.docx';

        return response()->streamDownload(function () {
            // 1 سم = 567 twip
            $cm = 567;

            // أبعاد البطاقة
            $cellWidthTwip = 9 * $cm; // عرض 9 سم
            $rowHeightTwip = 6 * $cm; // ارتفاع 6 سم

            // أبعاد الشعار
            $logoWidthTwip = 200; // عرض 5 سم
            $logoHeightTwip = 60; // ارتفاع 2 سم

            // 1. تهيئة PhpWord وإضافة Section A4 بهوامش 1.5 سم
            $phpWord = new PhpWord();
            $section = $phpWord->addSection([
                'size' => 'A4',
                'marginTop' => 1.5 * $cm,
                'marginBottom' => 1.5 * $cm,
                'marginLeft' => 1.5 * $cm,
                'marginRight' => 1.5 * $cm,
            ]);

            $logoPath = public_path('assets/img/logo.png');
            $appUrl = rtrim(env('APP_URL', url('/')), '/') . '/cv';

            // 2. تقسيم الـ pairs إلى صفحات من 8 بطاقات
            $chunks = array_chunk($this->newPairs, 8);

            foreach ($chunks as $pageCards) {
                // 3. إنشاء الجدول 2×4 بلا حدود خارجية
                $table = $section->addTable([
                    'unit' => TblWidth::PERCENT,
                    'tblW' => 100,
                    'alignment' => JcTable::CENTER,
                    'borderSize' => 0,
                    'cellMargin' => 100,
                ]);

                // 4. إضافة 4 صفوف × صفين
                for ($r = 0; $r < 4; $r++) {
                    $table->addRow($rowHeightTwip, [
                        'cantSplit' => true,
                        'exactHeight' => true,
                    ]);

                    for ($c = 0; $c < 2; $c++) {
                        $cell = $table->addCell($cellWidthTwip, [
                            'valign' => 'center',
                            'borderSize' => 6,
                            'borderColor' => 'CCCCCC',
                            'cellMargin' => 100,
                        ]);

                        $idx = $r * 2 + $c;
                        if (!isset($pageCards[$idx])) {
                            continue;
                        }

                        $p = $pageCards[$idx];

                        // 5. الشعار في منتصف البطاقة
                        if (file_exists($logoPath)) {
                            $cell->addImage($logoPath, [
                                'width' => $logoWidthTwip,
                                'height' => $logoHeightTwip,
                                'wrappingStyle' => 'behind',
                                'alignment' => Jc::CENTER,
                                'marginBottom' => 100,
                            ]);
                        }

                        // سطر فاصل
                        $cell->addTextBreak(1);
                        // رابط الصفحة
                        $cell->addLink($appUrl, $appUrl . ' : ' . __('backend.cv_code_Link'), ['size' => 8, 'color' => '0000FF', 'underline' => 'single'], ['alignment' => Jc::CENTER, 'spaceAfter' => 100]);
                        // 7. اسم الموظف (وسط، عريض)
                        $cell->addText(__('frontend.fullname') . ' : ' . $p['name'], ['bold' => true, 'size' => 12], ['alignment' => Jc::CENTER, 'spaceAfter' => 100]);
                        // 9. الكود (وسط، خط أحادي)
                        $cell->addText($p['code'] . ' : ' . __('backend.cv_code_code'), ['name' => 'Courier New', 'size' => 10], ['alignment' => Jc::CENTER]);
                    }
                }

                // 10. فاصل صفحة إذا لم تكن الأخيرة
                if ($pageCards !== end($chunks)) {
                    $section->addPageBreak();
                }
            }

            // 11. حفظ وإخراج الملف
            $writer = IOFactory::createWriter($phpWord, 'Word2007');
            $writer->save('php://output');
        }, $fileName);
    }

    public function nextStep(): void
    {
        if ($this->currentStep !== 1) {
            return;
        }

        // 0. يجب إضافة اسم واحد على الأقل
        if (empty($this->newNames)) {
            $this->dispatch('notify', type: 'error', message: __('backend.at_least_one_name'), duration: 4000);
            return;
        }

        // 1. تحقّق تكرار الأسماء محليًّا
        if (collect($this->newNames)->duplicates()->isNotEmpty()) {
            $this->dispatch('notify', type: 'error', message: __('backend.duplicate_name'), duration: 4000);
            return;
        }

        foreach ($this->newNames as $name) {
            // 2. عربي فقط
            if (!preg_match('/^[\p{Arabic}\s]+$/u', $name)) {
                $this->dispatch('notify', type: 'error', message: __('backend.only_arabic'), duration: 4000);
                return;
            }

            // 3. ثلاث كلمات على الأقل
            $words = preg_split('/\s+/u', trim($name));
            if (count($words) < 3) {
                $this->dispatch('notify', type: 'error', message: __('backend.at_least_three_words'), duration: 4000);
                return;
            }

            // 4. عدم وجود الاسم في الجدولين
            $existsInCodes = UserEmployeeCvCode::where('name', $name)->exists();
            $existsInEmployees = \App\Models\User\UserEmployee::whereRaw("JSON_VALUE(full_name, '$.ar') = ?", [$name])->exists();

            if ($existsInCodes || $existsInEmployees) {
                $this->dispatch('notify', type: 'error', message: __('backend.name_already_exists'), duration: 4000);
                return;
            }
        }

        // كلّ الشروط اجتازت → الانتقال للخطوة 2
        $this->newPairs = [];
        $targetProject = $this->canViewAll
            ? $this->new['project_code'] // ← نفس الـ binding
            : Auth::user()->project_code;
        foreach ($this->newNames as $name) {
            $code = $this->nextCode();
            $this->newPairs[] = ['name' => $name, 'code' => $code];
            UserEmployeeCvCode::create([
                'name' => $name,
                'code' => $code,
                'used' => false,
                'project_code' => $targetProject,
                'created_by' => Auth::id(),
            ]);
        }

        $this->currentStep = 2;
    }

    public function checkNameExists(string $name): bool
    {
        // 1. عربي فقط
        if (!preg_match('/^[\p{Arabic}\s]+$/u', $name)) {
            $this->dispatch('notify', type: 'error', message: __('backend.only_arabic'), duration: 6000);
            return true;
        }

        // 2. ثلاث كلمات على الأقل
        $words = preg_split('/\s+/u', trim($name));
        if (count($words) < 3) {
            $this->dispatch('notify', type: 'error', message: __('backend.at_least_three_words'), duration: 6000);
            return true;
        }

        // 3. عدم التكرار في الجدولين
        $existsInCodes = UserEmployeeCvCode::where('name', $name)->exists();
        $existsInEmployees = \App\Models\User\UserEmployee::whereRaw("JSON_VALUE(full_name, '$.ar') = ?", [$name])->exists();

        if ($existsInCodes || $existsInEmployees) {
            $this->dispatch('notify', type: 'error', message: __('backend.name_already_exists'), duration: 6000);
            return true;
        }

        return false;
    }

    public function update(): void
    {
        // 1. تحقق عام على الحقول
        $rules = [
            'id' => 'required|integer|exists:users_employees_cv_codes,id',
            'code' => 'required|string|unique:users_employees_cv_codes,code,' . $this->edit['id'],
            'name' => [
                'required',
                'string',
                'max:255',
                // عربي فقط
                'regex:/^[\p{Arabic}\s]+$/u',
                // ثلاث كلمات على الأقل
                function ($attribute, $value, $fail) {
                    if (count(preg_split('/\s+/u', trim($value))) < 3) {
                        $fail(__('backend.at_least_three_words'));
                    }
                },
            ],
            'used' => 'boolean',
        ];

        /* ➋ project_code حسب الصلاحيات */
        if ($this->canViewAll) {
            $rules['edit.project_code'] = ['required', 'exists:users_projects,code'];
        } else {
            // لا نسمح بتعديله أبداً
            $this->edit['project_code'] = Auth::user()->project_code;
            $rules['project_code'] = ['required', 'in:' . Auth::user()->project_code];
        }

        $data = validator($this->edit, $rules)->validate();

        // 2. تأكد من عدم وجود الاسم ذاته (في CV codes أو جدول الموظفين) لغير هذا السجل
        $exists = UserEmployeeCvCode::where('name', $data['name'])->where('id', '!=', $data['id'])->exists() || \App\Models\User\UserEmployee::whereRaw("JSON_VALUE(full_name, '$.ar') = ?", [$data['name']])->exists();

        if ($exists) {
            $this->dispatch('notify', type: 'error', message: __('backend.name_already_exists'), duration: 4000);
            return;
        }

        // 3. حدّث السجل
        UserEmployeeCvCode::findOrFail($data['id'])->update([
            'project_code' => $data['project_code'],
            'code' => $data['code'],
            'name' => $data['name'],
            'used' => $data['used'],
        ]);

        // 4. أغلق المودال
        $this->closeModal();

        // 5. إشعار نجاح
        $this->dispatch('notify', type: 'success', message: __('backend.cv_code_updated'), duration: 4000);
    }

    public function delete(): void
    {
        // خذ الـ id من المودال المفتوح
        [$mode, $id] = $this->activeModal;

        // تأكد أنّه مودال حذف
        if ($mode !== 'delete' || !$id) {
            return;
        }

        // احذف السجل
        UserEmployeeCvCode::findOrFail($id)->delete();

        // أغلق المودال
        $this->closeModal();

        // نبّه المستخدم بالنجاح
        $this->dispatch('notify', type: 'success', message: __('backend.cv_code_deleted'), duration: 4000);
    }
};
?>



<div class="p-4 max-w-5xl mx-auto" wire:init="$set('readyToLoad',true)">
    <h2 class="text-xl font-bold mb-4 text-zinc-800 dark:text-white">
        {{ __('backend.cv_codes') }}
    </h2>



    <div class="mb-4 max-w-md">
        <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
            {{ __('backend.search_by_code_or_name2') }}
        </label>
        <div class="relative">
            <input type="text" wire:model.live="search"
                class="w-full rounded-lg bg-white dark:bg-zinc-800 border border-zinc-300 dark:border-zinc-700 text-sm px-3 py-2"
                placeholder="{{ __('backend.search_placeholder2') }}" />

            @if (!empty($suggestions))
                <ul
                    class="absolute w-full bg-white dark:bg-zinc-900 border border-zinc-300 dark:border-zinc-700 mt-1 rounded-lg shadow z-50 text-sm">
                    @foreach ($suggestions as $suggestion)
                        <li class="px-3 py-1 hover:bg-zinc-100 dark:hover:bg-zinc-800 cursor-pointer"
                            wire:click="selectSuggestion('{{ addslashes($suggestion) }}')">

                            {{ $suggestion }}
                        </li>
                    @endforeach
                </ul>
            @endif
        </div>

    </div>


    @access('cv.genrate', 'cv.add_cv_code')
        <button wire:click="openAddModal()"
            class="bg-indigo-600 hover:bg-indigo-700 text-white text-sm font-semibold px-6 py-2 rounded-md transition duration-300 transform hover:scale-105 shadow hover:shadow-lg">
            + {{ __('backend.add_cv_code') }}
        </button>
    @endaccess

    <div class="bg-white dark:bg-zinc-900 shadow rounded-lg">
        <div class="mt-4">
            {{-- {{ $projects->links() }} --}}
        </div>
        <div class="bg-white dark:bg-zinc-900 shadow rounded-lg overflow-x-auto">
            <table class="min-w-[600px] w-full text-sm table-auto">
                <thead class="bg-zinc-100 dark:bg-zinc-800">
                    <tr>
                        @access('cv.genrate', 'cv.project_name')
                            <th class="px-4 py-2 text-start whitespace-nowrap" wire:click="sortBy('project_code')">
                                {{ __('backend.project_name') }}
                            </th>
                        @endaccess
                        @access('cv.genrate', 'cv.cv_code_code')
                            <th class="px-4 py-2 text-start whitespace-nowrap" wire:click="sortBy('code')">
                                {{ __('backend.cv_code_code') }}
                            </th>
                        @endaccess
                        @access('cv.genrate', 'cv.cv_code_name')
                            <th class="px-4 py-2 text-start whitespace-nowrap" wire:click="sortBy('name')">
                                {{ __('backend.cv_code_name') }}
                            </th>
                        @endaccess
                        @access('cv.genrate', 'cv.cv_code_used')
                            <th class="px-4 py-2 text-start whitespace-nowrap" wire:click="sortBy('used')">
                                {{ __('backend.cv_code_used') }}
                            </th>
                        @endaccess
                        <th class="px-4 py-2 text-start whitespace-nowrap">
                            <select wire:model.live="perPage"
                                class="w-full rounded-lg bg-white dark:bg-zinc-800 border border-zinc-300 dark:border-zinc-700 text-sm px-3 py-2">
                                <option value="5">5</option>
                                <option value="10">10</option>
                                <option value="25">25</option>
                                <option value="50">50</option>
                            </select>
                        </th>
                    </tr>
                </thead>
                <tbody>
                    @foreach ($this->codes as $row)
                        <tr class="border-t dark:border-zinc-700">
                            @access('cv.genrate', 'cv.project_name')
                            <td class="px-4 py-2 whitespace-nowrap">{{ $row->project?->name ?? '—' }}</td>
                            @endaccess
                            @access('cv.genrate', 'cv.cv_code_code')
                            <td class="px-4 py-2 whitespace-nowrap">{{ $row->code }}</td>
                            @endaccess
                            @access('cv.genrate', 'cv.cv_code_name')
                            <td class="px-4 py-2 whitespace-nowrap">{{ $row->name }}</td>
                            @endaccess
                            @access('cv.genrate', 'cv.cv_code_used')
                            <td class="px-4 py-2 whitespace-nowrap">
                                <span class="text-base">
                                    {{ $row->used ? '✅' : '❌' }}
                                </span>
                                <span
                                    class="inline-flex items-center gap-1 px-2 py-1 text-xs font-semibold rounded-full
                                         {{ $row->used ? 'bg-green-500 text-white' : 'bg-red-500 text-white' }}">
                                    {{ $row->used ? __('backend.yes') : __('backend.no') }}
                                </span>
                            </td>
                            @endaccess
                            <td class="px-3 py-2 text-end">
                                @access('cv.genrate', 'cv.edit')
                                <button wire:click="openEditModal({{ $row->id }})"
                                    class="text-xs bg-yellow-500 hover:bg-yellow-600 text-white px-3 py-1 rounded shadow">
                                    {{ __('backend.edit') }}
                                </button>
                                @endaccess
                                @access('cv.genrate', 'cv.delete')
                                <button wire:click="openDeleteModal({{ $row->id }})"
                                    class="text-xs bg-red-600 hover:bg-red-700 text-white px-3 py-1 rounded shadow">
                                    {{ __('backend.delete') }}
                                </button>
                                @endaccess
                            </td>
                        </tr>
                    @endforeach
                </tbody>
            </table>

            <div class="mt-4">{{ $this->codes->links() }}</div>
        </div>
        {{-- مودال الإضافة --}}
        @if ($activeModal && $activeModal[0] === 'add')
            @include('livewire.backend.employee.cv.modals.add')
        @endif

        {{-- مودال التعديل --}}
        @if ($activeModal && $activeModal[0] === 'edit')
            @include('livewire.backend.employee.cv.modals.edit')
        @endif

        {{-- مودال الحذف --}}
        @if ($activeModal && $activeModal[0] === 'delete')
            @include('livewire.backend.employee.cv.modals.delete')
        @endif


    </div>
</div>
