<?php

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Auth\Events\Lockout;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;
use Livewire\Attributes\Layout;
use Livewire\Volt\Component;
use Illuminate\View\View;

new #[Layout('components.frontend.layouts.struct')] class extends Component {
    public function rendering(View $view): void
    {
        $view->title(__('frontend.tittlewelcomepage') . ' | ' . __('frontend.login'));
    }
    public string $password = '';
    public bool $loginDisabled = false;

    public function unlock(): void
    {
        $user = Auth::user();

        $settings = $user->settings ?? [];
        $attempts = $settings['unlock_attempts'] ?? 0;

        if (!Hash::check($this->password, $user->password)) {
            $attempts++;
            $settings['unlock_attempts'] = $attempts;

            if ($attempts >= 5) {
                Auth::logout();
                Session::invalidate();
                Session::regenerateToken();

                $settings['is_locked'] = false;
                $settings['unlock_attempts'] = 0;

                $user->update([
                    'settings' => $settings,
                ]);

                $this->redirect(route('login'));
                session()->flash('toast', [
                    'type' => 'error',
                    'msg' => __('backend.too_many_unlock_attempts'),
                    'duration' => 5000,
                ]);
                return;
            }

            $user->update([
                'settings' => $settings,
            ]);

            $this->dispatch('notify', type: 'error', message: __('backend.incorrect_password_attempt', ['num' => $attempts]), duration: 5000);
            return;
        }

        $settings['is_locked'] = false;
        $settings['unlock_attempts'] = 0;

        $user->update([
            'settings' => $settings,
        ]);

        Session::regenerate();

        session()->flash('toast', [
            'type' => 'success',
            'msg' => __('backend.unlock_success'),
            'duration' => 5000,
        ]);

        $this->redirect(route('dashboard'));
    }
};
?>

<div x-data="{ locked: true }" class="relative min-h-screen overflow-hidden">

    <!-- الخلفية -->
    <img :src="darkMode ? '/assets/img/company-lock-bg.png' : '/assets/img/company-lock-bg-light.png'"
        alt="Lock Background"
        class="fixed inset-0 w-full h-full object-cover z-0 transition-all duration-700 ease-in-out pointer-events-none"
        :class="locked ? 'opacity-100 translate-y-0' : 'opacity-0 -translate-y-full'" />


    <!-- شاشة القفل -->
    <div x-show="locked" x-transition:leave="transition-all ease-in-out duration-700"
        x-transition:leave-start="opacity-100 translate-y-0" x-transition:leave-end="opacity-0 -translate-y-full"
        @click="locked = false"
        class="absolute inset-0 z-10 flex items-center justify-center cursor-pointer
               bg-black/60 backdrop-blur-md text-white select-none transition-all">

        <div class="text-center animate-fade-in-up">
            <svg class="w-16 h-16 mx-auto mb-4 text-white/80" fill="none" stroke="currentColor" stroke-width="1.5"
                viewBox="0 0 24 24">
                <rect x="3" y="11" width="18" height="11" rx="2" ry="2" />
                <path d="M7 11V7a5 5 0 0110 0v4" />
            </svg>
            <h2 class="text-xl md:text-2xl font-semibold">{{ __('backend.click_to_continue') }}</h2>
            <p class="text-sm text-zinc-300 mt-2">{{ __('backend.click_anywhere') }}</p>
        </div>
    </div>

    <!-- فورم كلمة المرور -->
    <div x-show="!locked" x-transition:enter="transition ease-out duration-700"
        x-transition:enter-start="opacity-0 translate-y-10 scale-95"
        x-transition:enter-end="opacity-100 translate-y-0 scale-100"
        class="relative z-20 flex items-center justify-center min-h-screen px-4">

        <div
            class="max-w-md w-full space-y-6
                    bg-white/80 dark:bg-zinc-900/80 backdrop-blur-md
                    p-8 rounded-xl shadow-2xl
                    border border-zinc-300 dark:border-zinc-700
                    text-center animate-fade-in-down">

            <!-- اللوجو -->
            <img src="/assets/img/logo.png" alt="Logo" class="w-[220px] h-auto mx-auto drop-shadow mb-4" />

            <h2 class="text-2xl font-semibold text-zinc-800 dark:text-white">{{ __('backend.locked_title') }}</h2>
            <!-- صورة المستخدم أو اختصار اسمه -->
            <div class="flex flex-col items-center gap-2 mb-6 relative">
                @php
                    $user = Auth::user();
                    $avatar = $user->settings['avatar'] ?? null;
                    $initials = collect(explode(' ', $user->name))
                        ->map(fn($word) => mb_substr($word, 0, 1))
                        ->join('');

                    // تحقق هل الرابط خارجي أو من التخزين
                    $isExternal = $avatar && Str::startsWith($avatar, ['http://', 'https://']);
                @endphp

                <div class="relative group w-24 h-24">
                    <!-- حلقة النبض -->
                    <div class="animate-ring"></div>

                    <!-- خلفية glow -->
                    <div
                        class="absolute inset-0 rounded-full blur-[45px] opacity-60
        bg-gradient-to-br from-[#3b82f6] via-[#60a5fa] to-[#93c5fd]
dark:from-[#2563eb] dark:via-[#3b82f6] dark:to-[#60a5fa]
        group-hover:opacity-80 transition duration-700 z-[-1]">
                    </div>

                    @if ($avatar)
                        <img src="{{ $isExternal ? $avatar : asset('storage/' . $avatar) }}" alt="User Photo"
                            class="w-24 h-24 rounded-full object-cover border-4 border-white dark:border-zinc-700 shadow-2xl transition-transform duration-300 group-hover:scale-105" />
                    @else
                        <div
                            class="w-24 h-24 flex items-center justify-center rounded-full
                        bg-blue-500 dark:bg-blue-600 text-white dark:bg-zinc-700 dark:text-white
                        text-2xl font-semibold border-4 border-white dark:border-zinc-700 shadow-2xl transition-transform duration-300 group-hover:scale-105">
                            {{ $initials }}
                        </div>
                    @endif
                </div>

                <!-- اسم المستخدم -->
                <h3 class="text-lg font-bold text-zinc-800 dark:text-white mt-2 animate-fade-in-up">
                    {{ $user->name }}
                </h3>
            </div>

            <p class="text-sm text-zinc-600 dark:text-zinc-300">{{ __('backend.locked_msg') }}</p>

            <form wire:submit.prevent="unlock" class="space-y-4" id="lock-form">
                <input type="password" wire:model.defer="password" required placeholder="••••••••"
                    autocomplete="current-password"
                    class="block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600
                       bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 ring-accent outline-none transition" />

                <button id="login-btn" type="submit"
                    class="neon-btn neon-blue w-full py-2 rounded-lg font-medium {{ $this->loginDisabled ? 'opacity-50 cursor-not-allowed' : '' }}"
                    {{ $this->loginDisabled ? 'disabled' : '' }}>
                    <span class="btn-text">{{ __('backend.unlock_continue') }}</span>
                    <span class="glow"></span>
                </button>
            </form>

            <form method="POST" action="{{ route('logout') }}">
                @csrf
                <button type="submit"
                    class="w-full py-2 mt-2 rounded-lg font-semibold transition duration-300
                           bg-red-500 hover:bg-red-600 text-white
                           dark:bg-red-400 dark:hover:bg-red-500 dark:text-black">
                    {{ __('frontend.logout') }}
                </button>
            </form>
        </div>
    </div>



</div>
