@props([
    'show' => false,
    'type' => 'default', // default, form, upload, processing
    'message' => null,
    'progress' => null,
    'backdrop' => true,
    'size' => 'md' // sm, md, lg
])

<div
    x-data="{
        show: @js($show),
        type: @js($type),
        message: @js($message),
        progress: @js($progress),
        networkSpeed: 'fast',
        dots: '',
        dotInterval: null,

        init() {
            // مراقبة سرعة الشبكة
            this.checkNetworkSpeed();

            // تحديث النقاط المتحركة - محسن
            this.startDotAnimation();

            // مراقبة تغييرات الشبكة
            this.setupNetworkListener();

            // تنظيف عند الإغلاق
            this.$watch('show', (value) => {
                if (!value) {
                    this.cleanup();
                }
            });
        },

        startDotAnimation() {
            if (this.dotInterval) clearInterval(this.dotInterval);
            this.dotInterval = setInterval(() => {
                this.dots = this.dots.length >= 3 ? '' : this.dots + '.';
            }, 400); // أسرع قليلاً
        },

        cleanup() {
            if (this.dotInterval) {
                clearInterval(this.dotInterval);
                this.dotInterval = null;
            }
        },

        setupNetworkListener() {
            if (navigator.connection) {
                navigator.connection.addEventListener('change', () => {
                    this.checkNetworkSpeed();
                });
            }
        },

        checkNetworkSpeed() {
            if (navigator.connection) {
                const connection = navigator.connection;
                const effectiveType = connection.effectiveType;

                if (effectiveType === 'slow-2g' || effectiveType === '2g') {
                    this.networkSpeed = 'slow';
                } else if (effectiveType === '3g') {
                    this.networkSpeed = 'medium';
                } else {
                    this.networkSpeed = 'fast';
                }
            }
        },

        getLoadingMessage() {
            if (this.message) return this.message;

            const messages = {
                fast: {
                    default: '{{ __('Loading...') }}',
                    form: '{{ __('Processing form...') }}',
                    upload: '{{ __('Uploading files...') }}',
                    processing: '{{ __('Processing data...') }}'
                },
                medium: {
                    default: '{{ __('Loading, please wait...') }}',
                    form: '{{ __('Processing form, please wait...') }}',
                    upload: '{{ __('Uploading files, please wait...') }}',
                    processing: '{{ __('Processing data, please wait...') }}'
                },
                slow: {
                    default: '{{ __('Loading, this may take a while...') }}',
                    form: '{{ __('Processing form, almost done...') }}',
                    upload: '{{ __('Uploading files, this may take a while...') }}',
                    processing: '{{ __('Processing data, almost done...') }}'
                }
            };

            return messages[this.networkSpeed][this.type] || messages[this.networkSpeed].default;
        },

        destroy() {
            this.cleanup();
        }
    }"
    x-show="show || ($wire?.loading && !$wire?.offline)"
    x-transition:enter="transition-all duration-200 ease-out"
    x-transition:enter-start="opacity-0 scale-95 backdrop-blur-0"
    x-transition:enter-end="opacity-100 scale-100 backdrop-blur-sm"
    x-transition:leave="transition-all duration-150 ease-in"
    x-transition:leave-start="opacity-100 scale-100 backdrop-blur-sm"
    x-transition:leave-end="opacity-0 scale-95 backdrop-blur-0"
    class="fixed inset-0 z-50 flex items-center justify-center"
    style="display: none;"
    {{ $attributes }}>

    @if($backdrop)
    <!-- Backdrop -->
    <div class="absolute inset-0 bg-black/40 backdrop-blur-sm"></div>
    @endif

    <!-- Loading Container -->
    <div class="relative bg-white/95 dark:bg-zinc-900/95 backdrop-blur-md rounded-2xl shadow-2xl border border-white/20 dark:border-zinc-700/50 w-full mx-4
        {{ $size === 'sm' ? 'p-6 max-w-xs' : ($size === 'lg' ? 'p-10 max-w-md' : 'p-8 max-w-sm') }}">

        <!-- Loading Animation -->
        <div class="flex flex-col items-center space-y-6">

            <!-- Spinner Types -->
            <div class="relative">
                @if($type === 'upload')
                    <!-- Upload Animation -->
                    <div class="relative w-20 h-20">
                        <!-- Cloud -->
                        <div class="absolute inset-0 flex items-center justify-center">
                            <svg class="w-16 h-16 text-indigo-500" fill="currentColor" viewBox="0 0 24 24">
                                <path d="M19.35 10.04A7.49 7.49 0 0 0 12 4C9.11 4 6.6 5.64 5.35 8.04A5.994 5.994 0 0 0 0 14c0 3.31 2.69 6 6 6h13c2.76 0 5-2.24 5-5 0-2.64-2.05-4.78-4.65-4.96z"/>
                            </svg>
                        </div>
                        <!-- Upload Arrow -->
                        <div class="absolute inset-0 flex items-center justify-center">
                            <svg class="w-8 h-8 text-white animate-bounce-light" fill="currentColor" viewBox="0 0 24 24">
                                <path d="M7 14l5-5 5 5H7z"/>
                            </svg>
                        </div>
                        <!-- Animated Dots -->
                        <div class="absolute -bottom-2 left-1/2 transform -translate-x-1/2">
                            <div class="flex space-x-1">
                                <div class="w-2 h-2 bg-indigo-500 rounded-full animate-pulse"></div>
                                <div class="w-2 h-2 bg-indigo-500 rounded-full animate-pulse" style="animation-delay: 0.2s"></div>
                                <div class="w-2 h-2 bg-indigo-500 rounded-full animate-pulse" style="animation-delay: 0.4s"></div>
                            </div>
                        </div>
                    </div>

                @elseif($type === 'form')
                    <!-- Form Processing Animation -->
                    <div class="relative w-20 h-20">
                        <div class="absolute inset-0 border-4 border-indigo-200 dark:border-indigo-800 rounded-full"></div>
                        <div class="absolute inset-0 border-4 border-indigo-600 rounded-full border-t-transparent animate-spin-smooth"></div>
                        <div class="absolute inset-0 flex items-center justify-center">
                            <svg class="w-8 h-8 text-indigo-600 animate-pulse" fill="currentColor" viewBox="0 0 24 24">
                                <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                        </div>
                    </div>

                @elseif($type === 'processing')
                    <!-- Data Processing Animation -->
                    <div class="relative w-20 h-20">
                        <div class="grid grid-cols-3 gap-1 w-12 h-12 mx-auto">
                            @for($i = 0; $i < 9; $i++)
                                <div class="w-3 h-3 bg-indigo-600 rounded-sm animate-pulse" style="animation-delay: {{ $i * 0.1 }}s"></div>
                            @endfor
                        </div>
                    </div>

                @else
                    <!-- Default Animation -->
                    <div class="relative w-20 h-20">
                        <!-- Outer Ring -->
                        <div class="absolute inset-0 border-4 border-indigo-200 dark:border-indigo-800 rounded-full"></div>
                        <!-- Spinning Ring -->
                        <div class="absolute inset-0 border-4 border-indigo-600 rounded-full border-t-transparent animate-spin-smooth"></div>
                        <!-- Inner Pulse -->
                        <div class="absolute inset-4 bg-indigo-600 rounded-full animate-pulse-step opacity-20"></div>
                        <!-- Center Dot -->
                        <div class="absolute inset-0 flex items-center justify-center">
                            <div class="w-3 h-3 bg-indigo-600 rounded-full animate-ping-soft"></div>
                        </div>
                    </div>
                @endif
            </div>

            <!-- Progress Bar (if provided) -->
            @if($progress !== null)
                <div class="w-full bg-zinc-200 dark:bg-zinc-700 rounded-full h-2 overflow-hidden">
                    <div class="bg-gradient-to-r from-indigo-500 to-purple-600 h-2 rounded-full transition-all duration-300 ease-out"
                         :style="`width: ${progress || 0}%`"></div>
                </div>
                <div class="text-sm text-zinc-600 dark:text-zinc-400" x-text="`${progress || 0}%`"></div>
            @endif

            <!-- Loading Message -->
            <div class="text-center space-y-2">
                <p class="text-lg font-semibold text-zinc-800 dark:text-white" x-text="getLoadingMessage() + dots"></p>

                <!-- Network Speed Indicator -->
                <div class="flex items-center justify-center space-x-2 text-sm text-zinc-500 dark:text-zinc-400">
                    <div class="flex space-x-1">
                        <div class="w-1 h-3 rounded-full"
                             :class="networkSpeed === 'slow' ? 'bg-red-500' : networkSpeed === 'medium' ? 'bg-yellow-500' : 'bg-green-500'"></div>
                        <div class="w-1 h-3 rounded-full"
                             :class="networkSpeed === 'medium' || networkSpeed === 'fast' ? (networkSpeed === 'medium' ? 'bg-yellow-500' : 'bg-green-500') : 'bg-zinc-300'"></div>
                        <div class="w-1 h-3 rounded-full"
                             :class="networkSpeed === 'fast' ? 'bg-green-500' : 'bg-zinc-300'"></div>
                    </div>
                    <span x-text="networkSpeed === 'slow' ? '{{ __('Slow Connection') }}' : networkSpeed === 'medium' ? '{{ __('Medium Connection') }}' : '{{ __('Fast Connection') }}'"></span>
                </div>

                <!-- Tips for slow connection -->
                <div x-show="networkSpeed === 'slow'"
                     x-transition:enter="transition-all duration-300"
                     x-transition:enter-start="opacity-0 transform translate-y-2"
                     x-transition:enter-end="opacity-100 transform translate-y-0"
                     class="mt-3 p-3 bg-amber-50 dark:bg-amber-900/20 rounded-lg border border-amber-200 dark:border-amber-800">
                    <p class="text-xs text-amber-700 dark:text-amber-300">
                        {{ __('Your connection seems slow. Please be patient while we process your request.') }}
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* محسن للأداء */
@keyframes spin-smooth {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

@keyframes pulse-step {
    0%, 100% {
        transform: scale(1);
        opacity: 1;
    }
    50% {
        transform: scale(1.05);
        opacity: 0.8;
    }
}

@keyframes bounce-light {
    0%, 100% {
        transform: translateY(0);
        animation-timing-function: cubic-bezier(0.8, 0, 1, 1);
    }
    50% {
        transform: translateY(-8px);
        animation-timing-function: cubic-bezier(0, 0, 0.2, 1);
    }
}

@keyframes ping-soft {
    0% {
        transform: scale(1);
        opacity: 1;
    }
    75%, 100% {
        transform: scale(1.5);
        opacity: 0;
    }
}

.animate-spin-smooth {
    animation: spin-smooth 1s linear infinite;
}

.animate-pulse-step {
    animation: pulse-step 1.5s ease-in-out infinite;
}

.animate-bounce-light {
    animation: bounce-light 1s infinite;
}

.animate-ping-soft {
    animation: ping-soft 1.5s cubic-bezier(0, 0, 0.2, 1) infinite;
}

/* تحسينات الأداء */
.advanced-loader {
    will-change: opacity, transform;
    contain: layout style paint;
}

.advanced-loader * {
    will-change: auto;
}

/* تحسين backdrop-blur للأداء */
.backdrop-blur-optimized {
    backdrop-filter: blur(4px);
    -webkit-backdrop-filter: blur(4px);
}

@media (prefers-reduced-motion: reduce) {
    .animate-spin-smooth,
    .animate-pulse-step,
    .animate-bounce-light,
    .animate-ping-soft {
        animation: none;
    }
}
</style>
