/**
 * Advanced Loader Utilities
 * نظام متطور لإدارة حالات التحميل مع دعم سرعة الشبكة
 */

class AdvancedLoader {
    constructor() {
        this.instances = new Map();
        this.networkSpeed = this.detectNetworkSpeed();
        this.setupNetworkMonitoring();
    }

    /**
     * إنشاء loader جديد
     */
    create(id, options = {}) {
        const defaultOptions = {
            type: 'default',
            size: 'md',
            message: null,
            progress: null,
            backdrop: true,
            target: null,
            autoShow: false,
            timeout: 10000, // Reduced to 10 seconds
            autoHide: true,
            maxRetries: 3
        };

        const config = { ...defaultOptions, ...options };
        const instance = {
            id,
            config,
            element: null,
            timeout: null,
            startTime: Date.now(),
            retryCount: 0,
            isVisible: false
        };        this.instances.set(id, instance);
  if (config.autoShow) {
                    this.show(id);
        }

        return instance;
    }

    /**
     * عرض الـ loader
     */
    show(id, message = null) {
        const instance = this.instances.get(id);
        if (!instance) return;

        // منع إظهار متكرر
        if (instance.isVisible) return;

        // إلغاء التايم أوت السابق
        if (instance.timeout) {
            clearTimeout(instance.timeout);
        }

        // تحديث الرسالة إذا تم تمريرها
        if (message) {
            instance.config.message = message;
        }

        // إنشاء العنصر إذا لم يكن موجوداً
        if (!instance.element) {
            instance.element = this.createElement(instance);
            document.body.appendChild(instance.element);
        }

        // عرض العنصر
        instance.element.style.display = 'flex';
        instance.startTime = Date.now();
        instance.isVisible = true;

        // تعيين timeout مع تحسينات
        const timeoutDuration = this.getAdaptiveTimeout(instance.config.timeout);
        instance.timeout = setTimeout(() => {
            this.handleTimeout(id);
        }, timeoutDuration);

        // إرسال event مخصص
        window.dispatchEvent(new CustomEvent('loader:show', { detail: { id, instance } }));
    }

    /**
     * إخفاء الـ loader
     */
    hide(id) {
        const instance = this.instances.get(id);
        if (!instance || !instance.element || !instance.isVisible) return;

        // إلغاء التايم أوت
        if (instance.timeout) {
            clearTimeout(instance.timeout);
            instance.timeout = null;
        }

        // إخفاء العنصر مع animation
        instance.element.style.opacity = '0';
        instance.element.style.transform = 'scale(0.95)';

        setTimeout(() => {
            if (instance.element) {
                instance.element.style.display = 'none';
                instance.element.style.opacity = '1';
                instance.element.style.transform = 'scale(1)';
            }
        }, 200);

        instance.isVisible = false;

        // حساب وقت التحميل
        const loadTime = Date.now() - instance.startTime;

        // إرسال event مخصص
        window.dispatchEvent(new CustomEvent('loader:hide', {
            detail: { id, instance, loadTime }
        }));

        // تنظيف إذا كان auto-cleanup مفعل
        if (instance.config.autoHide && loadTime > 100) {
            setTimeout(() => this.destroy(id), 1000);
        }
    }

    /**
     * تحديث التقدم
     */
    updateProgress(id, progress, message = null) {
        const instance = this.instances.get(id);
        if (!instance || !instance.element) return;

        const progressBar = instance.element.querySelector('.progress-bar');
        const progressText = instance.element.querySelector('.progress-text');
        const messageElement = instance.element.querySelector('.loading-message');

        if (progressBar) {
            progressBar.style.width = `${Math.min(100, Math.max(0, progress))}%`;
        }

        if (progressText) {
            progressText.textContent = `${Math.round(progress)}%`;
        }

        if (message && messageElement) {
            messageElement.textContent = message;
        }

        // Auto-hide when complete
        if (progress >= 100) {
            setTimeout(() => this.hide(id), 800);
        }
    }

    /**
     * إنشاء عنصر الـ loader
     */
    createElement(instance) {
        const div = document.createElement('div');
        div.className = 'advanced-loader fixed inset-0 z-50 flex items-center justify-center';
  const sizeClasses = {
            sm: 'p-6 max-w-xs',
            md: 'p-8 max-w-sm',
           lg: 'p-10 max-w-md'
        };

        const backdrop = instance.config.backdrop ?            '<div class="absolute inset-0 b g-black/40 backdrop-blur-sm"></div>' : '';

        div.innerHTML = `
            ${backdrop}
             <div class="relative bg-white/95 dark:bg-zinc-900/95 backdrop-blur-md rounded-2xl shadow-2xl border border-white/20 dark:border-zinc-700/50 w-full mx-4 ${sizeClasses[instance.config.size]}">
                <div class="flex flex-col items-center space-y-6">
                    ${this.getAnimationHTML(instance.config.type)}
  ${instance.config.progress !== null ? `
                        <div class="w-full bg-zinc-200 dark:bg-zinc-700 rounded-full h-2 ov                    erflow-hidden">
                            <div class="progress-bar bg-gradient-to-r from-indigo-500 to-purple-600 h-2 rounded-full transition-all duration-300 ease-out" style="width: ${instance.config.progress}%"></div>
                        </div>
                        <div class="progress-text text-sm text-zinc-600 dark:text-zinc-400">${instance.config.progress}%</div>
                    ` : ''}
  <div class="text-center space-y-2">
                        <p class="loading-message text-lg font-semibold text-zinc-800 dark:text-white"                    >
                            ${this.getLoadingMessage(instance.config)}
                        </p>
                        ${this.getNetworkIndicatorHTML()}
                    </div>
                </div>
            </div>
        `;

        return div;
    }

    /**
     * الحصول على HTML الخاص بالأنيميشن
     */
    getAnimationHTML(type) {
        const animations = {
            upload: `
                <div class="relative w-20 h-20">
                    <div class="absolute inset-0 flex items-center justify-center">
                        <svg class="w-16 h-16 text-indigo-500" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M19.35 10.04A7.49 7.49 0 0 0 12 4C9.11 4 6.6 5.64 5.35 8.04A5.994 5.994 0 0 0 0 14c0 3.31 2.69 6 6 6h13c2.76 0 5-2.24 5-5 0-2.64-2.05-4.78-4.65-4.96z"/>
                        </svg>
                    </div>
                    <div class="absolute inset-0 flex items-center justify-center">
                        <svg class="w-8 h-8 text-white animate-bounce" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M7 14l5-5 5 5H7z"/>
                        </svg>
                    </div>
                </div>
            `,
            form: `
                <div class="relative w-20 h-20">
                    <div class="absolute inset-0 border-4 border-indigo-200 dark:border-indigo-800 rounded-full"></div>
                    <div class="absolute inset-0 border-4 border-indigo-600 rounded-full border-t-transparent animate-spin"></div>
                    <div class="absolute inset-0 flex items-center justify-center">
                        <svg class="w-8 h-8 text-indigo-600 animate-pulse" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                    </div>
                </div>
            `,
            processing: `
                <div class="relative w-20 h-20">
                    <div class="grid grid-cols-3 gap-1 w-12 h-12 mx-auto">
                        ${Array.from({length: 9}, (_, i) =>
                            `<div class="w-3 h-3 bg-indigo-600 rounded-sm animate-pulse" style="animation-delay: ${i * 0.1}s"></div>`
                        ).join('')}
                    </div>
                 </div>
            `,
            default: `
                <div class="relative w-20 h-20">
                    <div class="absolute inset-0 border-4 border-indigo-200 dark:border-indigo-800 rounded-full"></div>
                    <div class="absolute inset-0 border-4 border-indigo-600 rounded-full border-t-transparent animate-spin"></div>
                    <div class="absolute inset-4 bg-indigo-600 rounded-full animate-pulse opacity-20"></div>
                    <div class="absolute inset-0 flex items-center justify-center">
                        <div class="w-3 h-3 bg-indigo-600 rounded-full animate-ping"></div>
                    </div>
                </div>
            `
        };

        return animations[type] || animations.default;
    }

    /**
     * الحصول على رسالة التحميل حسب نوع الشبكة
     */
    getLoadingMessage(config) {
        if (config.message) return config.message;

        const messages = {
            fast: {
                default: 'Loading...',
                form: 'Processing form...',
                upload: 'Uploading files...',
                processing: 'Processing data...'
            },
            medium: {
                default: 'Loading, please wait...',
                form: 'Processing form, please wait...',
                upload: 'Uploading files, please wait...',
                processing: 'Processing data, please wait...'
            },
            slow: {
                default: 'Loading... This may take a while',
                form: 'Processing form... Almost done',
                upload: 'Uploading files... This may take a while',
                processing: 'Processing data... Almost done'
            }
        };

        return messages[this.networkSpeed][config.type] || messages[this.networkSpeed].default;
    }

    /**
     * الحصول على HTML الخاص بمؤشر الشبكة
     */
    getNetworkIndicatorHTML() {
        const speedColors = {
            fast: 'bg-green-500',
            medium: 'bg-yellow-500',
            slow: 'bg-red-500'
        };

        const speedText = {
            fast: 'Fast Connection',
            medium: 'Medium Connection',
            slow: 'Slow Connection'
        };

        return `
            <div class="flex items-center justify-center space-x-2 text-sm text-zinc-500 dark:text-zinc-400">
                <div class="flex space-x-1">
                    <div class="w-1 h-3 rounded-full ${this.networkSpeed === 'slow' ? 'bg-red-500' : this.networkSpeed === 'medium' ? 'bg-yellow-500' : 'bg-green-500'}"></div>
                    <div class="w-1 h-3 rounded-full ${this.networkSpeed === 'medium' || this.networkSpeed === 'fast' ? (this.networkSpeed === 'medium' ? 'bg-yellow-500' : 'bg-green-500') : 'bg-zinc-300'}"></div>
                    <div class="w-1 h-3 rounded-full ${this.networkSpeed === 'fast' ? 'bg-green-500' : 'bg-zinc-300'}"></div>
                </div>
                <span>${speedText[this.networkSpeed]}</span>
            </div>
            ${this.networkSpeed === 'slow' ? `
                <div class="mt-3 p-3 bg-amber-50 dark:bg-amber-900/20 rounded-lg border border-amber-200 dark:border-amber-800">
                    <p class="text-xs text-amber-700 dark:text-amber-300">
                        Your connection seems slow. Please be patient while we process your request.
                    </p>
                </div>
            ` : ''}
        `;
    }

    /**
     * اكتشاف سرعة الشبكة
     */
    detectNetworkSpeed() {
        if (navigator.connection) {
            const connection = navigator.connection;
            if (connection.effectiveType === 'slow-2g' || connection.effectiveType === '2g') {
                return 'slow';
            } else if (connection.effectiveType === '3g') {
                return 'medium';
            }
        }
        return 'fast';
    }

    /**
     * مراقبة تغييرات الشبكة
     */
    setupNetworkMonitoring() {
        if (navigator.connection) {
            navigator.connection.addEventListener('change', () => {
                this.networkSpeed = this.detectNetworkSpeed();
                this.updateAllInstances();
            });
        }
    }

    /**
     * تحديث جميع الـ instances
     */
    updateAllInstances() {
        this.instances.forEach((instance, id) => {
            if (instance.element && instance.element.style.display !== 'none') {
                const messageElement = instance.element.querySelector('.loading-message');
                if (messageElement) {
                    messageElement.textContent = this.getLoadingMessage(instance.config);
                }

                const networkIndicator = instance.element.querySelector('.network-indicator');
                if (networkIndicator) {
                    networkIndicator.innerHTML = this.getNetworkIndicatorHTML();
                }
            }
        });
    }

    /**
     * عرض تحذير الاتصال البطيء مع إعادة المحاولة
     */
    handleTimeout(id) {
        const instance = this.instances.get(id);
        if (!instance || !instance.element) return;

        instance.retryCount++;

        if (instance.retryCount <= instance.config.maxRetries) {
            this.showRetryOption(id);
        } else {
            this.showTimeoutError(id);
        }
    }

    /**
     * عرض خيار إعادة المحاولة
     */
    showRetryOption(id) {
        const instance = this.instances.get(id);
        if (!instance || !instance.element) return;

        const messageElement = instance.element.querySelector('.loading-message');
        if (messageElement) {
            messageElement.innerHTML = `
                <div class="text-center">
                    <p class="text-amber-600 mb-3">Taking longer than expected...</p>
                    <button onclick="window.AdvancedLoader.retry('${id}')"
                            class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                        Try Again
                    </button>
                </div>
            `;
        }
    }

    /**
     * إعادة المحاولة
     */
    retry(id) {
        const instance = this.instances.get(id);
        if (!instance) return;

        // إخفاء الـ loader الحالي
        this.hide(id);

        // إعادة إنشاء الـ loader
        setTimeout(() => {
            instance.retryCount = 0;
            this.show(id);

            // إرسال event للتطبيق لإعادة المحاولة
            window.dispatchEvent(new CustomEvent('loader:retry', {
                detail: { id, instance }
            }));
        }, 500);
    }

    /**
     * عرض خطأ انتهاء الوقت
     */
    showTimeoutError(id) {
        const instance = this.instances.get(id);
        if (!instance || !instance.element) return;

        const messageElement = instance.element.querySelector('.loading-message');
        if (messageElement) {
            messageElement.innerHTML = `
                <div class="text-center text-red-600">
                    <p class="mb-2">Operation timed out</p>
                    <button onclick="window.AdvancedLoader.hide('${id}')"
                            class="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors">
                        Close
                    </button>
                </div>
            `;
        }
    }

    /**
     * الحصول على timeout تكيفي حسب سرعة الشبكة
     */
    getAdaptiveTimeout(baseTimeout) {
        const multipliers = {
            fast: 1,
            medium: 1.5,
            slow: 2.5
        };

        return Math.min(baseTimeout * multipliers[this.networkSpeed], 15000);
    }

    /**
     * تدمير loader
     */
    destroy(id) {
        const instance = this.instances.get(id);
        if (!instance) return;

        if (instance.timeout) {
            clearTimeout(instance.timeout);
        }

        if (instance.element) {
            instance.element.remove();
        }

        this.instances.delete(id);
    }

    /**
     * تدمير جميع الـ loaders
     */
    destroyAll() {
        this.instances.forEach((_, id) => {
            this.destroy(id);
        });
    }
}

// إنشاء instance عام
window.AdvancedLoader = new AdvancedLoader();

// إضافة shortcut functions
window.showLoader = (id, options) => window.AdvancedLoader.create(id, { ...options, autoShow: true });
window.hideLoader = (id) => window.AdvancedLoader.hide(id);
window.updateLoaderProgress = (id, progress, message) => window.AdvancedLoader.updateProgress(id, progress, message);

// Livewire integration - محسن ومتطور
document.addEventListener('livewire:init', () => {
    let activeLoaders = new Set();

    // إظهار الـ loader عند بدء الطلب
    Livewire.hook('request', ({ succeed, fail, payload }) => {
        try {
            const component = payload?.component;
            if (!component) return;

            // استخراج الـ target بشكل آمن
            let target = 'default';
            if (payload.updates && Array.isArray(payload.updates) && payload.updates.length > 0) {
                const firstUpdate = payload.updates[0];
                if (firstUpdate && firstUpdate.payload && firstUpdate.payload.method) {
                    target = firstUpdate.payload.method;
                }
            }

            const loaderId = `livewire-${component.id}-${target}`;

            // منع إظهار متكرر
            if (activeLoaders.has(loaderId)) return;

            activeLoaders.add(loaderId);

            window.AdvancedLoader.create(loaderId, {
                type: 'form',
                message: 'Processing...',
                autoShow: true,
                timeout: 8000,
                autoHide: true
            });

            // إخفاء عند النجاح
            succeed(() => {
                setTimeout(() => {
                    window.AdvancedLoader.hide(loaderId);
                    activeLoaders.delete(loaderId);
                }, 300);
            });

            // إخفاء عند الفشل
            fail(() => {
                setTimeout(() => {
                    window.AdvancedLoader.hide(loaderId);
                    activeLoaders.delete(loaderId);
                }, 300);
            });
        } catch (error) {
            console.warn('Advanced Loader: Error in Livewire hook:', error);
        }
    });

    // تنظيف الـ loaders المعلقة
    const cleanupInterval = setInterval(() => {
        try {
            activeLoaders.forEach(loaderId => {
                const instance = window.AdvancedLoader.instances.get(loaderId);
                if (instance && Date.now() - instance.startTime > 20000) {
                    window.AdvancedLoader.hide(loaderId);
                    activeLoaders.delete(loaderId);
                }
            });
        } catch (error) {
            console.warn('Advanced Loader: Error in cleanup:', error);
        }
    }, 5000);

    // مراقبة wire:loading elements
    const observer = new MutationObserver((mutations) => {
        try {
            mutations.forEach((mutation) => {
                mutation.addedNodes.forEach((node) => {
                    if (node.nodeType === 1 && node.hasAttribute?.('wire:loading')) {
                        const target = node.getAttribute('wire:target') || 'default';
                        const loaderId = `wire-loading-${target}`;

                        if (!activeLoaders.has(loaderId)) {
                            activeLoaders.add(loaderId);
                            window.AdvancedLoader.create(loaderId, {
                                type: 'processing',
                                autoShow: true,
                                timeout: 6000
                            });
                        }
                    }
                });

                mutation.removedNodes.forEach((node) => {
                    if (node.nodeType === 1 && node.hasAttribute?.('wire:loading')) {
                        const target = node.getAttribute('wire:target') || 'default';
                        const loaderId = `wire-loading-${target}`;

                        window.AdvancedLoader.hide(loaderId);
                        activeLoaders.delete(loaderId);
                    }
                });
            });
        } catch (error) {
            console.warn('Advanced Loader: Error in mutation observer:', error);
        }
    });

    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
});

export default AdvancedLoader;
