<?php

namespace App\Traits\Employwee;

use App\Models\User\UserEmployee;
use Carbon\Carbon;
use App\Models\User\UserEmployeeLeave;
use Illuminate\Support\Facades\Auth;



trait UserEmployeesLeaves
{




    public function openLeaveModal(int $empId)
    {
        $this->leaveEmployeeId = $empId;

        // إعادة الضبط كاملة
        $this->reset([
            'type', // leave_type
            'start_date',
            'end_date',
            'openEnded',
            'applyNow',
            'notes',
        ]);

        $this->activeModal = 'leaveEmp';
    }

    public function saveLeave(): void
    {
        // 1. إذا تم التقديم الآن، اجعلي تاريخ البداية اليوم
        if ($this->applyNow) {
            $this->start_date = now()->format('Y-m-d');
        }

        // 2. تأكد أن تاريخ النهاية ليس قبل البداية
        if ($this->start_date !== null && $this->end_date !== null && Carbon::parse($this->end_date)->lt(Carbon::parse($this->start_date))) {
            $this->addError('end_date', __('backend.error_end_after_start'));
            return;
        }

        // 3. إعداد قواعد التحقق
        $rules = [
            'type' => ['required', 'in:regular,project'],
            'notes' => ['nullable', 'string'],
        ];

        // 4. تحديد الحالة الافتراضية
        $status = $this->applyNow ? 'activated' : 'pending';

        if (!$this->applyNow) {
            $rules['start_date'] = ['required', 'date'];
        }

        // 5. إذا كان النوع عادي ويُطلب تحديد نهاية
        if (!$this->openEnded) {
            $rules['end_date'] = ['required', 'date', 'after_or_equal:start_date'];
        }

        // 6. نفّذ التحقق
        $this->validate($rules, [], $this->attributes());

        // 7. تخلية أي إجازات سابقة متداخلة
        UserEmployeeLeave::refreshStatusesForEmployee($this->leaveEmployeeId);

        // 8. جلب الإجازات المفعلة أو المعلقة للتحقق من التداخل
        $pending = UserEmployeeLeave::where('employee_id', $this->leaveEmployeeId)
            ->whereIn('status', ['pending', 'activated'])
            ->get();

        if ($pending->first(fn($l) => is_null($l->end_date))) {
            $this->dispatch('notify', type: 'error', message: __('backend.leave_error_end_open'), duration: 6000);
            return;
        }

        $newStart = Carbon::parse($this->start_date)->startOfDay();
        $newEnd = $this->end_date ? Carbon::parse($this->end_date)->endOfDay() : $newStart;

        foreach ($pending as $l) {
            $existStart = $l->start_date->startOfDay();
            $existEnd = $l->end_date ? $l->end_date->endOfDay() : $existStart;

            if ($newStart->lte($existEnd) && $existStart->lte($newEnd)) {
                $this->dispatch('notify', type: 'error', message: __('backend.leave_error_overlap'), duration: 6000);
                return;
            }
        }

        // 9. إنشاء سجل الإجازة
        UserEmployeeLeave::create([
            'created_by' => Auth::id(),
            'status' => $status,
            'leave_type' => $this->type,
            'employee_id' => $this->leaveEmployeeId,
            'start_date' => $this->start_date,
            'end_date' => $this->type === 'regular' && !$this->openEnded ? $this->end_date : null,
            'note' => $this->notes,
        ]);

        // 10. إبلاغ النجاح وإعادة تحميل البيانات
        $this->dispatch('notify', type: 'success', message: __('backend.leave_saved_successfully'), duration: 6000);

        $this->activeModal = null;
        $this->loadEmployees();
        $this->readyToLoad = true;
    }


    public string $employeeName = '';


    public function openLeavessModal(int $empId)
    {
        $this->leaveEmployeeId = $empId;
        $emp = UserEmployee::find($empId);
        $this->employeeName = $this->employeeName = $emp->full_name[app()->getLocale()] ?? ($emp->full_name['ar'] ?? '-');
        UserEmployeeLeave::refreshStatusesForEmployee($this->leaveEmployeeId);
        $this->currentStep = 1;
        $this->loadLeavess($empId);
        $this->activeModal = 'manageLeaves';
    }

    public function loadleavess(int $empId): void
    {
        $this->leaves = UserEmployeeLeave::getCurrentAndUpcoming($empId);
    }

    public function cancelLeave(): void
    {
        $leaveId = $this->deleteLeaveId;

        // 1️⃣ تحقق من سبب الإلغاء
        $this->validate(['cancelReason' => 'required|string|max:255'], [], ['cancelReason' => __('backend.reason_for_cancellation')]);

        // 2️⃣ حدّث حالات الإجازات قبل الإلغاء
        UserEmployeeLeave::refreshStatusesForEmployee($this->leaveEmployeeId);

        // 3️⃣ ابحث عن الإجازة وغيّرها إلى cancelled
        $leave = UserEmployeeLeave::findOrFail($leaveId);
        $leave->update([
            'status' => 'cancelled',
            'cancelled_by' => Auth::id(),
            'cancelled_at' => now(),
            'cancelled_reason' => $this->cancelReason,
        ]);

        // 4️⃣ نظّف الحقول وأعد تحميل القائمة
        $this->deleteLeaveId = null;
        $this->cancelReason = null;
        $this->loadLeavess($this->leaveEmployeeId);

        // 5️⃣ إشعار نجاح وإطلاق الحدث
        $this->dispatch('notify', type: 'success', message: __('backend.leave_cancelled_successfully'), duration: 6000);
        $this->dispatch('leave-updated');
    }

    public function editLeave(int $leaveId): void
    {
        $l = UserEmployeeLeave::findOrFail($leaveId);

        $this->editLeaveId = $leaveId;
        $this->editType = $l->leave_type;
        $this->editApplyNow = $l->status === 'activated' && is_null($l->start_date) === false;
        $this->editStartDate = $l->start_date?->toDateString();
        $this->editEndDate = $l->end_date?->toDateString();
        $this->editOpenEnded = is_null($l->end_date);
        $this->editNotes = $l->note;
        $this->currentStep = 2; // أو الخطوة المناسبة في ال wizard
    }




    public function updateLeave(): void
    {
        // 1️⃣ إذا "تطبيق الآن" فعطّل تاريخ البداية
        if ($this->editApplyNow) {
            $this->editStartDate = now()->format('Y-m-d');
        }

        // 2️⃣ تأكد أن النهاية ليست قبل البداية
        if (
            $this->editStartDate && $this->editEndDate
            && Carbon::parse($this->editEndDate)->lt(Carbon::parse($this->editStartDate))
        ) {
            $this->addError('editEndDate', __('backend.error_end_after_start'));
            return;
        }

        // 3️⃣ لو "تطبيق الآن"، نجعل البداية فارغة بعد الحفظ
        $status = 'applied';
        if (! $this->editApplyNow) {
            $status = 'pending';
        } elseif ($this->editApplyNow) {
            $status = 'activated';
        }

        // 4️⃣ إعداد قواعد التحقق
        $rules = [
            'editType' => ['required', 'in:regular,project'],
            'editNotes' => ['nullable', 'string'],
        ];

        if (! $this->editApplyNow) {
            $rules['editStartDate'] = ['required', 'date'];
        }
        if (! $this->editOpenEnded) {
            $rules['editEndDate'] = ['required', 'date', 'after_or_equal:editStartDate'];
        }

        // 5️⃣ نفّذ التحقق
        $this->validate($rules, [], $this->attributes());

        // 6️⃣ حدّث حالات الموظف قبل الفحص
        UserEmployeeLeave::refreshStatusesForEmployee($this->leaveEmployeeId);

        // 7️⃣ جلب الإجازات المعلقة والمفعلة (باستثناء هذه الإجازة)
        $pending = UserEmployeeLeave::where('employee_id', $this->leaveEmployeeId)
            ->where('id', '!=', $this->editLeaveId)
            ->whereIn('status', ['pending', 'activated'])
            ->get();

        // 8️⃣ خطأ لو في إجازة مفتوحة
        if ($pending->first(fn($l) => is_null($l->end_date))) {
            $this->dispatch('notify', type: 'error', message: __('backend.leave_error_end_open'), duration: 6000);
            return;
        }

        // 9️⃣ فحص التداخل الزمني
        $newStart = Carbon::parse($this->editStartDate)->startOfDay();
        $newEnd   = ! $this->editOpenEnded && $this->editEndDate
            ? Carbon::parse($this->editEndDate)->endOfDay()
            : $newStart;

        foreach ($pending as $l) {
            $existStart = Carbon::parse($l->start_date)->startOfDay();
            $existEnd   = $l->end_date
                ? Carbon::parse($l->end_date)->endOfDay()
                : $existStart;

            if ($newStart->lte($existEnd) && $existStart->lte($newEnd)) {
                $this->dispatch('notify', type: 'error', message: __('backend.leave_error_overlap'), duration: 6000);
                return;
            }
        }

        //  🔟 تطبيق التعديل على قاعدة البيانات
        $leave = UserEmployeeLeave::findOrFail($this->editLeaveId);
        $leave->update([
            'status'            => $status,
            'leave_type'        => $this->editType,
            'start_date'        => $this->editApplyNow ? null : $this->editStartDate,
            'end_date'          => $this->editOpenEnded ? null : $this->editEndDate,
            'note'              => $this->editNotes,
        ]);

        // ١١️⃣ إعادة تحميل القائمة
        $this->loadLeavess($this->leaveEmployeeId);

        // ١٢️⃣ إشعار نجاح
        $this->dispatch('notify', type: 'success', message: __('backend.leave_updated_successfully'), duration: 6000);
        $this->dispatch('leave-updated');
    }
}
