<?php

namespace App\Models\User;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;



class UserEmployeeLeave extends Model
{

    use HasFactory;
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'users_employee_leaves';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'created_by',
        'status',
        'leave_type',
        'employee_id',
        'start_date',
        'end_date',
        'job_uuid',
        'cancelled_by',
        'cancelled_at',
        'cancelled_reason',
        'note',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'start_date'     => 'date',
        'end_date'       => 'date',
        'cancelled_at'   => 'datetime',
    ];

   /**
     * المستخدم الذي أنشأ الإجازة.
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(\App\Models\User::class, 'created_by');
    }

    /**
     * الموظف المرتبط بهذه الإجازة.
     */
    public function employee(): BelongsTo
    {
        return $this->belongsTo(UserEmployee::class, 'employee_id');
    }

    /**
     * المستخدم الذي ألغى أو رفض الإجازة.
     */
    public function cancelledBy(): BelongsTo
    {
        return $this->belongsTo(\App\Models\User::class, 'cancelled_by');
    }



    /**
     * حالة الإجازة كنصّ مفصّل (يمكنك تعديلها حسب الحاجة).
     */
    public function getStatusLabelAttribute(): string
    {
        return match ($this->status) {
            'pending'   => __('backend.leave_status_pending'),
            'activated' => __('backend.leave_status_activated'),
            'applied'   => __('backend.leave_status_applied'),
            'cancelled' => __('backend.leave_status_cancelled'),
            default     => $this->status,
        };
    }

    /**
     * نوع الإجازة كنصّ.
     */
    public function getTypeLabelAttribute(): string
    {
        return match ($this->leave_type) {
            'regular' => __('backend.leave_type_regular'),
            'project' => __('backend.leave_type_project'),
            default   => $this->leave_type,
        };
    }


/**
     * 1. تحديث حالات جميع الإجازات في النظام:
     *    - pending → activated عندما يصير now >= start_date (و–إذا end_date موجود: now <= end_date، أو open-ended)
     *    - activated → applied عندما تجاوزنا end_date
     *    (الحالات الأخرى تبقى كما هي: cancelled)
     */
    public static function refreshStatuses(): void
    {
        $now = Carbon::now();

        // pending → activated (open-ended أو داخل النطاق)
        self::where('status', 'pending')
            ->whereDate('start_date', '<=', $now)
            ->where(function($q) use($now) {
                $q->whereNull('end_date')
                  ->orWhereDate('end_date', '>=', $now);
            })
            ->update(['status' => 'activated']);

        // activated → applied (انتهى end_date)
        self::where('status', 'activated')
            ->whereNotNull('end_date')
            ->whereDate('end_date', '<', $now)
            ->update(['status' => 'applied']);
    }

    /**
     * 2. تحديث حالات الإجازات لموظف واحد:
     */
    public static function refreshStatusesForEmployee(int $employeeId): void
    {
        $now = Carbon::now();

        // pending → activated
        self::where('employee_id', $employeeId)
            ->where('status', 'pending')
            ->whereDate('start_date', '<=', $now)
            ->where(function($q) use($now) {
                $q->whereNull('end_date')
                  ->orWhereDate('end_date', '>=', $now);
            })
            ->update(['status' => 'activated']);

        // activated → applied
        self::where('employee_id', $employeeId)
            ->where('status', 'activated')
            ->whereNotNull('end_date')
            ->whereDate('end_date', '<', $now)
            ->update(['status' => 'applied']);
    }

    /**
     * 3. جلب الإجازات الجارية والمقبلة لموظف:
     *    - اليوم أو بعده (start_date >= today)
     *    - أو تلك التي في حالة "activated"
     */
    public static function getCurrentAndUpcoming(int $employeeId): array
    {
        $today = Carbon::today();

        $leaves = self::with(['creator', 'cancelledBy'])
        ->where('employee_id', $employeeId)
        // ➊ الحالة المطلوبة فقط
        ->whereIn('status', ['activated', 'pending'])
        // ➋ شرط التاريخ (مستقبلية أو جارية الآن)
        ->where(function ($q) use ($today) {
            // أ – تبدأ اليوم أو بعده
            $q->whereDate('start_date', '>=', $today)
              // ب – أو جارية الآن (اليوم بين البداية والنهاية أو النهاية غير محددة)
              ->orWhere(function ($sub) use ($today) {
                  $sub->whereDate('start_date', '<=', $today)
                      ->where(function ($end) use ($today) {
                          $end->whereNull('end_date')                   // نهاية مفتوحة
                              ->orWhereDate('end_date', '>=', $today);  // أو اليوم ≤ النهاية
                      });
              });
        })
        ->orderBy('start_date')
        ->get();

        return $leaves->map(fn($l) => [
            'id'             => $l->id,
            'status'         => $l->status,
            'leave_type'     => $l->leave_type,
            'start_date'     => $l->start_date->toDateString(),
            'end_date'       => $l->end_date?->toDateString(),
            'note'           => $l->note,
            'created_by'     => optional($l->creator)->name,
            'cancelled_by'   => optional($l->cancelledBy)->name,
            'cancelled_at'   => $l->cancelled_at?->toDateTimeString(),
            'cancelled_reason'=> $l->cancelled_reason,
        ])->toArray();
    }

}
