<?php

namespace App\Models\User;

use App\Models\Emp\EmployeeDepartment;
use App\Models\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Carbon;

class UserEmployee extends Model
{
    protected $table = 'users_employees';

    protected $fillable = [
        'employee_code',
        'project_code',
        'department_id',
        'full_name',
        'phone',
        'job_title_id',
        'national_id',
        'birth_date',
        'governorate',
        'hiring_date',
        'contract_date',
        'gender',
        'id_card_image',
        'cv',
    ];

    protected $casts = [
        'full_name' => 'array',
        'governorate' => 'array',
        'id_card_image' => 'array',
        'phone' => 'array',
        'cv' => 'array',
        'birth_date' => 'date',
        'hiring_date' => 'date',
        'contract_date' => 'date',
    ];

    /**
     * علاقة الموظف بالمشروع.
     */
    public function project(): BelongsTo
    {
        return $this->belongsTo(UserProject::class, 'project_code', 'code');
    }

    public function jobTitle(): BelongsTo
    {
        return $this->belongsTo(UserJobTitle::class, 'job_title_id');
    }

    public function department(): BelongsTo
    {
        return $this->belongsTo(EmployeeDepartment::class, 'department_id');
    }


    public function user(): HasOne
    {
        return $this->hasOne(User::class, 'employee_id');
    }

    public function movements(): HasMany
    {
        return $this->hasMany(UserEmployeeMovement::class, 'employee_id');
    }

    public function leaves(): HasMany
    {
        return $this->hasMany(UserEmployeeLeave::class, 'employee_id');
    }


    public function getFullNameTranslatedAttribute(): string
    {
        $lang = app()->getLocale();
        $decoded = is_array($this->full_name) ? $this->full_name : json_decode($this->full_name, true);

        if (!is_array($decoded)) {
            return '';
        }

        return $decoded[$lang] ?? reset($decoded) ?? '';
    }



    /**
     * Accessor: اسم المحافظة حسب اللغة الحالية (ar|en).
     */
    public function getGovernorateNameAttribute(): string
    {
        $gov = $this->governorate;  // بالفعل مصفوفة ['code'=>'27','en'=>'Qena','ar'=>'قنا']
        if (! is_array($gov)) {
            return '-';
        }

        $locale = app()->getLocale(); // 'ar' أو 'en'
        // حاول ترجمة للّغة الحالية، أو fallback للعربية أو الإنجليزية
        return $gov[$locale]
            ?? $gov['ar']
            ?? $gov['en']
            ?? '-';
    }

    // لو حابة يظهر تلقائياً لما تعمل toArray/JSON
    protected $appends = [
        'governorate_name',
        'mission',
        'currentleave',
    ];

    public function getMissionAttribute(): ?string
{
    $type    = $this->current_movement_type;
    $end     = $this->current_movement_end_date;
    $project = $this->current_movement_project_name ?? '-';

    if (! $type) {
        return null;
    }

    $labelType = __("backend.{$type}");

    if (is_null($end)) {
        return __("backend.mission_open", [
            'type'    => $labelType,
            'project' => $project,
        ]);
    }

    $days = Carbon::today()->diffInDays(Carbon::parse($end));
    return __("backend.mission_ends_in", [
        'type'    => $labelType,
        'project' => $project,
        'days'    => $days,
    ]);
}


public function getCurrentLeaveAttribute(): ?string
{
    $type = $this->current_leave_type;
    $end  = $this->current_leave_end_date;

    // لو ما فيش إجازة حالية
    if (! $type) {
        return null;
    }

    // ترجمة نوع الإجازة، مثلاً 'annual' → __('backend.annual')
    $labelType = __("backend.{$type}");

    // إذا المفتوح بدون نهاية
    if (is_null($end)) {
        return __("backend.leave_open", [
            'type' => $labelType,
        ]);
    }

    // حساب عدد الأيام المتبقية
    $days = Carbon::today()->diffInDays(Carbon::parse($end), false);
    $days = $days < 0 ? 0 : $days;

    return __("backend.leave_ends_in", [
        'type' => $labelType,
        'days' => $days,
    ]);
}

}
