<?php

namespace App\Models\System;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SystemRoute extends Model
{
    protected $table = 'system_routes';

    protected $fillable = [
        'is_active',
        'prefix',
        'name',
        'url',
        'method',
        'type',
        'controller',
        'view_name',
        'middleware',
        'constraints',
        'route_properties',
        'auth',
        'main_group',
        'parent_id',
        'order',
    ];

    protected $casts = [
        'is_active'        => 'boolean',
        'auth'             => 'boolean',
        'controller'       => 'array',
        'middleware'       => 'array',
        'constraints'      => 'array',
        'route_properties' => 'array',
    ];

    /**
     * علاقة مع الـ Parent Group
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(SystemParent::class, 'parent_id');
    }

    /**
     * استخراج اسم العرض الحالي حسب اللغة
     */
    public function displayName(): string
    {
        $props = $this->route_properties ?? [];
        $locale = app()->getLocale();
        return $props['display_name'][$locale] ?? $this->name;
    }

    /**
     * استخراج الأيقونة
     */
    public function icon(): ?string
    {
        return $this->route_properties['icon'] ?? null;
    }

    /**
     * هل يظهر في القائمة؟
     */
    public function showInMenu(): bool
    {
        return $this->route_properties['is_menu'] ?? false;
    }

    public static function visibleFor($user)
{

    if (!$user) {
        return self::where('is_active', true)
            ->where('auth', false)
            ->get();
    }

    $userPages = [];

    if (isset($user->pages['page'])) {
        $raw = $user->pages['page'];
        $userPages = is_array($raw) ? $raw : [$raw];
    }

    if (in_array('[all]', $userPages)) {
        return self::where('is_active', true)->get();
    }

    if (!empty($userPages)) {
        return self::where('is_active', true)
            ->where(function ($query) use ($userPages) {
                $query->whereIn('name', $userPages)
                      ->orWhere('main_group', 'system');
            })
            ->get();
    }

    // fallback في حالة المستخدم موجود لكن مفيش صفحات محددة
    return self::where('is_active', true)
        ->where('main_group', 'system')
        ->get();
}



}
