<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Schema;
use App\Models\System\SystemRoute;
use Livewire\Volt\Volt;
use Illuminate\Support\Arr;



/**
 * 🔧 دالة مساعدة لتسجيل الروتات بوضوح ونظافة
 */
if (! function_exists('registerDynamicRoute')) {
    function registerDynamicRoute(SystemRoute $route)
    {
        $method      = strtolower($route->method ?? 'get');
        $url         = $route->url;
        $type        = strtolower($route->type);
        $prefix      = $route->prefix;
        $name        = $route->name;
        $middleware  = Arr::wrap($route->middleware);
        $controller  = $route->controller ?? [];
        $view        = $route->view_name;
        $constraints = Arr::wrap($route->constraints);

        $routeInstance = null;

        // 🔷 Volt Route
        if ($type === 'volt' && $view) {
            $routeInstance = Volt::route($url, $view);
        }

        // 🔷 Blade View
        elseif ($type === 'blade' && $view) {
            $routeInstance = Route::$method($url, fn() => view($view));
        }

        // 🔷 PDF via mPDF
        elseif ($type === 'pdf' && $view) {
            $routeInstance = Route::get($url, function () use ($view) {
                $mpdf = new \Mpdf\Mpdf([
                    'mode' => 'utf-8',
                    'default_font' => 'amiri',
                    'format' => 'A4',
                    'orientation' => 'P',
                ]);
                $html = view($view)->render();
                $mpdf->WriteHTML($html);
                return response($mpdf->Output('document.pdf', 'I'), 200)
                    ->header('Content-Type', 'application/pdf');
            });
        }

        // 🔷 Controller
        // 🔷 Controller
        elseif ($type === 'controller' && isset($controller['main'])) {
            $controllerClass = $controller['main'];

            // استخدام method من البيانات لو موجودة، ولو مش موجودة تبقى __invoke
            $controllerMethod = $controller['method'] ?? '__invoke';

            if (class_exists($controllerClass) && method_exists($controllerClass, $controllerMethod)) {
                $routeInstance = Route::$method($url, [$controllerClass, $controllerMethod]);
            } else {
                Route::$method($url, fn() => abort(404, 'Controller or method missing'))->name($name ?? null);
                return;
            }
        }


        // ❌ Fallback في حالة Route ناقص
        if (!$routeInstance) {
            Route::$method($url, fn() => abort(404, 'Route misconfigured'))->name($name ?? null);
            return;
        }

        // ✅ خصائص إضافية
        if ($name) {
            $routeInstance->name($name);
        }

        if (!empty($middleware)) {
            $routeInstance->middleware($middleware);
        }

        if (!empty($prefix)) {
            $routeInstance->prefix($prefix);
        }

        if (!empty($constraints)) {
            foreach ($constraints as $param => $pattern) {
                $routeInstance->where($param, $pattern);
            }
        }
    }
}

$isConsole = app()->runningInConsole();

if (Schema::hasTable('system_routes')) {

    // ⚙️ حالة console: تسجيل كل الروتات
    if ($isConsole) {
        $routes = SystemRoute::all();
        foreach ($routes as $route) {
            registerDynamicRoute($route);
        }
    } else {
        // 🌐 الروتات المتاحة للضيوف
        $guestRoutes = SystemRoute::where('is_active', true)
            ->where('auth', false)
            ->get();

        foreach ($guestRoutes as $route) {
            registerDynamicRoute($route);
        }

        // 🔐 الروتات التي تتطلب تسجيل دخول
        Route::middleware('auth')->group(function () {
            $authRoutes = SystemRoute::where('is_active', true)
                ->where('auth', true)
                ->get();

            foreach ($authRoutes as $route) {
                registerDynamicRoute($route);
            }
        });
    }
} else {
    Route::any('{any}', fn() => view('backsoon'))->where('any', '.*');
}
