<?php

use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Illuminate\Validation\Rules;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Locked;
use Livewire\Volt\Component;
use Illuminate\View\View;

new #[Layout('components.frontend.layouts.struct')] class extends Component {
    public function rendering(View $view): void
    {
        $view->title(__('frontend.tittlewelcomepage')  . ' | ' . __('frontend.reset_title'));
    }
    #[Locked]
    public string $token = '';
    public string $email = '';
    public string $password = '';
    public string $password_confirmation = '';

    /**
     * Mount the component.
     */
    public function mount(string $token): void
    {
        $this->token = $token;

        $this->email = request()->string('email');
    }

    /**
     * Reset the password for the given user.
     */
    public function resetPassword(): void
    {
        $this->validate([
            'token' => ['required'],
            'email' => ['required', 'string', 'email'],
            'password' => ['required', 'string', 'confirmed', Rules\Password::defaults()],
        ]);

        // Here we will attempt to reset the user's password. If it is successful we
        // will update the password on an actual user model and persist it to the
        // database. Otherwise we will parse the error and return the response.
        $status = Password::reset(
            $this->only('email', 'password', 'password_confirmation', 'token'),
            function ($user) {
                $user->forceFill([
                    'password' => Hash::make($this->password),
                    'remember_token' => Str::random(60),
                ])->save();

                event(new PasswordReset($user));
            }
        );

        // If the password was successfully reset, we will redirect the user back to
        // the application's home authenticated view. If there is an error we can
        // redirect them back to where they came from with their error message.
        if ($status != Password::PasswordReset) {
            $this->addError('email', __($status));

            return;
        }

        Session::flash('status', __($status));

        $this->redirectRoute('login', navigate: true);
    }
}; ?>

<div class="min-h-screen flex items-center justify-center bg-transparent transition duration-700 overflow-hidden relative">

    <div class="z-10 max-w-md w-full space-y-6 bg-white/80 dark:bg-zinc-900/80 backdrop-blur-md p-8 rounded-xl shadow-2xl border border-zinc-300 dark:border-zinc-700">
        <!-- 🟡 Logo + عنوان -->
        <div class="flex flex-col items-center gap-2 animate-fade-in-down">
            <img src="/assets/img/logo.png" alt="Logo" class="w-[300px] h-auto mb-3 drop-shadow-lg" />
            <h2 class="text-2xl font-semibold text-zinc-800 dark:text-white">
                {{ __('frontend.reset_title') }}
            </h2>
            <p class="text-sm text-zinc-500 dark:text-zinc-400 text-center">
                {{ __('frontend.reset_desc') }}
            </p>
        </div>


        <!-- 🔐 Form -->
        <form wire:submit="resetPassword" class="space-y-4 animate-fade-in-down">
            <!-- Email -->
            <div>
                <label for="email" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300">
                    {{ __('frontend.emailadress') }}
                </label>
                <input type="email" id="email" wire:model="email" required
                       class="mt-1 block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 ring-accent outline-none transition" />
            </div>

            <!-- Password -->
            <div>
                <label for="password" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300">
                    {{ __('frontend.password') }}
                </label>
                <input type="password" id="password" wire:model="password" required autocomplete="new-password"
                       placeholder="{{ __('frontend.password') }}"
                       class="mt-1 block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 ring-accent outline-none transition" />
            </div>

            <!-- Confirm Password -->
            <div>
                <label for="password_confirmation" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300">
                    {{ __('frontend.confirm') }}
                </label>
                <input type="password" id="password_confirmation" wire:model="password_confirmation" required autocomplete="new-password"
                       placeholder="{{ __('frontend.confirm') }}"
                       class="mt-1 block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 ring-accent outline-none transition" />
            </div>

            <!-- Submit -->
            <button type="submit" class="neon-btn neon-blue w-full delay-[500ms]">
                <span class="btn-text">{{ __('frontend.reset_button') }}</span>
                <span class="glow"></span>
            </button>
        </form>
    </div>
</div>
