<?php

use Livewire\Volt\Component;
use Livewire\WithPagination;
use App\Models\User\UserEmployee;
use Barryvdh\DomPDF\Facade\Pdf;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\EmployeesExport;
use Livewire\Attributes\Layout;
use Illuminate\Support\Facades\Response;
use Illuminate\View\View;

new #[Layout('components.backend.layouts.clear')] class extends Component {
    public string $search = '';
    public string $sortField = 'employee_code';
    public string $sortDirection = 'asc';
    public int $perPage = 10;

    public function mount(): void
    {
        // استقبلنا البيانات اللي جايه في URL
        $this->search = request()->filled('search') ? request('search') : '';
        $this->sortField = request()->filled('sortField') ? request('sortField') : 'employee_code';
        $this->sortDirection = request()->filled('sortDirection') ? request('sortDirection') : 'asc';
        $this->perPage = request()->filled('perPage') ? (int) request('perPage') : 10;
    }

    public function render(): mixed
    {
        // هذا الكومبوننت مش فيه واجهة عرض -> هو للتصدير فقط
        abort(404);
    }

    public function export()
    {
        $employees = UserEmployee::query()
            ->when($this->search, function ($query) {
                $query->where(function ($query) {
                    $query->where('employee_code', 'like', "%{$this->search}%")
                          ->orWhere('national_id', 'like', "%{$this->search}%")
                          ->orWhereRaw("LOWER(JSON_VALUE(full_name, '$.ar')) LIKE ?", ['%' . strtolower($this->search) . '%'])
                          ->orWhereRaw("LOWER(JSON_VALUE(full_name, '$.en')) LIKE ?", ['%' . strtolower($this->search) . '%']);
                });
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->limit($this->perPage)
            ->get();

        $pdf = Pdf::loadView('livewire.backend.employee.pdf', compact('employees'))
            ->setPaper('a4', 'portrait');

        return Response::streamDownload(function () use ($pdf) {
            echo $pdf->stream();
        }, 'employees_list.pdf');
    }
};

?>
<div>
    <style>
        body {
            font-family: 'DejaVu Sans', sans-serif;
            margin: 0;
            padding: 1cm;
            background: #fff;
            color: #000;
            font-size: 12px;
        }
        header {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            height: 120px;
            text-align: center;
            padding-top: 10px;
        }
        header img {
            height: 60px;
            margin-bottom: 5px;
        }
        header h1 {
            margin: 5px 0 0 0;
            font-size: 20px;
        }
        footer {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            height: 30px;
            text-align: center;
            font-size: 10px;
            color: #666;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 140px;
        }
        th, td {
            border: 1px solid #ccc;
            padding: 8px;
            text-align: center;
        }
        thead {
            background-color: #f0f0f0;
        }
        tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        .pagenum:before {
            content: counter(page);
        }
    </style>
    <header>
        {{-- 🔵 لوجو الشركة --}}
        <img src="{{ public_path('images/logo.png') }}" alt="Company Logo">
        {{-- 🔵 عنوان الصفحة --}}
        <h1>{{ __('Employees List') }}</h1>
    </header>

    <footer>
        {{ __('Page') }} <span class="pagenum"></span>
    </footer>

    <main>
        <table>
            <thead>
                <tr>
                    <th>{{ __('Employee Code') }}</th>
                    <th>{{ __('Full Name') }}</th>
                    <th>{{ __('Department') }}</th>
                    <th>{{ __('Job Title') }}</th>
                    <th>{{ __('Project') }}</th>
                    <th>{{ __('Gender') }}</th>
                </tr>
            </thead>
            <tbody>
                @foreach ($employees as $emp)
                    <tr>
                        <td>{{ $emp->employee_code }}</td>
                        <td>{{ $emp->full_name['ar'] ?? '—' }}</td>
                        <td>{{ $emp->department?->name_ar ?? '—' }}</td>
                        <td>{{ $emp->jobTitle?->name_ar ?? '—' }}</td>
                        <td>{{ $emp->project?->name_ar ?? '—' }}</td>
                        <td>{{ $emp->gender === 'male' ? __('Male') : __('Female') }}</td>
                    </tr>
                @endforeach
            </tbody>
        </table>
    </main>
</div>
