<?php
use Carbon\Carbon;
use Livewire\Volt\Component;
use Livewire\WithPagination;
use Livewire\Attributes\Layout;
use Illuminate\Support\Facades\Auth;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Response;
use Illuminate\View\View;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile as LivewireTemp;
use Mpdf\Mpdf;
use App\Models\User\UserEmployee;
use App\Models\User\UserProject;
use App\Models\User\UserEmployeeMovement;
use App\Models\User\UserEmployeeLeave;

//Traits
use App\Traits\test;
use App\Traits\Employwee\UserEmployees;
use App\Traits\Employwee\UserEmployeesMovements;
use App\Traits\Employwee\UserEmployeesLeaves;

new #[Layout('components.backend.layouts.struct')] class extends Component {
    use WithPagination;
    use WithFileUploads;
    //Add Details Edit Delete
    use UserEmployees, UserEmployeesMovements, UserEmployeesLeaves;

    public string $search = '';
    public string $sortField = 'employee_code';
    public string $sortDirection = 'asc';
    public bool $readyToLoad = false;
    public int $perPage = 10;
    protected array $messages = [];
    //----------------------------
    public $currentStep = 1;
    public $activeModal = null;
    //-----------------------------

    public $employee;
    public $editEmployeeCv = [];

    // ─── خصائص جديدة للحركة ───────────────────────────────────────────────
    public int $movementEmployeeId;
    public bool $applyNow = false;
    public string $type = '';
    public ?string $start_date = null;
    public ?string $end_date = null;
    public bool $openEnded = false;
    public ?int $to_project = null;
    public ?string $notes = null;
    public array $projects = [];
    protected $appends = ['mission', 'currentleave'];

    public array $movements = [];
    public ?int $editMovementId = null;
    public string $editType = '';
    public bool $editApplyNow = false;
    public ?string $editStartDate = null;
    public ?string $editEndDate = null;
    public bool $editOpenEnded = false;
    public ?int $editToProject = null;
    public ?string $editNotes = null;
    public ?int $deleteMovementId = null;
    public ?string $cancelReason = null;

    public int $leaveEmployeeId;
    public array $leaves = [];
    public ?int $deleteLeaveId = null;
    public ?int $editLeaveId = null;

    public function rendering(View $view): void
    {
        $view->title(__('backend.department_name') . ' | ' . __('backend.employees_count'));
    }
    public function loadEmployees()
    {
        UserEmployeeMovement::refreshStatuses();
        $this->readyToLoad = true;
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
        $this->readyToLoad = true;
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function print()
    {
        return redirect()->route('employees.print', [
            'search' => $this->search,
            'sortField' => $this->sortField,
            'sortDirection' => $this->sortDirection,
            'perPage' => $this->perPage,
        ]);
    }

    public function exportPdf()
    {
        $employees = $this->getEmployees(false);

        $html = view('livewire.backend.employee.pdf', compact('employees'))->render();

        $mpdf = new Mpdf([
            // 'format' => 'A4-L', // A4 Landscape
            'directionality' => 'rtl', // لو عايز الطباعة من اليمين لليسار
        ]);

        $mpdf->WriteHTML($html);

        return Response::streamDownload(function () use ($mpdf) {
            echo $mpdf->Output('', 'S');
        }, 'employees_list.pdf');
    }

    public function exportExcel()
    {
        $employees = $this->getEmployees(false);

        $data = $employees->map(function ($employee) {
            return [
                'Employee Code' => $employee->employee_code,
                'Full Name' => $employee->full_name['ar'] ?? '-',
                'Department' => $employee->department?->name_ar ?? '-',
                'Job Title' => $employee->jobTitle?->name_ar ?? '-',
                'Project' => $employee->project?->name_ar ?? '-',
                'Gender' => $employee->gender === 'male' ? __('Male') : __('Female'),
                'Governorate' => $employee->governorate['ar'] ?? '-',
            ];
        });

        return Excel::download(
            new class ($data) implements \Maatwebsite\Excel\Concerns\FromCollection, \Maatwebsite\Excel\Concerns\WithHeadings {
                protected $data;

                public function __construct($data)
                {
                    $this->data = $data;
                }

                public function collection()
                {
                    return $this->data;
                }

                public function headings(): array
                {
                    return ['Employee Code', 'Full Name', 'Department', 'Job Title', 'Project', 'Gender', 'Governorate'];
                }
            },
            'employees_list.xlsx',
        );
    }

    private function getEmployees($paginate = true)
    {
        // 1) الصفحات المسموحة (من عمود pages)
        $pagesData = Auth::user()->pages['page'] ?? [];
        // 2) العناصر المسموحة ضمن emp.index
        $elementsData = Auth::user()->pages['element']['emp.index'] ?? [];
        // 3) هل هو مدير عام ([all])؟
        $isSuperAdmin = in_array('[all]', $pagesData);
        $canViewAll = $isSuperAdmin || (in_array('emp.index', $pagesData) && in_array('employee.view_all', $elementsData));
        $canViewMission = $isSuperAdmin || (in_array('emp.index', $pagesData) && in_array('mission_employees_header', $elementsData));

        $query = UserEmployee::query()
            ->leftJoin('users_projects', 'users_employees.project_code', '=', 'users_projects.code')
            ->select('users_employees.*', 'users_projects.name as project_name', 'users_projects.code as project_code_search')
            // ➕ العمود الجديد: مأمورية حالية + عدد الأيام المتبقية
            ->selectSub(
                fn($q) => $q
                    ->from('users_employee_movements')
                    ->select('type')
                    ->whereColumn('users_employee_movements.employee_id', 'users_employees.id')
                    ->where('users_employee_movements.status', 'activated') // ← جدول.عمود + قيمة مقفولة
                    ->orderByDesc('start_date')
                    ->limit(1),
                'current_movement_type',
            )
            ->selectSub(fn($q) => $q->from('users_employee_movements')->select('end_date')->whereColumn('users_employee_movements.employee_id', 'users_employees.id')->where('users_employee_movements.status', 'activated')->orderByDesc('start_date')->limit(1), 'current_movement_end_date')
            ->selectSub(fn($q) => $q->from('users_employee_movements')->join('users_projects', 'users_employee_movements.to_project', '=', 'users_projects.id')->select('users_projects.name')->whereColumn('users_employee_movements.employee_id', 'users_employees.id')->where('users_employee_movements.status', 'activated')->orderByDesc('start_date')->limit(1), 'current_movement_project_name')

            ->selectSub(fn($q) => $q->from('users_employee_leaves')->select('leave_type')->whereColumn('employee_id', 'users_employees.id')->where('status', 'activated')->orderByDesc('start_date')->limit(1), 'current_leave_type')
            ->selectSub(fn($q) => $q->from('users_employee_leaves')->select('end_date')->whereColumn('employee_id', 'users_employees.id')->where('status', 'activated')->orderByDesc('start_date')->limit(1), 'current_leave_end_date');

        if (!$canViewAll) {
            // لو مسموح فقط بعرض الموظفين في مأمورية
            if ($canViewMission) {
                // نحصل على الـ project_id للموظف
                $myProjectCode = Auth::user()->project_code;
                $myProjectId = UserProject::where('code', $myProjectCode)->value('id');

                $query->where(function ($q) use ($myProjectCode, $myProjectId) {
                    $q->where('users_employees.project_code', $myProjectCode)->orWhereExists(function ($sub) use ($myProjectId) {
                        $sub->from('users_employee_movements')->whereColumn('users_employee_movements.employee_id', 'users_employees.id')->where('users_employee_movements.to_project', $myProjectId)->where('users_employee_movements.status', 'activated'); // ← الشرط الجديد
                    });
                });
            }
            // وإلا: عرض الموظفين فقط من نفس المشروع
            else {
                $query->where('users_employees.project_code', Auth::user()->project_code);
            }
        }

        $query->where(function ($query) {
            $query
                ->where('employee_code', 'like', "%{$this->search}%")
                ->orWhere('national_id', 'like', "%{$this->search}%")
                ->orWhereRaw("LOWER(JSON_VALUE(full_name, '$.ar')) LIKE ?", ['%' . strtolower($this->search) . '%'])
                ->orWhereRaw("LOWER(JSON_VALUE(full_name, '$.en')) LIKE ?", ['%' . strtolower($this->search) . '%'])
                ->orWhereRaw('LOWER(users_projects.name) LIKE ?', ['%' . strtolower($this->search) . '%'])
                ->orWhere('users_projects.code', 'like', "%{$this->search}%");
        });

        // دعم الـ sort على project_name
        if ($this->sortField === 'project_name') {
            $query->orderBy('users_projects.name', $this->sortDirection);
        } else {
            $query->orderBy($this->sortField, $this->sortDirection);
        }

        return $paginate ? $query->paginate($this->perPage) : $query->limit($this->perPage)->get();
    }

    public function render(): mixed
    {
        UserEmployeeMovement::refreshStatuses();
        $employees = collect();
        if ($this->readyToLoad) {
            $employees = $this->getEmployees();
        }

        return view('livewire.backend.employee.page', compact('employees'));
    }

    //----------  Defults ------------------------\\

    public function previousStep()
    {
        $this->currentStep--;
    }
    public function closeModal()
    {
        $this->activeModal = null;
    }

    protected function attributes(): array
    {
        return [
            'new.hire_date' => __('frontend.hire_date'),
            'new.name' => __('frontend.fullname'),
            'new.name_ar' => __('frontend.fullname'),
            'new.email' => __('frontend.emailadress'),
            'new.phone' => __('frontend.phone'),
            'new.phone.*' => __('frontend.phone'),
            'new.project_code' => __('frontend.project'),
            'new.national_id' => __('frontend.national_id'),
            'new.id_card_images' => __('frontend.attach_documents'),
            'new.job_title_id' => __('frontend.job_title'),
            'type' => __('backend.movement_type'),
            'start_date' => __('backend.start_date'),
            'end_date' => __('backend.end_date'),
            'to_project' => __('backend.to_project'),
            'notes' => __('backend.notes'),
            'applyNow' => __('backend.apply_now'),
            'openEnded' => __('backend.open_ended'),

            // حقول التعديل
            'editcv.name' => __('frontend.fullname'),
            'editcv.hire_date' => __('frontend.hire_date'),
            'editcv.contract_date' => __('frontend.contract_date'),
            'editType' => __('backend.movement_type'),
            'editStartDate' => __('backend.start_date'),
            'editEndDate' => __('backend.end_date'),
            'editToProject' => __('backend.to_project'),
            'editNotes' => __('backend.notes'),
            'editApplyNow' => __('backend.apply_now'),
            'editOpenEnded' => __('backend.open_ended'),

            // حقل سبب الإلغاء
            'cancelReason' => __('backend.reason_for_cancellation'),
        ];
    }
};

?>
<div class="p-4 bg-white dark:bg-zinc-900 rounded-xl shadow text-sm text-right text-zinc-800 dark:text-zinc-100"
    wire:init="loadEmployees">


    <div class="print:hidden">
        @access('emp.index', 'employee.search')
            <div
                class="bg-white dark:bg-zinc-900 border border-zinc-200 dark:border-zinc-700 rounded-xl shadow-sm px-4 py-5 transition-all">
                <!-- البحث -->

                <div class="flex flex-col md:flex-row rtl:md:flex-row-reverse justify-between items-center gap-4">
                    {{-- مربع البحث --}}

                    <div class="order-1 rtl:md:order-2 flex-1">
                        <label for="search"
                            class="block text-sm font-semibold text-gray-700 dark:text-zinc-300 mb-1 rtl:text-right ltr:text-left">
                            {{ __('backend.search_label') }}
                        </label>

                        <div class="relative">
                            <input id="search" type="text" wire:model.live="search"
                                placeholder="{{ __('backend.search_placeholder') }}"
                                class="peer w-full h-12 px-4 rounded-lg
                 border border-gray-300 bg-gray-50 text-gray-900
                 placeholder-transparent
                 focus:ring-2 focus:ring-indigo-200 focus:border-indigo-500
                 dark:border-zinc-600 dark:bg-zinc-800 dark:text-zinc-100 dark:focus:border-blue-400
                 transition shadow-sm text-start" />

                            {{-- Placeholder عائم --}}
                            <label for="search"
                                class="absolute start-3 top-1/2 -translate-y-1/2 text-sm text-gray-500 dark:text-zinc-400
                      peer-placeholder-shown:top-1/2 peer-placeholder-shown:text-base peer-placeholder-shown:-translate-y-1/2
                      transition-all">
                                {{ __('backend.search_placeholder') }}
                            </label>
                        </div>
                    </div>
                </div>


                {{-- جاري التحميل --}}
                <div wire:loading.delay wire:target="search"
                    class="mt-2 text-sm text-gray-500 dark:text-zinc-400 animate-pulse">
                    {{ __('Searching...') }}
                </div>


            </div>
        @endaccess


        <div
            class="bg-white dark:bg-zinc-900 border border-zinc-200 dark:border-zinc-700 rounded-xl shadow-sm px-4 py-5 transition-all">


            {{-- مجموعة أزرار جانبية --}}
            <div class="flex flex-wrap gap-2 items-center">


                @access('emp.index', 'employee.add_new')
                    <button wire:click="openAddEmpModal()"
                        class="bg-indigo-600 hover:bg-indigo-700 text-white text-sm font-semibold px-6 py-2 rounded-md transition duration-300 transform hover:scale-105 shadow hover:shadow-lg">
                        ➕ {{ __('backend.add_new') }}
                    </button>
                @endaccess
                @access('emp.index', 'employee.printpage')
                    {{-- 🔵 زر الطباعة --}}
                    <button onclick="window.print()"
                        class="inline-flex items-center gap-1 px-3 py-2 rounded-md bg-yellow-500 hover:bg-yellow-600 dark:bg-yellow-400 dark:hover:bg-yellow-500 text-white text-sm font-medium transition">

                        🖨️ {{ __('frontend.printpage') }}
                    </button>
                @endaccess
                @access('emp.index', 'employee.download_pdf')
                    {{-- 🔴 زر تصدير PDF --}}
                    <button wire:click="exportPdf"
                        class="inline-flex items-center gap-1 px-3 py-2 rounded-md bg-red-600 hover:bg-red-700 dark:bg-red-500 dark:hover:bg-red-600 text-white text-sm font-medium transition">
                        📄 {{ __('frontend.download_pdf') }}
                    </button>
                @endaccess
                @access('emp.index', 'employee.download_excel')
                    {{-- 🟡 زر تصدير Excel --}}
                    <button wire:click="exportExcel"
                        class="inline-flex items-center gap-1 px-3 py-2 rounded-md bg-green-600 hover:bg-green-700 dark:bg-green-500 dark:hover:bg-green-600 text-white text-sm font-medium transition">
                        📊 {{ __('frontend.download_excel') }}
                    </button>
                @endaccess

            </div>

        </div>


    </div>





    @if (!$readyToLoad)
        <div class="text-center text-gray-500 dark:text-gray-400 p-8">
            ⏳ جاري تحميل الموظفين...
        </div>
    @else
        {{-- resources/views/livewire/backend/user/employees-table.blade.php --}}
        <div id="printable-area"
            class="overflow-x-auto rounded-lg border border-zinc-300 dark:border-zinc-700 shadow-sm">

            {{-- Header for print --}}
            <div class="hidden print:block fixed top-0 w-full text-center mb-6">
                <img src="{{ asset('assets/img/logo.png') }}" class="h-20 mb-2" alt="Company Logo">
                <h1 class="text-2xl font-bold text-zinc-800 dark:text-zinc-100">
                    {{ __('Employees List') }}
                </h1>
            </div>
            <!-- فقط على الشاشات الصغيرة -->
            <div class="block sm:hidden mb-4 px-4">

                <select id="perPageMobile" wire:model.live="perPage"
                    class="w-full text-base rounded-lg bg-gray-50 border border-gray-300
                 focus:ring-2 focus:ring-indigo-200 focus:border-indigo-500
                 dark:bg-zinc-800 dark:border-zinc-600 dark:text-zinc-100 transition">
                    <option value="5">5</option>
                    <option value="10">10</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
            </div>

            <table
                class="min-w-full table-auto divide-y divide-gray-300 dark:divide-zinc-700 text-sm
           print:border print:border-gray-400 print:border-collapse">

                <thead
                    class="sticky top-0 z-20
                  bg-gray-200 dark:bg-zinc-900
                  border-b border-gray-200 dark:border-zinc-700
                  print:divide-y print:divide-gray-400">
                    <tr class="text-center print:text-center">
                        @access('emp.index', 'employee.employee_code')
                            <th wire:click="sortBy('employee_code')"
                                class="px-6 py-3 cursor-pointer whitespace-nowrap
                        text-sm sm:text-base lg:text-lg font-medium uppercase tracking-wide
                        text-gray-800 dark:text-zinc-200
                        hover:bg-gray-50 dark:hover:bg-zinc-800
                        transition-colors print:border print:border-gray-400">
                                {{ __('backend.employee_code') }}
                                @if ($sortField === 'employee_code')
                                    <span class="inline-block ml-1 text-xs">
                                        {!! $sortDirection === 'asc' ? '↑' : '↓' !!}
                                    </span>
                                @endif
                            </th>
                        @endaccess
                        @access('emp.index', 'employee.fullname')
                            <th wire:click="sortBy('full_name')"
                                class="px-6 py-3 cursor-pointer whitespace-nowrap
                        text-sm sm:text-base lg:text-lg font-medium uppercase tracking-wide
                        text-gray-600 dark:text-zinc-300
                        hover:bg-gray-50 dark:hover:bg-zinc-800
                        transition-colors print:border print:border-gray-400">
                                {{ __('frontend.fullname') }}
                                @if ($sortField === 'full_name')
                                    <span class="inline-block ml-1 text-xs">
                                        {!! $sortDirection === 'asc' ? '↑' : '↓' !!}
                                    </span>
                                @endif
                            </th>
                        @endaccess
                        @access('emp.index', 'employee.project_name')
                            <th wire:click="sortBy('project_name')"
                                class="hidden lg:table-cell px-6 py-3 cursor-pointer whitespace-nowrap
                        text-sm sm:text-base lg:text-lg font-medium uppercase tracking-wide
                        text-gray-600 dark:text-zinc-300
                        hover:bg-gray-50 dark:hover:bg-zinc-800
                        transition-colors print:border print:border-gray-400">
                                {{ __('backend.projects') }}
                                @if ($sortField === 'project_name')
                                    <span class="inline-block ml-1 text-xs">
                                        {!! $sortDirection === 'asc' ? '↑' : '↓' !!}
                                    </span>
                                @endif
                            </th>
                        @endaccess
                        @access('emp.index', 'employee.job_title')
                            <th
                                class="hidden md:table-cell px-6 py-3
                        text-sm sm:text-base lg:text-lg font-medium uppercase tracking-wide
                        text-gray-600 dark:text-zinc-300
                        print:border print:border-gray-400">
                                {{ __('frontend.job_title') }}
                            </th>
                        @endaccess
                        @access('emp.index', 'employee.mission')
                            <th
                                class="hidden md:table-cell px-6 py-3
                        text-sm sm:text-base lg:text-lg font-medium uppercase tracking-wide
                        text-gray-600 dark:text-zinc-300
                        print:border print:border-gray-400">
                                {{ __('backend.mission') }}
                            </th>
                        @endaccess
                        @access('emp.index', 'employee.current_leave')
                            <th
                                class="hidden md:table-cell px-6 py-3
                        text-sm sm:text-base lg:text-lg font-medium uppercase tracking-wide
                        text-gray-600 dark:text-zinc-300
                        print:border print:border-gray-400">
                                {{ __('backend.current_leave') }}
                            </th>
                        @endaccess
                        @access('emp.index', 'employee.add_more_details')
                            <th
                                class="px-6 py-3 text-sm sm:text-base lg:text-lg font-medium uppercase tracking-wide text-gray-600 dark:text-zinc-300 hover:bg-gray-50 dark:hover:bg-zinc-800 transition-colors print:border print:border-gray-400">
                                {{ __('backend.add_more_details') }}
                            </th>
                        @endaccess
                        <th class="hidden sm:table-cell print:hidden px-6 py-3 text-center w-[200px]">
                            <select id="perPage" wire:model.live="perPage"
                                class="w-full text-base lg:text-lg rounded-lg bg-gray-50 border border-gray-300
                              focus:ring-2 focus:ring-indigo-200 focus:border-indigo-500
                              dark:bg-zinc-800 dark:border-zinc-600 dark:text-zinc-100 transition">
                                <option value="5">5</option>
                                <option value="10">10</option>
                                <option value="25">25</option>
                                <option value="50">50</option>
                                <option value="100">100</option>
                            </select>
                        </th>
                    </tr>
                </thead>


                <tbody
                    class="divide-y divide-gray-100 dark:divide-zinc-700 text-zinc-900 dark:text-zinc-200
              print:divide-y print:divide-gray-300">
                    @forelse ($employees as $emp)
                        <tr class="hover:bg-zinc-50 dark:hover:bg-zinc-800 transition text-center ">
                            @access('emp.index', 'employee.employee_code')
                                <td data-label="{{ __('backend.employee_code') }}"
                                    class="px-4 py-2 print:border print:border-gray-400">
                                    {{ $emp->employee_code }}
                                </td>
                            @endaccess
                            @access('emp.index', 'employee.fullname')
                                <td data-label="{{ __('frontend.fullname') }}"
                                    class="px-4 py-2 print:border print:border-gray-400">
                                    {{ $emp?->full_name_translated ?? '-' }}
                                </td>
                            @endaccess
                            @access('emp.index', 'employee.project_name')
                                <td data-label="{{ __('backend.project_name') }}"
                                    class="px-4 py-2 hidden lg:table-cell print:border print:border-gray-400">
                                    {{ $emp->project?->name ?? '—' }}
                                </td>
                            @endaccess
                            @access('emp.index', 'employee.job_title')
                                <td data-label="{{ __('frontend.job_title') }}"
                                    class="px-4 py-2 hidden md:table-cell print:border print:border-gray-400">
                                    {{ $emp->jobTitle?->{'title_' . $lang} ?? '-' }}
                                </td>
                            @endaccess
                            @access('emp.index', 'employee.mission')
                                <td data-label="{{ __('backend.mission') }}"
                                    class="px-4 py-2 hidden md:table-cell print:border print:border-gray-400">
                                    {{ $emp->mission ?? '—' }}
                                </td>
                            @endaccess
                            @access('emp.index', 'employee.current_leave')
                                <td data-label="{{ __('backend.current_leave') }}"
                                    class="px-4 py-2 hidden md:table-cell print:border print:border-gray-400">
                                    {{ $emp->currentleave ?? '—' }}
                                </td>
                            @endaccess

                            @access('emp.index', 'employee.add_more_details')
                                <td data-label="{{ __('backend.add_more_details') }}"
                                    class="px-4 py-2 print:border print:border-gray-400">
                                    <div class="flex flex-col items-center space-y-2">
                                        @php
                                            $cvData = json_decode($emp->cv, true) ?? []; // فك تشفير JSON إلى مصفوفة
                                            $rate = ($cvData['rate'] ?? 0) / 2; // تحويل التقييم إلى 5 نجوم
                                            $fullStars = floor($rate); // عدد النجوم الكاملة
                                            $hasHalfStar = $rate - floor($rate) > 0; // هل يوجد جزء عشري؟
                                        @endphp
                                        <!-- نوع العقد -->
                                        <span class="text-sm font-medium text-gray-700 dark:text-zinc-300">
                                            @if (!empty($cvData['contract_type']))
                                                {{ __('backend.' . $cvData['contract_type']) }}
                                            @endif
                                        </span>
                                        <!-- الدرجة -->
                                        <span class="text-sm text-gray-500 dark:text-zinc-400">
                                            @if (!empty($cvData['grade']))
                                                {{ __('backend.grade') }}: {{ $cvData['grade'] ?? '-' }}
                                            @endif
                                        </span>
                                        <!-- التقييم بالنجوم -->
                                        <div class="flex justify-center space-x-1">
                                            @for ($i = 1; $i <= 5; $i++)
                                                <!-- عرض 5 نجوم فقط -->
                                                @if ($i <= $fullStars)
                                                    <!-- نجمة كاملة -->

                                                    <svg xmlns="http://www.w3.org/2000/svg" fill="currentColor"
                                                        viewBox="0 0 24 24" class="w-5 h-5 text-yellow-500">
                                                        <path
                                                            d="M12 .587l3.668 7.431 8.2 1.192-5.934 5.782 1.4 8.168L12 18.896l-7.334 3.864 1.4-8.168L.166 9.21l8.2-1.192z" />
                                                    </svg>
                                                @elseif ($hasHalfStar && $i <= $fullStars + 1)
                                                    <!-- نصف نجمة -->
                                                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"
                                                        class="w-5 h-5">
                                                        <defs>
                                                            <linearGradient id="halfStarGradient" x1="0%"
                                                                y1="0%" x2="100%" y2="0%">
                                                                <stop offset="50%"
                                                                    stop-color="{{ $lang === 'ar' ? 'gray' : '#F59E0B' }}" />
                                                                <stop offset="50%"
                                                                    stop-color="{{ $lang === 'ar' ? '#F59E0B' : 'gray' }}" />
                                                            </linearGradient>
                                                        </defs>
                                                        <path fill="url(#halfStarGradient)"
                                                            d="M12 .587l3.668 7.431 8.2 1.192-5.934 5.782 1.4 8.168L12 18.896l-7.334 3.864 1.4-8.168L.166 9.21l8.2-1.192z" />
                                                    </svg>
                                                @else
                                                    <!-- نجمة غير مضيئة -->
                                                    <svg xmlns="http://www.w3.org/2000/svg" fill="currentColor"
                                                        viewBox="0 0 24 24" class="w-5 h-5 text-gray-400">
                                                        <path
                                                            d="M12 .587l3.668 7.431 8.2 1.192-5.934 5.782 1.4 8.168L12 18.896l-7.334 3.864 1.4-8.168L.166 9.21l8.2-1.192z" />
                                                    </svg>
                                                @endif
                                            @endfor
                                        </div>
                                    </div>
                                </td>
                            @endaccess

                            @php
                                $isRtl = app()->getLocale() !== 'ar';
                                // نحصل على مصفوفة الصلاحيات الخاصة بالـ emp.index أو نجعلها فارغة إذا مش موجودة
                                $empPerms = Auth::user()->settings['element']['emp.index'] ?? [];
                                $pages = Auth::user()->pages['page'] ?? [];
                            @endphp
                            @if (in_array('[all]', $pages) ||
                                    in_array('employee.details', $empPerms) ||
                                    in_array('employee.edit', $empPerms) ||
                                    in_array('employee.transfer', $empPerms) ||
                                    in_array('employee.manage_movements', $empPerms) ||
                                    in_array('employee.apply_leave', $empPerms) ||
                                    in_array('employee.manage_leaves', $empPerms))
                                <td class="print:hidden px-4 py-2">
                                    <div x-data="{ open: false }" class="relative inline-block">
                                        <!-- الزر الرئيسي -->
                                        <button @click="open = !open"
                                            class="relative p-3 rounded-full
                                         bg-gradient-to-r from-sky-500 to-indigo-600
                                         text-white shadow-lg hover:shadow-2xl
                                         focus:outline-none focus:ring-4 focus:ring-sky-300
                                         transition-transform duration-300 ease-in-out transform"
                                            :class="{ 'rotate-90 scale-110': open }">
                                            <i class="ri-more-2-fill text-2xl"></i>
                                            <span
                                                class="absolute -top-1 -right-1 block w-2 h-2
                                           bg-red-500 rounded-full animate-ping"></span>
                                        </button>

                                        <!-- قائمة العناصر -->
                                        <div x-show="open" x-cloak @click.away="open = false"
                                            x-transition:enter="transition ease-out duration-100"
                                            x-transition:enter-start="opacity-0 scale-95"
                                            x-transition:enter-end="opacity-100 scale-100"
                                            x-transition:leave="transition ease-in duration-75"
                                            x-transition:leave-start="opacity-100 scale-100"
                                            x-transition:leave-end="opacity-0 scale-95"
                                            class="absolute top-full mt-2
                                         {{ $isRtl ? 'right-0 origin-top-right' : 'left-0 origin-top-left' }}
                                         bg-white dark:bg-zinc-900
                                         divide-y divide-gray-200 dark:divide-zinc-700
                                         rounded-lg shadow-lg ring-1 ring-black ring-opacity-5 z-20
                                         w-40 sm:w-56">
                                            <!-- التفاصيل -->
                                            @access('emp.index', 'employee.details')
                                                <button wire:click="openDetailsModal({{ $emp->id }})"
                                                    class="flex items-center gap-2 w-full px-4 py-3
                                                 hover:bg-gray-100 dark:hover:bg-zinc-800
                                                 border-l-4 border-yellow-500">
                                                    <i class="ri-file-list-line text-yellow-500 text-xl"></i>
                                                    📋 {{ __('backend.details') }}
                                                </button>
                                            @endaccess
                                            @access('emp.index', 'employee.add_more_deatils')
                                                <button wire:click="openAddMoreDetailsModal({{ $emp->id }})"
                                                    class="flex items-center gap-2 w-full px-4 py-3
                                             hover:bg-gray-100 dark:hover:bg-zinc-800
                                             border-l-4 border-yellow-500">
                                                    <i class="ri-file-list-line text-yellow-500 text-xl"></i>
                                                    📋 {{ __('backend.add_more_details') }}
                                                </button>
                                            @endaccess
                                            @access('emp.index', 'employee.edit')
                                                <!-- التعديل -->
                                                <button wire:click="openEditEmpModal({{ $emp->id }})"
                                                    class="flex items-center gap-2 w-full px-4 py-3
                                                 hover:bg-gray-100 dark:hover:bg-zinc-800
                                                 border-l-4 border-orange-500">
                                                    <i class="ri-edit-line text-orange-500 text-xl"></i>
                                                    ✏️ {{ __('backend.edit') }}
                                                </button>
                                            @endaccess
                                            @access('emp.index', 'employee.transfer')
                                                <!-- النقل -->
                                                <button wire:click="openTransferModal({{ $emp->id }})"
                                                    class="flex items-center gap-2 w-full px-4 py-3
                                                 hover:bg-gray-100 dark:hover:bg-zinc-800
                                                 border-l-4 border-green-500">
                                                    <i class="ri-refresh-line text-green-500 text-xl"></i>
                                                    🔄 {{ __('backend.transfer') }}
                                                </button>
                                            @endaccess
                                            @access('emp.index', 'employee.manage_movements')
                                                <!-- إدارة الحركات -->
                                                <button wire:click="openMovementsModal({{ $emp->id }})"
                                                    class="flex items-center gap-2 w-full px-4 py-3
                                                 hover:bg-gray-100 dark:hover:bg-zinc-800
                                                 border-l-4 border-emerald-500">
                                                    <i class="ri-folder-shield-line text-emerald-500 text-xl"></i>
                                                    🛞 {{ __('backend.manage_movements') }}
                                                </button>
                                            @endaccess
                                            @access('emp.index', 'employee.apply_leave')
                                                <!-- إضافة إجازة -->
                                                <button wire:click="openLeaveModal({{ $emp->id }})"
                                                    class="flex items-center gap-2 w-full px-4 py-3
                                                 hover:bg-gray-100 dark:hover:bg-zinc-800
                                                 border-l-4 border-sky-500">
                                                    <i class="ri-calendar-event-line text-sky-500 text-xl"></i>
                                                    🗓️ {{ __('backend.apply_leave') }}
                                                </button>
                                            @endaccess
                                            @access('emp.index', 'employee.manage_leaves')
                                                <!-- إدارة الإجازات -->
                                                <button wire:click="openLeavessModal({{ $emp->id }})"
                                                    class="flex items-center gap-2 w-full px-4 py-3
                                                 hover:bg-gray-100 dark:hover:bg-zinc-800
                                                 border-l-4 border-indigo-500">
                                                    <i class="ri-file-list-3-line text-indigo-500 text-xl"></i>
                                                    📋 {{ __('backend.manage_leaves') }}
                                                </button>
                                            @endaccess
                                        </div>
                                    </div>
                                </td>
                            @endif


                        </tr>
                    @empty
                        <tr>
                            <td colspan="9" class="px-4 py-3 text-center text-zinc-500 dark:text-zinc-400">
                                {{ __('No matching employees found') }}
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>



        <div class="mt-4 flex justify-center">
            {{ $employees->links() }}
            {{-- {{ $employees->links('vendor.pagination.tailwind') }} --}}
        </div>
    @endif



    @if ($activeModal === 'addEmp')
        @include('livewire.backend.employee.modals.add-new')
    @endif
    @if ($activeModal === 'addMoreDetails')
        @include('livewire.backend.employee.modals.add-more-details')
    @endif
    @if ($activeModal === 'editEmp')
        @include('livewire.backend.employee.modals.edit-cv')
    @endif
    @if ($activeModal === 'details')
        @include('livewire.backend.employee.modals.details')
    @endif
    @if ($activeModal === 'transferEmp')
        @include('livewire.backend.employee.modals.emp-moves')
    @endif
    @if ($activeModal === 'manageMovements')
        @include('livewire.backend.employee.modals.emp-moves-edit')
    @endif
    @if ($activeModal === 'leaveEmp')
        @include('livewire.backend.employee.modals.emp-leave')
    @endif
    @if ($activeModal === 'manageLeaves')
        @include('livewire.backend.employee.modals.emp-leave-edit')
    @endif
</div>
