<?php

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Auth\Events\Lockout;
use Livewire\Attributes\Layout;
use Livewire\Volt\Component;
use App\Traits\LogsActivity;
use Illuminate\View\View;

new #[Layout('components.frontend.layouts.struct')] class extends Component {
    use LogsActivity;


    public function rendering(View $view): void
    {
        $view->title(__('frontend.tittlewelcomepage') .' | '.__('frontend.2fa_title'));
    }
    public $code;
    public bool $loginDisabled = false;

    protected $rules = [
        'code' => 'required|digits:6',
    ];

    protected $messages = [
        'code.required' => 'The :attribute is required.',
        'code.digits' => 'The :attribute must be 6 digits.',
    ];
    protected bool $alreadyNotified = false;

    public function boot()
    {
        if (RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
            $seconds = RateLimiter::availableIn($this->throttleKey());

            $this->loginDisabled = true;

            if (!$this->alreadyNotified) {
                $this->alreadyNotified = true;
                $this->dispatch('notify', type: 'rate-limit', message: __('auth.throttle'), countdown: $seconds);
            }
        }
    }

    public function verify()
    {
        $this->ensureIsNotRateLimited();

        $this->validate();

        $google2fa = app('pragmarx.google2fa');
        $user = Auth::user();

        if ($google2fa->verifyKey($user->two_factor_secret, $this->code)) {
            RateLimiter::clear($this->throttleKey()); // ✅ مسح العدّاد بعد نجاح الدخول

            $user->two_factor_confirmed_at = now();
            $user->save();

            $this->logActivity('2FA Success', 'User passed two-factor authentication.', 'users');

            session()->flash('toast', [
                'type' => 'success',
                'msg' => 'Login successful!',
                'duration' => 5000,
            ]);

            // $this->redirectIntended(default: route('dashboard', absolute: false), navigate: true);
            $this->redirect(route('dashboard'));
        } else {
            RateLimiter::hit($this->throttleKey()); // 🔁 زيادة عدد المحاولات

            $this->logActivity('2FA Failed', 'User entered invalid 2FA code.', 'users');

            $this->dispatch('notify', type: 'error', message: '⚠️ The code is invalid.', duration: 5000);
            $this->dispatch('otp-invalid');
            return;
        }
    }

    protected function ensureIsNotRateLimited(): void
    {
        if (!RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
            return;
        }

        event(new Lockout(request()));

        $seconds = RateLimiter::availableIn($this->throttleKey());

        $this->loginDisabled = true;

        if (!$this->alreadyNotified) {
            $this->alreadyNotified = true;
            $this->dispatch('notify', type: 'rate-limit', message: __('auth.throttle'), countdown: $seconds);
        }
    }

    protected function throttleKey(): string
    {
        return strtolower(Auth::user()?->email ?? request()->ip()) . '|2fa';
    }
};
?>

<div
    class="min-h-screen flex items-center justify-center bg-transparent transition duration-700 overflow-hidden relative">
    <div
        class="z-10 max-w-md w-full space-y-6 bg-white/80 dark:bg-zinc-900/80 backdrop-blur-md p-8 rounded-xl shadow-2xl border border-zinc-300 dark:border-zinc-700">

        <!-- 🔐 العنوان والوصف -->
        <div class="flex flex-col items-center gap-2">
            <img src="/assets/img/logo.png" alt="Logo" class="w-[300px] h-auto mb-3 drop-shadow-lg" />
            <h2 class="text-2xl font-semibold text-zinc-800 dark:text-white">
                {{ __('frontend.2fa_title') }}
            </h2>
            <p class="text-sm text-zinc-500 dark:text-zinc-400 text-center">
                {{ __('frontend.2fa_desc') }}
            </p>
        </div>

        <!-- ✅ حالة الجلسة -->
        {{-- <x-auth-session-status class="text-center" :status="session('status')" /> --}}

        <!-- 🔢 نموذج الكود (2FA) -->
        {{-- <form wire:submit="verify" class="space-y-4"> --}}
        <form wire:submit="verify" class="space-y-4" id="otp-form">
            <!-- المنطقة دي هتتعامل مع إدخال الكود عن طريق المربعات -->
            <div>
                <label for="otp" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300">
                    {{ __('frontend.2fa_code') }}
                </label>
                <!-- نحاكي ستايل الـ input الأساسي هنا -->
                <div id="otp-container"
                    class="mt-1 block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 ring-accent outline-none transition flex gap-2 justify-center">
                    <input type="text" maxlength="1" class="otp-input" id="otp0"
                        style="width:3rem; text-align: center;" />
                    <input type="text" maxlength="1" class="otp-input" id="otp1"
                        style="width:3rem; text-align: center;" />
                    <input type="text" maxlength="1" class="otp-input" id="otp2"
                        style="width:3rem; text-align: center;" />
                    <input type="text" maxlength="1" class="otp-input" id="otp3"
                        style="width:3rem; text-align: center;" />
                    <input type="text" maxlength="1" class="otp-input" id="otp4"
                        style="width:3rem; text-align: center;" />
                    <input type="text" maxlength="1" class="otp-input" id="otp5"
                        style="width:3rem; text-align: center;" />
                </div>
            </div>
            <input type="hidden" name="code" id="otp-code" wire:model="code" value="">

            {{-- <button type="submit" class="neon-btn neon-blue w-full"> --}}

            <button id="login-btn" type="submit"
                class="neon-btn delay-[1000ms] neon-blue {{ $this->loginDisabled ? 'opacity-50 cursor-not-allowed grayscale text-zinc-400' : '' }}  w-full"
                {{ $this->loginDisabled ? 'disabled' : '' }}>
                <span class="btn-text">{{ __('frontend.2fa_verify') }}</span>
                <span class="glow"></span>
            </button>
        </form>

        <!-- 🚪 تسجيل الخروج -->
        <form method="POST" action="{{ route('logout') }}">
            @csrf
            <button type="submit" class="neon-btn neon-red w-full">
                <span class="btn-text">{{ __('frontend.logout') }}</span>
            </button>
        </form>

    </div>
</div>
