<?php

use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Layout;
use Livewire\Volt\Component;
use Illuminate\View\View;
use PragmaRX\Recovery\Recovery;

new #[Layout('components.frontend.layouts.struct')] class extends Component {
    public function routeName(): void
    {
        $this->dispatch('notify', type: 'info', message: 'Current route: ' . request()->route()->getName());
    }

    public $secret;
    public $qrCodeUrl;
    public $showingQrCode = false;
    public $showingRecoveryCodes = false;
    public $recoveryCodes = [];
    public $code;
    public bool $agreed = false;
    public bool $showHelpsTFA = false;
    public function rendering(View $view): void
    {
        $view->title(__('frontend.tittlewelcomepage') . ' | ' . __('frontend.2fa_title'));
    }

    public function mount()
    {
        $google2fa = app('pragmarx.google2fa');
        $user = Auth::user();

        $this->secret = $user->two_factor_secret;
        $this->recoveryCodes = $user->two_factor_recovery_codes;

        if (!$this->secret) {
            $this->secret = $google2fa->generateSecretKey();
            $this->recoveryCodes = (new Recovery())->toJson();
            $this->qrCodeUrl = $google2fa->getQRCodeInline(config('app.name'), $user->email, $this->secret);
        }
    }

    public function toggleQrCode()
    {
        $this->showingQrCode = !$this->showingQrCode;
    }

    public function toggleRecoveryCodes()
    {
        $this->showingRecoveryCodes = !$this->showingRecoveryCodes;
    }

    public function enableTwoFactor()
    {
        $google2fa = app('pragmarx.google2fa');

        $this->validate([
            'code' => [
                'required',
                'digits:6',
                function ($attribute, $value, $fail) use ($google2fa) {
                    if (!$google2fa->verifyKey($this->secret, $value)) {
                        $this->dispatch('notify', type: 'error', message: __('The code is invalid.'), duration: 5000);
                        $fail(__('The code is invalid.'));
                    }
                },
            ],
        ]);

        $user = Auth::user();
        $user->two_factor_secret = $this->secret;
        $user->two_factor_recovery_codes = $this->recoveryCodes;
        $user->two_factor_confirmed_at = now();
        $user->save();

        $this->dispatch('notify', type: 'success', message: __('2FA Enabled Successfully'));
        $this->dispatch('fireworks');
    }

    public function disableTwoFactor()
    {
        $user = Auth::user();
        $user->two_factor_secret = null;
        $user->two_factor_recovery_codes = null;
        $user->two_factor_confirmed_at = null;
        $user->save();

        $this->dispatch('notify', type: 'success', message: __('2FA Disabled Successfully'));
    }
};
?>

<div x-data="{ showHelpsTFA: @entangle('showHelpsTFA'), currentTab: 'android' }"
    class="min-h-screen flex items-center justify-center bg-transparent transition duration-700 overflow-hidden relative">

    <!-- لوجو + عنوان -->
    <div
        class="z-10 max-w-md w-full space-y-6 bg-white/80 dark:bg-zinc-900/80 backdrop-blur-md p-8 rounded-xl shadow-2xl border border-zinc-300 dark:border-zinc-700">

        <div class="flex flex-col items-center gap-2 animate-fade-in-down">
            <img src="/assets/img/logo.png" alt="Logo" class="w-[300px] h-auto mb-3 drop-shadow-lg" />
            <h2 class="text-2xl font-semibold text-zinc-800 dark:text-white">
                {{ __('frontend.2fa_title') }}
            </h2>
            <p class="text-sm text-zinc-600 dark:text-zinc-400 text-center">
                {{ __('frontend.2fa_desc2') }}
            </p>
        </div>

        {{-- المحتوى الديناميكي حسب حالة التفعيل --}}
        @if (!Auth::user()->two_factor_secret)

            @if ($showingQrCode)
                <p class="text-zinc-700 dark:text-zinc-200 text-center">{{ __('frontend.scan_qr_instruction') }}</p>


                <div class="flex justify-center">{!! $qrCodeUrl !!}</div>

                <input wire:model="code" type="text" placeholder="{{ __('frontend.enter_auth_code') }}"
                    class="w-full px-4 py-2 mt-3 rounded-lg border dark:bg-zinc-800 dark:text-white">

                <button wire:click="enableTwoFactor" class="neon-btn delay-[1000ms] neon-green w-full">
                    {{ __('frontend.enable_2fa') }}
                    <span class="glow"></span>
                </button>



                <div class="text-center mt-2">
                    <button @click.prevent="showHelpsTFA=true" class="neon-info text-sm text-blue-500 hover:underline">
                        {{ __('frontend.how_to_use_auth_app') }}
                    </button>
                </div>
            @else
                <button wire:click="toggleQrCode" class="neon-btn delay-[1000ms] neon-blue w-full">
                    {{ __('frontend.show_qr') }}
                    <span class="glow"></span>
                </button>
            @endif
        @else
            <p class="text-zinc-700 dark:text-zinc-200 text-center">{{ __('frontend.2fa_enabled') }}</p>
            <script>
                function launchConfetti() {
                    confetti({
                        particleCount: 200,
                        spread: 100,
                        origin: {
                            y: 0.6
                        }
                    });
                }
            </script>
            <a href="{{ route('dashboard') }}" class="neon-btn neon-blue w-full mb-2 text-center block">
                {{ __('frontend.go_to_dashboard') }}
                <span class="glow"></span>
            </a>
            @if (!$showingRecoveryCodes)
                <button wire:click="toggleRecoveryCodes" class="neon-btn neon-blue w-full">
                    {{ __('frontend.show_recovery') }}
                </button>
            @else
                <ul class="text-sm text-zinc-700 dark:text-zinc-200 grid grid-cols-2 gap-2">
                    @foreach (json_decode($recoveryCodes) as $code)
                        <li class="bg-zinc-200 dark:bg-zinc-800 px-2 py-1 rounded text-center font-mono">
                            {{ $code }}</li>
                    @endforeach
                </ul>
                <button wire:click="toggleRecoveryCodes" class="neon-btn neon-gray w-full mt-2">
                    {{ __('frontend.hide_recovery') }}
                </button>
            @endif

        @endif
    </div>

    <script src="https://cdn.jsdelivr.net/npm/canvas-confetti@1.6.0/dist/confetti.browser.min.js"></script>
    <!-- تأكد إنك محطوط الكود ده داخل عنصر فيه x-data لتعريف المتغيرات -->

    <!-- ======= Modal with Tabs ======= -->
    <div x-show="showHelpsTFA" x-transition:enter="transition ease-out duration-300"
        x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100"
        x-transition:leave="transition ease-in duration-200" x-transition:leave-start="opacity-100"
        x-transition:leave-end="opacity-0"
        class="fixed inset-0 bg-black/60 backdrop-blur-sm flex items-center justify-center z-50 px-4 sm:px-6"
        wire:ignore x-cloak>
        <div
            class="bg-white dark:bg-zinc-900 rounded-2xl shadow-2xl border border-zinc-200 dark:border-zinc-700
               w-full max-w-lg max-h-[70vh] flex flex-col overflow-hidden">
            <!-- Header -->
            <div class="flex flex-col items-center justify-center p-4 border-b border-zinc-100 dark:border-zinc-800">
                <img src="/assets/img/logo.png" class="h-10 w-auto mb-2" alt="logo">
                <h3 class="text-lg font-bold text-zinc-800 dark:text-white transition-all duration-300">
                    <template x-if="currentTab === 'android'">{{ __('frontend.android') }}</template>
                    <template x-if="currentTab === 'iphone'">{{ __('frontend.iphone_title') }}</template>
                </h3>
            </div>

            <!-- Tabs Buttons -->
            <div
                class="flex items-center justify-center gap-3 border-b border-zinc-200 dark:border-zinc-700 p-3 bg-zinc-50 dark:bg-zinc-800">
                <!-- Android Button -->
                <button @click="currentTab = 'android'"
                    class="flex items-center gap-2 px-4 py-2 rounded-full text-sm font-semibold transition-all duration-300
                       shadow-sm border border-transparent"
                    :class="currentTab === 'android'
                        ?
                        'bg-green-600 text-white shadow-lg scale-[1.02]' :
                        'bg-green-100 dark:bg-green-900 text-green-700 dark:text-green-200 hover:bg-green-200 dark:hover:bg-green-800'">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-4 w-4" fill="currentColor" viewBox="0 0 24 24">
                        <path
                            d="M17.6 9.48c.18-.31.28-.66.28-1.03 0-1.27-1.04-2.31-2.31-2.31-.81 0-1.53.42-1.96 1.06a2.314 2.314 0 00-3.92 0A2.314 2.314 0 007.3 6.14C6.03 6.14 5 7.17 5 8.45c0 .37.1.72.28 1.03-.87.45-1.48 1.36-1.48 2.42 0 1.5 1.22 2.73 2.73 2.73H16.3c1.5 0 2.73-1.22 2.73-2.73 0-1.06-.61-1.97-1.43-2.42z" />
                    </svg>
                    Android
                </button>

                <!-- iPhone/iPad Button -->
                <button @click="currentTab = 'iphone'"
                    class="flex items-center gap-2 px-4 py-2 rounded-full text-sm font-semibold transition-all duration-300
                       shadow-sm border border-transparent"
                    :class="currentTab === 'iphone'
                        ?
                        'bg-blue-600 text-white shadow-lg scale-[1.02]' :
                        'bg-blue-100 dark:bg-blue-900 text-blue-700 dark:text-blue-200 hover:bg-blue-200 dark:hover:bg-blue-800'">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-4 w-4" fill="currentColor" viewBox="0 0 24 24">
                        <path
                            d="M19.17 12.47c-.34.78-.91 1.69-1.62 2.72-.75 1.1-1.51 2.19-2.37 2.19-.87 0-1.15-.59-2.14-.59-.99 0-1.31.59-2.13.59-.83 0-1.49-.99-2.36-2.2C7.08 14.17 6 12.12 6 9.98c0-2.17 1.57-3.32 3.1-3.32.85 0 1.53.5 2.07.5.51 0 1.32-.52 2.25-.52.62 0 2.06.07 2.93 1.6-2.55 1.27-2.13 4.84.82 4.23z" />
                    </svg>
                    iPhone / iPad
                </button>
            </div>


            <!-- Tabs Content Container -->
            <div
                class="flex-1 min-h-0 overflow-y-auto px-6 py-4 scrollbar-thin scrollbar-thumb-indigo-400 dark:scrollbar-thumb-indigo-600 scrollbar-track-transparent">
                <!-- android Content -->
                <div x-show="currentTab === 'android'"
                    class="break-words space-y-8 text-sm text-zinc-700 dark:text-zinc-300 leading-relaxed">

                    <!-- عنوان رئيسي -->
                    <h2 class="text-center text-lg sm:text-xl font-bold text-zinc-800 dark:text-white">
                        {{ __('frontend.auth_android_title') }}
                    </h2>

                    <!-- وصف -->
                    <p class="text-center max-w-2xl mx-auto">
                        {{ __('frontend.auth_android_desc') }}
                    </p>

                    <!-- زر التحميل -->
                    <div class="text-center">
                        <a href="https://play.google.com/store/apps/details?id=com.google.android.apps.authenticator2"
                            target="_blank"
                            class="inline-flex items-center gap-2 px-5 py-2.5 bg-green-600 hover:bg-green-700 text-white rounded-lg font-semibold shadow transition-all duration-300">
                            <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="currentColor"
                                viewBox="0 0 24 24">
                                <path d="M12 2C6.486 2 2 6.486 2 12s4.486 10 10 10 10-4.486 10-10S17.514 2 12 2z" />
                            </svg>
                            {{ __('frontend.auth_android_download_btn') }}
                        </a>
                    </div>

                    <!-- خطوات مع الصور -->
                    <div class="space-y-6 max-w-3xl mx-auto">
                        <!-- خطوة 1 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">📍 {{ __('frontend.auth_android_step1') }}</h3>
                        </div>

                        <!-- خطوة 2 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">🔍 {{ __('frontend.auth_android_step2') }}</h3>
                        </div>

                        <!-- خطوة 3 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">🚀 {{ __('frontend.auth_android_step3') }}</h3>
                        </div>

                        <!-- خطوة 4 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">📷 {{ __('frontend.auth_android_step4') }}</h3>
                        </div>

                        <!-- خطوة 5 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">🎯 {{ __('frontend.auth_android_step5') }}</h3>
                        </div>

                        <!-- خطوة 6 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">✅ {{ __('frontend.auth_android_step6') }}</h3>
                        </div>
                    </div>

                    <!-- فيديو -->
                    <div
                        class="border border-zinc-200 dark:border-zinc-700 rounded-xl bg-white/70 dark:bg-zinc-900/70 p-4 shadow-md max-w-3xl mx-auto">
                        <h3 class="text-center font-bold text-base sm:text-lg text-zinc-800 dark:text-white mb-4">
                            📽️ {{ __('frontend.auth_android_video_title') }}
                        </h3>
                        <div class="rounded-lg overflow-hidden">
                            <iframe class="w-full aspect-video" src="https://www.youtube.com/embed/G4peeVyW9Ys"
                                frameborder="0" allowfullscreen title="Google Authenticator Android Guide">
                            </iframe>
                        </div>
                    </div>

                </div>




                <!-- iphone Content -->
                <div x-show="currentTab === 'iphone'"
                    class="break-words space-y-8 text-sm text-zinc-700 dark:text-zinc-300 leading-relaxed">

                    <!-- عنوان رئيسي -->
                    <h2 class="text-center text-lg sm:text-xl font-bold text-zinc-800 dark:text-white">
                        {{ __('frontend.auth_ios_title') }}
                    </h2>

                    <!-- وصف -->
                    <p class="text-center max-w-2xl mx-auto">
                        {{ __('frontend.auth_ios_desc') }}
                    </p>

                    <!-- زر التحميل -->
                    <div class="text-center">
                        <a href="https://apps.apple.com/app/google-authenticator/id388497605" target="_blank"
                            class="inline-flex items-center gap-2 px-5 py-2.5 bg-blue-600 hover:bg-blue-700 text-white rounded-lg font-semibold shadow transition-all duration-300">
                            <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="currentColor"
                                viewBox="0 0 24 24">
                                <path d="M12 2C6.486 2 2 6.486 2 12s4.486 10 10 10 10-4.486 10-10S17.514 2 12 2z" />
                            </svg>
                            {{ __('frontend.auth_ios_download_btn') }}
                        </a>
                    </div>

                    <!-- خطوات مع الصور -->
                    <div class="space-y-6 max-w-3xl mx-auto">
                        <!-- خطوة 1 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">📍 {{ __('frontend.auth_ios_step1') }}</h3>

                        </div>

                        <!-- خطوة 2 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">🔍 {{ __('frontend.auth_ios_step2') }}</h3>

                        </div>

                        <!-- خطوة 3 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">🚀 {{ __('frontend.auth_ios_step3') }}</h3>

                        </div>

                        <!-- خطوة 4 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">📷 {{ __('frontend.auth_ios_step4') }}</h3>

                        </div>

                        <!-- خطوة 5 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">🎯 {{ __('frontend.auth_ios_step5') }}</h3>

                        </div>

                        <!-- خطوة 6 -->
                        <div
                            class="border border-zinc-200 dark:border-zinc-700 rounded-xl p-4 bg-white/60 dark:bg-zinc-900/60 shadow-md">
                            <h3 class="font-bold text-center mb-2">✅ {{ __('frontend.auth_ios_step6') }}</h3>
                            {{-- <img src="https://i.imgur.com/ZcpKwFE.png" alt="رؤية كود التحقق"
                                class="w-full rounded-lg shadow"> --}}
                        </div>
                    </div>

                    <!-- فيديو -->
                    <div
                        class="border border-zinc-200 dark:border-zinc-700 rounded-xl bg-white/70 dark:bg-zinc-900/70 p-4 shadow-md max-w-3xl mx-auto">
                        <h3 class="text-center font-bold text-base sm:text-lg text-zinc-800 dark:text-white mb-4">
                            📽️ {{ __('frontend.auth_ios_video_title') }}
                        </h3>
                        <div class="rounded-lg overflow-hidden">
                            <iframe class="w-full aspect-video" src="https://www.youtube.com/embed/G4peeVyW9Ys"
                                frameborder="0" allowfullscreen title="Google Authenticator iOS Guide">
                            </iframe>
                        </div>
                    </div>

                </div>
            </div>

            <!-- Footer -->
            <div class="flex justify-end gap-2 border-t border-zinc-100 dark:border-zinc-800 p-4">
                <button @click="$wire.set('agreed', false); showHelpsTFA = false"
                    class="px-4 py-2 rounded-lg text-sm bg-zinc-200 dark:bg-zinc-700 hover:bg-zinc-300 dark:hover:bg-zinc-600">
                    {{ __('frontend.cancel') }}
                </button>
                <button @click="$wire.set('agreed', true); showHelpsTFA = false"
                    class="px-4 py-2 rounded-lg text-sm bg-indigo-600 text-white hover:bg-indigo-700">
                    {{ __('frontend.agree') }}
                </button>
            </div>
        </div>
    </div>




</div>
