<?php

namespace App\Traits\Employwee;

use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use App\Models\User\UserEmployee;
use App\Models\User\UserProject;
use App\Models\User\UserEmployeeMovement;

trait UserEmployeesMovements
{



    public function openTransferModal(int $empId)
    {
        $this->movementEmployeeId = $empId;
        // إعادة الضبط
        $this->reset(['type', 'start_date', 'end_date', 'openEnded', 'to_project', 'notes']);
        $this->loadProjectsForMovement();
        // تعيين الموظف والنوع

        // افتح المودال
        $this->activeModal = 'transferEmp';
    }

    protected function loadProjectsForMovement(): void
    {
        // 1️⃣ جلب مشروع الموظف الحالي (قد يكون code أو id حسب Relation)
        $emp = UserEmployee::find($this->movementEmployeeId);
        $currentProjId = optional($emp->project)->id;

        // 2️⃣ جلب آخر مأمورية مفعّلة للمستخدم
        $activeTo = UserEmployeeMovement::where('employee_id', $this->movementEmployeeId)->where('status', 'activated')->value('to_project');

        // 3️⃣ بناء الاستعلام
        $query = UserProject::query();

        // استبعد مشروع الموظف الحالي
        if ($currentProjId) {
            $query->where('id', '!=', $currentProjId);
        }

        // إذا في مأمورية مفعّلة، استبعد المشروع الوجهة أيضاً
        if ($activeTo) {
            $query->where('id', '!=', $activeTo);
        }

        // 4️⃣ أخيراً خزن النتيجة
        $this->projects = $query->get()->toArray();
    }


    public function saveMovement(): void
    {
        if ($this->applyNow) {
            // نحفظ التاريخ بصيغة Y-m-d ليتوافق مع حقل الـ date
            $this->start_date = Carbon::now()->format('Y-m-d');
        }

        if ($this->start_date !== null && $this->end_date !== null && Carbon::parse($this->end_date)->lt(Carbon::parse($this->start_date))) {
            // تضيفين خطأ لحقل end_date وتخرجي
            $this->addError(
                'end_date',
                __('backend.error_end_after_start'), // أو الرسالة الملائمة لك
            );
            return;
        }
        // 1. قواعد التحقق الأساسية
        $rules = [
            'type' => ['required', 'in:mission,transfer,termination'],
            'notes' => ['nullable', 'string'],
        ];
        $status = 'applied';
        if (!$this->applyNow) {
            $rules['start_date'] = ['required', 'date'];
            $status = 'pending';
        } elseif ($this->end_date) {
            $status = 'activated';
        }
        if (in_array($this->type, ['mission', 'transfer'])) {
            $rules['to_project'] = ['required', 'exists:users_projects,id'];
        }
        if ($this->type === 'mission' && !$this->openEnded) {
            $rules['end_date'] = ['required', 'date', 'after_or_equal:start_date'];
        }
        if ($this->type === 'mission' && $this->applyNow) {
            $status = 'activated';
        }
        $this->validate($rules, [], $this->attributes());

        // 2. حدِّث حالات الموظف أولًا
        UserEmployeeMovement::refreshStatusesForEmployee($this->movementEmployeeId);

        // 3. جلب الحركات المعلقة بعد التحديث
        $pending = UserEmployeeMovement::where('employee_id', $this->movementEmployeeId)
            ->whereIn('status', ['pending', 'activated'])
            ->get();

        // 4. إذا هناك pending بدون end_date → خطأ
        if ($pending->first(fn($m) => is_null($m->end_date))) {
            $this->dispatch('notify', type: 'error', message: __('backend.error_end_open'), duration: 6000);
            return;
        }

        // 5. تحقق من تضارب النطاقات مع أي pending end_date
        $newStart = Carbon::parse($this->start_date)->startOfDay();
        $newEnd = $this->end_date ? Carbon::parse($this->end_date)->endOfDay() : $newStart;

        foreach ($pending as $m) {
            $existStart = $m->start_date->startOfDay();
            $existEnd = $m->end_date ? $m->end_date->endOfDay() : $existStart;

            // تقاطع إذا:
            // newStart ≤ existEnd  && existStart ≤ newEnd
            if ($newStart->lte($existEnd) && $existStart->lte($newEnd)) {
                $this->dispatch('notify', type: 'error', message: __('backend.error_overlap'), duration: 6000);
                return;
            }
        }

        $employee = UserEmployee::findOrFail($this->movementEmployeeId);
        $fromProjId = optional($employee->project)->id;
        // 6. إنشاء الحركة فوريًا (status = applied)
        $movement = UserEmployeeMovement::create([
            'created_by' => Auth::id(),
            'status' => $status,
            'type' => $this->type,
            'employee_id' => $this->movementEmployeeId,
            'start_date' => $this->start_date ?? now(),
            'end_date' => $this->type === 'mission' && !$this->openEnded ? $this->end_date : null,
            'from_project' => $fromProjId,
            'to_project' => $this->to_project,
            'notes' => $this->notes,
        ]);

        // 7. إذا كان transfer حدّث مشروع الموظف مباشرة
        if ($this->type === 'transfer' && $status === 'applied') {
            $proj = UserProject::find($this->to_project);
            if ($employee && $proj) {
                $employee->project_code = $proj->code;
                $employee->save();
            }
        }

        // 8. نجاح
        $this->dispatch('notify', type: 'success', message: __('backend.movement_saved_successfully'), duration: 6000);

        $this->activeModal = null;
        $this->loadEmployees();
        $this->readyToLoad = true;
    }

    public string $employeeName = '';

    public function openMovementsModal(int $empId)
    {
        $this->movementEmployeeId = $empId;
        $emp = UserEmployee::find($empId);
        $this->employeeName = $this->employeeName = $emp->full_name[app()->getLocale()] ?? ($emp->full_name['ar'] ?? '-');
        UserEmployeeMovement::refreshStatusesForEmployee($this->movementEmployeeId);
        $this->loadProjectsForMovement();
        $this->currentStep = 1;
        $this->loadMovements($empId);
        $this->activeModal = 'manageMovements';
    }
    public function loadMovements(int $empId): void
    {
        $this->movements = UserEmployeeMovement::getCurrentAndUpcoming($empId);
    }

    public function editMovement(int $moveId): void
    {
        $m = UserEmployeeMovement::findOrFail($moveId);
        $this->editMovementId = $moveId;
        $this->editType = $m->type;
        $this->editApplyNow = $m->status === 'activated' && is_null($m->start_date) === false;
        $this->editStartDate = $m->start_date?->toDateString();
        $this->editEndDate = $m->end_date?->toDateString();
        $this->editOpenEnded = is_null($m->end_date);
        $this->editToProject = $m->to_project;
        $this->editNotes = $m->notes;
        $this->currentStep = 2;
    }

    // حفظ التعديلات
    public function updateMovement(): void
    {
        if ($this->editApplyNow) {
            // نحفظ التاريخ بصيغة Y-m-d ليتوافق مع حقل الـ date
            $this->editStartDate = Carbon::now()->format('Y-m-d');
        }

        if ($this->editStartDate !== null && $this->editEndDate !== null && Carbon::parse($this->editEndDate)->lt(Carbon::parse($this->editStartDate))) {
            $this->addError(
                'editEndDate',
                __('backend.error_end_after_start'), // أو أي رسالة ترغبين بها
            );
            return;
        }
        if ($this->editApplyNow) {
            $this->editStartDate = null;
        }

        // 2. لو النوع مش 'mission' أو 'transfer'، يبقى مش محتاجين مشروع وجهة
        if (!in_array($this->editType, ['mission', 'transfer'], true)) {
            $this->editToProject = null;
        }

        // 3. لو مش ‘mission’ أو لو المهمة مفتوحة النهاية، يبقى مش محتاجين تاريخ نهاية
        if ($this->editType !== 'mission' || $this->editOpenEnded) {
            $this->editEndDate = null;
        }
        // 1️⃣ إعداد قواعد التحقق وحساب الحالة
        $rules = [
            'editType' => ['required', 'in:mission,transfer,termination'],
            'editNotes' => ['nullable', 'string'],
        ];
        $status = 'applied';

        if (!$this->editApplyNow) {
            $rules['editStartDate'] = ['required', 'date'];
            $status = 'pending';
        } elseif ($this->editType === 'mission' && $this->editApplyNow) {
            $status = 'activated';
        }

        if (in_array($this->editType, ['mission', 'transfer'])) {
            $rules['editToProject'] = ['required', 'exists:users_projects,id'];
        }

        if ($this->editType === 'mission' && !$this->editOpenEnded) {
            $rules['editEndDate'] = ['required', 'date', 'after_or_equal:editStartDate'];
        }

        // تمرير الأسماء المترجمة من الـ attributes()
        $this->validate($rules, [], $this->attributes());

        // 2️⃣ حدّث حالات الموظف قبل الفحص
        // مهم: نمرّر employeeId، لا movementId
        UserEmployeeMovement::refreshStatusesForEmployee($this->movementEmployeeId);

        // 3️⃣ جلب الحركات المعلقة والمفعّلة (باستثناء الحركة الراهنة)
        $pending = UserEmployeeMovement::where('employee_id', $this->movementEmployeeId)
            ->where('id', '!=', $this->editMovementId)
            ->whereIn('status', ['pending', 'activated'])
            ->get();

        // 4️⃣ خطأ إذا وُجد pending بدون end_date
        if ($pending->first(fn($m) => is_null($m->end_date))) {
            $this->dispatch('notify', type: 'error', message: __('backend.error_end_open'), duration: 6000);
            return;
        }

        // 5️⃣ حساب نطاق الحركة الجديدة وفحص التقاطع (overlap)
        $newStart = Carbon::parse($this->editStartDate)->startOfDay();
        $newEnd = $this->editType === 'mission' && !$this->editOpenEnded && $this->editEndDate ? Carbon::parse($this->editEndDate)->endOfDay() : $newStart;

        foreach ($pending as $m) {
            $existStart = $m->start_date->startOfDay();
            $existEnd = $m->end_date ? $m->end_date->endOfDay() : $existStart;

            if ($newStart->lte($existEnd) && $existStart->lte($newEnd)) {
                $this->dispatch('notify', type: 'error', message: __('backend.error_overlap'), duration: 6000);
                return;
            }
        }

        // 6️⃣ تطبيق التعديل على السجل
        $movement = UserEmployeeMovement::findOrFail($this->editMovementId);
        $movement->update([
            'type' => $this->editType,
            'status' => $status,
            'start_date' => $this->editStartDate ?: $movement->start_date,
            'end_date' => $this->editOpenEnded ? null : $this->editEndDate,
            'to_project' => $this->editToProject,
            'notes' => $this->editNotes,
        ]);

        // 7️⃣ (اختياري) لو نقل مباشر حدّث project_code في جدول الموظف
        if ($this->editType === 'transfer' && $status === 'applied') {
            $employee = UserEmployee::find($this->movementEmployeeId);
            $proj = UserProject::find($this->editToProject);
            if ($employee && $proj) {
                $employee->project_code = $proj->code;
                $employee->save();
            }
        }

        // 8️⃣ العودة للخطوة 1 وإعادة تحميل الحركات
        $this->loadMovements($this->movementEmployeeId);

        // إشعار نجاح
        $this->dispatch('notify', type: 'success', message: __('backend.movement_updated_successfully'), duration: 6000);
        $this->dispatch('movement-updated');
    }

    // حذف حركة
    public function cancelMovement(): void
    {
        $moveId = $this->deleteMovementId;

        $rules = [
            'cancelReason' => ['required|string|max:255'],
        ];
        // 1️⃣ Validate سبب الإلغاء
        $this->validate(
            ['cancelReason' => 'required|string|max:255'],
            [], // هنا لو عندك رسائل مخصصة
            ['cancelReason' => __('backend.reason_for_cancellation')],
        );

        // 2️⃣ حدّث حالات الموظف قبل الإلغاء
        UserEmployeeMovement::refreshStatusesForEmployee($moveId);

        // 3️⃣ ابحث عن الحركة وغيّرها لـ cancelled
        $movement = UserEmployeeMovement::findOrFail($moveId);
        $movement->update([
            'status' => 'cancelled',
            'cancelled_by' => Auth::id(),
            'cancelled_at' => now(),
            'cancelled_reason' => $this->cancelReason,
        ]);

        // 4️⃣ نظّف الحقول وأعد تحميل القائمة
        $this->deleteMovementId = null;
        $this->cancelReason = null;
        $this->loadMovements($this->movementEmployeeId);

        // 5️⃣ إشعار نجاح وإطلاق الحدث
        $this->dispatch('notify', type: 'success', message: __('backend.movement_cancelled_successfully'), duration: 6000);
        $this->dispatch('movement-updated');
    }
}
