<?php

namespace App\Traits\Employwee;

use App\Models\User\UserEmployee;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile as LivewireTemp;

trait UserEmployees
{

    //Details
    public $selectedEmp;
    //----------------------
    //Add

    //----------------------
    //Edit
    //----------------------
    //----------------------Details section ----------------------------------
    public function openDetailsModal($id)
    {
        $employee = UserEmployee::with(['project', 'department', 'jobTitle'])->findOrFail($id);

        // تأكد من أن الصور مصفوفة فعلًا
        if (is_string($employee->id_card_image)) {
            $employee->id_card_image = json_decode($employee->id_card_image, true);
        }

        $this->selectedEmp = $employee;
        $this->activeModal = 'details';
    }

    //----------------------Add section ----------------------------------\\

    // دوال فتح المودالات
    public function openAddEmpModal()
    {
        $this->activeModal = 'addEmp';
        $this->currentStep = 1;
    }

    public function openAddMoreDetailsModal($id)
    {
        $this->editingEmployee = $id; // تخزين ID الموظف
        $employee = UserEmployee::findOrFail($id); // جلب بيانات الموظف

        // جلب البيانات القديمة من خانة الـ cv إذا كانت موجودة
        $cvData = json_decode($employee->cv, true) ?? [];

        // تعبئة الحقول بالبيانات القديمة
        $this->editcv = [
            'contract_type' => $cvData['contract_type'] ?? '',
            'grade' => $cvData['grade'] ?? '',
            'rate' => $cvData['rate'] ?? 0, // تعيين القيمة الافتراضية للتقييم
        ];

        $this->activeModal = 'addMoreDetails'; // تحديد اسم المودال
    }

    public $new = [
        'name' => '',
        'name_ar' => '',
        'email' => '',
        'phone' => [''],
        'project_code' => '',
        'job_title_id' => '',
        'national_id' => '',
        'profile_picture' => null,
        'id_card_images' => [],
        'id_card_images_buffer' => [], // لو عايز buffer
        'skills' => [],
        'hire_date' => '', // ده مهم جدًا!
        'contract_date' => '', // ده مهم جدًا!
        'experience_roads' => [],
        'experience_airports' => [],
    ];

    protected function rules(): array
    {
        return [
            'new.hire_date' => ['required', 'date', 'before_or_equal:today'],
            'new.contract_date' => ['nullable', 'date', 'after_or_equal:new.hire_date'],
            'new.name' => [
                'required',
                'string',
                'max:255',
                function ($attribute, $value, $fail) {
                    if (str_word_count($value) < 3) {
                        $fail(__('validation.fullname_required_three_words'));
                    }
                },
            ],
            'new.name_ar' => [
                'required',
                'string',
                'max:255',
                'regex:/^[\p{Arabic}\s]+$/u',
                function ($attribute, $value, $fail) {
                    if (count(preg_split('/\s+/u', trim($value))) < 3) {
                        $fail(__('validation.fullname_required_three_words_ar'));
                    }
                },
            ],

            'new.email' => ['nullable', 'string', 'lowercase', 'email', 'max:255', Rule::unique('users', 'email')],
            'new.phone' => ['required', 'array', 'min:1'],
            'new.phone.*' => ['required', 'string', 'regex:/^(\+?20|0020)?1?\d{9}$/'],
            'new.project_code' => ['required', 'exists:users_projects,code'],
            'new.national_id' => [
                'required',
                'digits:14',
                'unique:users_employees,national_id',
                function ($attribute, $value, $fail) {
                    if (!preg_match('/^[23][0-9]{13}$/', $value)) {
                        return $fail(__('validation.national_id_format_invalid'));
                    }

                    $century = match (substr($value, 0, 1)) {
                        '2' => 1900,
                        '3' => 2000,
                        default => null,
                    };

                    $year = (int) substr($value, 1, 2);
                    $month = (int) substr($value, 3, 2);
                    $day = (int) substr($value, 5, 2);

                    if (!checkdate($month, $day, $century + $year)) {
                        return $fail(__('validation.national_id_birthdate_invalid'));
                    }

                    $govCode = substr($value, 7, 2);
                    $validGovCodes = ['01', '02', '03', '04', '11', '12', '13', '14', '15', '16', '17', '18', '19', '21', '22', '23', '24', '25', '26', '27', '28', '29', '31', '32', '33', '34', '35', '88'];

                    if (!in_array($govCode, $validGovCodes)) {
                        return $fail(__('validation.national_id_governorate_invalid'));
                    }
                    // ✅ تحقق النوع (ذكر/أنثى)
                    $genderDigit = substr($value, 12, 1);
                    if (!is_numeric($genderDigit)) {
                        return $fail(__('validation.national_id_gender_invalid'));
                    }
                },
            ],
            'new.id_card_images' => ['required', 'array', 'min:1'],
            'new.id_card_images.*' => ['file', 'mimes:jpeg,jpg,png,pdf', 'max:4096'],
            'new.job_title_id' => ['required', 'exists:users_job_titles,id'],
            'new.profile_picture' => ['nullable', 'image', 'mimes:jpeg,jpg,png', 'max:2048'],
            'new.skills' => ['nullable', 'array'],
            'new.skills.*' => ['string', 'max:50'],
        ];
    }

    public function storeEmp()
    {
        try {
            // 1️⃣ Validate
            $this->validate($this->rules(), $this->messages, $this->attributes());

            // 2️⃣ Transactional save
            DB::transaction(function () {
                $nextId = (UserEmployee::max('id') ?? 0) + 1;
                $employeeCode = 'EMP-' . str_pad((string) $nextId, 6, '0', STR_PAD_LEFT);

                while (UserEmployee::where('employee_code', $employeeCode)->exists()) {
                    $nextId++;
                    $employeeCode = 'EMP-' . str_pad((string) $nextId, 6, '0', STR_PAD_LEFT);
                }

                // رفع المستندات
                $idCardPaths = [];
                $folder = 'employees/' . $employeeCode;

                foreach ($this->new['id_card_images'] as $file) {
                    if ($file instanceof LivewireTemp) {
                        $nameForFile = Str::slug($this->new['name']);
                        $randomCode = Str::random(6);
                        $extension = $file->getClientOriginalExtension();
                        $filename = "{$nameForFile}_{$randomCode}.{$extension}";

                        $idCardPaths[] = $file->storeAs($folder, $filename, 'public');
                    }
                }

                // استخراج معلومات الرقم القومي
                $info = $this->parseNationalId($this->new['national_id']);

                // رفع الصورة الرسمية
                $profilePicturePath = null;
                if ($this->new['profile_picture'] instanceof LivewireTemp) {
                    $nameForFile = Str::slug($this->new['name']);
                    $randomCode = Str::random(6);
                    $extension = $this->new['profile_picture']->getClientOriginalExtension();
                    $filename = "{$nameForFile}_profile_{$randomCode}.{$extension}";

                    $profilePicturePath = $this->new['profile_picture']->storeAs($folder, $filename, 'public');
                }

                // إنشاء السجل
                UserEmployee::create([
                    'project_code' => $this->new['project_code'],
                    'job_title_id' => $this->new['job_title_id'],
                    'employee_code' => $employeeCode,
                    'full_name' => [
                        'ar' => $this->new['name_ar'],
                        'en' => $this->new['name'] ?? null,
                    ],
                    'phone' => json_encode($this->new['phone']),
                    'national_id' => $this->new['national_id'],
                    'birth_date' => $info['birth_date'],
                    'governorate' => $info['governorate'],
                    'gender' => $info['gender'],
                    'id_card_image' => json_encode($idCardPaths),
                    'hiring_date' => $this->new['hire_date'],
                    'contract_date' => $this->new['contract_date'] ?? null,
                    'cv' => json_encode(
                        array_filter(
                            [
                                'work_experience' => [
                                    'roads' => $this->new['experience_roads'],
                                    'airports' => $this->new['experience_airports'],
                                ],
                                'email' => $this->new['email'] ?? null,
                                'profile_picture' => $profilePicturePath ?? null,
                                'skills' => $this->new['skills'],
                                'qualification' => $this->new['qualification'], // المؤهل الدراسي
                                'graduation_year' => $this->new['graduation_year'], // سنة التخرج
                                'graduation_project' => $this->new['graduation_project'], // مشروع التخرج
                                'military_status' => $this->new['military_status'], // الموقف من التجنيد
                            ],
                            fn($v) => !is_null($v) && $v !== '',
                        ),
                        JSON_UNESCAPED_UNICODE,
                    ),
                ]);
            });
            $this->dispatch('notify', type: 'success', message: __('backend.cv_saved_successfully'), duration: 6000);

            $this->activeModal = null; // ✅ إغلاق المودال
            $this->reset('new'); // ✅ تفريغ النموذج
            $this->readyToLoad = true;
        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('notify', type: 'error', message: $e->validator->errors()->first(), duration: 6000);
        } catch (\Throwable $e) {
            $this->dispatch('notify', type: 'error', message: __('frontend.anothererror'), duration: 6000);

            $this->dispatch('notify', type: 'error', message: $e->getMessage(), duration: 10000);
        }
    }

    public function addmoredetail()
    {
        try {
            // التحقق من صحة البيانات المدخلة
            $this->validate([
                'editcv.contract_type' => ['required', 'string', Rule::in(['contract', 'serky'])],
                'editcv.grade' => ['required', 'integer', 'min:1'],
                'editcv.rate' => ['required', 'integer', 'min:1', 'max:10'],
            ]);

            // جلب الموظف المحدد
            $employee = UserEmployee::findOrFail($this->editingEmployee);

            // جلب البيانات القديمة من خانة الـ cv (إذا كانت موجودة)
            $cvData = json_decode($employee->cv, true) ?? [];

            // دمج البيانات الجديدة مع البيانات القديمة
            $cvData = array_merge($cvData, [
                'contract_type' => $this->editcv['contract_type'],
                'grade' => $this->editcv['grade'],
                'rate' => $this->editcv['rate'],
            ]);

            // تحديث سجل الموظف
            $employee->update([
                'cv' => json_encode($cvData, JSON_UNESCAPED_UNICODE),
            ]);

            // عرض رسالة نجاح
            $this->dispatch('notify', type: 'success', message: __('backend.details_added_successfully'), duration: 6000);

            // إغلاق المودال
            $this->activeModal = null;
            $this->reset('editcv'); // تفريغ النموذج
        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('notify', type: 'error', message: $e->validator->errors()->first(), duration: 6000);
        } catch (\Throwable $e) {
            $this->dispatch('notify', type: 'error', message: __('frontend.anothererror'), duration: 6000);
        }
    }

    //-------- add process section ----------------------\\


    public function nextStep()
    {
        try {
            if ($this->activeModal === 'addEmp') {
                $this->validate($this->stepRules(), $this->messages, $this->attributes());
            } elseif ($this->activeModal === 'editEmp') {
                $this->validate($this->stepEditRules(), $this->messages, $this->attributes());
            }

            $this->currentStep++;
        } catch (\Illuminate\Validation\ValidationException $e) {
            $firstError = $e->validator->errors()->first();
            $this->dispatch('notify', type: 'error', message: $firstError, duration: 6000);
        }
    }

    private function stepRules(): array
    {
        if ($this->currentStep === 1) {
            return [
                'new.name' => [
                    'required',
                    'string',
                    'max:255',
                    function ($attribute, $value, $fail) {
                        if (str_word_count($value) < 3) {
                            $fail(__('validation.fullname_required_three_words'));
                        }
                    },
                ],
                'new.name_ar' => [
                    'required',
                    'string',
                    'max:255',
                    'regex:/^[\p{Arabic}\s]+$/u',
                    function ($attribute, $value, $fail) {
                        if (count(preg_split('/\s+/u', trim($value))) < 3) {
                            $fail(__('validation.fullname_required_three_words_ar'));
                        }
                    },
                ],
                'new.email' => ['nullable', 'string', 'lowercase', 'email', 'max:255', Rule::unique('users', 'email')],
                'new.phone' => ['required', 'array', 'min:1'],
                'new.phone.*' => ['required', 'string', 'regex:/^(\+?20|0020)?1?\d{9}$/'],
            ];
        }

        if ($this->currentStep === 2) {
            return [
                'new.national_id' => [
                    'required',
                    'digits:14',
                    'unique:users_employees,national_id',
                    function ($attribute, $value, $fail) {
                        if (!preg_match('/^[23][0-9]{13}$/', $value)) {
                            return $fail(__('validation.national_id_format_invalid'));
                        }

                        $century = match (substr($value, 0, 1)) {
                            '2' => 1900,
                            '3' => 2000,
                            default => null,
                        };

                        $year = (int) substr($value, 1, 2);
                        $month = (int) substr($value, 3, 2);
                        $day = (int) substr($value, 5, 2);

                        if (!checkdate($month, $day, $century + $year)) {
                            return $fail(__('validation.national_id_birthdate_invalid'));
                        }

                        $govCode = substr($value, 7, 2);
                        $validGovCodes = ['01', '02', '03', '04', '11', '12', '13', '14', '15', '16', '17', '18', '19', '21', '22', '23', '24', '25', '26', '27', '28', '29', '31', '32', '33', '34', '35', '88'];

                        if (!in_array($govCode, $validGovCodes)) {
                            return $fail(__('validation.national_id_governorate_invalid'));
                        }

                        $genderDigit = substr($value, 12, 1);
                        if (!is_numeric($genderDigit)) {
                            return $fail(__('validation.national_id_gender_invalid'));
                        }
                    },
                ],
                'new.project_code' => ['required', 'exists:users_projects,code'],
                'new.hire_date' => ['required', 'date', 'before_or_equal:today'],
                'new.job_title_id' => ['required', 'exists:users_job_titles,id'],
            ];
        }

        if ($this->currentStep === 3) {
            return [
                'new.id_card_images' => ['required', 'array', 'min:1'],
                'new.id_card_images.*' => ['file', 'mimes:jpeg,jpg,png,pdf', 'max:4096'],
                'new.profile_picture' => ['nullable', 'image', 'mimes:jpeg,jpg,png', 'max:2048'],
                'new.skills' => ['nullable', 'array'],
                'new.skills.*' => ['string', 'max:50'],
            ];
        }

        return [];
    }


    //----------------------Edit section ----------------------------------\\
    public $editingEmployee = null;




    public function openEditEmpModal(UserEmployee $employee)
    {
        $cvArray = json_decode($employee->cv, true);
        $this->editEmployeeCv = $cvArray;
        $this->editcv = [
            'name' => $employee->full_name['en'] ?? '',
            'name_ar' => $employee->full_name['ar'] ?? '',
            'email' => $cvArray['email'] ?? null,
            'phone' => json_decode($employee->phone ?? '[]', true),
            'project_code' => $employee->project_code,
            'job_title_id' => $employee->job_title_id,
            'national_id' => $employee->national_id,
            'profile_picture' => $cvArray['profile_picture'] ?? null,
            'id_card_images' => json_decode($employee->id_card_image ?? '[]', true),
            'id_card_images_buffer' => json_decode($employee->id_card_image ?? '[]', true),
            'skills' => $cvArray['skills'] ?? [],
            'hire_date' => optional($employee->hiring_date)->format('Y-m-d'),
            'contract_date' => optional($employee->contract_date)->format('Y-m-d'),
            'experience_roads' => $cvArray['work_experience']['roads'] ?? [],
            'experience_airports' => $cvArray['work_experience']['airports'] ?? [],
        ];

        $this->editingEmployee = $employee->id; // ضروري نعرف من بنعدل
        $this->activeModal = 'editEmp';
        $this->currentStep = 1;
    }

    private function editRules(): array
    {
        return [
            'editcv.hire_date' => ['required', 'date', 'before_or_equal:today'],
            'editcv.name' => [
                'required',
                'string',
                'max:255',
                function ($attribute, $value, $fail) {
                    if (str_word_count($value) < 3) {
                        $fail(__('validation.fullname_required_three_words'));
                    }
                },
            ],
            'editcv.name_ar' => [
                'required',
                'string',
                'max:255',
                'regex:/^[\p{Arabic}\s]+$/u',
                function ($attribute, $value, $fail) {
                    if (count(preg_split('/\s+/u', trim($value))) < 3) {
                        $fail(__('validation.fullname_required_three_words_ar'));
                    }
                },
            ],
            'editcv.email' => ['nullable', 'string', 'lowercase', 'email', 'max:255', Rule::unique('users', 'email')->ignore($this->editingEmployee, 'id')],
            'editcv.phone' => ['required', 'array', 'min:1'],
            'editcv.phone.*' => ['required', 'string', 'regex:/^(\+?20|0020)?1?\d{9}$/'],
            'editcv.project_code' => ['required', 'exists:users_projects,code'],
            'editcv.national_id' => [
                'required',
                'digits:14',
                Rule::unique('users_employees', 'national_id')->ignore($this->editingEmployee),
                function ($attribute, $value, $fail) {
                    if (!preg_match('/^[23][0-9]{13}$/', $value)) {
                        return $fail(__('validation.national_id_format_invalid'));
                    }

                    $century = match (substr($value, 0, 1)) {
                        '2' => 1900,
                        '3' => 2000,
                        default => null,
                    };

                    $year = (int) substr($value, 1, 2);
                    $month = (int) substr($value, 3, 2);
                    $day = (int) substr($value, 5, 2);

                    if (!checkdate($month, $day, $century + $year)) {
                        return $fail(__('validation.national_id_birthdate_invalid'));
                    }

                    $govCode = substr($value, 7, 2);
                    $validGovCodes = ['01', '02', '03', '04', '11', '12', '13', '14', '15', '16', '17', '18', '19', '21', '22', '23', '24', '25', '26', '27', '28', '29', '31', '32', '33', '34', '35', '88'];

                    if (!in_array($govCode, $validGovCodes)) {
                        return $fail(__('validation.national_id_governorate_invalid'));
                    }

                    $genderDigit = substr($value, 12, 1);
                    if (!is_numeric($genderDigit)) {
                        return $fail(__('validation.national_id_gender_invalid'));
                    }
                },
            ],
            'editcv.hire_date' => ['required', 'date', 'before_or_equal:today'],
            'editcv.job_title_id' => ['required', 'exists:users_job_titles,id'],
            'editcv.skills' => ['nullable', 'array'],
            'editcv.skills.*' => ['string', 'max:50'],
        ];
    }


    public $editcv = [
        'name' => '',
        'name_ar' => '',
        'email' => '',
        'phone' => [''],
        'project_code' => '',
        'job_title_id' => '',
        'national_id' => '',
        'profile_picture' => null,
        'id_card_images' => [],
        'id_card_images_buffer' => [], // لو عايز buffer
        'skills' => [],
        'hire_date' => '', // ده مهم جدًا!
        'contract_date' => '', // ده مهم جدًا!
        'experience_roads' => [],
        'experience_airports' => [],
    ];


    public function updateEmp()
    {
        try {
            $this->validate($this->editRules(), $this->messages, $this->attributes());

            DB::transaction(function () {
                $employee = UserEmployee::findOrFail($this->editingEmployee);

                $folder = 'employees/' . $employee->employee_code;

                $newUploaded = collect($this->editcv['id_card_images'])
                    ->filter(fn($f) => $f instanceof LivewireTemp)
                    ->map(function ($file) use ($folder) {
                        $filename = Str::slug($this->editcv['name']) . '_' . Str::random(6) . '.' . $file->getClientOriginalExtension();
                        return $file->storeAs($folder, $filename, 'public');
                    })
                    ->all();

                // 🟢 دمج القديم مع الجديد
                $existing = array_filter($this->editcv['id_card_images_buffer'], fn($f) => is_string($f));
                $idCardImageFinal = array_merge($existing, $newUploaded);

                $oldCv = json_decode($employee->cv, true);
                $profilePicturePath = $this->processProfilePic($folder, $oldCv['profile_picture'] ?? null);

                // استخراج معلومات الرقم القومي
                $info = $this->parseNationalId($this->editcv['national_id']);

                // تحديث البيانات
                $employee->update([
                    'project_code' => $this->editcv['project_code'],
                    'job_title_id' => $this->editcv['job_title_id'],
                    'full_name' => [
                        'ar' => $this->editcv['name_ar'],
                        'en' => $this->editcv['name'],
                    ],
                    'phone' => json_encode($this->editcv['phone']),
                    'national_id' => $this->editcv['national_id'],
                    'birth_date' => $info['birth_date'],
                    'governorate' => $info['governorate'],
                    'gender' => $info['gender'],
                    'id_card_image' => json_encode($idCardImageFinal),
                    'hiring_date' => $this->editcv['hire_date'],
                    'contract_date' => $this->editcv['contract_date'],
                    'cv' => json_encode(
                        array_filter(
                            [
                                'work_experience' => [
                                    'roads' => $this->editcv['experience_roads'],
                                    'airports' => $this->editcv['experience_airports'],
                                ],
                                'email' => $this->editcv['email'] ?? null,
                                'profile_picture' => $profilePicturePath,
                                'skills' => $this->editcv['skills'],
                                'qualification' => $this->editcv['qualification'], // المؤهل الدراسي
                                'graduation_year' => $this->editcv['graduation_year'], // سنة التخرج
                                'graduation_project' => $this->editcv['graduation_project'], // مشروع التخرج
                                'military_status' => $this->editcv['military_status'], // الموقف من التجنيد

                            ],
                            fn($v) => !is_null($v) && $v !== '',
                        ),
                        JSON_UNESCAPED_UNICODE,
                    ),
                ]);
            });

            // Success toast
            $this->dispatch('notify', type: 'success', message: __('backend.cv_updated_successfully'), duration: 6000);

            $this->activeModal = null; // ✅ إغلاق المودال
            $this->reset('editcv', 'editingEmployee'); // ✅ تفريغ النموذج
            $this->resetPage();
            $this->loadEmployees();
            $this->readyToLoad = true;
        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('notify', type: 'error', message: $e->validator->errors()->first(), duration: 6000);
        } catch (\Throwable $e) {
            $this->dispatch('notify', type: 'error', message: __('frontend.anothererror'), duration: 6000);
            $this->dispatch('notify', type: 'error', message: $e->getMessage(), duration: 10000);
        }
    }


    //-------------- edit process section ------------\\

    private function stepEditRules(): array
    {
        if ($this->currentStep === 1) {
            return [
                'editcv.name' => [
                    'required',
                    'string',
                    'max:255',
                    function ($attribute, $value, $fail) {
                        if (str_word_count($value) < 3) {
                            $fail(__('validation.fullname_required_three_words'));
                        }
                    },
                ],
                'editcv.name_ar' => [
                    'required',
                    'string',
                    'max:255',
                    'regex:/^[\p{Arabic}\s]+$/u',
                    function ($attribute, $value, $fail) {
                        if (count(preg_split('/\s+/u', trim($value))) < 3) {
                            $fail(__('validation.fullname_required_three_words_ar'));
                        }
                    },
                ],
                'editcv.email' => ['nullable', 'string', 'lowercase', 'email', 'max:255', Rule::unique('users', 'email')->ignore($this->editingEmployee, 'id')],
                'editcv.phone' => ['required', 'array', 'min:1'],
                'editcv.phone.*' => ['required', 'string', 'regex:/^(\+?20|0020)?1?\d{9}$/'],
            ];
        }

        if ($this->currentStep === 2) {
            return [
                'editcv.national_id' => [
                    'required',
                    'digits:14',
                    Rule::unique('users_employees', 'national_id')->ignore($this->editingEmployee),
                    // باقي الفاليديشين زي الموجود
                ],
                'editcv.project_code' => ['required', 'exists:users_projects,code'],
                'editcv.hire_date' => ['required', 'date', 'before_or_equal:today'],
                'editcv.contract_date' => ['nullable', 'date', 'before_or_equal:today'],
                'editcv.job_title_id' => ['required', 'exists:users_job_titles,id'],
            ];
        }

        if ($this->currentStep === 3) {
            return [
                'editcv.skills' => ['nullable', 'array'],
                'editcv.skills.*' => ['string', 'max:50'],
                'editcv.qualification' => 'required|string|max:255',
                'editcv.graduation_year' => 'required|integer|min:1900|max:' . date('Y'),
                'editcv.graduation_project' => 'nullable|string|max:255',
                'editcv.military_status' => 'required|in:exempted,completed,postponed,not_required',
                'editcv.contract_type' => 'required|in:contract,serky',
            ];
        }

        return [];
    }

    // ------------------ process section ------------------------ \\

    private function processProfilePic(string $folder, ?string $oldPath): ?string
    {
        // لو المستخدم رفع صورة جديدة
        if ($this->editcv['profile_picture'] instanceof LivewireTemp) {
            $file = $this->editcv['profile_picture'];
            $filename = Str::slug($this->editcv['name']) . '_profile_' . Str::random(6) . '.' . $file->getClientOriginalExtension();

            // خزّن الصورة وأرجع المسار
            return $file->storeAs($folder, $filename, 'public');
        }

        // مفيش صورة جديدة → نحتفظ بالمسار القديم
        return $oldPath;
    }

    private function parseNationalId(string $id): array
    {
        // 🗓️ تاريخ الميلاد
        $century = match (substr($id, 0, 1)) {
            '2' => 1900,
            '3' => 2000,
            default => null,
        };

        $year = (int) substr($id, 1, 2);
        $month = (int) substr($id, 3, 2);
        $day = (int) substr($id, 5, 2);

        $birthDate = $century ? sprintf('%04d-%02d-%02d', $century + $year, $month, $day) : null;

        // 🏙️ المحافظة
        $govCode = substr($id, 7, 2);
        $governorates = [
            '01' => ['ar' => 'القاهرة', 'en' => 'Cairo'],
            '02' => ['ar' => 'الإسكندرية', 'en' => 'Alexandria'],
            '03' => ['ar' => 'بورسعيد', 'en' => 'Port Said'],
            '04' => ['ar' => 'السويس', 'en' => 'Suez'],
            '11' => ['ar' => 'دمياط', 'en' => 'Damietta'],
            '12' => ['ar' => 'الدقهلية', 'en' => 'Dakahlia'],
            '13' => ['ar' => 'الشرقية', 'en' => 'Sharqia'],
            '14' => ['ar' => 'القليوبية', 'en' => 'Qalyubia'],
            '15' => ['ar' => 'كفر الشيخ', 'en' => 'Kafr El-Sheikh'],
            '16' => ['ar' => 'الغربية', 'en' => 'Gharbia'],
            '17' => ['ar' => 'المنوفية', 'en' => 'Monufia'],
            '18' => ['ar' => 'البحيرة', 'en' => 'Beheira'],
            '19' => ['ar' => 'الإسماعيلية', 'en' => 'Ismailia'],
            '21' => ['ar' => 'الجيزة', 'en' => 'Giza'],
            '22' => ['ar' => 'بني سويف', 'en' => 'Beni Suef'],
            '23' => ['ar' => 'الفيوم', 'en' => 'Faiyum'],
            '24' => ['ar' => 'المنيا', 'en' => 'Minya'],
            '25' => ['ar' => 'أسيوط', 'en' => 'Assiut'],
            '26' => ['ar' => 'سوهاج', 'en' => 'Sohag'],
            '27' => ['ar' => 'قنا', 'en' => 'Qena'],
            '28' => ['ar' => 'أسوان', 'en' => 'Aswan'],
            '29' => ['ar' => 'الأقصر', 'en' => 'Luxor'],
            '31' => ['ar' => 'البحر الأحمر', 'en' => 'Red Sea'],
            '32' => ['ar' => 'الوادي الجديد', 'en' => 'New Valley'],
            '33' => ['ar' => 'مطروح', 'en' => 'Matrouh'],
            '34' => ['ar' => 'شمال سيناء', 'en' => 'North Sinai'],
            '35' => ['ar' => 'جنوب سيناء', 'en' => 'South Sinai'],
            '88' => ['ar' => 'خارج الجمهورية', 'en' => 'Outside Egypt'],
        ];

        $governorate = [
            'code' => $govCode,
            'en' => $governorates[$govCode]['en'] ?? null,
            'ar' => $governorates[$govCode]['ar'] ?? null,
        ];

        // 👤 النوع (رقم فردي = ذكر / زوجي = أنثى)
        $genderDigit = (int) substr($id, 12, 1);
        $gender = null;
        if (strlen($id) >= 13) {
            $genderDigit = (int) substr($id, 12, 1);
            $gender = $genderDigit % 2 === 0 ? 'female' : 'male';
        }

        return [
            'birth_date' => $birthDate,
            'governorate' => $governorate,
            'gender' => $gender,
        ];
    }

    public function updatedIdCardImages($newFiles)
    {
        // Merge القيمة المؤقتة مع الملفات الجديدة
        $merged = collect($this->new['id_card_images_buffer'])->merge($newFiles)->unique(fn($file) => $file->getClientOriginalName() . $file->getSize())->values()->all();

        $this->new['id_card_images'] = $merged;
        $this->new['id_card_images_buffer'] = $merged; // تحديث النسخة المؤقتة
    }

    public function removeImage($index)
    {
        unset($this->new['id_card_images'][$index]);
        $this->new['id_card_images'] = array_values($this->new['id_card_images']);

        unset($this->new['id_card_images_buffer'][$index]);
        $this->new['id_card_images_buffer'] = array_values($this->new['id_card_images_buffer']);
    }



    public function removeEditcvImage($index)
    {
        unset($this->editcv['id_card_images'][$index]);
        $this->editcv['id_card_images'] = array_values($this->editcv['id_card_images']);

        unset($this->editcv['id_card_images_buffer'][$index]);
        $this->editcv['id_card_images_buffer'] = array_values($this->editcv['id_card_images_buffer']);
    }


    public function updatedEditcvIdCardImages($newFiles)
    {
        $merged = collect($this->editcv['id_card_images_buffer'])->merge($newFiles)->unique(fn($file) => $file->getClientOriginalName() . $file->getSize())->values()->all();

        $this->editcv['id_card_images'] = $merged;
        $this->editcv['id_card_images_buffer'] = $merged;
    }
}
