<?php

namespace App\Models\User;


use App\Models\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

class UserEmployeeMovement extends Model
{
    use HasFactory;

    protected $table = 'users_employee_movements';

    protected $fillable = [
        'created_by',
        'status',
        'type',
        'employee_id',
        'start_date',
        'end_date',
        'from_project',
        'to_project',
        'job_uuid',
        'cancelled_by',
        'cancelled_at',
        'cancelled_reason',
        'notes',
    ];

    protected $casts = [
        'start_date'   => 'date',
        'end_date'     => 'date',
        'cancelled_at' => 'datetime',
    ];

    /**
     * المستخدم الذي أنشأ الحركة
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * الموظف المرتبط بهذه الحركة
     */
    public function employee(): BelongsTo
    {
        return $this->belongsTo(UserEmployee::class, 'employee_id');
    }

    /**
     * المشروع الأصلي قبل الحركة
     */
    public function fromProject(): BelongsTo
    {
        return $this->belongsTo(UserProject::class, 'from_project');
    }

    /**
     * المشروع الوجهة
     */
    public function toProject(): BelongsTo
    {
        return $this->belongsTo(UserProject::class, 'to_project');
    }

    /**
     * المستخدم الذي ألغى الحركة
     */
    public function canceller(): BelongsTo
    {
        return $this->belongsTo(User::class, 'cancelled_by');
    }


    /**
     * تحديث حالات الحركات وفق السيناريو الموصوف
     * - pending إذا لم يحن وقت البدء بعد
     * - activated إذا دخل نطاق start_date <= now < end_date
     * - applied إذا لا توجد end_date أو تجاوزنا end_date
     * وعند النوع transfer: تحديث project_code في جدول الموظف
     */
    public static function refreshStatuses(): void
    {
        $now = Carbon::now();

        // 1. pending → applied (open-ended أو تجاوز end_date)
        $openOrPast = self::where('status', 'pending')
            ->whereDate('start_date', '<=', $now)
            ->where(function ($q) use ($now) {
                $q->whereNull('end_date')
                  ->orWhereDate('end_date', '<', $now);
            });
        $openOrPast->update(['status' => 'activated']);

        // 2. pending → activated (داخل النطاق)
        self::where('status', 'pending')
            ->whereDate('start_date', '<=', $now)
            ->whereNotNull('end_date')
            ->whereDate('end_date', '>=', $now)
            ->update(['status' => 'activated']);

        // 3. activated → applied (انتهت مأموريات مغلقة)
        self::where('status', 'activated')
            ->whereNotNull('end_date')
            ->whereDate('end_date', '<', $now)
            ->update(['status' => 'applied']);

        // 4. تحديث دفعة لحركات transfer: آخر حركة لكل موظف
        $latestTransfers = self::select('employee_id', DB::raw('MAX(start_date) as max_start'))
            ->where('type', 'transfer')
            ->whereIn('status', ['activated','applied'])
            ->groupBy('employee_id')
            ->pluck('max_start','employee_id');

        // جلب الحركات بناءً على start_date الأقصى
        $transfers = self::where('type','transfer')
            ->whereIn('status',['activated','applied'])
            ->whereIn('employee_id', array_keys($latestTransfers->toArray()))
            ->get()
            ->filter(function($m) use ($latestTransfers) {
                return $m->start_date->eq($latestTransfers[$m->employee_id]);
            });

        // تحديث دفعة لحقل project_code
        $updates = [];
        foreach ($transfers as $movement) {
            if ($movement->toProject) {
                $updates[$movement->employee_id] = ['project_code' => $movement->toProject->code];
            }
        }
        foreach ($updates as $empId => $data) {
            UserEmployee::where('id',$empId)->update($data);
        }
    }

    /**
     * تحديث حالات الحركات لموظف معين بكفاءة وحفظ دفعة
     *
     * @param int $employeeId
     */
    public static function refreshStatusesForEmployee(int $employeeId): void
    {
        $now = Carbon::now();

        // 1. pending → applied (open-ended أو تجاوز end_date)
        self::where('employee_id', $employeeId)
            ->where('status', 'pending')
            ->whereDate('start_date', '<=', $now)
            ->where(function ($q) use ($now) {
                $q->whereNull('end_date')
                  ->orWhereDate('end_date', '<', $now);
            })
            ->update(['status' => 'activated']);

        // 2. pending → activated (داخل النطاق)
        self::where('employee_id', $employeeId)
            ->where('status', 'pending')
            ->whereDate('start_date', '<=', $now)
            ->whereNotNull('end_date')
            ->whereDate('end_date', '>=', $now)
            ->update(['status' => 'activated']);

        // 3. activated → applied (انتهت مأموريات مغلقة)
        self::where('employee_id', $employeeId)
            ->where('status', 'activated')
            ->whereNotNull('end_date')
            ->whereDate('end_date', '<', $now)
            ->update(['status' => 'applied']);

        // 4. تحديث نقل الموظف لآخر حركة transfer
        $latestStart = self::where('employee_id', $employeeId)
            ->where('type', 'transfer')
            ->whereIn('status', ['activated','applied'])
            ->max('start_date');

        if ($latestStart) {
            $movement = self::where('employee_id', $employeeId)
                ->where('type', 'transfer')
                ->whereIn('status', ['activated','applied'])
                ->whereDate('start_date', $latestStart)
                ->first();

            if ($movement && $movement->toProject) {
                UserEmployee::where('id', $employeeId)
                    ->update(['project_code' => $movement->toProject->code]);
            }
        }
    }



    /**
     * جلب الحركات الجارية والمستقبلية لموظف معين
     *
     * @param  int  $employeeId
     * @return array
     */
    public static function getCurrentAndUpcoming(int $employeeId): array
    {
        $today = Carbon::today();

        $movements = self::with(['fromProject', 'toProject', 'creator', 'canceller'])
    ->where('employee_id', $employeeId)
    // ➊ الحالات المسموح بها
    ->whereIn('status', ['activated', 'pending'])
    // ➋ شرط التاريخ (مستقبلية أو جارية)
    ->where(function ($q) use ($today) {
        $q->whereDate('start_date', '>=', $today)           // حركات تبدأ اليوم أو بعده
          ->orWhere(function ($sub) use ($today) {          // أو حركات جارية الآن
              $sub->whereDate('start_date', '<=', $today)   //   البداية ≤ اليوم
                  ->where(function ($end) use ($today) {
                      $end->whereNull('end_date')           //   نهاية مفتوحة
                          ->orWhereDate('end_date', '>=', $today); // أو اليوم ≤ النهاية
                  });
          });
    })
    ->orderBy('start_date')
    ->get();




        // يرجّع مجموعة النتائج كمصفوفة
        return $movements->map(function ($m) {
            return [
                'id'               => $m->id,
                'status'           => $m->status,
                'type'             => $m->type,
                'start_date'       => $m->start_date->toDateString(),
                'end_date'         => $m->end_date?->toDateString(),
                'from_project'     => [
                    'id'   => $m->fromProject->id,
                    'name' => $m->fromProject->name,
                ],
                'to_project'       => $m->toProject ? [
                    'id'   => $m->toProject->id,
                    'name' => $m->toProject->name,
                ] : null,
                'notes'            => $m->notes,
                'created_by'       => $m->creator?->name,
                'cancelled_by'     => $m->canceller?->name,
                'cancelled_at'     => $m->cancelled_at?->toDateTimeString(),
                'cancelled_reason' => $m->cancelled_reason,
            ];
        })->toArray();
    }
}
