<?php
use Livewire\Volt\Component;
use App\Models\Z_Website\Market\MarketItem;
use App\Models\Z_Website\Market\MarketTransaction;
use App\Models\SRO_VT_SHARD\RefObjCommon;
use App\Models\SRO_VT_SHARD\Chest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\WithPagination;

new class extends Component {
    use WithPagination;

    public $searchTerm = '';
    public $filterType = 'all'; // all, available, expiring
    public $priceRange = 'all'; // all, low, medium, high
    public $sortType = 'all'; // all, premium, normal, event
    public $selectedImageUrl = '';
    public $selectedImageAlt = '';
    public $perPage = 12; // عدد العناصر في كل صفحة

    public function mount()
    {
        // لا نحتاج لتحميل البيانات في mount مع pagination
    }

    //load data with pagination - استخدام render بدلاً من خاصية
    public function render() : mixed
    {
        $query = MarketItem::where('balance_type', '!=', 'balance')
            ->where('stock', '!=', 0) // Exclude out of stock items
            ->where('price', '>', 0) // Exclude free items
            ->where('sort_type', '!=', '') // Exclude items without a sort type
            //check expiry_date if not null Check if expiry_date is after now
            ->where(function ($query) {
                $query->whereNull('expiry_date')
                      ->orWhere('expiry_date', '>', now());
            })
            ->where('package_id', 0); // Exclude package items

        // تطبيق المرشحات على الاستعلام
        $query = $this->applyFiltersToQuery($query);

        // الحصول على العناصر مع pagination
        $filteredItems = $query->orderBy('sort_order')
            ->paginate($this->perPage)
            ->withQueryString(); // للحفاظ على معاملات الاستعلام في URL

        return view('livewire.backend.pages.gift-item-shop', [
            'filteredItems' => $filteredItems,
            'totalItems' => $query->count()
        ]);
    }

    private function applyFiltersToQuery($query)
    {
        // تطبيق مرشح البحث
        if (!empty($this->searchTerm)) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . $this->searchTerm . '%')
                  ->orWhere('description', 'like', '%' . $this->searchTerm . '%');
            });
        }

        // تطبيق مرشح النوع (حالة المخزون)
        if ($this->filterType === 'available') {
            $query->where(function ($q) {
                $q->where('stock', '>', 0)->orWhere('stock', -1);
            });
        } elseif ($this->filterType === 'expiring') {
            $query->whereNotNull('expiry_date')
                  ->where('expiry_date', '>', now())
                  ->where('expiry_date', '<', now()->addDays(7));
        }

        // تطبيق مرشح نطاق السعر
        if ($this->priceRange !== 'all') {
            switch ($this->priceRange) {
                case 'low':
                    $query->where('price', '<=', 1000);
                    break;
                case 'medium':
                    $query->whereBetween('price', [1001, 10000]);
                    break;
                case 'high':
                    $query->where('price', '>', 10000);
                    break;
            }
        }

        // تطبيق مرشح نوع الفرز
        if ($this->sortType !== 'all' && !empty($this->sortType)) {
            $query->where('sort_type', $this->sortType);
        }

        return $query;
    }

    // دالة مبسطة للفلاتر التي لا تحتاج pagination كامل
    public function applyFilters()
    {
        $this->resetPage(); // إعادة تعيين الصفحة عند تطبيق المرشحات
    }

    public function updatedSearchTerm()
    {
        $this->applyFilters();
    }

    public function updatedPriceRange()
    {
        $this->applyFilters();
    }

    public function updatedSortType()
    {
        $this->applyFilters();
    }

    public function setFilter($type)
    {
        $this->filterType = $type;
        $this->applyFilters();
    }

    public function setPriceRange($range)
    {
        $this->priceRange = $range;
        $this->applyFilters();
    }

    public function setSortType($type)
    {
        $this->sortType = $type;
        $this->applyFilters();
    }

    public function resetFilters()
    {
        $this->searchTerm = '';
        $this->filterType = 'all';
        $this->priceRange = 'all';
        $this->sortType = 'all';
        $this->applyFilters();
    }

    // إضافة دالة لتغيير عدد العناصر في الصفحة
    public function changePerPage($perPage)
    {
        $this->perPage = $perPage;
        $this->resetPage();
    }


    private function getImageUrl($imagePath)
    {
        if (empty($imagePath)) {
            return asset('assets/images/default-item.svg');
        }

        // إذا كان الرابط يحتوي على http أو https فهو رابط خارجي
        if (str_starts_with($imagePath, 'http')) {
            return $imagePath;
        }

        // إذا كان يبدأ بـ storage فهو ملف محلي من storage
        if (str_starts_with($imagePath, 'storage/')) {
            return asset($imagePath);
        }

        // إذا كان ملف من storage بدون البادئة
        if (str_contains($imagePath, '.')) {
            return asset('storage/' . $imagePath);
        }

        // افتراضي
        return asset('assets/images/default-item.svg');
    }

    public function getAvailableSortTypesProperty()
    {
        return MarketItem::where('balance_type', '!=', 'balance')
            ->where('stock', '!=', 0)
            ->where('price', '>', 0)
            ->where('sort_type', '!=', '')
            ->where(function ($query) {
                $query->whereNull('expiry_date')
                      ->orWhere('expiry_date', '>', now());
            })
            ->where('package_id', 0)
            ->pluck('sort_type')
            ->filter()
            ->unique()
            ->sort()
            ->values()
            ->toArray();
    }

    //buy item
    public function buyItem($itemId, $quantity = 1)
    {
        // التحقق من صحة الكمية
        if ($quantity <= 0) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:giftShop.notifications.invalidQuantity',
                'duration' => 5000
            ]);
            return;
        }

        //Get Item Information by itemId
        $item = MarketItem::find($itemId);
        if (!$item) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:giftShop.notifications.itemNotFound',
                'duration' => 5000
            ]);
            return;
        }

        //check user balance
        $user = Auth::user();
        if (!$user || !$user->webUser) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:giftShop.notifications.loginRequired',
                'duration' => 5000
            ]);
            return;
        }

        // التحقق من وجود JID
        if (!$user->JID) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:giftShop.notifications.loginRequired',
                'duration' => 5000
            ]);
            return;
        }

        // إضافة debugging للـ user data
        \Log::info('User data for purchase:', [
            'auth_id' => Auth::id(),
            'user_jid' => $user->JID,
            'balance_type' => $item->balance_type,
            'item_price' => $item->price,
            'quantity' => $quantity
        ]);

        // التحقق من الرصيد بناء على نوع العملة المطلوبة
        $totalCost = $item->price * $quantity;
        $hasEnoughBalance = false;

        switch($item->balance_type) {
            case 'gift_balance':
                $hasEnoughBalance = $user->webUser->gift_balance >= $totalCost;
                $currentBalance = $user->webUser->gift_balance;
                break;
            case 'silk':
                $hasEnoughBalance = $user->skSilk->silk_own >= $totalCost;
                $currentBalance = $user->skSilk->silk_own;
                break;
            case 'donate':
                $hasEnoughBalance = $user->skSilk->silk_gift >= $totalCost;
                $currentBalance = $user->skSilk->silk_gift;
                break;
            default:
                // إذا كان balance عادي، نرفض العملية
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'trans:giftShop.notifications.invalidCurrency',
                    'duration' => 5000
                ]);
                return;
        }

        if (!$hasEnoughBalance) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:giftShop.notifications.insufficientBalance',
                'duration' => 5000,
                'params' => [
                    'currency' => ucfirst(str_replace('_', ' ', $item->balance_type)),
                    'required' => number_format($totalCost, 2),
                    'available' => number_format($currentBalance, 2)
                ]
            ]);
            return;
        }

        //check stock if out of stock
        if ($item->stock !== -1 && $item->stock <= 0) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:giftShop.notifications.itemOutOfStock',
                'duration' => 5000
            ]);
            return;
        }

        //check request user from stock
        if ($item->stock !== -1 && $item->stock < $quantity) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:giftShop.notifications.notEnoughStock',
                'duration' => 5000,
                'params' => [
                    'available' => $item->stock
                ]
            ]);
            return;
        }

        //check expiry date
        if ($item->expiry_date) {
            $expiryDate = is_string($item->expiry_date)
                ? \Carbon\Carbon::parse($item->expiry_date)
                : $item->expiry_date;

            if ($expiryDate < now()) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'trans:giftShop.notifications.itemExpired',
                    'message' => 'Error: Item has expired!',
                    'duration' => 5000
                ]);
                return;
            }
        }

        //Before add Item to user inventory, check if user has enough space in inventory
        if ($item->item_type === 'item') {
            try {
                // جلب MaxStack من RefObjCommon + RefObjItem
                $refObjCommon = RefObjCommon::where('ID', $item->RefItemID)->with('getRefObjItem')->first();

                if (!$refObjCommon || !$refObjCommon->getRefObjItem) {
                    $this->dispatch('show-notification', [
                        'type' => 'error',
                        'message' => 'Error: Item reference data not found!',
                        'duration' => 5000
                    ]);
                    return;
                }

                $maxStack = $refObjCommon->getRefObjItem->MaxStack ?? 1;
                if ($maxStack <= 0) {
                    $maxStack = 1;
                }

                $totalNeeded = $item->Count * $quantity;
                $newSlotsNeeded = ceil($totalNeeded / $maxStack);

                $availableSlots = Chest::where('UserJID', $user->JID)
                    ->where(function ($query) {
                        $query->whereNull('ItemID')->orWhere('ItemID', 0);
                    })
                    ->where('slot', '<=', 149)
                    ->count();

                // إذا عدد الخانات المطلوبة أكبر من المتاحة => رفض العملية
                if ($newSlotsNeeded > $availableSlots) {
                    $missingSlots = $newSlotsNeeded - $availableSlots;
                    $this->dispatch('show-notification', [
                        'type' => 'error',
                        'message' => 'No available space in Storage page 1 ~ 5. You need ' . $missingSlots . ' more slots.',
                        'duration' => 5000
                    ]);
                    return;
                }
            } catch (\Exception $e) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Error checking inventory space: ' . $e->getMessage(),
                    'duration' => 5000
                ]);
                return;
            }
        }

        //Start transaction
        DB::beginTransaction();
        try {
            if($item->item_type === 'item'){
                // البروسيدور يقوم بخصم الرصيد بنفسه للعناصر

                // تسجيل البيانات المرسلة للـ stored procedure
                \Log::info('Calling stored procedure with data:', [
                    'JID' => $user->JID,
                    'type' => 'item',
                    'id' => $item->id,
                    'quantity' => $quantity,
                    'balance_type' => $item->balance_type,
                    'user_balance' => $user->webUser->balance,
                    'item_price' => $item->price
                ]);

                $results = DB::connection('sqlsrv')->select('EXEC [market_buy] ?, ?, ?, ?, ?', [
                    $user->JID,        // استخدام JID بدلاً من Auth::id()
                    'item',
                    $item->id,
                    $quantity,         // إرسال الكمية المطلوبة بدلاً من total_needed
                    $item->balance_type
                ]);

                // تسجيل النتيجة المرجعة
                \Log::info('Stored procedure result:', ['results' => $results]);

            }else if ($item->item_type === 'Donate'){
                // خصم الرصيد يدوياً للأنواع الأخرى
                $user->webUser->balance -= $item->price * $quantity;
                $user->webUser->save();

                $user->skSilk->silk_gift += $item->Count * $quantity;
                $user->skSilk->save();
            }else if ($item->item_type === 'Silk'){
                // خصم الرصيد يدوياً للأنواع الأخرى
                $user->webUser->balance -= $item->price * $quantity;
                $user->webUser->save();

                $user->skSilk->silk_own += $item->Count * $quantity;
                $user->skSilk->save();
            }

            // التحقق من نتيجة الإجراء المخزن للعناصر فقط
            if($item->item_type === 'item'){
                if (!$results || !isset($results[0])) {
                    throw new \Exception('No response from stored procedure');
                }

                $result = $results[0];

                // التحقق من وجود status بدلاً من Result
                if (!isset($result->status)) {
                    throw new \Exception('Invalid stored procedure response - missing status field');
                }

                // التحقق من حالة النجاح (status = 1 يعني نجاح)
                if ($result->status != 1) {
                    $errorMessage = isset($result->message) ? $result->message : 'Unknown error';
                    throw new \Exception($errorMessage . ' (Status: ' . $result->status . ')');
                }

                // تسجيل النجاح
                \Log::info('Item purchase successful', [
                    'item_id' => $item->id,
                    'user_jid' => $user->JID,
                    'quantity' => $quantity,
                    'message' => $result->message ?? 'Success'
                ]);
            }

            // تحديث المخزون إذا لم يكن لا نهائي
            if ($item->stock !== -1) {
                $item->stock -= $quantity;
                $item->save();
            }

            DB::commit();

            // إرسال رسالة نجاح
            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'trans:giftShop.notifications.purchaseSuccess',
                'duration' => 5000
            ]);

            // تسجيل المعاملة بعد النجاح
            // نحتاج نجيب الرصيد الجديد بعد الخصم
            $user->refresh(); // تحديث بيانات المستخدم
            $newBalance = 0;
            switch ($item->balance_type) {
                case 'gift_balance':
                    $newBalance = $user->webUser->gift_balance ?? 0;
                    break;
                case 'silk':
                    $newBalance = $user->skSilk->silk_own ?? 0;
                    break;
                case 'donate':
                    $newBalance = $user->skSilk->silk_gift ?? 0;
                    break;
            }

            // تسجيل المعاملة
            MarketTransaction::createLog([
                'user_jid' => $user->JID,
                'username' => $user->username ?? $user->UserID ?? 'Unknown',
                'transaction_type' => 'item',
                'item_id' => $item->id,
                'item_name' => $item->name,
                'quantity' => $quantity,
                'unit_price' => $item->price,
                'total_price' => $totalCost,
                'currency_type' => $item->balance_type,
                'balance_before' => $currentBalance,
                'balance_after' => $newBalance,
                'status' => 'success',
                'transaction_details' => [
                    'item_name' => $item->name,
                    'item_type' => $item->item_type,
                    'purchase_method' => $item->item_type === 'item' ? 'stored_procedure' : 'manual',
                    'item_count' => $item->Count ?? 1,
                    'shop_type' => 'item shop',
                ]
            ]);

            // تحديث الأرصدة في الواجهة الأمامية
            $this->dispatch('updateBalances');

        } catch (\Exception $e) {
            DB::rollBack();

            // تسجيل المعاملة الفاشلة
            try {
                MarketTransaction::createLog([
                    'user_jid' => $user->JID,
                    'username' => $user->username ?? $user->UserID ?? 'Unknown',
                    'transaction_type' => 'item',
                    'item_id' => $item->id,
                    'item_name' => $item->name,
                    'quantity' => $quantity,
                    'unit_price' => $item->price,
                    'total_price' => $totalCost,
                    'currency_type' => $item->balance_type,
                    'balance_before' => $currentBalance,
                    'balance_after' => $currentBalance, // لم يتغير الرصيد بسبب الفشل
                    'status' => 'failed',
                    'error_message' => $e->getMessage(),
                    'transaction_details' => [
                        'item_name' => $item->name,
                        'item_type' => $item->item_type,
                        'failure_reason' => $e->getMessage(),
                        'attempted_method' => $item->item_type === 'item' ? 'stored_procedure' : 'manual',
                        'shop_type' => 'item shop',
                    ]
                ]);
            } catch (\Exception $logException) {
                // إذا فشل تسجيل الـ log، لا نريد أن يتوقف التطبيق
                \Log::error('Failed to log failed gift item transaction: ' . $logException->getMessage());
            }

            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:giftShop.notifications.purchaseFailed',
                'duration' => 5000,
                'params' => [
                    'error' => $e->getMessage()
                ]
            ]);
        }
    }
};
?>
<div>


    <div class="gift-shop-container p-4"
         x-data="{
            selectedImageUrl: <?php if ((object) ('selectedImageUrl') instanceof \Livewire\WireDirective) : ?>window.Livewire.find('<?php echo e($__livewire->getId()); ?>').entangle('<?php echo e('selectedImageUrl'->value()); ?>')<?php echo e('selectedImageUrl'->hasModifier('live') ? '.live' : ''); ?><?php else : ?>window.Livewire.find('<?php echo e($__livewire->getId()); ?>').entangle('<?php echo e('selectedImageUrl'); ?>')<?php endif; ?>,
            selectedImageAlt: <?php if ((object) ('selectedImageAlt') instanceof \Livewire\WireDirective) : ?>window.Livewire.find('<?php echo e($__livewire->getId()); ?>').entangle('<?php echo e('selectedImageAlt'->value()); ?>')<?php echo e('selectedImageAlt'->hasModifier('live') ? '.live' : ''); ?><?php else : ?>window.Livewire.find('<?php echo e($__livewire->getId()); ?>').entangle('<?php echo e('selectedImageAlt'); ?>')<?php endif; ?>,
            isLoading: false
         }"
         x-init="isLoading = true; setTimeout(() => isLoading = false, 1000)">

        <!-- Loading State -->
        <div x-show="isLoading" class="flex items-center justify-center py-20">
            <div class="text-center">
                <div class="animate-spin rounded-full h-16 w-16 border-b-4 border-cyan-400 mx-auto mb-4"></div>
                <p class="text-purple-200" wire:ignore  data-translate="itemShop.status.loading">Loading items...</p>
            </div>
        </div>

        <div x-show="!isLoading" x-transition class="max-w-7xl mx-auto">

            <!-- Page Header -->
                        <!-- Gift Shop Header -->
            <div class="gift-shop-header text-center mb-8 p-8 rounded-2xl">
                <div class="w-20 h-20 mx-auto mb-6 bg-gradient-to-r from-orange-500 to-pink-600 rounded-3xl flex items-center justify-center shadow-lg shadow-pink-500/25">
                    <!-- Heart SVG Icon -->
                    <svg class="w-10 h-10 text-white" fill="currentColor" viewBox="0 0 24 24">
                        <path d="M12 21.35l-1.45-1.32C5.4 15.36 2 12.28 2 8.5 2 5.42 4.42 3 7.5 3c1.74 0 3.41.81 4.5 2.09C13.09 3.81 14.76 3 16.5 3 19.58 3 22 5.42 22 8.5c0 3.78-3.4 6.86-8.55 11.54L12 21.35z"/>
                    </svg>
                </div>
                <h1 class="gift-shop-title mb-4">Item Shop</h1>
                <p class="gift-shop-subtitle mb-6">Purchase items with Gift Balance, Silk, or Donate points</p>

                <!-- Filter Stats -->
                <div class="flex justify-center gap-4 text-sm text-purple-300">
                    <span>Total Items: <strong class="text-cyan-400"><?php echo e($totalItems); ?></strong></span>
                    <span>•</span>
                    <span>Page: <strong class="text-cyan-400"><?php echo e($filteredItems->currentPage()); ?></strong> of <strong class="text-cyan-400"><?php echo e($filteredItems->lastPage()); ?></strong></span>
                    <span>•</span>
                    <span>Showing: <strong class="text-cyan-400"><?php echo e($filteredItems->count()); ?></strong> items per page</span>
                </div>
            </div>

            <!-- Filters Section -->
            <div class="gift-filter-section p-6 mb-8">
                <div class="space-y-6">
                    <!-- Row 1: Stock Status Filters -->
                    <div class="flex flex-col lg:flex-row gap-6 items-start lg:items-center">
                        <!-- Stock Status Filter Buttons -->
                        <div class="flex flex-wrap gap-3">
                            <button @click="$wire.setFilter('all')"
                                    class="gift-filter-button px-6 py-3 rounded-xl font-medium <?php echo e($filterType === 'all' ? 'active' : ''); ?>"
                                    wire:ignore  data-translate="giftShop.filters.showAll">
                                All Items
                            </button>
                            <button @click="$wire.setFilter('available')"
                                    class="gift-filter-button px-6 py-3 rounded-xl font-medium <?php echo e($filterType === 'available' ? 'active' : ''); ?>"
                                    wire:ignore  data-translate="giftShop.filters.showAvailable">
                                In Stock
                            </button>
                            <button @click="$wire.setFilter('expiring')"
                                    class="gift-filter-button px-6 py-3 rounded-xl font-medium <?php echo e($filterType === 'expiring' ? 'active' : ''); ?>"
                                    wire:ignore  data-translate="giftShop.filters.showExpiring">
                                Expiring Soon
                            </button>
                        </div>

                        <!-- Search Input -->
                        <div class="flex-1 max-w-md">
                            <input type="text"
                                   wire:model.live="searchTerm"
                                   class="gift-search-input w-full px-4 py-3 text-white placeholder-gray-400"
                                   placeholder="Search items by name..."
                                   wire:ignore  data-translate-placeholder="giftShop.filters.searchPlaceholder">
                        </div>

                        <!-- Reset Button -->
                        <button @click="$wire.resetFilters()"
                                class="reset-button px-6 py-3 rounded-xl font-medium"
                                wire:ignore  data-translate="itemShop.filters.resetFilters">
                            Reset Filters
                        </button>
                    </div>

                    <!-- Row 2: Price and Sort Type Filters -->
                    <div class="flex flex-col lg:flex-row gap-6 items-start lg:items-center">
                        <!-- Price Range Filter -->
                        <div class="flex flex-col gap-2">
                            <label class="text-sm font-medium text-orange-200" wire:ignore  data-translate="giftShop.filters.priceRange">Price Range:</label>
                            <div class="flex flex-wrap gap-2">
                                <button @click="$wire.setPriceRange('all')"
                                        class="gift-filter-button px-4 py-2 rounded-lg text-sm <?php echo e($priceRange === 'all' ? 'active' : ''); ?>"
                                        wire:ignore  data-translate="giftShop.filters.priceRanges.all">
                                    All Prices
                                </button>
                                <button @click="$wire.setPriceRange('low')"
                                        class="gift-filter-button px-4 py-2 rounded-lg text-sm <?php echo e($priceRange === 'low' ? 'active' : ''); ?>"
                                        wire:ignore  data-translate="giftShop.filters.priceRanges.low">
                                    Low (≤ 1,000)
                                </button>
                                <button @click="$wire.setPriceRange('medium')"
                                        class="gift-filter-button px-4 py-2 rounded-lg text-sm <?php echo e($priceRange === 'medium' ? 'active' : ''); ?>"
                                        wire:ignore  data-translate="giftShop.filters.priceRanges.medium">
                                    Medium (1,001 - 10,000)
                                </button>
                                <button @click="$wire.setPriceRange('high')"
                                        class="gift-filter-button px-4 py-2 rounded-lg text-sm <?php echo e($priceRange === 'high' ? 'active' : ''); ?>"
                                        wire:ignore  data-translate="giftShop.filters.priceRanges.high">
                                    High (> 10,000)
                                </button>
                            </div>
                        </div>

                        <!-- Sort Type Filter -->
                        <div class="flex flex-col gap-2">
                            <label class="text-sm font-medium text-orange-200" wire:ignore  data-translate="giftShop.filters.sortType">Sort Type:</label>
                            <div class="flex flex-wrap gap-2">
                                <button @click="$wire.setSortType('all')"
                                        class="gift-filter-button px-4 py-2 rounded-lg text-sm <?php echo e($sortType === 'all' ? 'active' : ''); ?>"
                                        wire:ignore  data-translate="giftShop.filters.sortTypes.all">
                                    All Types
                                </button>
                                <!--[if BLOCK]><![endif]--><?php $__currentLoopData = $this->availableSortTypes; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $type): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?>
                                    <!--[if BLOCK]><![endif]--><?php if(!empty($type)): ?>
                                        <button @click="$wire.setSortType('<?php echo e($type); ?>')"
                                                class="gift-filter-button px-4 py-2 rounded-lg text-sm <?php echo e($sortType === $type ? 'active' : ''); ?>">
                                            <?php echo e(ucfirst($type)); ?>

                                        </button>
                                    <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                                <?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><!--[if ENDBLOCK]><![endif]-->
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Pagination Controls and Per Page Selector -->
            <div class="flex flex-col lg:flex-row justify-between items-center gap-4 mb-6 p-4 bg-gray-800/30 rounded-xl">
                <!-- Per Page Selector -->
                <div class="flex items-center gap-3">
                    <span class="text-purple-200 text-sm">Items per page:</span>
                    <div class="flex gap-2">
                        <!--[if BLOCK]><![endif]--><?php $__currentLoopData = [6, 12, 24, 48]; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $pageSize): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?>
                            <button @click="$wire.changePerPage(<?php echo e($pageSize); ?>)"
                                    class="px-3 py-1 rounded-lg text-sm font-medium transition-all
                                           <?php echo e($perPage === $pageSize
                                              ? 'bg-purple-600 text-white shadow-lg'
                                              : 'bg-gray-700 text-gray-300 hover:bg-gray-600'); ?>">
                                <?php echo e($pageSize); ?>

                            </button>
                        <?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><!--[if ENDBLOCK]><![endif]-->
                    </div>
                </div>

                <!-- Navigation Info -->
                <!--[if BLOCK]><![endif]--><?php if($totalItems > 0): ?>
                    <div class="text-sm text-purple-200">
                        Showing <?php echo e($filteredItems->firstItem()); ?> to <?php echo e($filteredItems->lastItem()); ?>

                        of <?php echo e($totalItems); ?> results
                    </div>
                <?php endif; ?><!--[if ENDBLOCK]><![endif]-->

                <!-- Quick Page Navigation -->
                <!--[if BLOCK]><![endif]--><?php if($filteredItems->hasPages()): ?>
                    <div class="flex items-center gap-2">
                        <!-- Previous Page -->
                        <!--[if BLOCK]><![endif]--><?php if($filteredItems->onFirstPage()): ?>
                            <span class="px-3 py-1 rounded-lg bg-gray-700 text-gray-500 cursor-not-allowed">
                                Previous
                            </span>
                        <?php else: ?>
                            <button wire:click="previousPage"
                                    class="px-3 py-1 rounded-lg bg-purple-600 text-white hover:bg-purple-700 transition-colors">
                                Previous
                            </button>
                        <?php endif; ?><!--[if ENDBLOCK]><![endif]-->

                        <!-- Page Numbers (show only a few) -->
                        <?php
                            $start = max(1, $filteredItems->currentPage() - 2);
                            $end = min($filteredItems->lastPage(), $filteredItems->currentPage() + 2);
                        ?>

                        <!--[if BLOCK]><![endif]--><?php if($start > 1): ?>
                            <button wire:click="gotoPage(1)" class="px-3 py-1 rounded-lg bg-gray-700 text-gray-300 hover:bg-gray-600">1</button>
                            <!--[if BLOCK]><![endif]--><?php if($start > 2): ?>
                                <span class="text-gray-500">...</span>
                            <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                        <?php endif; ?><!--[if ENDBLOCK]><![endif]-->

                        <!--[if BLOCK]><![endif]--><?php for($i = $start; $i <= $end; $i++): ?>
                            <button wire:click="gotoPage(<?php echo e($i); ?>)"
                                    class="px-3 py-1 rounded-lg <?php echo e($i === $filteredItems->currentPage()
                                           ? 'bg-purple-600 text-white'
                                           : 'bg-gray-700 text-gray-300 hover:bg-gray-600'); ?>">
                                <?php echo e($i); ?>

                            </button>
                        <?php endfor; ?><!--[if ENDBLOCK]><![endif]-->

                        <!--[if BLOCK]><![endif]--><?php if($end < $filteredItems->lastPage()): ?>
                            <?php if($end < $filteredItems->lastPage() - 1): ?>
                                <span class="text-gray-500">...</span>
                            <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                            <button wire:click="gotoPage(<?php echo e($filteredItems->lastPage()); ?>)"
                                    class="px-3 py-1 rounded-lg bg-gray-700 text-gray-300 hover:bg-gray-600">
                                <?php echo e($filteredItems->lastPage()); ?>

                            </button>
                        <?php endif; ?><!--[if ENDBLOCK]><![endif]-->

                        <!-- Next Page -->
                        <!--[if BLOCK]><![endif]--><?php if($filteredItems->hasMorePages()): ?>
                            <button wire:click="nextPage"
                                    class="px-3 py-1 rounded-lg bg-purple-600 text-white hover:bg-purple-700 transition-colors">
                                Next
                            </button>
                        <?php else: ?>
                            <span class="px-3 py-1 rounded-lg bg-gray-700 text-gray-500 cursor-not-allowed">
                                Next
                            </span>
                        <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                    </div>
                <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
            </div>

            <!-- Items Grid -->
            <!--[if BLOCK]><![endif]--><?php if($filteredItems && $filteredItems->count() > 0): ?>
                <div class="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-8">
                    <!--[if BLOCK]><![endif]--><?php $__currentLoopData = $filteredItems; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $item): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?>
                        <?php
                            $hasStock = $item->stock !== -1 && $item->stock > 0;
                            $hasExpiry = $item->expiry_date !== null;
                            $cardClass = '';

                            if ($hasStock && $hasExpiry) {
                                $cardClass = 'has-both';
                            } elseif ($hasStock) {
                                $cardClass = 'has-stock';
                            } elseif ($hasExpiry) {
                                $cardClass = 'has-expiry';
                            }

                            // Calculate expiry information
                            $expiryData = null;
                            if ($item->expiry_date) {
                                $expiryDate = is_string($item->expiry_date)
                                    ? \Carbon\Carbon::parse($item->expiry_date)
                                    : $item->expiry_date;
                                $daysLeft = now()->diffInDays($expiryDate, false);
                                $hoursLeft = now()->diffInHours($expiryDate, false);
                                $expiryData = [
                                    'days' => $daysLeft,
                                    'hours' => $hoursLeft,
                                    'isUrgent' => $daysLeft <= 1
                                ];
                            }
                        ?>

                        <div class="gift-item-card <?php echo e($cardClass); ?> p-6" x-data="{
                            showDetails: false,
                            quantity: 1,

                         }">

                            <!-- Stock Counter Badge -->
                            <!--[if BLOCK]><![endif]--><?php if($item->stock !== -1): ?>
                                <div class="gift-counter-badge gift-stock-counter <?php echo e($item->stock <= 10 ? 'low' : ''); ?>">
                                    <!--[if BLOCK]><![endif]--><?php if($item->stock <= 10): ?>
                                        <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.728-.833-2.5 0L4.268 18.5c-.77.833.192 2.5 1.732 2.5z"></path>
                                        </svg>
                                    <?php else: ?>
                                        <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"></path>
                                        </svg>
                                    <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                                    <?php echo e(number_format($item->stock)); ?>

                                </div>
                            <?php else: ?>
                                <div class="gift-counter-badge gift-stock-counter unlimited">
                                    <svg class="w-3 h-3" fill="currentColor" viewBox="0 0 24 24">
                                        <path d="M18.6 6.62c-1.44 0-2.8.56-3.77 1.53L12 10.66 10.48 9.14c-.64-.64-1.49-.99-2.4-.99-1.87 0-3.39 1.51-3.39 3.38S6.21 14.91 8.08 14.91c.91 0 1.76-.35 2.4-.99L12 12.41l1.52 1.51c.64.64 1.49.99 2.4.99 1.87 0 3.39-1.51 3.39-3.38s-1.52-3.37-3.39-3.37z"/>
                                    </svg>
                                    ∞
                                </div>
                            <?php endif; ?><!--[if ENDBLOCK]><![endif]-->

                            <!-- Expiry Counter Badge -->
                            <!--[if BLOCK]><![endif]--><?php if($expiryData): ?>
                                <div class="gift-counter-badge gift-expiry-counter <?php echo e($expiryData['isUrgent'] ? 'urgent' : ''); ?>">
                                    <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                    </svg>
                                    <!--[if BLOCK]><![endif]--><?php if($expiryData['days'] > 0): ?>
                                        <?php echo e(floor($expiryData['days'])); ?> <span wire:ignore  data-translate="giftShop.counter.days">Days</span>
                                    <?php else: ?>
                                        <?php echo e(floor($expiryData['hours'])); ?> <span wire:ignore  data-translate="giftShop.counter.hours">Hours</span>
                                    <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                                </div>
                            <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                            <!-- Item Image -->
                            <div class="gift-item-image-container mb-6 h-48 flex items-center justify-center relative">
                                <div class="gift-item-rarity-glow"></div>
                                <img src="<?php echo e($this->getImageUrl($item->image)); ?>"
                                     alt="<?php echo e($item->name); ?>"
                                     class="item-image max-h-full max-w-full object-contain relative z-10"
                                     onerror="this.src='<?php echo e(asset('assets/images/default-item.svg')); ?>'">
                                <div class="absolute top-3 right-3 z-20">
                                    <!-- SVG Search Plus Icon -->
                                    <svg class="w-5 h-5 text-white opacity-75 hover:opacity-100 transition-opacity" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0zM10 7v3m0 0v3m0-3h3m-3 0H7"></path>
                                    </svg>
                                </div>
                            </div>

                            <!-- Item Info -->
                            <div class="space-y-4 relative z-10">
                                <!-- Name and Type -->
                                <div>
                                    <h3 class="gift-item-name text-xl font-bold mb-2"><?php echo e($item->name); ?></h3>
                                    <span class="gift-item-type-badge"><?php echo e(ucfirst($item->item_type)); ?></span>
                                </div>

                                <!-- Description -->
                                <!--[if BLOCK]><![endif]--><?php if($item->description): ?>
                                    <p class="item-description text-sm text-gray-300"><?php echo e(Str::limit($item->description, 100)); ?></p>
                                <?php endif; ?><!--[if ENDBLOCK]><![endif]-->

                                <!-- Stock Status -->
                                <div class="flex items-center justify-between">
                                    <!--[if BLOCK]><![endif]--><?php if($item->stock === -1): ?>
                                        <span class="gift-stock-available">
                                            <!-- SVG Infinity Icon -->
                                            <svg class="w-4 h-4" fill="currentColor" viewBox="0 0 24 24">
                                                <path d="M18.6 6.62c-1.44 0-2.8.56-3.77 1.53L12 10.66 10.48 9.14c-.64-.64-1.49-.99-2.4-.99-1.87 0-3.39 1.51-3.39 3.38S6.21 14.91 8.08 14.91c.91 0 1.76-.35 2.4-.99L12 12.41l1.52 1.51c.64.64 1.49.99 2.4.99 1.87 0 3.39-1.51 3.39-3.38s-1.52-3.37-3.39-3.37z"/>
                                            </svg>
                                            <span wire:ignore  data-translate="giftShop.details.unlimited">Unlimited</span>
                                        </span>
                                    <?php elseif($item->stock > 10): ?>
                                        <span class="gift-stock-available">
                                            <!-- SVG Check Icon -->
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                                            </svg>
                                            <span wire:ignore  data-translate="giftShop.card.stockAvailable">In Stock</span>
                                        </span>
                                    <?php elseif($item->stock > 0): ?>
                                        <span class="gift-stock-low">
                                            <!-- SVG Warning Icon -->
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.728-.833-2.5 0L4.268 18.5c-.77.833.192 2.5 1.732 2.5z"></path>
                                            </svg>
                                            <span wire:ignore  data-translate="giftShop.card.stockLow">Low Stock</span>
                                        </span>
                                    <?php else: ?>
                                        <span class="gift-stock-out">
                                            <!-- SVG X Icon -->
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                            </svg>
                                            <span wire:ignore  data-translate="giftShop.card.stockOut">Out of Stock</span>
                                        </span>
                                    <?php endif; ?><!--[if ENDBLOCK]><![endif]-->

                                    <!-- Details Toggle -->
                                    <button @click="showDetails = !showDetails"
                                            class="gift-details-toggle">
                                        <span x-show="!showDetails" wire:ignore  data-translate="giftShop.card.showDetails">Show Details</span>
                                        <span x-show="showDetails" wire:ignore  data-translate="giftShop.card.hideDetails">Hide Details</span>
                                        <!-- SVG Chevron Down Icon -->
                                        <svg class="w-4 h-4 ml-1 transition-transform" :class="showDetails ? 'rotate-180' : ''" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                                        </svg>
                                    </button>
                                </div>

                                <!-- Item Details (Expandable) -->
                                <div x-show="showDetails" x-transition class="gift-item-details">
                                    <div class="space-y-2">
                                        <!--[if BLOCK]><![endif]--><?php if($item->Count): ?>
                                            <div class="gift-detail-item">
                                                <span class="gift-detail-label" wire:ignore  data-translate="giftShop.details.count">Count per Purchase:</span>
                                                <span class="gift-detail-value"><?php echo e($item->Count); ?></span>
                                            </div>
                                        <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                                        <div class="gift-detail-item">
                                            <span class="gift-detail-label" wire:ignore  data-translate="giftShop.details.balanceType">Currency Type:</span>
                                            <span class="gift-detail-value"><?php echo e(ucfirst(str_replace('_', ' ', $item->balance_type))); ?></span>
                                        </div>
                                        <!--[if BLOCK]><![endif]--><?php if($item->sort_type): ?>
                                            <div class="gift-detail-item">
                                                <span class="gift-detail-label" wire:ignore  data-translate="giftShop.details.sortType">Sort Type:</span>
                                                <span class="gift-detail-value">
                                                    <?php
                                                        $sortTypeClass = 'bg-blue-500/20 text-blue-300'; // default
                                                        if(strtolower($item->sort_type) === 'premium') {
                                                            $sortTypeClass = 'bg-orange-500/20 text-orange-300';
                                                        } elseif(strtolower($item->sort_type) === 'event') {
                                                            $sortTypeClass = 'bg-pink-500/20 text-pink-300';
                                                        }
                                                    ?>
                                                    <span class="px-2 py-1 text-xs rounded-full <?php echo e($sortTypeClass); ?>">
                                                        <?php echo e(ucfirst($item->sort_type)); ?>

                                                    </span>
                                                </span>
                                            </div>
                                        <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                                        <!--[if BLOCK]><![endif]--><?php if($item->expiry_date): ?>
                                            <div class="gift-detail-item">
                                                <span class="gift-detail-label" wire:ignore  data-translate="giftShop.details.expiryDate">Expiry Date:</span>
                                                <span class="gift-detail-value">
                                                    <!--[if BLOCK]><![endif]--><?php if(is_string($item->expiry_date)): ?>
                                                        <?php echo e(\Carbon\Carbon::parse($item->expiry_date)->format('Y-m-d H:i')); ?>

                                                    <?php else: ?>
                                                        <?php echo e($item->expiry_date->format('Y-m-d H:i')); ?>

                                                    <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                                                </span>
                                            </div>
                                        <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                                    </div>
                                </div>

                                <!-- Expiry Warning -->
                                <!--[if BLOCK]><![endif]--><?php if($item->expiry_date): ?>
                                    <?php
                                        $expiryDate = is_string($item->expiry_date)
                                            ? \Carbon\Carbon::parse($item->expiry_date)
                                            : $item->expiry_date;
                                        $isExpiring = $expiryDate->isBefore(now()->addDays(7)) && $expiryDate->isAfter(now());
                                    ?>
                                    <!--[if BLOCK]><![endif]--><?php if($isExpiring): ?>
                                        <div class="gift-expiry-warning">
                                            <!-- SVG Clock Icon -->
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                            </svg>
                                            <span wire:ignore  data-translate="giftShop.card.itemExpiring">Expires Soon</span>
                                        </div>
                                    <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                                <?php endif; ?><!--[if ENDBLOCK]><![endif]-->

                                <!-- Price Section -->
                                <div class="price-section">
                                    <div class="flex items-center justify-between mb-4">
                                        <div>
                                            <p class="text-sm text-gray-400 mb-1" wire:ignore  data-translate="giftShop.card.price">Price:</p>
                                            <p class="gift-price-display">
                                                <!--[if BLOCK]><![endif]--><?php if($item->balance_type === 'gift_balance'): ?>
                                                    <div class="relative inline-block mr-1">
                                                  <svg class="balance-icon-svg" width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><defs><linearGradient id="starGradient" x1="0%" y1="0%" x2="100%" y2="100%"><stop offset="0%" style="stop-color:#ffcc02"/><stop offset="50%" style="stop-color:#ff9800"/><stop offset="100%" style="stop-color:#f57c00"/></linearGradient><filter id="starGlow"><feGaussianBlur stdDeviation="2" result="coloredBlur"/><feMerge><feMergeNode in="coloredBlur"/><feMergeNode in="SourceGraphic"/></feMerge></filter></defs><polygon points="12,2 15.09,8.26 22,9.27 17,14.14 18.18,21.02 12,17.77 5.82,21.02 7,14.14 2,9.27 8.91,8.26 12,2" fill="url(#starGradient)" stroke="#f57c00" stroke-width="1" filter="url(#starGlow)"/><circle cx="12" cy="12" r="2" fill="#fff"/><path d="M10 12l1-1 1 1-1 1-1-1z" fill="#f57c00"/></svg>
                                                </div>
                                                <?php elseif($item->balance_type === 'silk'): ?>
                                                   <div class="relative inline-block mr-1">
                                                  <svg class="balance-icon-svg" width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><defs><linearGradient id="silkGradient" x1="0%" y1="0%" x2="100%" y2="100%"><stop offset="0%" style="stop-color:#fff9c4"/><stop offset="50%" style="stop-color:#ffeb3b"/><stop offset="100%" style="stop-color:#f57f17"/></linearGradient><filter id="sparkle"><feGaussianBlur stdDeviation="1" result="coloredBlur"/><feMerge><feMergeNode in="coloredBlur"/><feMergeNode in="SourceGraphic"/></feMerge></filter></defs><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z" fill="url(#silkGradient)" stroke="#fbc02d" stroke-width="1.5" filter="url(#sparkle)"/><circle cx="8" cy="6" r="1.5" fill="#fbc02d" opacity="0.8"/><circle cx="16" cy="6" r="1.5" fill="#fbc02d" opacity="0.8"/><circle cx="6" cy="18" r="1" fill="#fbc02d" opacity="0.6"/><circle cx="18" cy="18" r="1" fill="#fbc02d" opacity="0.6"/><path d="M10 12l1-1 1 1-1 1-1-1z" fill="#f57f17"/><circle cx="12" cy="12" r="0.5" fill="#fff"/></svg>
                                                </div>
                                                <?php elseif($item->balance_type === 'donate'): ?>
                                                   <div class="relative inline-block mr-1">
                                                  <svg class="balance-icon-svg mr-1" width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><defs><linearGradient id="heartGradient" x1="0%" y1="0%" x2="100%" y2="100%"><stop offset="0%" style="stop-color:#f8bbd9"/><stop offset="50%" style="stop-color:#e91e63"/><stop offset="100%" style="stop-color:#ad1457"/></linearGradient></defs><path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z" fill="url(#heartGradient)" stroke="#c2185b" stroke-width="1.5"/><path d="M12 7v6M9 10h6" stroke="#fff" stroke-width="2" stroke-linecap="round"/><text x="12" y="16" text-anchor="middle" fill="#fff" font-size="6" font-weight="bold">4T</text><circle cx="8" cy="6" r="1" fill="#fff" opacity="0.7"/><circle cx="16" cy="6" r="1" fill="#fff" opacity="0.7"/></svg>
                                                </div>
                                                <?php else: ?>
                                                    <!-- Default Diamond for other currencies -->
                                                    <svg class="w-4 h-4 inline-block mr-1 text-yellow-400" fill="currentColor" viewBox="0 0 24 24">
                                                        <path d="M6,2L18,2L22,8L12,22L2,8L6,2M12,4.5L16.75,4.5L19.75,8.5L12,18.5L4.25,8.5L7.25,4.5L12,4.5Z"/>
                                                    </svg>
                                                <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                                                <?php echo e(number_format($item->price, 2)); ?>

                                            </p>
                                        </div>

                                        <!-- Quantity Controls -->
                                        <div class="gift-quantity-controls">
                                            <button @click="quantity = Math.max(1, quantity - 1)"
                                                    class="gift-quantity-btn"
                                                    :disabled="quantity <= 1">
                                                <!-- SVG Minus Icon -->
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4"></path>
                                                </svg>
                                            </button>
                                            <input type="number"
                                                   x-model.number="quantity"
                                                   class="gift-quantity-input"
                                                   min="1" max="999">
                                            <button @click="quantity = Math.min(999, quantity + 1)"
                                                    class="gift-quantity-btn"
                                                    :disabled="quantity >= 999">
                                                <!-- SVG Plus Icon -->
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                                                </svg>
                                            </button>
                                        </div>
                                    </div>

                                    <!-- Total Price -->
                                    <div class="text-center mb-4">
                                        <p class="text-sm text-gray-400 mb-1" wire:ignore  data-translate="giftShop.card.totalPrice">Total Price:</p>
                                        <b class="gift-total-price flex items-center justify-center">
                                            <!--[if BLOCK]><![endif]--><?php if($item->balance_type === 'gift_balance'): ?>
                                               <div class="relative inline-block mr-1">
                                                  <svg class="balance-icon-svg" width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><defs><linearGradient id="starGradient" x1="0%" y1="0%" x2="100%" y2="100%"><stop offset="0%" style="stop-color:#ffcc02"/><stop offset="50%" style="stop-color:#ff9800"/><stop offset="100%" style="stop-color:#f57c00"/></linearGradient><filter id="starGlow"><feGaussianBlur stdDeviation="2" result="coloredBlur"/><feMerge><feMergeNode in="coloredBlur"/><feMergeNode in="SourceGraphic"/></feMerge></filter></defs><polygon points="12,2 15.09,8.26 22,9.27 17,14.14 18.18,21.02 12,17.77 5.82,21.02 7,14.14 2,9.27 8.91,8.26 12,2" fill="url(#starGradient)" stroke="#f57c00" stroke-width="1" filter="url(#starGlow)"/><circle cx="12" cy="12" r="2" fill="#fff"/><path d="M10 12l1-1 1 1-1 1-1-1z" fill="#f57c00"/></svg>
                                                </div>
                                            <?php elseif($item->balance_type === 'silk'): ?>
                                                <div class="relative inline-block mr-1">
                                                  <svg class="balance-icon-svg" width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><defs><linearGradient id="silkGradient" x1="0%" y1="0%" x2="100%" y2="100%"><stop offset="0%" style="stop-color:#fff9c4"/><stop offset="50%" style="stop-color:#ffeb3b"/><stop offset="100%" style="stop-color:#f57f17"/></linearGradient><filter id="sparkle"><feGaussianBlur stdDeviation="1" result="coloredBlur"/><feMerge><feMergeNode in="coloredBlur"/><feMergeNode in="SourceGraphic"/></feMerge></filter></defs><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z" fill="url(#silkGradient)" stroke="#fbc02d" stroke-width="1.5" filter="url(#sparkle)"/><circle cx="8" cy="6" r="1.5" fill="#fbc02d" opacity="0.8"/><circle cx="16" cy="6" r="1.5" fill="#fbc02d" opacity="0.8"/><circle cx="6" cy="18" r="1" fill="#fbc02d" opacity="0.6"/><circle cx="18" cy="18" r="1" fill="#fbc02d" opacity="0.6"/><path d="M10 12l1-1 1 1-1 1-1-1z" fill="#f57f17"/><circle cx="12" cy="12" r="0.5" fill="#fff"/></svg>
                                                </div>
                                            <?php elseif($item->balance_type === 'donate'): ?>
                                                <div class="relative inline-block mr-1">
                                                  <svg class="balance-icon-svg mr-1" width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><defs><linearGradient id="heartGradient" x1="0%" y1="0%" x2="100%" y2="100%"><stop offset="0%" style="stop-color:#f8bbd9"/><stop offset="50%" style="stop-color:#e91e63"/><stop offset="100%" style="stop-color:#ad1457"/></linearGradient></defs><path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z" fill="url(#heartGradient)" stroke="#c2185b" stroke-width="1.5"/><path d="M12 7v6M9 10h6" stroke="#fff" stroke-width="2" stroke-linecap="round"/><text x="12" y="16" text-anchor="middle" fill="#fff" font-size="6" font-weight="bold">4T</text><circle cx="8" cy="6" r="1" fill="#fff" opacity="0.7"/><circle cx="16" cy="6" r="1" fill="#fff" opacity="0.7"/></svg>
                                                </div>

                                            <?php else: ?>
                                                <!-- Default Diamond for other currencies -->
                                                <svg class="w-4 h-4 mr-1 text-yellow-400" fill="currentColor" viewBox="0 0 24 24">
                                                    <path d="M6,2L18,2L22,8L12,22L2,8L6,2M12,4.5L16.75,4.5L19.75,8.5L12,18.5L4.25,8.5L7.25,4.5L12,4.5Z"/>
                                                </svg>
                                            <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                                            <span x-text="(<?php echo e($item->price); ?> * quantity).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',')">
                                                <?php echo e(number_format($item->price, 2)); ?>

                                            </span>
                                        </b>
                                    </div>

                                    <!-- Buy Button -->
                                    <?php
                                        $isExpired = false;
                                        if ($item->expiry_date) {
                                            $expiryDate = is_string($item->expiry_date)
                                                ? \Carbon\Carbon::parse($item->expiry_date)
                                                : $item->expiry_date;
                                            $isExpired = $expiryDate->isPast();
                                        }
                                    ?>
                                    <button @click="$wire.buyItem(<?php echo e($item->id); ?>, quantity)"
                                            class="gift-buy-button w-full"
                                            :disabled="isLoading || <?php echo e($item->stock === 0 ? 'true' : 'false'); ?> || <?php echo e($isExpired ? 'true' : 'false'); ?>">
                                        <span x-show="!isLoading">
                                            <!--[if BLOCK]><![endif]--><?php if($item->stock === 0): ?>
                                                <span wire:ignore  data-translate="giftShop.card.outOfStock">Out of Stock</span>
                                            <?php elseif($isExpired): ?>
                                                <span wire:ignore  data-translate="giftShop.card.expired">Expired</span>
                                            <?php else: ?>
                                                <!-- SVG Shopping Cart Icon -->
                                                <svg class="w-4 h-4 mr-2 inline-block" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4m0 0L7 13m0 0l-2.5-5M7 13l-2.5 5M17 17a2 2 0 100 4 2 2 0 000-4zM9 17a2 2 0 100 4 2 2 0 000-4z"></path>
                                                </svg>
                                                <span wire:ignore  data-translate="giftShop.card.buyNow">Buy Now</span>
                                            <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
                                        </span>
                                        <span x-show="isLoading">
                                            <!-- SVG Spinner Icon -->
                                            <svg class="w-4 h-4 mr-2 inline-block animate-spin" fill="none" viewBox="0 0 24 24">
                                                <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                                                <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                            </svg>
                                            Processing...
                                        </span>
                                    </button>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?><!--[if ENDBLOCK]><![endif]-->
                </div>

                <!-- Bottom Pagination -->
                <!--[if BLOCK]><![endif]--><?php if($filteredItems->hasPages()): ?>
                    <div class="bottom-pagination">
                        <div class="pagination-wrapper">
                            <?php echo e($filteredItems->links('pagination.custom')); ?>

                        </div>
                    </div>
                <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
            <?php else: ?>
                <!-- No Items State -->
                <div class="text-center py-20">
                    <div class="w-24 h-24 mx-auto mb-6 bg-gradient-to-r from-orange-500 to-pink-600 rounded-3xl flex items-center justify-center shadow-lg shadow-pink-500/25">
                        <!-- SVG Gift Box Icon -->
                        <svg class="w-12 h-12 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v13m0-13V6a2 2 0 112 2h-2zm0 0V5.5A2.5 2.5 0 109.5 8H12zm-7 4h14M5 12a2 2 0 110-4h14a2 2 0 110 4M5 12v7a2 2 0 002 2h10a2 2 0 002-2v-7"></path>
                        </svg>
                    </div>
                    <h3 class="text-2xl font-bold text-white mb-4" wire:ignore  data-translate="giftShop.status.noItems">No items found</h3>
                    <p class="text-gray-400 mb-8" wire:ignore  data-translate="giftShop.status.noItemsDesc">Try adjusting your search or filter criteria</p>
                    <button @click="$wire.resetFilters()"
                            class="gift-reset-button px-8 py-3"
                            wire:ignore  data-translate="giftShop.filters.resetFilters">
                        Reset Filters
                    </button>
                </div>
            <?php endif; ?><!--[if ENDBLOCK]><![endif]-->
        </div>


    </div>
</div>
<?php /**PATH C:\laragon\www\4TresSro-PVE\resources\views/livewire/backend/pages/gift-item-shop.blade.php ENDPATH**/ ?>