<?php

use Livewire\Volt\Component;
use Livewire\WithFileUploads;
use App\Models\FastEventsWelcome;
use App\Models\SRO_VT_SHARD\Char;
use App\Models\SRO_VT_PROXY\_Players;
use App\Models\SRO_VT_SHARD\_User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

new class extends Component {
    use WithFileUploads;

    public $selectedPlayerType = null;
    public $selectedCharacter = null;
    public $oldCharacterName = '';
    public $oldCharacterProofs = [];
    public $donationAmount = 0;
    public $selectedCurrency = 'USD';
    public $donationProofs = [];
    public $oldCharacterImages = [];
    public $calculatedBalance = 0;
    public $userApplication = null;
    public $userCharacters = [];
    public $isSubmitting = false;
    public $isClaiming = false;

    public function mount()
    {
        $this->loadUserData();
    }

    public function loadUserData()
    {
        $user = Auth::user();

        // Check if user already has an application
        $this->userApplication = FastEventsWelcome::where('JID', $user->JID)->first();

        // Load user characters: Get CharIDs from _User table first, then get character details
        $userCharIds = _User::where('UserJID', $user->JID)->pluck('CharID');
        $this->userCharacters = Char::whereIn('CharID', $userCharIds)
            ->select('CharName16', 'CharID', 'RefObjID', 'CurLevel')
            ->orderBy('CharName16')
            ->get()
            ->map(function($char) {
                return [
                    'CharID' => $char->CharID,
                    'CharName16' => $char->CharName16,
                    'RefObjID' => $char->RefObjID,
                    'CurLevel' => $char->CurLevel,
                    'avatar' => Char::getCharAvatarGifByCharname($char->CharName16)
                ];
            });
    }

    public function selectPlayerType($type)
    {
        $this->selectedPlayerType = $type;
        $this->calculateReward();
    }

    public function selectCharacter($characterId)
    {
        // Check if this character belongs to the user by checking through _User table
        $userCharIds = _User::where('UserJID', Auth::user()->JID)->pluck('CharID');

        if ($userCharIds->contains($characterId)) {
            $this->selectedCharacter = $characterId;
        }
    }

    public function updatedDonationAmount()
    {
        $this->calculateReward();
    }

    public function updatedSelectedCurrency()
    {
        $this->calculateReward();
    }

    public function calculateReward()
    {
        if ($this->selectedPlayerType === 'donate' && $this->donationAmount > 0) {
            $rate = $this->selectedCurrency === 'USD' ? 40 : 1;
            $this->calculatedBalance = floor(($this->donationAmount * $rate) * 0.5);
        } else {
            $this->calculatedBalance = 0;
        }
    }

    public function submitApplication()
    {
        $this->isSubmitting = true;

        try {
            // Validate based on player type
            $rules = [
                'selectedPlayerType' => 'required|in:new,old,donate',
                'selectedCharacter' => 'required|exists:SRO_VT_SHARD.dbo.Char,CharID',
            ];

            if ($this->selectedPlayerType === 'old') {
                $rules['oldCharacterName'] = 'required|string|max:255';
                $rules['oldCharacterProofs'] = 'required|array|min:1';
                $rules['oldCharacterProofs.*'] = 'image|mimes:jpeg,png,jpg,gif|max:5120';
            }

            if ($this->selectedPlayerType === 'donate') {
                $rules['donationAmount'] = 'required|numeric|min:1';
                $rules['selectedCurrency'] = 'required|in:USD,EGP';
                $rules['donationProofs'] = 'required|array|min:1';
                $rules['donationProofs.*'] = 'image|mimes:jpeg,png,jpg,gif|max:5120';
            }

            $this->validate($rules);

            // Check character ownership
            $character = Char::where('CharID', $this->selectedCharacter)
                ->where('UserJID', Auth::user()->JID)
                ->first();

            if (!$character) {
                session()->flash('error', __('welcomeEvent.messages.characterNotFound'));
                return;
            }

            // Check if character is online (optional requirement)
            $player = _Players::where('CharID', $this->selectedCharacter)->first();
            if (!$player) {
                session()->flash('error', __('welcomeEvent.messages.characterOffline'));
                return;
            }

            // Store uploaded files
            $oldCharacterProofPaths = [];
            $donationProofPaths = [];
            $oldCharacterImagePaths = [];

            if ($this->selectedPlayerType === 'old') {
                foreach ($this->oldCharacterProofs as $file) {
                    $path = $file->store('welcome-event/old-character-proofs', 'public');
                    $oldCharacterProofPaths[] = $path;
                }
            }

            if ($this->selectedPlayerType === 'donate') {
                foreach ($this->donationProofs as $file) {
                    $path = $file->store('welcome-event/donation-proofs', 'public');
                    $donationProofPaths[] = $path;
                }
            }

            foreach ($this->oldCharacterImages as $file) {
                $path = $file->store('welcome-event/old-character-images', 'public');
                $oldCharacterImagePaths[] = $path;
            }

            // Calculate reward amount
            $rewardAmount = $this->getRewardAmount();

            // Create application
            FastEventsWelcome::create([
                'JID' => Auth::user()->JID,
                'HWID' => Auth::user()->latest_hwid ?? '',
                'player_type' => $this->selectedPlayerType,
                'CharID' => $this->selectedCharacter,
                'old_character_name' => $this->oldCharacterName,
                'donation_amount' => $this->selectedPlayerType === 'donate' ? $this->donationAmount : null,
                'currency' => $this->selectedPlayerType === 'donate' ? $this->selectedCurrency : null,
                'reward_amount' => $rewardAmount,
                'old_character_proofs' => json_encode($oldCharacterProofPaths),
                'donation_proofs' => json_encode($donationProofPaths),
                'old_character_images' => json_encode($oldCharacterImagePaths),
                'status' => 'pending',
                'submitted_at' => now(),
            ]);

            session()->flash('success', __('welcomeEvent.messages.applicationSubmitted'));
            $this->loadUserData();

        } catch (\Exception $e) {
            session()->flash('error', __('welcomeEvent.messages.applicationFailed'));
        } finally {
            $this->isSubmitting = false;
        }
    }

    public function claimReward()
    {
        $this->isClaiming = true;

        try {
            if (!$this->userApplication || !$this->userApplication->canBeClaimed()) {
                session()->flash('error', __('welcomeEvent.messages.claimFailed'));
                return;
            }

            // Update user balance
            $user = _User::where('JID', Auth::user()->JID)->first();
            if ($user) {
                $user->increment('GMoney', $this->userApplication->reward_amount);
            }

            // Update application status
            $this->userApplication->update([
                'status' => 'claimed',
                'claimed_at' => now(),
            ]);

            session()->flash('success', __('welcomeEvent.messages.rewardClaimed'));
            $this->loadUserData();

        } catch (\Exception $e) {
            session()->flash('error', __('welcomeEvent.messages.claimFailed'));
        } finally {
            $this->isClaiming = false;
        }
    }

    private function getRewardAmount()
    {
        switch ($this->selectedPlayerType) {
            case 'new':
                return 10;
            case 'old':
                return 40;
            case 'donate':
                return $this->calculatedBalance;
            default:
                return 0;
        }
    }

    public function resetApplication()
    {
        $this->selectedPlayerType = null;
        $this->selectedCharacter = null;
        $this->oldCharacterName = '';
        $this->oldCharacterProofs = [];
        $this->donationAmount = 0;
        $this->selectedCurrency = 'USD';
        $this->donationProofs = [];
        $this->oldCharacterImages = [];
        $this->calculatedBalance = 0;
    }

    public function with()
    {
        return [
            'hasApplication' => !is_null($this->userApplication),
        ];
    }
}; ?>

<script>
function welcomeEventData() {
    return {
        selectedPlayer: @entangle('selectedPlayerType'),
        selectedCharacter: @entangle('selectedCharacter'),
        donationAmount: @entangle('donationAmount'),
        currency: @entangle('currency'),
        oldCharacterName: @entangle('oldCharacterName'),

        // Helper methods
        selectPlayerType(type) {
            this.selectedPlayer = type;
        },

        selectCharacterFromList(charId) {
            this.selectedCharacter = charId;
            $wire.selectCharacter(charId);
        }
    }
}

function fileUpload(fieldName) {
    return {
        handleDrop(event) {
            const files = Array.from(event.dataTransfer.files);
            // File handling would need to be implemented through Livewire
            console.log('Files dropped:', files);
        }
    }
}

function countdown(deadline) {
    return {
        timeRemaining: '',
        interval: null,

        init() {
            this.updateCountdown();
            this.interval = setInterval(() => {
                this.updateCountdown();
            }, 1000);
        },

        destroy() {
            if (this.interval) {
                clearInterval(this.interval);
            }
        },

        updateCountdown() {
            const now = new Date().getTime();
            const deadlineTime = new Date(deadline).getTime();
            const distance = deadlineTime - now;

            if (distance > 0) {
                const hours = Math.floor(distance / (1000 * 60 * 60));
                const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
                const seconds = Math.floor((distance % (1000 * 60)) / 1000);

                this.timeRemaining = `${hours}h ${minutes}m ${seconds}s`;
            } else {
                this.timeRemaining = 'Review period expired';
                this.destroy();
            }
        }
    }
}
</script>

<div class="welcome-event-container" x-data="welcomeEventData()">
    {{-- Header Section --}}
    <div class="welcome-event-header">
        <h1 class="welcome-title">{{ __('welcomeEvent.title') }}</h1>
        <p class="welcome-description">{{ __('welcomeEvent.description') }}</p>
    </div>

    {{-- Flash Messages --}}
    @if (session('success'))
        <div class="flash-message success" x-data="{ show: true }" x-show="show" x-transition>
            <span>{{ session('success') }}</span>
            <button @click="show = false" class="close-btn">&times;</button>
        </div>
    @endif

    @if (session('error'))
        <div class="flash-message error" x-data="{ show: true }" x-show="show" x-transition>
            <span>{{ session('error') }}</span>
            <button @click="show = false" class="close-btn">&times;</button>
        </div>
    @endif

    {{-- Main Content --}}
    @if (!$hasApplication)
        {{-- Player Type Selection --}}
        @if (!$selectedPlayerType)
            <div class="player-type-selection">
                <h2 class="section-title">{{ __('welcomeEvent.selectPlayerType') }}</h2>
                <div class="player-types-grid">
                    {{-- New Player Card --}}
                    <div class="player-type-card new-player"
                         @click="$wire.selectPlayerType('new')"
                         x-transition:enter="card-enter">
                        <div class="card-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12 2L13.09 8.26L18 9L13.09 9.74L12 16L10.91 9.74L6 9L10.91 8.26L12 2Z"/>
                            </svg>
                        </div>
                        <h3 class="card-title">{{ __('welcomeEvent.newPlayer.title') }}</h3>
                        <p class="card-subtitle">{{ __('welcomeEvent.newPlayer.subtitle') }}</p>
                        <div class="card-reward">{{ __('welcomeEvent.newPlayer.reward') }}</div>
                        <p class="card-description">{{ __('welcomeEvent.newPlayer.description') }}</p>
                        <button class="select-type-btn">{{ __('welcomeEvent.buttons.selectType') }}</button>
                    </div>

                    {{-- Old Player Card --}}
                    <div class="player-type-card old-player"
                         @click="$wire.selectPlayerType('old')"
                         x-transition:enter="card-enter"
                         x-transition:enter.delay.100ms>
                        <div class="card-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12 2C13.1 2 14 2.9 14 4C14 5.1 13.1 6 12 6C10.9 6 10 5.1 10 4C10 2.9 10.9 2 12 2ZM21 9V7L15 1H5C3.89 1 3 1.89 3 3V21C3 22.11 3.89 23 5 23H19C20.11 23 21 22.11 21 21V9H21ZM17 7V3.5L20.5 7H17Z"/>
                            </svg>
                        </div>
                        <h3 class="card-title">{{ __('welcomeEvent.oldPlayer.title') }}</h3>
                        <p class="card-subtitle">{{ __('welcomeEvent.oldPlayer.subtitle') }}</p>
                        <div class="card-reward">{{ __('welcomeEvent.oldPlayer.reward') }}</div>
                        <p class="card-description">{{ __('welcomeEvent.oldPlayer.description') }}</p>
                        <button class="select-type-btn">{{ __('welcomeEvent.buttons.selectType') }}</button>
                    </div>

                    {{-- donate Player Card --}}
                    <div class="player-type-card donate-player"
                         @click="$wire.selectPlayerType('donate')"
                         x-transition:enter="card-enter"
                         x-transition:enter.delay.200ms>
                        <div class="card-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12,2A3,3 0 0,1 15,5V11A3,3 0 0,1 12,14A3,3 0 0,1 9,11V5A3,3 0 0,1 12,2M19,11C19,14.53 16.39,17.44 13,17.93V21H11V17.93C7.61,17.44 5,14.53 5,11H7A5,5 0 0,0 12,16A5,5 0 0,0 17,11H19Z"/>
                            </svg>
                        </div>
                        <h3 class="card-title">{{ __('welcomeEvent.donatePlayer.title') }}</h3>
                        <p class="card-subtitle">{{ __('welcomeEvent.donatePlayer.subtitle') }}</p>
                        <div class="card-reward">{{ __('welcomeEvent.donatePlayer.reward') }}</div>
                        <p class="card-description">{{ __('welcomeEvent.donatePlayer.description') }}</p>
                        <button class="select-type-btn">{{ __('welcomeEvent.buttons.selectType') }}</button>
                    </div>
                </div>
            </div>
        @else
            {{-- Application Form --}}
            <div class="application-form" x-transition:enter="form-enter">
                <h2 class="section-title">
                    {{ __('welcomeEvent.' . $selectedPlayerType . 'Player.title') }} {{ __('welcomeEvent.forms.characterSelection') }}
                </h2>

                {{-- Character Selection --}}
                <div class="form-section">
                    <label class="form-label">{{ __('welcomeEvent.forms.selectCharacter') }}</label>
                    <p class="form-help">{{ __('welcomeEvent.forms.selectCharacterDesc') }}</p>

                    <div class="character-selection">
                        @forelse($userCharacters as $character)
                            <div class="character-option @if($selectedCharacter == $character->CharID) selected @endif"
                                 @click="selectCharacterFromList({{ $character->CharID }})">
                                <div class="character-avatar">
                                    <img src="{{ $character['avatar'] }}" alt="{{ $character['name'] }}" />
                                </div>
                                <div class="character-info">
                                    <div class="character-name">{{ $character['name'] }}</div>
                                    <div class="character-level">Level {{ $character['level'] }}</div>
                                    <div class="character-id">ID: {{ $character['id'] }}</div>
                                </div>
                            </div>
                        @empty
                            <div class="no-characters">
                                <p>No characters found. Please create a character first.</p>
                            </div>
                        @endforelse
                    </div>
                </div>

                {{-- Old Player Specific Fields --}}
                @if($selectedPlayerType === 'old')
                    <div class="form-section">
                        <label class="form-label">{{ __('welcomeEvent.forms.oldCharacterName') }}</label>
                        <input type="text"
                               wire:model="oldCharacterName"
                               class="form-input"
                               placeholder="{{ __('welcomeEvent.forms.oldCharacterNamePlaceholder') }}">
                    </div>

                    <div class="form-section">
                        <label class="form-label">{{ __('welcomeEvent.forms.oldCharacterProof') }}</label>
                        <p class="form-help">{{ __('welcomeEvent.forms.oldCharacterProofDesc') }}</p>

                        <div class="file-upload-container" x-data="fileUpload('oldCharacterProofs')">
                            <div class="file-upload-area"
                                 @drop.prevent="handleDrop($event)"
                                 @dragover.prevent
                                 @click="$refs.fileInput.click()">
                                <div class="upload-icon">📁</div>
                                <p>{{ __('welcomeEvent.forms.dragAndDrop') }}</p>
                                <p class="upload-help">{{ __('welcomeEvent.forms.supportedFormats') }}</p>
                            </div>
                            <input type="file"
                                   x-ref="fileInput"
                                   wire:model="oldCharacterProofs"
                                   multiple
                                   accept="image/*"
                                   style="display: none;">
                        </div>

                        @if($oldCharacterProofs)
                            <div class="uploaded-files">
                                @foreach($oldCharacterProofs as $file)
                                    <div class="uploaded-file">
                                        <span>{{ $file->getClientOriginalName() }}</span>
                                        <span class="file-size">({{ number_format($file->getSize() / 1024, 1) }}KB)</span>
                                    </div>
                                @endforeach
                            </div>
                        @endif
                    </div>
                @endif

                {{-- donate Player Specific Fields --}}
                @if($selectedPlayerType === 'donate')
                    <div class="form-section">
                        <label class="form-label">{{ __('welcomeEvent.forms.donationAmount') }}</label>
                        <div class="amount-input-group">
                            <input type="number"
                                   wire:model.live="donationAmount"
                                   class="form-input amount-input"
                                   placeholder="{{ __('welcomeEvent.forms.donationAmountPlaceholder') }}"
                                   min="1"
                                   step="0.01">

                            <select wire:model.live="selectedCurrency" class="currency-select">
                                <option value="USD">USD</option>
                                <option value="EGP">EGP</option>
                            </select>
                        </div>

                        <div class="currency-rates">
                            <span class="rate usd">{{ __('welcomeEvent.forms.usdRate') }}</span>
                            <span class="rate egp">{{ __('welcomeEvent.forms.egpRate') }}</span>
                        </div>

                        @if($calculatedBalance > 0)
                            <div class="calculated-reward">
                                <h4>{{ __('welcomeEvent.forms.calculatedReward') }}</h4>
                                <div class="balance-display">{{ $calculatedBalance }} Balance</div>
                                <p class="balance-breakdown">
                                    {{ __('welcomeEvent.forms.balanceBreakdown', [
                                        'amount' => $donationAmount,
                                        'currency' => $selectedCurrency,
                                        'balance' => $calculatedBalance
                                    ]) }}
                                </p>
                            </div>
                        @endif
                    </div>

                    <div class="form-section">
                        <label class="form-label">{{ __('welcomeEvent.forms.donationProofs') }}</label>
                        <p class="form-help">{{ __('welcomeEvent.forms.donationProofsDesc') }}</p>

                        <div class="file-upload-container" x-data="fileUpload('donationProofs')">
                            <div class="file-upload-area"
                                 @drop.prevent="handleDrop($event)"
                                 @dragover.prevent
                                 @click="$refs.fileInput.click()">
                                <div class="upload-icon">💰</div>
                                <p>{{ __('welcomeEvent.forms.dragAndDrop') }}</p>
                                <p class="upload-help">{{ __('welcomeEvent.forms.supportedFormats') }}</p>
                            </div>
                            <input type="file"
                                   x-ref="fileInput"
                                   wire:model="donationProofs"
                                   multiple
                                   accept="image/*"
                                   style="display: none;">
                        </div>

                        @if($donationProofs)
                            <div class="uploaded-files">
                                @foreach($donationProofs as $file)
                                    <div class="uploaded-file">
                                        <span>{{ $file->getClientOriginalName() }}</span>
                                        <span class="file-size">({{ number_format($file->getSize() / 1024, 1) }}KB)</span>
                                    </div>
                                @endforeach
                            </div>
                        @endif
                    </div>
                @endif

                {{-- Optional Old Character Images --}}
                <div class="form-section">
                    <label class="form-label">{{ __('welcomeEvent.forms.oldCharacterImages') }} (Optional)</label>
                    <p class="form-help">{{ __('welcomeEvent.forms.oldCharacterImagesDesc') }}</p>

                    <div class="file-upload-container" x-data="fileUpload('oldCharacterImages')">
                        <div class="file-upload-area"
                             @drop.prevent="handleDrop($event)"
                             @dragover.prevent
                             @click="$refs.fileInput.click()">
                            <div class="upload-icon">🖼️</div>
                            <p>{{ __('welcomeEvent.forms.dragAndDrop') }}</p>
                            <p class="upload-help">{{ __('welcomeEvent.forms.supportedFormats') }}</p>
                        </div>
                        <input type="file"
                               x-ref="fileInput"
                               wire:model="oldCharacterImages"
                               multiple
                               accept="image/*"
                               style="display: none;">
                    </div>

                    @if($oldCharacterImages)
                        <div class="uploaded-files">
                            @foreach($oldCharacterImages as $file)
                                <div class="uploaded-file">
                                    <span>{{ $file->getClientOriginalName() }}</span>
                                    <span class="file-size">({{ number_format($file->getSize() / 1024, 1) }}KB)</span>
                                </div>
                            @endforeach
                        </div>
                    @endif
                </div>

                {{-- Submit Button --}}
                <div class="form-actions">
                    <button wire:click="resetApplication" class="btn btn-secondary">
                        {{ __('Back') }}
                    </button>
                    <button wire:click="submitApplication"
                            class="btn btn-primary submit-btn"
                            :disabled="$wire.isSubmitting">
                        <span x-show="!$wire.isSubmitting">{{ __('welcomeEvent.forms.submitApplication') }}</span>
                        <span x-show="$wire.isSubmitting">{{ __('welcomeEvent.forms.submitting') }}</span>
                    </button>
                </div>
            </div>
        @endif
    @else
        {{-- Application Status Display --}}
        <div class="application-status" x-transition:enter="status-enter">
            @if($userApplication->status === 'pending')
                <div class="status-card pending">
                    <div class="status-icon">⏳</div>
                    <h2>{{ __('welcomeEvent.status.pending.title') }}</h2>
                    <p>{{ __('welcomeEvent.status.pending.description') }}</p>

                    <div class="time-remaining" x-data="countdown('{{ $userApplication->getReviewDeadline() }}')">
                        <span>{{ __('welcomeEvent.status.pending.timeRemaining') }}:</span>
                        <div class="countdown-display" x-text="timeRemaining"></div>
                    </div>
                </div>

            @elseif($userApplication->status === 'approved')
                <div class="status-card approved">
                    <div class="status-icon">✅</div>
                    <h2>{{ __('welcomeEvent.status.approved.title') }}</h2>
                    <p>{{ __('welcomeEvent.status.approved.description') }}</p>

                    <div class="reward-info">
                        <div class="reward-amount">{{ $userApplication->reward_amount }} Balance</div>
                    </div>

                    <button wire:click="claimReward"
                            class="btn btn-success claim-btn"
                            :disabled="$wire.isClaiming">
                        <span x-show="!$wire.isClaiming">{{ __('welcomeEvent.status.approved.claimReward') }}</span>
                        <span x-show="$wire.isClaiming">{{ __('Claiming...') }}</span>
                    </button>
                </div>

            @elseif($userApplication->status === 'rejected')
                <div class="status-card rejected">
                    <div class="status-icon">❌</div>
                    <h2>{{ __('welcomeEvent.status.rejected.title') }}</h2>
                    <p>{{ __('welcomeEvent.status.rejected.description') }}</p>

                    @if($userApplication->rejection_reason)
                        <div class="rejection-reason">
                            <strong>{{ __('welcomeEvent.status.rejected.reason') }}:</strong>
                            <p>{{ $userApplication->rejection_reason }}</p>
                        </div>
                    @endif
                </div>

            @elseif($userApplication->status === 'claimed')
                <div class="status-card claimed">
                    <div class="status-icon">🎉</div>
                    <h2>{{ __('welcomeEvent.status.claimed.title') }}</h2>
                    <p>{{ __('welcomeEvent.status.claimed.description') }}</p>

                    <div class="claim-info">
                        <div class="claimed-amount">{{ $userApplication->reward_amount }} Balance</div>
                        <p>{{ __('welcomeEvent.status.claimed.claimedAt') }}: {{ $userApplication->claimed_at->format('Y-m-d H:i:s') }}</p>
                    </div>
                </div>
            @endif

            {{-- Application Details --}}
            <div class="application-details">
                <h3>{{ __('welcomeEvent.details.playerType') }}</h3>
                <div class="details-grid">
                    <div class="detail-item">
                        <span class="label">{{ __('welcomeEvent.details.playerType') }}:</span>
                        <span class="value">{{ __('welcomeEvent.' . $userApplication->player_type . 'Player.title') }}</span>
                    </div>
                    <div class="detail-item">
                        <span class="label">{{ __('welcomeEvent.details.characterName') }}:</span>
                        <span class="value">{{ $userApplication->character->CharName16 ?? 'N/A' }}</span>
                    </div>
                    <div class="detail-item">
                        <span class="label">{{ __('welcomeEvent.details.rewardAmount') }}:</span>
                        <span class="value">{{ $userApplication->reward_amount }} Balance</span>
                    </div>
                    <div class="detail-item">
                        <span class="label">{{ __('welcomeEvent.details.submittedAt') }}:</span>
                        <span class="value">{{ $userApplication->submitted_at->format('Y-m-d H:i:s') }}</span>
                    </div>
                    @if($userApplication->reviewed_at)
                        <div class="detail-item">
                            <span class="label">{{ __('welcomeEvent.details.reviewedAt') }}:</span>
                            <span class="value">{{ $userApplication->reviewed_at->format('Y-m-d H:i:s') }}</span>
                        </div>
                    @endif
                    @if($userApplication->donation_amount)
                        <div class="detail-item">
                            <span class="label">{{ __('welcomeEvent.details.donationAmount') }}:</span>
                            <span class="value">{{ $userApplication->donation_amount }} {{ $userApplication->currency }}</span>
                        </div>
                    @endif
                    @if($userApplication->old_character_name)
                        <div class="detail-item">
                            <span class="label">{{ __('welcomeEvent.details.oldCharacterName') }}:</span>
                            <span class="value">{{ $userApplication->old_character_name }}</span>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    @endif
</div>
