<?php
use Livewire\Volt\Component;
use Illuminate\Support\Facades\Auth;
use App\Models\FastEventsWelcome;
use App\Models\Z_Website\User;

new class extends Component {
    public $user;
    public $userRole;
    public $hasAccess = false;
    public $applications = [];
    public $stats = [];
    public $selectedFilter = 'all';
    public $selectedApplication = null;
    public $showApproveModal = false;
    public $showRejectModal = false;
    public $showDetailsModal = false;
    public $rejectionReason = '';

    public function mount()
    {
        $this->user = Auth::user();

        // Check if user has required role level (level 2+ for access)
        if ($this->user && $this->user->hasRoleLevel(2)) {
            $this->hasAccess = true;
            $this->userRole = $this->user->getRoleName();
            $this->loadData();
        }
    }

    public function loadData()
    {
        $this->loadStats();
        $this->loadApplications();
        FastEventsWelcome::updateAllHwid();

    }

    public function loadStats()
    {
        $this->stats = [
            'pending' => FastEventsWelcome::where('status', 'pending')->count(),
            'approved' => FastEventsWelcome::where('status', 'approved')->count(),
            'rejected' => FastEventsWelcome::where('status', 'rejected')->count(),
            'claimed' => FastEventsWelcome::whereNotNull('claimed_at')->count(),
        ];
    }

    public function loadApplications()
    {
        $query = FastEventsWelcome::with(['user', 'character']);

        if ($this->selectedFilter !== 'all') {
            if ($this->selectedFilter === 'claimed') {
                $query->whereNotNull('claimed_at');
            } else {
                $query->where('status', $this->selectedFilter);
            }
        }

        $this->applications = $query->orderBy('submitted_at', 'desc')->get();
    }

    public function filterApplications($filter)
    {
        $this->selectedFilter = $filter;
        $this->loadApplications();
    }

    public function viewDetails($applicationId)
    {
        $this->selectedApplication = FastEventsWelcome::with(['user', 'character'])->find($applicationId);
        $this->showDetailsModal = true;
    }

    public function openApproveModal($applicationId)
    {
        $this->selectedApplication = FastEventsWelcome::find($applicationId);
        $this->showApproveModal = true;
    }

    public function openRejectModal($applicationId)
    {
        $this->selectedApplication = FastEventsWelcome::find($applicationId);
        $this->rejectionReason = '';
        $this->showRejectModal = true;
    }

    public function approveApplication()
    {
        if ($this->selectedApplication) {
            $this->selectedApplication->approve($this->user->id);
            $this->showApproveModal = false;
            $this->selectedApplication = null;
            $this->loadData();
            $this->dispatch('notification', [
                'type' => 'success',
                'message' => 'Application approved successfully'
            ]);
        }
    }

    public function rejectApplication()
    {
        if ($this->selectedApplication && $this->rejectionReason) {
            $this->selectedApplication->reject($this->rejectionReason, $this->user->id);
            $this->showRejectModal = false;
            $this->selectedApplication = null;
            $this->rejectionReason = '';
            $this->loadData();
            $this->dispatch('notification', [
                'type' => 'success',
                'message' => 'Application rejected successfully'
            ]);
        }
    }

    public function closeModals()
    {
        $this->showApproveModal = false;
        $this->showRejectModal = false;
        $this->showDetailsModal = false;
        $this->selectedApplication = null;
        $this->rejectionReason = '';
    }

    public function refreshData()
    {
        $this->loadData();
        $this->dispatch('notification', [
            'type' => 'info',
            'message' => 'Data refreshed successfully'
        ]);
    }

    public function formatPlayerType($type)
    {
        return match($type) {
            'new' => 'New Player',
            'old' => 'Old Player',
            'donate' => 'Donating Player',
            default => ucfirst($type)
        };
    }

    public function formatCurrency($amount, $currency)
    {
        return $currency === 'USD' ? '$' . number_format($amount, 2) : number_format($amount) . ' EGP';
    }

    public function getProofsArray($proofs)
    {
        if (is_array($proofs)) {
            return $proofs;
        }

        if (is_string($proofs)) {
            // Try to decode JSON first
            $decoded = json_decode($proofs, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                return $decoded;
            }
            // If not JSON, treat as single string
            return [$proofs];
        }

        return [];
    }

    public function getStorageUrl($filePath)
    {
        if (!$filePath) {
            return null;
        }

        // If it's already a full URL, return as is
        if (str_starts_with($filePath, 'http')) {
            return $filePath;
        }

        // If it starts with storage/, create the correct URL
        if (str_starts_with($filePath, 'storage/')) {
            return asset($filePath);
        }

        // If it's just a filename, assume it's in welcome-event folder
        if (!str_contains($filePath, '/')) {
            return asset('storage/welcome-event/' . $filePath);
        }

        // Default: prepend storage path
        return asset('storage/' . $filePath);
    }

    public function getProofType($filePath)
    {
        if (!$filePath) {
            return 'document';
        }

        $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));

        return match($extension) {
            'jpg', 'jpeg', 'png', 'gif', 'webp' => 'image',
            'pdf' => 'pdf',
            'doc', 'docx' => 'document',
            'txt' => 'text',
            default => 'document'
        };
    }

    public function getProofDisplayName($filePath, $type = 'document')
    {
        if (!$filePath) {
            return 'Unknown File';
        }

        $filename = basename($filePath);

        // Try to create a more user-friendly name
        if (str_contains($filePath, 'donation-proofs')) {
            return 'Donation Proof - ' . $filename;
        }

        if (str_contains($filePath, 'old-character-proofs')) {
            return 'Old Character Proof - ' . $filename;
        }

        if (str_contains($filePath, 'character-images')) {
            return 'Character Image - ' . $filename;
        }

        return $filename;
    }

    public function getOtherCharactersByHwid($hwid)
    {
        if (!$hwid) {
            return [];
        }

        // Get all applications with the same HWID and status  approved and JID != current user's JID
        $otherApplications = FastEventsWelcome::where('HWID', $hwid)
            ->with(['character'])
            ->where('status', 'approved')
            ->where('JID', '!=', $this->user->JID)
            ->orderBy('submitted_at', 'desc')
            ->get();

        $characters = [];
        foreach ($otherApplications as $app) {
            if ($app->character) {
                $characters[] = [
                    'CharID' => $app->CharID,
                    'char_name' => $app->character->CharName16,
                    'player_type' => $app->player_type,
                    'status' => $app->status,
                    'reward_amount' => $app->reward_amount,
                    'submitted_at' => $app->submitted_at,
                    'claimed_at' => $app->claimed_at,
                    'application_id' => $app->id
                ];
            }
        }

        return $characters;
    }

    public function checkExistingCharacterInfo($hwid)
    {
        if (!$hwid) {
            return null;
        }

        return FastEventsWelcome::getExistsCharsInfoByHwid($hwid);
    }
}; ?>
<div>
<div x-data="{
    loaded: false,
    currentTime: new Date().toLocaleTimeString(),
    animationDelay: 0,

    init() {
        setTimeout(() => { this.loaded = true; }, 100);
        setInterval(() => {
            this.currentTime = new Date().toLocaleTimeString();
        }, 1000);
    },

    getAnimationDelay() {
        this.animationDelay += 100;
        return this.animationDelay;
    }
}" x-cloak class="team-fast-event-welcome">

    @if (!$hasAccess)
        <!-- Access Denied -->
        <div class="team-access-denied" x-show="loaded" x-transition:enter="team-fade-in"
            x-transition:enter-start="team-fade-out" x-transition:enter-end="team-fade-in">
            <div class="team-access-denied-content">
                <div class="team-access-denied-icon">
                    <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="10" />
                        <line x1="4.93" y1="4.93" x2="19.07" y2="19.07" />
                    </svg>
                </div>
                <h2 data-translate="team.access.denied.title">Access Denied</h2>
                <p data-translate="team.access.denied.message">You don't have sufficient privileges to access this page.</p>
                <p data-translate="team.access.denied.requirement">Minimum required level: 2</p>
            </div>
        </div>
    @else
        <!-- Fast Event Welcome Management -->
        <div x-show="loaded" x-transition:enter="team-slide-up"
            x-transition:enter-start="team-slide-down" x-transition:enter-end="team-slide-up">

            <!-- Header Section -->
            <div class="team-fast-event-header" x-show="loaded" x-transition:enter="team-fade-in"
                x-transition:enter-start="team-fade-out" x-transition:enter-end="team-fade-in"
                x-transition:enter.delay.200ms>
                <div class="team-fast-event-title">
                    <div class="team-fast-event-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M16 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                            <circle cx="8.5" cy="7" r="4"/>
                            <line x1="20" y1="8" x2="20" y2="14"/>
                            <line x1="23" y1="11" x2="17" y2="11"/>
                        </svg>
                    </div>
                    <div>
                        <h1 data-translate="team.fast_event_welcome.title">Fast Event Welcome Management</h1>
                        <p class="team-fast-event-subtitle" data-translate="team.fast_event_welcome.subtitle">Review and manage welcome event applications</p>
                    </div>
                </div>
                <div class="team-header-actions">
                    <button wire:click="refreshData" class="team-refresh-btn team-btn-glow" x-data="{ refreshing: false }"
                        @click="refreshing = true; setTimeout(() => refreshing = false, 1000)"
                        :class="{ 'team-btn-loading': refreshing }">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor"
                            stroke-width="2" :class="{ 'team-spin': refreshing }">
                            <polyline points="23 4 23 10 17 10" />
                            <path d="M20.49 15a9 9 0 1 1-2.12-9.36L23 10" />
                        </svg>
                        <span data-translate="team.actions.refresh">Refresh Data</span>
                    </button>
                </div>
            </div>

            <!-- Statistics Cards -->
            <div class="team-welcome-stats" x-show="loaded" x-transition:enter="team-fade-in"
                x-transition:enter-start="team-fade-out" x-transition:enter-end="team-fade-in"
                x-transition:enter.delay.300ms>

                <!-- Pending Applications -->
                <div class="team-welcome-stat-card" x-transition:enter="team-scale-in"
                    x-transition:enter-start="team-scale-out" x-transition:enter-end="team-scale-in"
                    x-transition:enter.delay.400ms>
                    <div class="team-welcome-stat-header">
                        <div class="team-welcome-stat-icon pending">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="10"/>
                                <polyline points="12,6 12,12 16,14"/>
                            </svg>
                        </div>
                        <div>
                            <p class="team-welcome-stat-number">{{ $stats['pending'] }}</p>
                            <p class="team-welcome-stat-label" data-translate="team.fast_event_welcome.stats.pending">Pending Applications</p>
                        </div>
                    </div>
                </div>

                <!-- Approved Applications -->
                <div class="team-welcome-stat-card" x-transition:enter="team-scale-in"
                    x-transition:enter-start="team-scale-out" x-transition:enter-end="team-scale-in"
                    x-transition:enter.delay.500ms>
                    <div class="team-welcome-stat-header">
                        <div class="team-welcome-stat-icon approved">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="9,11 12,14 22,4"/>
                                <path d="M21,12v7a2,2 0 0,1 -2,2H5a2,2 0 0,1 -2,-2V5a2,2 0 0,1 2,-2h11"/>
                            </svg>
                        </div>
                        <div>
                            <p class="team-welcome-stat-number">{{ $stats['approved'] }}</p>
                            <p class="team-welcome-stat-label" data-translate="team.fast_event_welcome.stats.approved">Approved Applications</p>
                        </div>
                    </div>
                </div>

                <!-- Rejected Applications -->
                <div class="team-welcome-stat-card" x-transition:enter="team-scale-in"
                    x-transition:enter-start="team-scale-out" x-transition:enter-end="team-scale-in"
                    x-transition:enter.delay.600ms>
                    <div class="team-welcome-stat-header">
                        <div class="team-welcome-stat-icon rejected">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="10"/>
                                <line x1="15" y1="9" x2="9" y2="15"/>
                                <line x1="9" y1="9" x2="15" y2="15"/>
                            </svg>
                        </div>
                        <div>
                            <p class="team-welcome-stat-number">{{ $stats['rejected'] }}</p>
                            <p class="team-welcome-stat-label" data-translate="team.fast_event_welcome.stats.rejected">Rejected Applications</p>
                        </div>
                    </div>
                </div>

                <!-- Claimed Rewards -->
                <div class="team-welcome-stat-card" x-transition:enter="team-scale-in"
                    x-transition:enter-start="team-scale-out" x-transition:enter-end="team-scale-in"
                    x-transition:enter.delay.700ms>
                    <div class="team-welcome-stat-header">
                        <div class="team-welcome-stat-icon claimed">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="20,12 20,22 4,22 4,12"/>
                                <rect x="2" y="7" width="20" height="5"/>
                                <line x1="12" y1="22" x2="12" y2="7"/>
                                <path d="M12 7H7.5a2.5 2.5 0 0 1 0-5C11 2 12 7 12 7z"/>
                                <path d="M12 7h4.5a2.5 2.5 0 0 0 0-5C13 2 12 7 12 7z"/>
                            </svg>
                        </div>
                        <div>
                            <p class="team-welcome-stat-number">{{ $stats['claimed'] }}</p>
                            <p class="team-welcome-stat-label" data-translate="team.fast_event_welcome.stats.claimed">Claimed Rewards</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Applications Section -->
            <div class="team-applications-section" x-show="loaded" x-transition:enter="team-fade-in"
                x-transition:enter-start="team-fade-out" x-transition:enter-end="team-fade-in"
                x-transition:enter.delay.800ms>

                <div class="team-applications-header">
                    <h2 class="team-applications-title" data-translate="team.fast_event_welcome.applications.title">Applications List</h2>
                    <div class="team-filter-controls">
                        <select wire:change="filterApplications($event.target.value)" class="team-filter-select">
                            <option value="all" data-translate="team.fast_event_welcome.applications.filter.all">All Applications</option>
                            <option value="pending" data-translate="team.fast_event_welcome.applications.filter.pending">Pending Review</option>
                            <option value="approved" data-translate="team.fast_event_welcome.applications.filter.approved">Approved</option>
                            <option value="rejected" data-translate="team.fast_event_welcome.applications.filter.rejected">Rejected</option>
                            <option value="claimed" data-translate="team.fast_event_welcome.applications.filter.claimed">Claimed</option>
                        </select>
                    </div>
                </div>

                <!-- Applications Grid -->
                <div class="team-applications-grid">
                    @forelse ($applications as $index => $application)
                        <div class="team-application-card" x-transition:enter="team-scale-in"
                            x-transition:enter-start="team-scale-out" x-transition:enter-end="team-scale-in"
                            x-transition:enter.delay.{{ 900 + $index * 100 }}ms>

                            <!-- Application Header -->
                            <div class="team-application-header">
                                <div class="team-application-user">
                                    <div class="team-application-avatar">
                                        {{ substr($application->character?->CharName16 ?? 'U', 0, 2) }}
                                    </div>
                                    <div class="team-application-info">
                                        <h3>{{ $application->character?->CharName16 ?? 'Unknown Character' }}</h3>
                                        <p>{{ $this->formatPlayerType($application->player_type) }}</p>
                                    </div>
                                </div>
                                <span class="team-application-status {{ $application->status }}">
                                    {{ $application->claimed_at ? 'claimed' : $application->status }}
                                </span>
                            </div>

                            <!-- Application Body -->
                            <div class="team-application-body">
                                <div class="team-application-details">
                                    <div class="team-application-detail">
                                        <svg class="team-application-detail-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                                            <circle cx="12" cy="7" r="4"/>
                                        </svg>
                                        <span class="team-application-detail-label" data-translate="team.fast_event_welcome.applications.details.jid">JID:</span>
                                        <span class="team-application-detail-value">{{ $application->JID }}</span>
                                    </div>

                                    @if($application->player_type === 'old')
                                        <div class="team-application-detail">
                                            <svg class="team-application-detail-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                                                <circle cx="9" cy="7" r="4"/>
                                                <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                                                <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                                            </svg>
                                            <span class="team-application-detail-label" data-translate="team.fast_event_welcome.applications.details.old_character">Old Character:</span>
                                            <span class="team-application-detail-value">{{ $application->old_character_name }}</span>
                                        </div>
                                    @endif

                                    @if($application->player_type === 'donate')
                                        <div class="team-application-detail">
                                            <svg class="team-application-detail-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <line x1="12" y1="1" x2="12" y2="23"/>
                                                <path d="M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>
                                            </svg>
                                            <span class="team-application-detail-label" data-translate="team.fast_event_welcome.applications.details.donation_amount">Donation:</span>
                                            <span class="team-application-detail-value">{{ $this->formatCurrency($application->donation_amount, $application->currency) }}</span>
                                        </div>
                                    @endif

                                    <div class="team-application-detail">
                                        <svg class="team-application-detail-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <polyline points="20,12 20,22 4,22 4,12"/>
                                            <rect x="2" y="7" width="20" height="5"/>
                                            <line x1="12" y1="22" x2="12" y2="7"/>
                                        </svg>
                                        <span class="team-application-detail-label" data-translate="team.fast_event_welcome.applications.details.reward_amount">Reward:</span>
                                        <span class="team-application-detail-value">{{ number_format($application->reward_amount) }} Balance</span>
                                    </div>

                                    <div class="team-application-detail">
                                        <svg class="team-application-detail-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <circle cx="12" cy="12" r="10"/>
                                            <polyline points="12,6 12,12 16,14"/>
                                        </svg>
                                        <span class="team-application-detail-label" data-translate="team.fast_event_welcome.applications.details.submitted_at">Submitted:</span>
                                        <span class="team-application-detail-value">{{ $application->submitted_at?->format('M d, Y H:i') }}</span>
                                    </div>
                                </div>

                                <!-- Proof Documents -->
                                @if($application->old_character_proofs || $application->donation_proofs || $application->old_character_images)
                                    <div class="team-application-proofs">
                                        <h4 data-translate="team.fast_event_welcome.applications.proofs.title">Proof Documents</h4>

                                        @if($application->old_character_proofs)
                                            @foreach($this->getProofsArray($application->old_character_proofs) as $proof)
                                                @if($proof)
                                                    @php
                                                        $proofType = $this->getProofType($proof);
                                                        $storageUrl = $this->getStorageUrl($proof);
                                                        $displayName = $this->getProofDisplayName($proof, $proofType);
                                                    @endphp
                                                    <div class="team-proof-item" data-proof-type="{{ $proofType }}">
                                                        @if($proofType === 'image')
                                                            <svg class="team-proof-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                                <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                                                                <circle cx="8.5" cy="8.5" r="1.5"/>
                                                                <polyline points="21,15 16,10 5,21"/>
                                                            </svg>
                                                        @elseif($proofType === 'pdf')
                                                            <svg class="team-proof-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                                <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                                                <polyline points="14,2 14,8 20,8"/>
                                                                <line x1="16" y1="13" x2="8" y2="13"/>
                                                                <line x1="16" y1="17" x2="8" y2="17"/>
                                                                <polyline points="10,9 9,9 8,9"/>
                                                            </svg>
                                                        @else
                                                            <svg class="team-proof-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                                <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                                                <polyline points="14,2 14,8 20,8"/>
                                                                <line x1="16" y1="13" x2="8" y2="13"/>
                                                                <line x1="16" y1="17" x2="8" y2="17"/>
                                                                <polyline points="10,9 9,9 8,9"/>
                                                            </svg>
                                                        @endif

                                                        @if($proofType === 'image')
                                                            <div class="team-proof-preview">
                                                                <img src="{{ $storageUrl }}" alt="{{ $displayName }}" class="team-proof-thumbnail">
                                                            </div>
                                                        @endif

                                                        <span class="team-proof-name">{{ $displayName }}</span>
                                                        <div class="team-proof-actions">
                                                            <a href="{{ $storageUrl }}" target="_blank" class="team-proof-view" data-translate="team.fast_event_welcome.applications.proofs.view">View</a>
                                                            @if($proofType === 'image')
                                                                <button onclick="openImageModal('{{ $storageUrl }}', '{{ $displayName }}')" class="team-proof-preview-btn">Preview</button>
                                                            @endif
                                                        </div>
                                                    </div>
                                                @endif
                                            @endforeach
                                        @endif

                                        @if($application->donation_proofs)
                                            @foreach($this->getProofsArray($application->donation_proofs) as $proof)
                                                @if($proof)
                                                    @php
                                                        $proofType = $this->getProofType($proof);
                                                        $storageUrl = $this->getStorageUrl($proof);
                                                        $displayName = $this->getProofDisplayName($proof, $proofType);
                                                    @endphp
                                                    <div class="team-proof-item" data-proof-type="{{ $proofType }}">
                                                        @if($proofType === 'image')
                                                            <svg class="team-proof-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                                <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                                                                <circle cx="8.5" cy="8.5" r="1.5"/>
                                                                <polyline points="21,15 16,10 5,21"/>
                                                            </svg>
                                                        @elseif($proofType === 'pdf')
                                                            <svg class="team-proof-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                                <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                                                <polyline points="14,2 14,8 20,8"/>
                                                                <line x1="16" y1="13" x2="8" y2="13"/>
                                                                <line x1="16" y1="17" x2="8" y2="17"/>
                                                                <polyline points="10,9 9,9 8,9"/>
                                                            </svg>
                                                        @else
                                                            <svg class="team-proof-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                                <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                                                <polyline points="14,2 14,8 20,8"/>
                                                                <line x1="16" y1="13" x2="8" y2="13"/>
                                                                <line x1="16" y1="17" x2="8" y2="17"/>
                                                                <polyline points="10,9 9,9 8,9"/>
                                                            </svg>
                                                        @endif

                                                        @if($proofType === 'image')
                                                            <div class="team-proof-preview">
                                                                <img src="{{ $storageUrl }}" alt="{{ $displayName }}" class="team-proof-thumbnail">
                                                            </div>
                                                        @endif

                                                        <span class="team-proof-name">{{ $displayName }}</span>
                                                        <div class="team-proof-actions">
                                                            <a href="{{ $storageUrl }}" target="_blank" class="team-proof-view" data-translate="team.fast_event_welcome.applications.proofs.view">View</a>
                                                            @if($proofType === 'image')
                                                                <button onclick="openImageModal('{{ $storageUrl }}', '{{ $displayName }}')" class="team-proof-preview-btn">Preview</button>
                                                            @endif
                                                        </div>
                                                    </div>
                                                @endif
                                            @endforeach
                                        @endif

                                        @if($application->old_character_images)
                                            @foreach($this->getProofsArray($application->old_character_images) as $image)
                                                @if($image)
                                                    @php
                                                        $proofType = $this->getProofType($image);
                                                        $storageUrl = $this->getStorageUrl($image);
                                                        $displayName = $this->getProofDisplayName($image, $proofType);
                                                    @endphp
                                                    <div class="team-proof-item" data-proof-type="{{ $proofType }}">
                                                        <svg class="team-proof-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                                                            <circle cx="8.5" cy="8.5" r="1.5"/>
                                                            <polyline points="21,15 16,10 5,21"/>
                                                        </svg>

                                                        <div class="team-proof-preview">
                                                            <img src="{{ $storageUrl }}" alt="{{ $displayName }}" class="team-proof-thumbnail">
                                                        </div>

                                                        <span class="team-proof-name">{{ $displayName }}</span>
                                                        <div class="team-proof-actions">
                                                            <a href="{{ $storageUrl }}" target="_blank" class="team-proof-view" data-translate="team.fast_event_welcome.applications.proofs.view">View</a>
                                                            <button onclick="openImageModal('{{ $storageUrl }}', '{{ $displayName }}')" class="team-proof-preview-btn">Preview</button>
                                                        </div>
                                                    </div>
                                                @endif
                                            @endforeach
                                        @endif
                                    </div>
                                @endif
                            </div>

                            <!-- HWID Warning Check -->
                            @php
                                $otherCharsCount = collect($this->getOtherCharactersByHwid($application->HWID))
                                    ->filter(function($char) use ($application) {
                                        return $char['application_id'] !== $application->id;
                                    })->count();
                            @endphp

                            @if($otherCharsCount > 0)
                                <div class="team-hwid-alert">
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                                        <line x1="12" y1="9" x2="12" y2="13"/>
                                        <line x1="12" y1="17" x2="12.01" y2="17"/>
                                    </svg>
                                    <span>{{ $otherCharsCount }} other character(s) with same HWID</span>
                                </div>
                            @endif

                            <!-- Action Buttons -->
                            <div class="team-application-actions">
                                <button wire:click="viewDetails({{ $application->id }})" class="team-action-btn view">
                                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/>
                                        <circle cx="12" cy="12" r="3"/>
                                    </svg>
                                    <span data-translate="team.fast_event_welcome.applications.actions.view_details">View Details</span>
                                </button>

                                @if($application->status === 'pending')
                                    <button wire:click="openApproveModal({{ $application->id }})" class="team-action-btn approve">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <polyline points="9,11 12,14 22,4"/>
                                            <path d="M21,12v7a2,2 0 0,1 -2,2H5a2,2 0 0,1 -2,-2V5a2,2 0 0,1 2,-2h11"/>
                                        </svg>
                                        <span data-translate="team.fast_event_welcome.applications.actions.approve">Approve</span>
                                    </button>

                                    <button wire:click="openRejectModal({{ $application->id }})" class="team-action-btn reject">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <circle cx="12" cy="12" r="10"/>
                                            <line x1="15" y1="9" x2="9" y2="15"/>
                                            <line x1="9" y1="9" x2="15" y2="15"/>
                                        </svg>
                                        <span data-translate="team.fast_event_welcome.applications.actions.reject">Reject</span>
                                    </button>
                                @endif
                            </div>
                        </div>
                    @empty
                        <div class="team-empty-state" x-transition:enter="team-fade-in">
                            <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1">
                                <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                                <circle cx="12" cy="7" r="4"/>
                            </svg>
                            <h3 data-translate="team.fast_event_welcome.messages.no_applications">No applications found</h3>
                            <p>Try adjusting your filters or check back later.</p>
                        </div>
                    @endforelse
                </div>
            </div>
        </div>

        <!-- Approve Modal -->
        @if($showApproveModal && $selectedApplication)
            <div class="team-modal-overlay" x-show="true" x-transition:enter="team-fade-in"
                x-transition:enter-start="team-fade-out" x-transition:enter-end="team-fade-in"
                @click.self="$wire.closeModals()">
                <div class="team-modal" x-transition:enter="team-scale-in"
                    x-transition:enter-start="team-scale-out" x-transition:enter-end="team-scale-in">
                    <div class="team-modal-header">
                        <h3 class="team-modal-title" data-translate="team.fast_event_welcome.applications.modals.approve.title">Approve Application</h3>
                        <button wire:click="closeModals" class="team-modal-close">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"/>
                                <line x1="6" y1="6" x2="18" y2="18"/>
                            </svg>
                        </button>
                    </div>
                    <div class="team-modal-body">
                        <p data-translate="team.fast_event_welcome.applications.modals.approve.message">This action will approve the application and allow the user to claim their reward.</p>
                        <div class="team-application-summary">
                            <p><strong>Character:</strong> {{ $selectedApplication->character?->CharName16 }}</p>
                            <p><strong>Player Type:</strong> {{ $this->formatPlayerType($selectedApplication->player_type) }}</p>
                            <p><strong>Reward Amount:</strong> {{ number_format($selectedApplication->reward_amount) }} Balance</p>
                        </div>
                    </div>
                    <div class="team-modal-actions">
                        <button wire:click="closeModals" class="team-action-btn">
                            <span data-translate="team.fast_event_welcome.applications.modals.approve.cancel">Cancel</span>
                        </button>
                        <button wire:click="approveApplication" class="team-action-btn approve">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="9,11 12,14 22,4"/>
                            </svg>
                            <span data-translate="team.fast_event_welcome.applications.modals.approve.confirm">Approve Application</span>
                        </button>
                    </div>
                </div>
            </div>
        @endif

        <!-- Reject Modal -->
        @if($showRejectModal && $selectedApplication)
            <div class="team-modal-overlay" x-show="true" x-transition:enter="team-fade-in"
                x-transition:enter-start="team-fade-out" x-transition:enter-end="team-fade-in"
                @click.self="$wire.closeModals()">
                <div class="team-modal" x-transition:enter="team-scale-in"
                    x-transition:enter-start="team-scale-out" x-transition:enter-end="team-scale-in">
                    <div class="team-modal-header">
                        <h3 class="team-modal-title" data-translate="team.fast_event_welcome.applications.modals.reject.title">Reject Application</h3>
                        <button wire:click="closeModals" class="team-modal-close">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"/>
                                <line x1="6" y1="6" x2="18" y2="18"/>
                            </svg>
                        </button>
                    </div>
                    <div class="team-modal-body">
                        <div class="team-form-group">
                            <label class="team-form-label" data-translate="team.fast_event_welcome.applications.modals.reject.reason_label">Rejection Reason</label>
                            <textarea wire:model="rejectionReason" class="team-form-textarea"
                                placeholder="Please provide a detailed reason for rejection..."
                                data-translate="team.fast_event_welcome.applications.modals.reject.reason_placeholder" rows="4"></textarea>
                        </div>
                    </div>
                    <div class="team-modal-actions">
                        <button wire:click="closeModals" class="team-action-btn">
                            <span data-translate="team.fast_event_welcome.applications.modals.reject.cancel">Cancel</span>
                        </button>
                        <button wire:click="rejectApplication" class="team-action-btn reject" :disabled="!$wire.rejectionReason">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="10"/>
                                <line x1="15" y1="9" x2="9" y2="15"/>
                                <line x1="9" y1="9" x2="15" y2="15"/>
                            </svg>
                            <span data-translate="team.fast_event_welcome.applications.modals.reject.confirm">Reject Application</span>
                        </button>
                    </div>
                </div>
            </div>
        @endif

        <!-- Details Modal -->
        @if($showDetailsModal && $selectedApplication)
            <div class="team-modal-overlay" x-show="true" x-transition:enter="team-fade-in"
                x-transition:enter-start="team-fade-out" x-transition:enter-end="team-fade-in"
                @click.self="$wire.closeModals()">
                <div class="team-modal" x-transition:enter="team-scale-in"
                    x-transition:enter-start="team-scale-out" x-transition:enter-end="team-scale-in">
                    <div class="team-modal-header">
                        <h3 class="team-modal-title" data-translate="team.fast_event_welcome.applications.modals.details.title">Application Details</h3>
                        <button wire:click="closeModals" class="team-modal-close">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"/>
                                <line x1="6" y1="6" x2="18" y2="18"/>
                            </svg>
                        </button>
                    </div>
                    <div class="team-modal-body">
                        <div class="team-application-full-details">
                            <!-- Full application details content here -->
                            <p><strong>Character Name:</strong> {{ $selectedApplication->character?->CharName16 }}</p>
                            <p><strong>JID:</strong> {{ $selectedApplication->JID }}</p>
                            <p><strong>HWID:</strong> {{ $selectedApplication->HWID }}</p>
                            <p><strong>Player Type:</strong> {{ $this->formatPlayerType($selectedApplication->player_type) }}</p>

                            @if($selectedApplication->player_type === 'old')
                                <p><strong>Old Character Name:</strong> {{ $selectedApplication->old_character_name }}</p>
                            @endif

                            @if($selectedApplication->player_type === 'donate')
                                <p><strong>Donation Amount:</strong> {{ $this->formatCurrency($selectedApplication->donation_amount, $selectedApplication->currency) }}</p>
                            @endif

                            <p><strong>Reward Amount:</strong> {{ number_format($selectedApplication->reward_amount) }} Balance</p>
                            <p><strong>Status:</strong> {{ ucfirst($selectedApplication->status) }}</p>
                            <p><strong>Submitted At:</strong> {{ $selectedApplication->submitted_at?->format('M d, Y H:i:s') }}</p>

                            @if($selectedApplication->reviewed_at)
                                <p><strong>Reviewed At:</strong> {{ $selectedApplication->reviewed_at->format('M d, Y H:i:s') }}</p>
                            @endif

                            @if($selectedApplication->rejection_reason)
                                <p><strong>Rejection Reason:</strong> {{ $selectedApplication->rejection_reason }}</p>
                            @endif

                            @if($selectedApplication->claimed_at)
                                <p><strong>Claimed At:</strong> {{ $selectedApplication->claimed_at->format('M d, Y H:i:s') }}</p>
                            @endif

                            <!-- Other Characters with Same HWID -->
                            @php
                                $otherCharacters = $this->getOtherCharactersByHwid($selectedApplication->HWID);
                                $filteredCharacters = collect($otherCharacters)->filter(function($char) {
                                    return $char['application_id'] !== $this->selectedApplication->id;
                                });
                            @endphp

                            @if($filteredCharacters->count() > 0)
                                <div class="team-other-characters-section">
                                    <h4 class="team-section-title">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                                            <circle cx="12" cy="7" r="4"/>
                                        </svg>
                                        Other Characters (Same HWID)
                                        <span class="team-count-badge">{{ $filteredCharacters->count() }}</span>
                                    </h4>

                                    <div class="team-other-characters-list">
                                        @foreach($filteredCharacters as $char)
                                            <div class="team-other-character-item">
                                                <div class="team-character-info">
                                                    <div class="team-character-name">
                                                        <strong>{{ $char['char_name'] }}</strong>
                                                        <span class="team-character-type">{{ $this->formatPlayerType($char['player_type']) }}</span>
                                                    </div>
                                                    <div class="team-character-details">
                                                        <span class="team-character-status team-status-{{ $char['status'] }}">
                                                            {{ ucfirst($char['status']) }}
                                                        </span>
                                                        <span class="team-character-reward">
                                                            {{ number_format($char['reward_amount']) }} Balance
                                                        </span>
                                                        @if($char['claimed_at'])
                                                            <span class="team-character-claimed">
                                                                <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                                    <polyline points="20,6 9,17 4,12"/>
                                                                </svg>
                                                                Claimed
                                                            </span>
                                                        @endif
                                                    </div>
                                                </div>
                                                <div class="team-character-date">
                                                    {{ $char['submitted_at']?->format('M d, Y') }}
                                                </div>
                                            </div>
                                        @endforeach
                                    </div>
                                </div>
                            @endif

                            <!-- HWID Character Check -->
                            @php
                                $existingCharInfo = $this->checkExistingCharacterInfo($selectedApplication->HWID);
                            @endphp

                            @if($existingCharInfo && $existingCharInfo['CharID'] !== $selectedApplication->CharID)
                                <div class="team-hwid-warning">
                                    <div class="team-warning-header">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                                            <line x1="12" y1="9" x2="12" y2="13"/>
                                            <line x1="12" y1="17" x2="12.01" y2="17"/>
                                        </svg>
                                        HWID Already Exists
                                    </div>
                                    <p class="team-warning-text">
                                        This HWID is associated with character: <strong>{{ $existingCharInfo['char_name'] }}</strong>
                                    </p>
                                </div>
                            @endif
                        </div>
                    </div>
                    <div class="team-modal-actions">
                        <button wire:click="closeModals" class="team-action-btn">
                            <span data-translate="team.fast_event_welcome.applications.modals.details.close">Close</span>
                        </button>
                    </div>
                </div>
            </div>
        @endif
    @endif
</div>

<!-- Image Preview Modal -->
<div id="imagePreviewModal" class="team-modal-overlay" style="display: none;" onclick="closeImageModal()">
    <div class="team-modal team-image-modal" onclick="event.stopPropagation()">
        <div class="team-modal-header">
            <h3 class="team-modal-title" id="imageModalTitle">Image Preview</h3>
            <button onclick="closeImageModal()" class="team-modal-close">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="18" y1="6" x2="6" y2="18"/>
                    <line x1="6" y1="6" x2="18" y2="18"/>
                </svg>
            </button>
        </div>
        <div class="team-modal-body team-image-modal-body">
            <img id="modalImage" src="" alt="" class="team-modal-image">
        </div>
        <div class="team-modal-actions">
            <a id="downloadImageBtn" href="" download class="team-action-btn">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                    <polyline points="7,10 12,15 17,10"/>
                    <line x1="12" y1="15" x2="12" y2="3"/>
                </svg>
                Download
            </a>
            <button onclick="closeImageModal()" class="team-action-btn">Close</button>
        </div>
    </div>
</div>

<script>
function openImageModal(imageUrl, imageName) {
    const modal = document.getElementById('imagePreviewModal');
    const modalImage = document.getElementById('modalImage');
    const modalTitle = document.getElementById('imageModalTitle');
    const downloadBtn = document.getElementById('downloadImageBtn');

    modalImage.src = imageUrl;
    modalImage.alt = imageName;
    modalTitle.textContent = imageName;
    downloadBtn.href = imageUrl;
    downloadBtn.download = imageName;

    modal.style.display = 'flex';
    document.body.style.overflow = 'hidden';
}

function closeImageModal() {
    const modal = document.getElementById('imagePreviewModal');
    modal.style.display = 'none';
    document.body.style.overflow = 'auto';
}

// Close modal with Escape key
document.addEventListener('keydown', function(event) {
    if (event.key === 'Escape') {
        closeImageModal();
    }
});
</script>
</div>
