<?php

use Livewire\Volt\Component;
use Illuminate\Support\Facades\Auth;
use App\Models\FastEventsTreasure;
use App\Models\Z_Website\User;

new class extends Component {
    public $user;
    public $userRole;
    public $hasAccess = false;
    public $treasures = [];
    public $stats = [];
    public $selectedFilter = 'all';
    public $selectedTreasure = null;
    public $showApproveModal = false;
    public $showRejectModal = false;
    public $showDetailsModal = false;
    public $rejectionReason = '';
    public $rejectionTreasureId = null;

    public function mount()
    {
        $this->user = Auth::user();

        // Check if user has required role level (level 2+ for access)
        if ($this->user && $this->user->hasRoleLevel(2)) {
            $this->hasAccess = true;
            $this->userRole = $this->user->getRoleName();
            $this->loadData();
        }
    }

    public function loadData()
    {
        $this->loadStats();
        $this->loadTreasures();
    }

    public function loadStats()
    {
        $this->stats = [
            'total' => FastEventsTreasure::count(),
            'pending' => FastEventsTreasure::pending()->count(),
            'approved' => FastEventsTreasure::byStatus(FastEventsTreasure::STATUS_APPROVED)->count(),
            'rejected' => FastEventsTreasure::rejected()->count(),
            'collected' => FastEventsTreasure::collected()->count(),
            'uncollected' => FastEventsTreasure::uncollected()->count(),
            'total_points' => FastEventsTreasure::sum('total_points'),
            'approved_points' => FastEventsTreasure::byStatus(FastEventsTreasure::STATUS_APPROVED)->sum('total_points'),
            'collected_points' => FastEventsTreasure::collected()->sum('total_points'),
        ];
    }

    public function loadTreasures()
    {
        $query = FastEventsTreasure::with(['user', 'acceptedBy']);

        if ($this->selectedFilter !== 'all') {
            switch ($this->selectedFilter) {
                case 'pending':
                    $query->pending();
                    break;
                case 'approved':
                    $query->byStatus(FastEventsTreasure::STATUS_APPROVED);
                    break;
                case 'rejected':
                    $query->rejected();
                    break;
                case 'collected':
                    $query->collected();
                    break;
                case 'uncollected':
                    $query->uncollected();
                    break;
            }
        }

        $this->treasures = $query->orderBy('created_at', 'desc')->get();
    }

    public function filterTreasures($filter)
    {
        $this->selectedFilter = $filter;
        $this->loadTreasures();
    }

    public function viewDetails($treasureId)
    {
        $this->selectedTreasure = FastEventsTreasure::with(['user', 'acceptedBy'])->find($treasureId);
        $this->showDetailsModal = true;
    }

    public function openApproveModal($treasureId)
    {
        $this->selectedTreasure = FastEventsTreasure::find($treasureId);
        $this->showApproveModal = true;
    }

    public function openRejectModal($treasureId)
    {
        $this->rejectionTreasureId = $treasureId;
        $this->selectedTreasure = FastEventsTreasure::find($treasureId);
        $this->rejectionReason = '';
        $this->showRejectModal = true;
    }

    public function approveTreasure()
    {
        if ($this->selectedTreasure) {
            $this->selectedTreasure->markAsAccepted($this->user->JID);
            $this->showApproveModal = false;
            $this->selectedTreasure = null;
            $this->loadData();
            $this->dispatch('notification', [
                'type' => 'success',
                'message' => 'Treasure submission approved successfully'
            ]);
        }
    }

    public function rejectTreasure()
    {
        if ($this->selectedTreasure && !empty($this->rejectionReason)) {
            $this->selectedTreasure->markAsRejected($this->rejectionReason, $this->user->JID);
            $this->showRejectModal = false;
            $this->selectedTreasure = null;
            $this->rejectionReason = '';
            $this->rejectionTreasureId = null;
            $this->loadData();
            $this->dispatch('notification', [
                'type' => 'success',
                'message' => 'Treasure submission rejected successfully'
            ]);
        }
    }

    public function rejectTreasureById()
    {
        if ($this->rejectionTreasureId && !empty($this->rejectionReason)) {
            $treasure = FastEventsTreasure::find($this->rejectionTreasureId);
            if ($treasure) {
                $treasure->markAsRejected($this->rejectionReason, $this->user->JID);
                $this->showRejectModal = false;
                $this->rejectionTreasureId = null;
                $this->rejectionReason = '';
                $this->loadData();
                $this->dispatch('notification', [
                    'type' => 'success',
                    'message' => 'Treasure submission rejected successfully'
                ]);
            }
        }
    }

    public function closeModals()
    {
        $this->showApproveModal = false;
        $this->showRejectModal = false;
        $this->showDetailsModal = false;
        $this->selectedTreasure = null;
        $this->rejectionReason = '';
        $this->rejectionTreasureId = null;
    }

    public function refreshData()
    {
        $this->loadData();
        $this->dispatch('notification', [
            'type' => 'info',
            'message' => 'Data refreshed successfully'
        ]);
    }

    public function getActivityBadges($treasure)
    {
        $badges = [];
        if ($treasure->like) $badges[] = ['text' => 'Like', 'color' => 'blue', 'points' => 5];
        if ($treasure->comment) $badges[] = ['text' => 'Comment', 'color' => 'green', 'points' => 7];
        if ($treasure->share) $badges[] = ['text' => 'Share', 'color' => 'purple', 'points' => 13];
        return $badges;
    }
}; ?>

<div x-data="{
    loaded: false,
    currentTime: new Date().toLocaleTimeString(),
    animationDelay: 0,

    init() {
        setTimeout(() => { this.loaded = true; }, 100);
        setInterval(() => {
            this.currentTime = new Date().toLocaleTimeString();
        }, 1000);
    },

    getAnimationDelay() {
        this.animationDelay += 100;
        return this.animationDelay;
    }
}" x-cloak class="team-fast-event-treasure">

    @if (!$hasAccess)
        <!-- Access Denied -->
        <div class="team-access-denied" x-show="loaded" x-transition:enter="team-fade-in">
            <div class="access-denied-card">
                <div class="access-denied-icon">
                    <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="10"/>
                        <line x1="4.93" y1="4.93" x2="19.07" y2="19.07"/>
                    </svg>
                </div>
                <h2 data-translate="team.access.denied.title">Access Denied</h2>
                <p data-translate="team.access.denied.message">You don't have sufficient privileges to access this page.</p>
                <p data-translate="team.access.denied.requirement">Minimum required level: 2</p>
            </div>
        </div>
    @else
        <!-- Fast Event Treasure Management -->
        <div x-show="loaded" x-transition:enter="team-slide-up">
            <!-- Header Section -->
            <div class="treasure-header-section" x-transition:enter="team-fade-in" :style="`animation-delay: ${getAnimationDelay()}ms`">
                <div class="treasure-header-content">
                    <div class="treasure-title-section">
                        <div class="treasure-icon">
                            <svg width="40" height="40" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
                            </svg>
                        </div>
                        <div class="treasure-title-text">
                            <h1 data-translate="team.fast_event_treasure.title">Fast Event Treasure Management</h1>
                            <p data-translate="team.fast_event_treasure.subtitle">Review and manage treasure hunt submissions</p>
                        </div>
                    </div>
                    <div class="treasure-header-actions">
                        <button wire:click="refreshData" class="treasure-refresh-btn">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="23 4 23 10 17 10"/>
                                <polyline points="1 20 1 14 7 14"/>
                                <path d="M20.49 9A9 9 0 0 0 5.64 5.64L1 10m22 4l-4.64 4.36A9 9 0 0 1 3.51 15"/>
                            </svg>
                            <span data-translate="team.common.refresh">Refresh</span>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Stats Cards -->
            <div class="treasure-stats-grid" x-transition:enter="team-fade-in" :style="`animation-delay: ${getAnimationDelay()}ms`">
                <div class="treasure-stat-card total">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value">{{ $stats['total'] ?? 0 }}</div>
                        <div class="stat-label" data-translate="team.fast_event_treasure.stats.total">Total Submissions</div>
                    </div>
                </div>

                <div class="treasure-stat-card pending">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="10"/>
                            <polyline points="12 6 12 12 16 14"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value">{{ $stats['pending'] ?? 0 }}</div>
                        <div class="stat-label" data-translate="team.fast_event_treasure.stats.pending">Pending Review</div>
                    </div>
                </div>

                <div class="treasure-stat-card accepted">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value">{{ $stats['approved'] ?? 0 }}</div>
                        <div class="stat-label" data-translate="team.fast_event_treasure.stats.approved">Approved</div>
                    </div>
                </div>

                <div class="treasure-stat-card rejected">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="10"/>
                            <line x1="15" y1="9" x2="9" y2="15"/>
                            <line x1="9" y1="9" x2="15" y2="15"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value">{{ $stats['rejected'] ?? 0 }}</div>
                        <div class="stat-label" data-translate="team.fast_event_treasure.stats.rejected">Rejected</div>
                    </div>
                </div>

                <div class="treasure-stat-card collected">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
                            <polyline points="7.5 4.21 12 6.81 16.5 4.21"/>
                            <polyline points="7.5 19.79 7.5 14.6 3 12"/>
                            <polyline points="21 12 16.5 14.6 16.5 19.79"/>
                            <polyline points="12 22.81 12 17"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value">{{ $stats['collected'] ?? 0 }}</div>
                        <div class="stat-label" data-translate="team.fast_event_treasure.stats.collected">Collected</div>
                    </div>
                </div>

                <div class="treasure-stat-card points">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value">{{ $stats['total_points'] ?? 0 }}</div>
                        <div class="stat-label" data-translate="team.fast_event_treasure.stats.total_points">Total Points</div>
                    </div>
                </div>

                <div class="treasure-stat-card collected-points">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="3"/>
                            <path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value">{{ $stats['collected_points'] ?? 0 }}</div>
                        <div class="stat-label" data-translate="team.fast_event_treasure.stats.collected_points">Collected Points</div>
                    </div>
                </div>
            </div>

            <!-- Filter Tabs -->
            <div class="treasure-filter-tabs" x-transition:enter="team-fade-in" :style="`animation-delay: ${getAnimationDelay()}ms`">
                <button wire:click="filterTreasures('all')" class="filter-tab {{ $selectedFilter === 'all' ? 'active' : '' }}">
                    <span data-translate="team.fast_event_treasure.filter.all">All Submissions</span>
                    <span class="filter-count">{{ $stats['total'] ?? 0 }}</span>
                </button>
                <button wire:click="filterTreasures('pending')" class="filter-tab {{ $selectedFilter === 'pending' ? 'active' : '' }}">
                    <span data-translate="team.fast_event_treasure.filter.pending">Pending</span>
                    <span class="filter-count">{{ $stats['pending'] ?? 0 }}</span>
                </button>
                <button wire:click="filterTreasures('approved')" class="filter-tab {{ $selectedFilter === 'approved' ? 'active' : '' }}">
                    <span data-translate="team.fast_event_treasure.filter.approved">Approved</span>
                    <span class="filter-count">{{ $stats['approved'] ?? 0 }}</span>
                </button>
                <button wire:click="filterTreasures('rejected')" class="filter-tab {{ $selectedFilter === 'rejected' ? 'active' : '' }}">
                    <span data-translate="team.fast_event_treasure.filter.rejected">Rejected</span>
                    <span class="filter-count">{{ $stats['rejected'] ?? 0 }}</span>
                </button>
                <button wire:click="filterTreasures('collected')" class="filter-tab {{ $selectedFilter === 'collected' ? 'active' : '' }}">
                    <span data-translate="team.fast_event_treasure.filter.collected">Collected</span>
                    <span class="filter-count">{{ $stats['collected'] ?? 0 }}</span>
                </button>
            </div>

            <!-- Treasures List -->
            <div class="treasure-list-container" x-transition:enter="team-fade-in" :style="`animation-delay: ${getAnimationDelay()}ms`">
                @if(count($treasures) > 0)
                    <div class="treasure-list">
                        @foreach($treasures as $index => $treasure)
                            <div class="treasure-item" x-transition:enter="team-slide-up" :style="`animation-delay: ${50 * {{ $index }}}ms`">
                                <!-- Treasure Header -->
                                <div class="treasure-item-header">
                                    <div class="treasure-user-info">
                                        <div class="treasure-user-avatar">
                                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                                                <circle cx="12" cy="7" r="4"/>
                                            </svg>
                                        </div>
                                        <div class="treasure-user-details">
                                            <div class="treasure-username">{{ $treasure->user->username ?? 'Unknown User' }}</div>
                                            <div class="treasure-user-id">ID: {{ $treasure->user_id }}</div>
                                        </div>
                                    </div>
                                    <div class="treasure-status-badge status-{{ $treasure->getStatusColor() }}">
                                        @if($treasure->status === 'approved')
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                            </svg>
                                            <span data-translate="team.fast_event_treasure.status.approved">{{ $treasure->getStatusLabel() }}</span>
                                        @elseif($treasure->status === 'rejected')
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <circle cx="12" cy="12" r="10"/>
                                                <line x1="15" y1="9" x2="9" y2="15"/>
                                                <line x1="9" y1="9" x2="15" y2="15"/>
                                            </svg>
                                            <span data-translate="team.fast_event_treasure.status.rejected">{{ $treasure->getStatusLabel() }}</span>
                                        @elseif($treasure->status === 'collected')
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
                                            </svg>
                                            <span data-translate="team.fast_event_treasure.status.collected">{{ $treasure->getStatusLabel() }}</span>
                                        @else
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <circle cx="12" cy="12" r="10"/>
                                                <polyline points="12 6 12 12 16 14"/>
                                            </svg>
                                            <span data-translate="team.fast_event_treasure.status.pending">{{ $treasure->getStatusLabel() }}</span>
                                        @endif
                                    </div>
                                </div>

                                <!-- Treasure Content -->
                                <div class="treasure-item-content">
                                    <div class="treasure-link-section">
                                        <div class="treasure-link-icon">
                                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"/>
                                                <path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"/>
                                            </svg>
                                        </div>
                                        <a href="{{ $treasure->link }}" target="_blank" class="treasure-link">
                                            {{ $treasure->link }}
                                        </a>
                                    </div>

                                    <!-- Activity Badges -->
                                    <div class="treasure-activities">
                                        @foreach($this->getActivityBadges($treasure) as $badge)
                                            <div class="activity-badge {{ $badge['color'] }}">
                                                <span class="activity-text">{{ $badge['text'] }}</span>
                                                <span class="activity-points">+{{ $badge['points'] }}</span>
                                            </div>
                                        @endforeach
                                    </div>

                                    <!-- Points Summary -->
                                    <div class="treasure-points-summary">
                                        <div class="points-total">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"/>
                                            </svg>
                                            <span>{{ $treasure->total_points }} <span data-translate="team.fast_event_treasure.points">Points</span></span>
                                        </div>
                                        @if($treasure->status === 'collected')
                                            <div class="collection-status collected">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
                                                </svg>
                                                <span data-translate="team.fast_event_treasure.status.collected">Collected</span>
                                            </div>
                                        @elseif($treasure->status === 'approved')
                                            <div class="collection-status approved">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                                </svg>
                                                <span data-translate="team.fast_event_treasure.status.ready_to_collect">Ready to Collect</span>
                                            </div>
                                        @elseif($treasure->status === 'rejected')
                                            <div class="collection-status rejected">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <circle cx="12" cy="12" r="10"/>
                                                    <line x1="15" y1="9" x2="9" y2="15"/>
                                                    <line x1="9" y1="9" x2="15" y2="15"/>
                                                </svg>
                                                <span data-translate="team.fast_event_treasure.status.rejected">Rejected</span>
                                            </div>
                                        @else
                                            <div class="collection-status pending">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <circle cx="12" cy="12" r="10"/>
                                                    <polyline points="12 6 12 12 16 14"/>
                                                </svg>
                                                <span data-translate="team.fast_event_treasure.status.pending_review">Pending Review</span>
                                            </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- Treasure Footer -->
                                <div class="treasure-item-footer">
                                    <div class="treasure-timestamp">
                                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <circle cx="12" cy="12" r="10"/>
                                            <polyline points="12 6 12 12 16 14"/>
                                        </svg>
                                        <span>{{ $treasure->created_at->format('M d, Y H:i') }}</span>
                                    </div>
                                    <div class="treasure-actions">
                                        <button wire:click="viewDetails({{ $treasure->id }})" class="action-btn view">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/>
                                                <circle cx="12" cy="12" r="3"/>
                                            </svg>
                                            <span data-translate="team.fast_event_treasure.actions.view_details">View</span>
                                        </button>
                                        @if($treasure->status === 'pending')
                                            <button wire:click="openApproveModal({{ $treasure->id }})" class="action-btn approve">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                                </svg>
                                                <span data-translate="team.fast_event_treasure.actions.approve">Approve</span>
                                            </button>
                                            <button wire:click="openRejectModal({{ $treasure->id }})" class="action-btn reject">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <circle cx="12" cy="12" r="10"/>
                                                    <line x1="15" y1="9" x2="9" y2="15"/>
                                                    <line x1="9" y1="9" x2="15" y2="15"/>
                                                </svg>
                                                <span data-translate="team.fast_event_treasure.actions.reject">Reject</span>
                                            </button>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <!-- Empty State -->
                    <div class="treasure-empty-state">
                        <div class="empty-state-icon">
                            <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
                            </svg>
                        </div>
                        <h3 data-translate="team.fast_event_treasure.messages.no_treasures">No treasure submissions found</h3>
                        <p data-translate="team.fast_event_treasure.messages.no_treasures_desc">There are no treasure hunt submissions to display at the moment.</p>
                    </div>
                @endif
            </div>
        </div>

        <!-- Approve Modal -->
        @if($showApproveModal && $selectedTreasure)
            <div class="team-modal-overlay" @click="$wire.closeModals()">
                <div class="team-modal approve-modal" @click.stop>
                    <div class="team-modal-header">
                        <h3 data-translate="team.fast_event_treasure.modals.approve.title">Approve Treasure Submission</h3>
                        <button @click="$wire.closeModals()" class="team-modal-close">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"/>
                                <line x1="6" y1="6" x2="18" y2="18"/>
                            </svg>
                        </button>
                    </div>
                    <div class="team-modal-body">
                        <div class="approve-confirmation">
                            <div class="confirmation-icon">
                                <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                </svg>
                            </div>
                            <p data-translate="team.fast_event_treasure.modals.approve.message">This action will approve the treasure submission and allow the user to collect their points.</p>
                            <div class="submission-summary">
                                <p><strong data-translate="team.fast_event_treasure.details.user">User:</strong> {{ $selectedTreasure->user->username ?? 'Unknown' }}</p>
                                <p><strong data-translate="team.fast_event_treasure.details.points">Points:</strong> {{ $selectedTreasure->total_points }}</p>
                                <p><strong data-translate="team.fast_event_treasure.details.link">Link:</strong> <a href="{{ $selectedTreasure->link }}" target="_blank">{{ $selectedTreasure->link }}</a></p>
                            </div>
                        </div>
                    </div>
                    <div class="team-modal-actions">
                        <button @click="$wire.closeModals()" class="btn-secondary" data-translate="team.common.cancel">Cancel</button>
                        <button wire:click="approveTreasure" class="btn-primary approve" data-translate="team.fast_event_treasure.modals.approve.confirm">Approve Submission</button>
                    </div>
                </div>
            </div>
        @endif

        <!-- Reject Modal -->
        @if($showRejectModal && $selectedTreasure)
            <div class="team-modal-overlay" @click="$wire.closeModals()">
                <div class="team-modal reject-modal" @click.stop>
                    <div class="team-modal-header">
                        <h3 data-translate="team.fast_event_treasure.modals.reject.title">Reject Treasure Submission</h3>
                        <button @click="$wire.closeModals()" class="team-modal-close">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"/>
                                <line x1="6" y1="6" x2="18" y2="18"/>
                            </svg>
                        </button>
                    </div>
                    <div class="team-modal-body">
                        <div class="reject-confirmation">
                            <div class="confirmation-icon">
                                <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <circle cx="12" cy="12" r="10"/>
                                    <line x1="15" y1="9" x2="9" y2="15"/>
                                    <line x1="9" y1="9" x2="15" y2="15"/>
                                </svg>
                            </div>
                            <p data-translate="team.fast_event_treasure.modals.reject.message">This action will reject the treasure submission. The user will not be able to collect points for this submission.</p>
                            <div class="submission-summary">
                                <p><strong data-translate="team.fast_event_treasure.details.user">User:</strong> {{ $selectedTreasure->user->username ?? 'Unknown' }}</p>
                                <p><strong data-translate="team.fast_event_treasure.details.points">Points:</strong> {{ $selectedTreasure->total_points }}</p>
                                <p><strong data-translate="team.fast_event_treasure.details.link">Link:</strong> <a href="{{ $selectedTreasure->link }}" target="_blank">{{ $selectedTreasure->link }}</a></p>
                            </div>

                            <!-- Rejection Reason Field -->
                            <div class="rejection-reason-field">
                                <label for="rejectionReason" class="rejection-reason-label" data-translate="team.fast_event_treasure.modals.reject.reason_label">Rejection Reason:</label>
                                <textarea
                                    id="rejectionReason"
                                    wire:model="rejectionReason"
                                    rows="4"
                                    class="rejection-reason-textarea"
                                    placeholder="{{ __('Please provide a reason for rejection...', [], 'en') }}"
                                    data-translate-placeholder="team.fast_event_treasure.modals.reject.reason_placeholder"
                                ></textarea>
                            </div>
                        </div>
                    </div>
                    <div class="team-modal-actions">
                        <button @click="$wire.closeModals()" class="btn-secondary" data-translate="team.common.cancel">Cancel</button>
                        <button wire:click="rejectTreasure" class="btn-danger reject" data-translate="team.fast_event_treasure.modals.reject.confirm">Reject Submission</button>
                    </div>
                </div>
            </div>
        @endif

        <!-- Details Modal -->
        @if($showDetailsModal && $selectedTreasure)
            <div class="team-modal-overlay" @click="$wire.closeModals()">
                <div class="team-modal details-modal" @click.stop>
                    <div class="team-modal-header">
                        <h3 data-translate="team.fast_event_treasure.modals.details.title">Treasure Submission Details</h3>
                        <button @click="$wire.closeModals()" class="team-modal-close">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"/>
                                <line x1="6" y1="6" x2="18" y2="18"/>
                            </svg>
                        </button>
                    </div>
                    <div class="team-modal-body">
                        <div class="details-grid">
                            <div class="detail-section">
                                <h4 data-translate="team.fast_event_treasure.details.user_info">User Information</h4>
                                <div class="detail-items">
                                    <div class="detail-item">
                                        <span class="detail-label" data-translate="team.fast_event_treasure.details.username">Username:</span>
                                        <span class="detail-value">{{ $selectedTreasure->user->username ?? 'Unknown' }}</span>
                                    </div>
                                    <div class="detail-item">
                                        <span class="detail-label" data-translate="team.fast_event_treasure.details.user_id">User ID:</span>
                                        <span class="detail-value">{{ $selectedTreasure->user_id }}</span>
                                    </div>
                                </div>
                            </div>

                            <div class="detail-section">
                                <h4 data-translate="team.fast_event_treasure.details.submission_info">Submission Information</h4>
                                <div class="detail-items">
                                    <div class="detail-item">
                                        <span class="detail-label" data-translate="team.fast_event_treasure.details.link">Link:</span>
                                        <a href="{{ $selectedTreasure->link }}" target="_blank" class="detail-link">{{ $selectedTreasure->link }}</a>
                                    </div>
                                    <div class="detail-item">
                                        <span class="detail-label" data-translate="team.fast_event_treasure.details.submitted_at">Submitted At:</span>
                                        <span class="detail-value">{{ $selectedTreasure->created_at->format('M d, Y H:i:s') }}</span>
                                    </div>
                                </div>
                            </div>

                            <div class="detail-section">
                                <h4 data-translate="team.fast_event_treasure.details.activities">Activities</h4>
                                <div class="activities-list">
                                    <div class="activity-item {{ $selectedTreasure->like ? 'completed' : 'incomplete' }}">
                                        <div class="activity-icon">
                                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"/>
                                            </svg>
                                        </div>
                                        <span data-translate="team.fast_event_treasure.activities.like">Like</span>
                                        <span class="activity-points">+5</span>
                                    </div>
                                    <div class="activity-item {{ $selectedTreasure->comment ? 'completed' : 'incomplete' }}">
                                        <div class="activity-icon">
                                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/>
                                            </svg>
                                        </div>
                                        <span data-translate="team.fast_event_treasure.activities.comment">Comment</span>
                                        <span class="activity-points">+7</span>
                                    </div>
                                    <div class="activity-item {{ $selectedTreasure->share ? 'completed' : 'incomplete' }}">
                                        <div class="activity-icon">
                                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <circle cx="18" cy="5" r="3"/>
                                                <circle cx="6" cy="12" r="3"/>
                                                <circle cx="18" cy="19" r="3"/>
                                                <line x1="8.59" y1="13.51" x2="15.42" y2="17.49"/>
                                                <line x1="15.41" y1="6.51" x2="8.59" y2="10.49"/>
                                            </svg>
                                        </div>
                                        <span data-translate="team.fast_event_treasure.activities.share">Share</span>
                                        <span class="activity-points">+13</span>
                                    </div>
                                </div>
                                <div class="total-points-display">
                                    <span data-translate="team.fast_event_treasure.details.total_points">Total Points:</span>
                                    <span class="points-value">{{ $selectedTreasure->total_points }}</span>
                                </div>
                            </div>

                            <div class="detail-section">
                                <h4 data-translate="team.fast_event_treasure.details.status_info">Status Information</h4>
                                <div class="detail-items">
                                    <div class="detail-item">
                                        <span class="detail-label" data-translate="team.fast_event_treasure.details.status">Status:</span>
                                        <span class="detail-value status-badge {{ $selectedTreasure->accepted ? 'accepted' : 'pending' }}">
                                            {{ $selectedTreasure->accepted ? 'Accepted' : 'Pending' }}
                                        </span>
                                    </div>
                                    <div class="detail-item">
                                        <span class="detail-label" data-translate="team.fast_event_treasure.details.collected">Collected:</span>
                                        <span class="detail-value status-badge {{ $selectedTreasure->collected ? 'collected' : 'uncollected' }}">
                                            {{ $selectedTreasure->collected ? 'Yes' : 'No' }}
                                        </span>
                                    </div>
                                    @if($selectedTreasure->accepted_at)
                                        <div class="detail-item">
                                            <span class="detail-label" data-translate="team.fast_event_treasure.details.accepted_at">Accepted At:</span>
                                            <span class="detail-value">{{ $selectedTreasure->accepted_at->format('M d, Y H:i:s') }}</span>
                                        </div>
                                    @endif
                                    @if($selectedTreasure->collected_at)
                                        <div class="detail-item">
                                            <span class="detail-label" data-translate="team.fast_event_treasure.details.collected_at">Collected At:</span>
                                            <span class="detail-value">{{ $selectedTreasure->collected_at->format('M d, Y H:i:s') }}</span>
                                        </div>
                                    @endif
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="team-modal-actions">
                        <button @click="$wire.closeModals()" class="btn-secondary" data-translate="team.common.close">Close</button>
                        @if(!$selectedTreasure->accepted)
                            <button wire:click="openApproveModal({{ $selectedTreasure->id }})" class="btn-primary approve" data-translate="team.fast_event_treasure.actions.approve">Approve</button>
                            <button wire:click="openRejectModal({{ $selectedTreasure->id }})" class="btn-danger reject" data-translate="team.fast_event_treasure.actions.reject">Reject</button>
                        @endif
                    </div>
                </div>
            </div>
        @endif
    @endif
</div>
