<?php

use Livewire\Volt\Component;
use Illuminate\Support\Facades\Auth;
use App\Models\FastEventsDuo;
use App\Models\Z_Website\User;

new class extends Component {
    public $user;
    public $userRole;
    public $hasAccess = false;
    public $duoRequests = [];
    public $stats = [];
    public $selectedFilter = 'all';
    public $selectedDuo = null;
    public $showApproveModal = false;
    public $showRejectModal = false;
    public $showDetailsModal = false;
    public $rejectionReason = '';
    public $rejectionDuoId = null;

    public function mount()
    {
        $this->user = Auth::user();

        // Check if user has required role level (level 2+ for access)
        if ($this->user && $this->user->hasRoleLevel(2)) {
            $this->hasAccess = true;
            $this->userRole = $this->user->getRoleName();
            $this->loadData();
        }
    }

    public function loadData()
    {
        $this->loadStats();
        $this->loadDuoRequests();
    }

    public function loadStats()
    {
        $this->stats = [
            'total' => FastEventsDuo::count(),
            'pending' => FastEventsDuo::pending()->count(),
            'accepted' => FastEventsDuo::accepted()->count(),
            'rejected' => FastEventsDuo::rejected()->count(),
            'collected' => FastEventsDuo::collected()->count(),
            'uncollected' => FastEventsDuo::uncollected()->count(),
            'senders' => FastEventsDuo::senders()->count(),
            'receivers' => FastEventsDuo::receivers()->count(),
            'total_rewards' => FastEventsDuo::accepted()->sum('reward_amount'),
            'collected_rewards' => FastEventsDuo::collected()->sum('reward_amount'),
        ];
    }

    public function loadDuoRequests()
    {
        $query = FastEventsDuo::with(['user', 'partner', 'reviewer']);

        if ($this->selectedFilter !== 'all') {
            switch ($this->selectedFilter) {
                case 'pending':
                    $query->pending();
                    break;
                case 'accepted':
                    $query->accepted();
                    break;
                case 'rejected':
                    $query->rejected();
                    break;
                case 'collected':
                    $query->collected();
                    break;
                case 'uncollected':
                    $query->uncollected();
                    break;
                case 'senders':
                    $query->senders();
                    break;
                case 'receivers':
                    $query->receivers();
                    break;
            }
        }

        $this->duoRequests = $query->orderBy('created_at', 'desc')->get();
    }

    public function filterDuoRequests($filter)
    {
        $this->selectedFilter = $filter;
        $this->loadDuoRequests();
    }

    public function viewDetails($duoId)
    {
        $this->selectedDuo = FastEventsDuo::with(['user', 'partner', 'reviewer'])->find($duoId);
        $this->showDetailsModal = true;
    }

    public function openApproveModal($duoId)
    {
        $this->selectedDuo = FastEventsDuo::find($duoId);
        $this->showApproveModal = true;
    }

    public function openRejectModal($duoId)
    {
        $this->rejectionDuoId = $duoId;
        $this->selectedDuo = FastEventsDuo::find($duoId);
        $this->rejectionReason = '';
        $this->showRejectModal = true;
    }

    public function approveDuo()
    {
        if ($this->selectedDuo) {
            $this->selectedDuo->markAsAccepted($this->user->JID);
            $this->showApproveModal = false;
            $this->selectedDuo = null;
            $this->loadData();
            $this->dispatch('notification', [
                'type' => 'success',
                'message' => 'Duo request approved successfully'
            ]);
        }
    }

    public function rejectDuo()
    {
        if ($this->selectedDuo && !empty($this->rejectionReason)) {
            $this->selectedDuo->markAsRejected($this->user->JID);
            $this->selectedDuo->rejection_reason = $this->rejectionReason;
            $this->selectedDuo->save();

            $this->showRejectModal = false;
            $this->selectedDuo = null;
            $this->rejectionReason = '';
            $this->rejectionDuoId = null;
            $this->loadData();
            $this->dispatch('notification', [
                'type' => 'success',
                'message' => 'Duo request rejected successfully'
            ]);
        }
    }

    public function rejectDuoById()
    {
        if ($this->rejectionDuoId && !empty($this->rejectionReason)) {
            $duo = FastEventsDuo::find($this->rejectionDuoId);
            if ($duo) {
                $duo->markAsRejected($this->user->JID);
                $duo->rejection_reason = $this->rejectionReason;
                $duo->save();

                $this->showRejectModal = false;
                $this->rejectionDuoId = null;
                $this->rejectionReason = '';
                $this->loadData();
                $this->dispatch('notification', [
                    'type' => 'success',
                    'message' => 'Duo request rejected successfully'
                ]);
            }
        }
    }

    public function closeModals()
    {
        $this->showApproveModal = false;
        $this->showRejectModal = false;
        $this->showDetailsModal = false;
        $this->selectedDuo = null;
        $this->rejectionReason = '';
        $this->rejectionDuoId = null;
    }

    public function refreshData()
    {
        $this->loadData();
        $this->dispatch('notification', [
            'type' => 'info',
            'message' => 'Data refreshed successfully'
        ]);
    }

    public function getActivityBadges($duo)
    {
        $badges = [];
        for ($i = 1; $i <= 5; $i++) {
            $linkField = "facebook_link_{$i}";
            $likeField = "link{$i}_like";
            $commentField = "link{$i}_comment";
            $shareField = "link{$i}_share";

            if (!empty($duo->$linkField)) {
                $linkBadges = [];
                if ($duo->$likeField) $linkBadges[] = ['text' => 'Like', 'color' => 'blue'];
                if ($duo->$commentField) $linkBadges[] = ['text' => 'Comment', 'color' => 'green'];
                if ($duo->$shareField) $linkBadges[] = ['text' => 'Share', 'color' => 'purple'];
                $badges["Link {$i}"] = $linkBadges;
            }
        }
        return $badges;
    }
}; ?>

<div x-data="{
    loaded: false,
    currentTime: new Date().toLocaleTimeString(),
    animationDelay: 0,

    init() {
        setTimeout(() => { this.loaded = true; }, 100);
        setInterval(() => {
            this.currentTime = new Date().toLocaleTimeString();
        }, 1000);
    },

    getAnimationDelay() {
        this.animationDelay += 100;
        return this.animationDelay;
    }
}" x-cloak class="team-fast-event-duo">

    @if (!$hasAccess)
        <!-- Access Denied -->
        <div class="team-access-denied" x-show="loaded" x-transition:enter="team-fade-in">
            <div class="access-denied-card">
                <div class="access-denied-icon">
                    <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="10"/>
                        <path d="m4.9 4.9 14.2 14.2"/>
                    </svg>
                </div>
                <h2 data-translate="team.access.denied.title">Access Denied</h2>
                <p data-translate="team.access.denied.message">You don't have sufficient privileges to access this page.</p>
                <p data-translate="team.access.denied.requirement">Minimum required level: 2</p>
            </div>
        </div>
    @else
        <!-- Fast Event Duo Management -->
        <div x-show="loaded" x-transition:enter="team-slide-up">
            <!-- Header Section -->
            <div class="duo-header-section" x-transition:enter="team-fade-in" :style="`animation-delay: ${getAnimationDelay()}ms`">
                <div class="duo-header-content">
                    <div class="duo-title-section">
                        <div class="duo-icon">
                            <svg width="32" height="32" viewBox="0 0 24 24" fill="currentColor">
                                <path d="M16 4c0-1.11.89-2 2-2s2 .89 2 2c0 1.11-.89 2-2 2s-2-.89-2-2zM4 18v-6h3v7c0 1.1.9 2 2 2h1v-13c0-1.1-.9-2-2-2H4c-1.1 0-2 .9-2 2v8c0 1.1.9 2 2 2zM12.5 11.5c.83 0 1.5-.67 1.5-1.5s-.67-1.5-1.5-1.5S11 9.17 11 10s.67 1.5 1.5 1.5zM15 20c1.1 0 2-.9 2-2v-7h3v6c1.1 0 2-.9 2-2V7c0-1.1-.9-2-2-2h-4c-1.1 0-2 .9-2 2v11c0 1.1.9 2 2 2z"/>
                            </svg>
                        </div>
                        <div class="duo-title-content">
                            <h1 data-translate="team.fast_event_duo.title">Fast Event Duo Management</h1>
                            <p data-translate="team.fast_event_duo.subtitle">Review and manage duo power partnership requests</p>
                        </div>
                    </div>
                    <div class="duo-header-actions">
                        <button wire:click="refreshData" class="refresh-btn" data-translate="team.common.refresh">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="23 4 23 10 17 10"/>
                                <polyline points="1 20 1 14 7 14"/>
                                <path d="m3.51 9a9 9 0 0 1 14.85-3.36L23 10M1 14l4.64 4.36A9 9 0 0 0 20.49 15"/>
                            </svg>
                            Refresh
                        </button>
                        <div class="current-time">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="10"/>
                                <polyline points="12 6 12 12 16 14"/>
                            </svg>
                            <span x-text="currentTime"></span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Stats Cards -->
            <div class="duo-stats-grid" x-transition:enter="team-fade-in" :style="`animation-delay: ${getAnimationDelay()}ms`">
                <div class="duo-stat-card total">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M16 4c0-1.11.89-2 2-2s2 .89 2 2c0 1.11-.89 2-2 2s-2-.89-2-2zM4 18v-6h3v7c0 1.1.9 2 2 2h1v-13c0-1.1-.9-2-2-2H4c-1.1 0-2 .9-2 2v8c0 1.1.9 2 2 2z"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <h3 data-translate="team.fast_event_duo.stats.total">Total Requests</h3>
                        <span class="stat-value">{{ $stats['total'] ?? 0 }}</span>
                    </div>
                </div>

                <div class="duo-stat-card pending">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
                            <circle cx="12" cy="12" r="10"/>
                            <polyline points="12 6 12 12 16 14"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <h3 data-translate="team.fast_event_duo.stats.pending">Pending Review</h3>
                        <span class="stat-value">{{ $stats['pending'] ?? 0 }}</span>
                    </div>
                </div>

                <div class="duo-stat-card accepted">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <h3 data-translate="team.fast_event_duo.stats.accepted">Accepted</h3>
                        <span class="stat-value">{{ $stats['accepted'] ?? 0 }}</span>
                    </div>
                </div>

                <div class="duo-stat-card rejected">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
                            <circle cx="12" cy="12" r="10"/>
                            <line x1="15" y1="9" x2="9" y2="15"/>
                            <line x1="9" y1="9" x2="15" y2="15"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <h3 data-translate="team.fast_event_duo.stats.rejected">Rejected</h3>
                        <span class="stat-value">{{ $stats['rejected'] ?? 0 }}</span>
                    </div>
                </div>

                <div class="duo-stat-card senders">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
                            <path d="m22 2-7 20-4-9-9-4 20-7z"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <h3 data-translate="team.fast_event_duo.stats.senders">Senders</h3>
                        <span class="stat-value">{{ $stats['senders'] ?? 0 }}</span>
                    </div>
                </div>

                <div class="duo-stat-card receivers">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M20 6L9 17l-5-5"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <h3 data-translate="team.fast_event_duo.stats.receivers">Receivers</h3>
                        <span class="stat-value">{{ $stats['receivers'] ?? 0 }}</span>
                    </div>
                </div>

                <div class="duo-stat-card rewards">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <h3 data-translate="team.fast_event_duo.stats.total_rewards">Total Rewards</h3>
                        <span class="stat-value">{{ $stats['total_rewards'] ?? 0 }}</span>
                    </div>
                </div>

                <div class="duo-stat-card collected-rewards">
                    <div class="stat-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M20.285 2l-11.285 11.567-5.286-5.011-3.714 3.716 9 8.728 15-15.285z"/>
                        </svg>
                    </div>
                    <div class="stat-content">
                        <h3 data-translate="team.fast_event_duo.stats.collected_rewards">Collected Rewards</h3>
                        <span class="stat-value">{{ $stats['collected_rewards'] ?? 0 }}</span>
                    </div>
                </div>
            </div>

            <!-- Filter Tabs -->
            <div class="duo-filter-tabs" x-transition:enter="team-fade-in" :style="`animation-delay: ${getAnimationDelay()}ms`">
                <button wire:click="filterDuoRequests('all')" class="filter-tab {{ $selectedFilter === 'all' ? 'active' : '' }}">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                        <path d="M3 4h18v2H3V4zm0 7h18v2H3v-2zm0 7h18v2H3v-2z"/>
                    </svg>
                    <span data-translate="team.fast_event_duo.filter.all">All Requests</span>
                </button>
                <button wire:click="filterDuoRequests('pending')" class="filter-tab {{ $selectedFilter === 'pending' ? 'active' : '' }}">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                        <circle cx="12" cy="12" r="10"/>
                        <polyline points="12 6 12 12 16 14"/>
                    </svg>
                    <span data-translate="team.fast_event_duo.filter.pending">Pending</span>
                </button>
                <button wire:click="filterDuoRequests('accepted')" class="filter-tab {{ $selectedFilter === 'accepted' ? 'active' : '' }}">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                        <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    <span data-translate="team.fast_event_duo.filter.accepted">Accepted</span>
                </button>
                <button wire:click="filterDuoRequests('rejected')" class="filter-tab {{ $selectedFilter === 'rejected' ? 'active' : '' }}">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                        <circle cx="12" cy="12" r="10"/>
                        <line x1="15" y1="9" x2="9" y2="15"/>
                        <line x1="9" y1="9" x2="15" y2="15"/>
                    </svg>
                    <span data-translate="team.fast_event_duo.filter.rejected">Rejected</span>
                </button>
                <button wire:click="filterDuoRequests('senders')" class="filter-tab {{ $selectedFilter === 'senders' ? 'active' : '' }}">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                        <path d="m22 2-7 20-4-9-9-4 20-7z"/>
                    </svg>
                    <span data-translate="team.fast_event_duo.filter.senders">Senders</span>
                </button>
                <button wire:click="filterDuoRequests('receivers')" class="filter-tab {{ $selectedFilter === 'receivers' ? 'active' : '' }}">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                        <path d="M20 6L9 17l-5-5"/>
                    </svg>
                    <span data-translate="team.fast_event_duo.filter.receivers">Receivers</span>
                </button>
            </div>

            <!-- Duo Requests List -->
            <div class="duo-list-container" x-transition:enter="team-fade-in" :style="`animation-delay: ${getAnimationDelay()}ms`">
                @if(count($duoRequests) > 0)
                    <div class="duo-list">
                        @foreach($duoRequests as $duo)
                            <div class="duo-item" x-transition:enter="team-slide-in-left" :style="`animation-delay: ${getAnimationDelay()}ms`">
                                <div class="duo-item-header">
                                    <div class="duo-user-info">
                                        <div class="user-avatar {{ $duo->role }}">
                                            @if($duo->role === 'sender')
                                                <svg width="32" height="32" viewBox="0 0 24 24" fill="currentColor">
                                                    <path d="m22 2-7 20-4-9-9-4 20-7z"/>
                                                </svg>
                                            @else
                                                <svg width="32" height="32" viewBox="0 0 24 24" fill="currentColor">
                                                    <path d="M20 6L9 17l-5-5"/>
                                                </svg>
                                            @endif
                                        </div>
                                        <div class="user-details">
                                            <h4>
                                                {{ $duo->user->username ?? 'Unknown User' }}
                                                @if($duo->role === 'sender')
                                                    <span class="role-badge sender">
                                                        <svg width="14" height="14" viewBox="0 0 24 24" fill="currentColor">
                                                            <path d="m22 2-7 20-4-9-9-4 20-7z"/>
                                                        </svg>
                                                        Sender
                                                    </span>
                                                @else
                                                    <span class="role-badge receiver">
                                                        <svg width="14" height="14" viewBox="0 0 24 24" fill="currentColor">
                                                            <path d="M20 6L9 17l-5-5"/>
                                                        </svg>
                                                        Receiver
                                                    </span>
                                                @endif
                                            </h4>
                                            <span class="user-id">ID: {{ $duo->user_id }}</span>
                                            @if($duo->partner_referrer_id)
                                                <span class="partner-info">
                                                    <svg width="12" height="12" viewBox="0 0 24 24" fill="currentColor">
                                                        <path d="M16 4c0-1.11.89-2 2-2s2 .89 2 2c0 1.11-.89 2-2 2s-2-.89-2-2zM4 18v-6h3v7c0 1.1.9 2 2 2h1v-13c0-1.1-.9-2-2-2H4c-1.1 0-2 .9-2 2v8c0 1.1.9 2 2 2z"/>
                                                    </svg>
                                                    Partner: {{ $duo->partner_referrer_id }}
                                                </span>
                                            @endif
                                        </div>
                                    </div>
                                    <div class="duo-status">
                                        @if($duo->status === 'accepted')
                                            <span class="status-indicator accepted">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                                    <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                                </svg>
                                                <span data-translate="team.fast_event_duo.status.accepted">Accepted</span>
                                            </span>
                                        @elseif($duo->status === 'rejected')
                                            <span class="status-indicator rejected">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                                    <circle cx="12" cy="12" r="10"/>
                                                    <line x1="15" y1="9" x2="9" y2="15"/>
                                                    <line x1="9" y1="9" x2="15" y2="15"/>
                                                </svg>
                                                <span data-translate="team.fast_event_duo.status.rejected">Rejected</span>
                                            </span>
                                        @else
                                            <span class="status-indicator pending">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                                    <circle cx="12" cy="12" r="10"/>
                                                    <polyline points="12 6 12 12 16 14"/>
                                                </svg>
                                                <span data-translate="team.fast_event_duo.status.pending">Pending Review</span>
                                            </span>
                                        @endif
                                    </div>
                                </div>

                                <div class="duo-item-content">
                                    <div class="duo-info-row">
                                        <div class="info-item">
                                            <span class="info-label" data-translate="team.fast_event_duo.details.role">Role:</span>
                                            <span class="info-value role-{{ $duo->role }}">
                                                @if($duo->role === 'sender')
                                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="currentColor">
                                                        <path d="m22 2-7 20-4-9-9-4 20-7z"/>
                                                    </svg>
                                                    <span data-translate="team.fast_event_duo.roles.sender">Sender</span>
                                                @else
                                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="currentColor">
                                                        <path d="M20 6L9 17l-5-5"/>
                                                    </svg>
                                                    <span data-translate="team.fast_event_duo.roles.receiver">Receiver</span>
                                                @endif
                                            </span>
                                        </div>
                                        @if($duo->partner_referrer_id)
                                            <div class="info-item partner-details">
                                                <span class="info-label" data-translate="team.fast_event_duo.details.partner">Partner:</span>
                                                <span class="info-value partner-id">
                                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="currentColor">
                                                        <path d="M16 4c0-1.11.89-2 2-2s2 .89 2 2c0 1.11-.89 2-2 2s-2-.89-2-2zM4 18v-6h3v7c0 1.1.9 2 2 2h1v-13c0-1.1-.9-2-2-2H4c-1.1 0-2 .9-2 2v8c0 1.1.9 2 2 2z"/>
                                                    </svg>
                                                    {{ $duo->partner_referrer_id }}
                                                </span>
                                            </div>
                                        @endif
                                        <div class="info-item">
                                            <span class="info-label" data-translate="team.fast_event_duo.details.reward">Reward:</span>
                                            <span class="info-value reward-amount">
                                                <svg width="14" height="14" viewBox="0 0 24 24" fill="currentColor">
                                                    <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
                                                </svg>
                                                {{ $duo->reward_amount ?? 0 }} 4T$
                                            </span>
                                        </div>
                                        <div class="info-item">
                                            <span class="info-label" data-translate="team.fast_event_duo.details.submitted_at">Submitted:</span>
                                            <span class="info-value">
                                                <svg width="14" height="14" viewBox="0 0 24 24" fill="currentColor">
                                                    <circle cx="12" cy="12" r="10"/>
                                                    <polyline points="12 6 12 12 16 14"/>
                                                </svg>
                                                {{ $duo->created_at->format('M d, Y H:i') }}
                                            </span>
                                        </div>
                                    </div>

                                    <!-- Facebook Links Summary -->
                                    @php
                                        $linkCount = 0;
                                        $totalActivities = 0;
                                        for($i = 1; $i <= 5; $i++) {
                                            $linkField = "facebook_link_{$i}";
                                            if(!empty($duo->$linkField)) {
                                                $linkCount++;
                                                if($duo->{"link{$i}_like"}) $totalActivities++;
                                                if($duo->{"link{$i}_comment"}) $totalActivities++;
                                                if($duo->{"link{$i}_share"}) $totalActivities++;
                                            }
                                        }
                                    @endphp

                                    @if($linkCount > 0)
                                        <div class="facebook-links-summary">
                                            <div class="links-header">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                                    <path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/>
                                                </svg>
                                                <span class="links-title">Facebook Links</span>
                                                <div class="links-stats">
                                                    <span class="link-count">{{ $linkCount }} Links</span>
                                                    <span class="activity-count">{{ $totalActivities }} Activities</span>
                                                </div>
                                            </div>
                                            <div class="links-preview">
                                                @for($i = 1; $i <= 5; $i++)
                                                    @php $linkField = "facebook_link_{$i}"; @endphp
                                                    @if(!empty($duo->$linkField))
                                                        <div class="link-preview-item">
                                                            <div class="link-number">{{ $i }}</div>
                                                            <div class="link-info">
                                                                <a href="{{ $duo->$linkField }}" target="_blank" class="link-url" title="{{ $duo->$linkField }}">
                                                                    {{ Str::limit($duo->$linkField, 40) }}
                                                                    <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                                        <path d="M18 13v6a2 2 0 01-2 2H5a2 2 0 01-2-2V8a2 2 0 012-2h6"/>
                                                                        <polyline points="15,3 21,3 21,9"/>
                                                                        <line x1="10" y1="14" x2="21" y2="3"/>
                                                                    </svg>
                                                                </a>
                                                                <div class="link-activities">
                                                                    @if($duo->{"link{$i}_like"})
                                                                        <span class="activity-badge like">
                                                                            <svg width="10" height="10" viewBox="0 0 24 24" fill="currentColor">
                                                                                <path d="M14 9V5a3 3 0 0 0-3-3l-4 9v11h11.28a2 2 0 0 0 2-1.7l1.38-9a2 2 0 0 0-2-2.3zM7 22H4a2 2 0 0 1-2-2v-7a2 2 0 0 1 2-2h3"/>
                                                                            </svg>
                                                                            Like
                                                                        </span>
                                                                    @endif
                                                                    @if($duo->{"link{$i}_comment"})
                                                                        <span class="activity-badge comment">
                                                                            <svg width="10" height="10" viewBox="0 0 24 24" fill="currentColor">
                                                                                <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/>
                                                                            </svg>
                                                                            Comment
                                                                        </span>
                                                                    @endif
                                                                    @if($duo->{"link{$i}_share"})
                                                                        <span class="activity-badge share">
                                                                            <svg width="10" height="10" viewBox="0 0 24 24" fill="currentColor">
                                                                                <circle cx="18" cy="5" r="3"/>
                                                                                <circle cx="6" cy="12" r="3"/>
                                                                                <circle cx="18" cy="19" r="3"/>
                                                                                <line x1="8.59" y1="13.51" x2="15.42" y2="17.49"/>
                                                                                <line x1="15.41" y1="6.51" x2="8.59" y2="10.49"/>
                                                                            </svg>
                                                                            Share
                                                                        </span>
                                                                    @endif
                                                                </div>
                                                            </div>
                                                        </div>
                                                    @endif
                                                @endfor
                                            </div>
                                        </div>
                                    @endif

                                    @if($duo->status === 'rejected' && !empty($duo->rejection_reason))
                                        <div class="rejection-reason">
                                            <div class="rejection-icon">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                                    <path d="M1 21h22L12 2 1 21zm12-3h-2v-2h2v2zm0-4h-2v-4h2v4z"/>
                                                </svg>
                                            </div>
                                            <div class="rejection-content">
                                                <span class="rejection-label" data-translate="team.fast_event_duo.details.rejection_reason">Rejection Reason:</span>
                                                <span class="rejection-text">{{ $duo->rejection_reason }}</span>
                                            </div>
                                        </div>
                                    @endif

                                    <div class="duo-item-actions">
                                        <button wire:click="viewDetails({{ $duo->id }})" class="action-btn details">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                <circle cx="12" cy="12" r="3"/>
                                                <path d="M12 1c-6.6 0-12 5.4-12 12s5.4 12 12 12 12-5.4 12-12-5.4-12-12-12zm0 17c-2.8 0-5-2.2-5-5s2.2-5 5-5 5 2.2 5 5-2.2 5-5 5z"/>
                                            </svg>
                                            <span data-translate="team.fast_event_duo.actions.view_details">View Details</span>
                                        </button>

                                        @if($duo->status === 'pending')
                                            <button wire:click="openApproveModal({{ $duo->id }})" class="action-btn approve">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                                </svg>
                                                <span data-translate="team.fast_event_duo.actions.approve">Approve</span>
                                            </button>
                                            <button wire:click="openRejectModal({{ $duo->id }})" class="action-btn reject">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <circle cx="12" cy="12" r="10"/>
                                                    <line x1="15" y1="9" x2="9" y2="15"/>
                                                    <line x1="9" y1="9" x2="15" y2="15"/>
                                                </svg>
                                                <span data-translate="team.fast_event_duo.actions.reject">Reject</span>
                                            </button>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="empty-state">
                        <div class="empty-icon">
                            <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1">
                                <path d="M16 4c0-1.11.89-2 2-2s2 .89 2 2c0 1.11-.89 2-2 2s-2-.89-2-2z"/>
                                <path d="M4 18v-6h3v7c0 1.1.9 2 2 2h1v-13c0-1.1-.9-2-2-2H4c-1.1 0-2 .9-2 2v8c0 1.1.9 2 2 2z"/>
                                <path d="M12.5 11.5c.83 0 1.5-.67 1.5-1.5s-.67-1.5-1.5-1.5S11 9.17 11 10s.67 1.5 1.5 1.5z"/>
                            </svg>
                        </div>
                        <h3 data-translate="team.fast_event_duo.messages.no_requests">No Duo Requests Found</h3>
                        <p data-translate="team.fast_event_duo.messages.no_requests_desc">There are no duo power requests to display at the moment.</p>
                    </div>
                @endif
            </div>
        </div>

        <!-- Approve Modal -->
        @if($showApproveModal && $selectedDuo)
            <div class="team-modal-overlay" @click="$wire.closeModals()">
                <div class="team-modal approve-modal" @click.stop>
                    <div class="team-modal-header">
                        <h3 data-translate="team.fast_event_duo.modals.approve.title">Approve Duo Request</h3>
                        <button @click="$wire.closeModals()" class="team-modal-close">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"/>
                                <line x1="6" y1="6" x2="18" y2="18"/>
                            </svg>
                        </button>
                    </div>
                    <div class="team-modal-content">
                        <p data-translate="team.fast_event_duo.modals.approve.message">Are you sure you want to approve this duo power request?</p>
                        <div class="duo-approve-details">
                            <div class="detail-row">
                                <span data-translate="team.fast_event_duo.details.user">User:</span>
                                <span>{{ $selectedDuo->user->username ?? 'Unknown' }}</span>
                            </div>
                            <div class="detail-row">
                                <span data-translate="team.fast_event_duo.details.role">Role:</span>
                                <span>{{ ucfirst($selectedDuo->role ?? 'Unknown') }}</span>
                            </div>
                            <div class="detail-row">
                                <span data-translate="team.fast_event_duo.details.reward">Reward:</span>
                                <span>{{ $selectedDuo->reward_amount ?? 0 }} 4T$</span>
                            </div>
                        </div>
                    </div>
                    <div class="team-modal-actions">
                        <button @click="$wire.closeModals()" class="btn-secondary" data-translate="team.common.cancel">Cancel</button>
                        <button wire:click="approveDuo" class="btn-primary approve" data-translate="team.fast_event_duo.modals.approve.confirm">Approve Request</button>
                    </div>
                </div>
            </div>
        @endif

        <!-- Reject Modal -->
        @if($showRejectModal && $selectedDuo)
            <div class="team-modal-overlay" @click="$wire.closeModals()">
                <div class="team-modal reject-modal" @click.stop>
                    <div class="team-modal-header">
                        <h3 data-translate="team.fast_event_duo.modals.reject.title">Reject Duo Request</h3>
                        <button @click="$wire.closeModals()" class="team-modal-close">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"/>
                                <line x1="6" y1="6" x2="18" y2="18"/>
                            </svg>
                        </button>
                    </div>
                    <div class="team-modal-content">
                        <p data-translate="team.fast_event_duo.modals.reject.message">Please provide a reason for rejecting this duo power request:</p>
                        <div class="duo-reject-details">
                            <div class="detail-row">
                                <span data-translate="team.fast_event_duo.details.user">User:</span>
                                <span>{{ $selectedDuo->user->username ?? 'Unknown' }}</span>
                            </div>
                            <div class="detail-row">
                                <span data-translate="team.fast_event_duo.details.role">Role:</span>
                                <span>{{ ucfirst($selectedDuo->role ?? 'Unknown') }}</span>
                            </div>
                        </div>
                        <div class="rejection-reason-field">
                            <label for="rejectionReason" data-translate="team.fast_event_duo.modals.reject.reason_label">Rejection Reason:</label>
                            <textarea
                                id="rejectionReason"
                                wire:model="rejectionReason"
                                placeholder="Enter the reason for rejection..."
                                data-translate-placeholder="team.fast_event_duo.modals.reject.reason_placeholder"
                                rows="4"
                                required
                            ></textarea>
                        </div>
                    </div>
                    <div class="team-modal-actions">
                        <button @click="$wire.closeModals()" class="btn-secondary" data-translate="team.common.cancel">Cancel</button>
                        <button wire:click="rejectDuoById" class="btn-danger reject" data-translate="team.fast_event_duo.modals.reject.confirm">Reject Request</button>
                    </div>
                </div>
            </div>
        @endif

        <!-- Details Modal -->
        @if($showDetailsModal && $selectedDuo)
            <div class="team-modal-overlay" @click="$wire.closeModals()">
                <div class="team-modal details-modal" @click.stop>
                    <div class="team-modal-header">
                        <h3 data-translate="team.fast_event_duo.modals.details.title">Duo Request Details</h3>
                        <button @click="$wire.closeModals()" class="team-modal-close">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <line x1="18" y1="6" x2="6" y2="18"/>
                                <line x1="6" y1="6" x2="18" y2="18"/>
                            </svg>
                        </button>
                    </div>
                    <div class="team-modal-content details-content">
                        <div class="details-section">
                            <h4 data-translate="team.fast_event_duo.details.user_info">User Information</h4>
                            <div class="details-grid">
                                <div class="detail-item">
                                    <span class="detail-label" data-translate="team.fast_event_duo.details.username">Username:</span>
                                    <span class="detail-value">{{ $selectedDuo->user->username ?? 'Unknown' }}</span>
                                </div>
                                <div class="detail-item">
                                    <span class="detail-label" data-translate="team.fast_event_duo.details.user_id">User ID:</span>
                                    <span class="detail-value">{{ $selectedDuo->user_id }}</span>
                                </div>
                                <div class="detail-item">
                                    <span class="detail-label" data-translate="team.fast_event_duo.details.role">Role:</span>
                                    <span class="detail-value">{{ ucfirst($selectedDuo->role ?? 'Unknown') }}</span>
                                </div>
                                <div class="detail-item">
                                    <span class="detail-label" data-translate="team.fast_event_duo.details.partner">Partner ID:</span>
                                    <span class="detail-value">{{ $selectedDuo->partner_referrer_id ?? 'Not Found' }}</span>
                                </div>
                            </div>
                        </div>

                        <div class="details-section">
                            <h4 data-translate="team.fast_event_duo.details.facebook_links">Facebook Links</h4>
                            <div class="links-grid">
                                @for($i = 1; $i <= 5; $i++)
                                    @php $linkField = "facebook_link_{$i}"; @endphp
                                    @if(!empty($selectedDuo->$linkField))
                                        <div class="link-item">
                                            <div class="link-header">
                                                <span class="link-number">Link {{ $i }}</span>
                                                <div class="link-activities">
                                                    @if($selectedDuo->{"link{$i}_like"})
                                                        <span class="activity-badge like">Like</span>
                                                    @endif
                                                    @if($selectedDuo->{"link{$i}_comment"})
                                                        <span class="activity-badge comment">Comment</span>
                                                    @endif
                                                    @if($selectedDuo->{"link{$i}_share"})
                                                        <span class="activity-badge share">Share</span>
                                                    @endif
                                                </div>
                                            </div>
                                            <a href="{{ $selectedDuo->$linkField }}" target="_blank" class="link-url">
                                                {{ Str::limit($selectedDuo->$linkField, 60) }}
                                            </a>
                                        </div>
                                    @endif
                                @endfor
                            </div>
                        </div>

                        <div class="details-section">
                            <h4 data-translate="team.fast_event_duo.details.status_info">Status Information</h4>
                            <div class="details-grid">
                                <div class="detail-item">
                                    <span class="detail-label" data-translate="team.fast_event_duo.details.status">Status:</span>
                                    <span class="detail-value">{{ ucfirst($selectedDuo->status ?? 'pending') }}</span>
                                </div>
                                <div class="detail-item">
                                    <span class="detail-label" data-translate="team.fast_event_duo.details.reward">Reward Amount:</span>
                                    <span class="detail-value">{{ $selectedDuo->reward_amount ?? 0 }} 4T$</span>
                                </div>
                                <div class="detail-item">
                                    <span class="detail-label" data-translate="team.fast_event_duo.details.submitted_at">Submitted At:</span>
                                    <span class="detail-value">{{ $selectedDuo->created_at->format('M d, Y H:i') }}</span>
                                </div>
                                @if($selectedDuo->reviewed_at)
                                    <div class="detail-item">
                                        <span class="detail-label" data-translate="team.fast_event_duo.details.reviewed_at">Reviewed At:</span>
                                        <span class="detail-value">{{ $selectedDuo->reviewed_at->format('M d, Y H:i') }}</span>
                                    </div>
                                @endif
                            </div>
                        </div>

                        @if($selectedDuo->status === 'rejected' && !empty($selectedDuo->rejection_reason))
                            <div class="details-section rejection-section">
                                <h4 data-translate="team.fast_event_duo.details.rejection_reason">Rejection Reason</h4>
                                <div class="rejection-reason-display">
                                    <p>{{ $selectedDuo->rejection_reason }}</p>
                                </div>
                            </div>
                        @endif
                    </div>
                    <div class="team-modal-actions">
                        <button @click="$wire.closeModals()" class="btn-secondary" data-translate="team.common.close">Close</button>
                    </div>
                </div>
            </div>
        @endif
    @endif
</div>
