<?php

use Livewire\Volt\Component;
use App\Models\SRO_VT_PROXY\_Players;
use App\Models\Z_Website\SiteSettings;
use Illuminate\Support\Facades\Cache;

new class extends Component {
    public $onlinePlayers = 0;
    public $totalPlayers = 0;
    public $fakeCount = 0;
    public $maxCapacity = 5000;
    public $isLoading = true;

    public function mount()
    {
        $this->updatePlayerCount();
        $this->isLoading = false;
    }

    public function updatePlayerCount()
    {
        try {
            // Get settings including max capacity
            $settings = Cache::remember('site_settings', 10, function () {
                return SiteSettings::first();
            });

            // Get max capacity from settings
            if ($settings && isset($settings->settings['sro_max_server'])) {
                $this->maxCapacity = (int) $settings->settings['sro_max_server'];
            }

            // Get online players count with shorter cache for more real-time feel
            $this->onlinePlayers = Cache::remember('online_players_count', 15, function () {
                return _Players::where('cur_status', 1)->count();
            });

            // Get fake count from site settings with much shorter cache (5 seconds for more responsive updates)
            $this->fakeCount = Cache::remember('fake_players_count', 5, function () use ($settings) {
                if (!$settings) {
                    $settings = SiteSettings::first();
                }
                return $settings ? (int) $settings->fake : 0;
            });

            // Calculate total displayed players
            $this->totalPlayers = $this->onlinePlayers + $this->fakeCount;
        } catch (\Exception $e) {
            // Fallback values if database connection fails
            $this->onlinePlayers = 0;
            $this->fakeCount = rand(50, 150);
            $this->totalPlayers = $this->fakeCount;
            $this->maxCapacity = 5000; // fallback capacity
        }
    }

    // Method to force refresh the cache (useful for admin updates)
    public function forceRefresh()
    {
        Cache::forget('fake_players_count');
        Cache::forget('online_players_count');
        Cache::forget('site_settings');
        $this->updatePlayerCount();
    }

    public function getFormattedTotalPlayersProperty()
    {
        return number_format($this->totalPlayers);
    }
}; ?>
<div>
<div class="gaming-player-counter @if($totalPlayers == 0) game-offline @endif">
    <!-- Gaming Style Counter Main Container -->
    <div class="counter-main">
        <!-- Counter Icon -->


        <!-- Digital Counter Display -->
        <div class="counter-display">
            <div class="digital-counter">
                @php
                    $percentage = $maxCapacity > 0 ? min(($totalPlayers / $maxCapacity) * 100, 100) : 0;
                @endphp

                <!-- Progress Border around digits -->
                <div class="progress-border capacity-@if($percentage < 70)low @elseif($percentage < 99)medium @else high @endif" style="--progress: {{ $percentage }}">
                    <div class="digit-row">
                        @php
                            $digits = str_split(str_pad($totalPlayers, 4, '0', STR_PAD_LEFT));
                        @endphp
                        @foreach($digits as $digit)
                            <div class="digit @if($percentage > 25) filled @endif">{{ $digit }}</div>
                        @endforeach
                    </div>

                    <!-- Progress overlay -->
                    <div class="progress-overlay">
                        <div class="progress-text @if($percentage > 15) filled @endif">{{ number_format($percentage, 1) }}% <span data-translate="welcome.capacity" >CAPACITY</span></div>
                    </div>
                </div>
            </div>

            <!-- Counter Label -->
            <div class="counter-label">
                @if($totalPlayers > 0)
                    <h3  wire:ignore  data-translate="hero.gameOnline" >GAME ONLINE</h3>
                    <p   wire:ignore  data-translate="hero.battleReady" >BATTLE READY</p>
                @else
                    <h3 wire:ignore data-translate="hero.gameOffline" >GAME OFFLINE</h3>
                    <p wire:ignore data-translate="hero.serverMaintenance" >SERVER MAINTENANCE</p>
                @endif
            </div>
        </div>
    </div>

    <!-- Simple Admin Panel -->
    <div class="admin-panel" style="display: none;" id="adminPanel">
        <button wire:click="forceRefresh">🔄 Refresh</button>
        <div>Real: {{ $onlinePlayers }}</div>
        <div>Fake: {{ $fakeCount }}</div>
        <div>Total: {{ $totalPlayers }}</div>
        <div>Capacity: {{ $maxCapacity }}</div>
    </div>

</div>


    <!-- Simple admin trigger -->
    <script>
        (function() {
            let adminKeys = '';
            document.addEventListener('keydown', function(e) {
                adminKeys += e.key;
                if (adminKeys.includes('admin')) {
                    document.getElementById('adminPanel').style.display = 'block';
                    setTimeout(() => {
                        document.getElementById('adminPanel').style.display = 'none';
                        adminKeys = '';
                    }, 5000);
                }
                if (adminKeys.length > 10) adminKeys = adminKeys.slice(-5);
            });
        })();
    </script>

    <style>
        /* Simple Gaming Player Counter - Updated Colors */
        .gaming-player-counter {
            background: linear-gradient(145deg,
                rgba(10, 10, 15, 0.95),
                rgba(26, 26, 46, 0.9));
            border: 2px solid rgba(168, 85, 247, 0.3);
            border-radius: 20px;
            padding: 2rem;
            backdrop-filter: blur(10px);
            box-shadow:
                0 25px 50px rgba(0, 0, 0, 0.5),
                0 0 30px rgba(168, 85, 247, 0.2);
            min-width: 300px;
            text-align: center;
        }

        .gaming-player-counter.game-offline {
            border-color: rgba(239, 68, 68, 0.3);
            background: linear-gradient(145deg,
                rgba(42, 15, 15, 0.95),
                rgba(59, 30, 30, 0.9));
            box-shadow:
                0 25px 50px rgba(0, 0, 0, 0.5),
                0 0 30px rgba(239, 68, 68, 0.2);
        }

        .counter-main {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 1.5rem;
        }

        .counter-icon {
            width: 60px;
            height: 60px;
            background: linear-gradient(135deg, rgba(59, 130, 246, 0.2), rgba(147, 51, 234, 0.2));
            border: 2px solid rgba(59, 130, 246, 0.4);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.8rem;
            color: #3b82f6;
        }

        .counter-display {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 1rem;
        }

        .digital-counter {
            background: linear-gradient(135deg,
                rgba(10, 10, 15, 0.8),
                rgba(26, 26, 46, 0.8));
            border-radius: 15px;
            padding: 1.5rem;
            direction: ltr;
            position: relative;
            overflow: hidden;
            box-shadow: inset 0 2px 10px rgba(0, 0, 0, 0.5);
        }

        /* Glass Container with Liquid Fill Effect - Color System */
        .progress-border {
            position: relative;
            border: 3px solid rgba(168, 85, 247, 0.4);
            border-radius: 15px;
            padding: 1.5rem;
            background: rgba(10, 10, 15, 0.1);
            backdrop-filter: blur(10px);
            transition: all 0.8s cubic-bezier(0.4, 0, 0.2, 1);
            overflow: hidden;
            box-shadow:
                inset 0 2px 10px rgba(0, 0, 0, 0.3),
                0 0 20px rgba(168, 85, 247, 0.2);
        }

        /* Low Capacity (< 70%) - Green */
        .progress-border.capacity-low {
            border-color: rgba(57, 255, 20, 0.4);
            box-shadow:
                inset 0 2px 10px rgba(0, 0, 0, 0.3),
                0 0 20px rgba(57, 255, 20, 0.2);
        }

        .progress-border.capacity-low::before {
            background: linear-gradient(180deg,
                rgba(57, 255, 20, 0.8) 0%,
                rgba(57, 255, 20, 0.6) 50%,
                rgba(57, 255, 20, 0.9) 100%);
        }

        .progress-border.capacity-low::after {
            background: linear-gradient(90deg,
                transparent 0%,
                rgba(57, 255, 20, 0.3) 25%,
                rgba(57, 255, 20, 0.5) 50%,
                rgba(57, 255, 20, 0.3) 75%,
                transparent 100%);
        }

        /* Medium Capacity (70-98%) - Orange */
        .progress-border.capacity-medium {
            border-color: rgba(255, 165, 0, 0.4);
            box-shadow:
                inset 0 2px 10px rgba(0, 0, 0, 0.3),
                0 0 20px rgba(255, 165, 0, 0.2);
        }

        .progress-border.capacity-medium::before {
            background: linear-gradient(180deg,
                rgba(255, 165, 0, 0.8) 0%,
                rgba(255, 165, 0, 0.6) 50%,
                rgba(255, 165, 0, 0.9) 100%);
        }

        .progress-border.capacity-medium::after {
            background: linear-gradient(90deg,
                transparent 0%,
                rgba(255, 165, 0, 0.3) 25%,
                rgba(255, 165, 0, 0.5) 50%,
                rgba(255, 165, 0, 0.3) 75%,
                transparent 100%);
        }

        /* High Capacity (99%+) - Red */
        .progress-border.capacity-high {
            border-color: rgba(255, 68, 68, 0.4);
            box-shadow:
                inset 0 2px 10px rgba(0, 0, 0, 0.3),
                0 0 20px rgba(255, 68, 68, 0.2);
        }

        .progress-border.capacity-high::before {
            background: linear-gradient(180deg,
                rgba(255, 68, 68, 0.8) 0%,
                rgba(255, 68, 68, 0.6) 50%,
                rgba(255, 68, 68, 0.9) 100%);
        }

        .progress-border.capacity-high::after {
            background: linear-gradient(90deg,
                transparent 0%,
                rgba(255, 68, 68, 0.3) 25%,
                rgba(255, 68, 68, 0.5) 50%,
                rgba(255, 68, 68, 0.3) 75%,
                transparent 100%);
        }

        /* Liquid Fill Effect - Default Purple */
        .progress-border::before {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            height: calc(var(--progress) * 1%);
            background: linear-gradient(180deg,
                rgba(168, 85, 247, 0.8) 0%,
                rgba(168, 85, 247, 0.6) 50%,
                rgba(168, 85, 247, 0.9) 100%);
            border-radius: 0 0 12px 12px;
            transition: height 1.2s cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 1;
            animation: liquidWave 3s ease-in-out infinite;
        }

        /* Liquid Surface Wave Effect - Default Purple */
        .progress-border::after {
            content: '';
            position: absolute;
            bottom: calc(var(--progress) * 1%);
            left: -50%;
            right: -50%;
            height: 20px;
            background: linear-gradient(90deg,
                transparent 0%,
                rgba(168, 85, 247, 0.3) 25%,
                rgba(168, 85, 247, 0.5) 50%,
                rgba(168, 85, 247, 0.3) 75%,
                transparent 100%);
            border-radius: 50%;
            transition: bottom 1.2s cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 2;
            animation: surfaceWave 2s ease-in-out infinite;
        }

        /* Offline state overrides */
        .gaming-player-counter.game-offline .progress-border::before {
            background: linear-gradient(180deg,
                rgba(255, 68, 68, 0.8) 0%,
                rgba(255, 68, 68, 0.6) 50%,
                rgba(255, 68, 68, 0.9) 100%);
        }

        .gaming-player-counter.game-offline .progress-border::after {
            background: linear-gradient(90deg,
                transparent 0%,
                rgba(255, 68, 68, 0.3) 25%,
                rgba(255, 68, 68, 0.5) 50%,
                rgba(255, 68, 68, 0.3) 75%,
                transparent 100%);
        }

        /* Liquid Wave Animation */
        @keyframes liquidWave {
            0%, 100% {
                transform: translateX(0px);
                filter: hue-rotate(0deg);
            }
            25% {
                transform: translateX(-3px);
                filter: hue-rotate(5deg);
            }
            50% {
                transform: translateX(0px);
                filter: hue-rotate(10deg);
            }
            75% {
                transform: translateX(3px);
                filter: hue-rotate(5deg);
            }
        }

        @keyframes surfaceWave {
            0%, 100% {
                transform: translateX(0px) scaleY(1);
                opacity: 0.6;
            }
            50% {
                transform: translateX(10px) scaleY(1.2);
                opacity: 0.8;
            }
        }        .progress-overlay {
            position: absolute;
            bottom: 0.5rem;
            left: 50%;
            transform: translateX(-50%);
            text-align: center;
            z-index: 3;
        }

        .progress-text {
            font-family: 'Orbitron', sans-serif;
            font-size: 0.7rem;
            letter-spacing: 1px;
            font-weight: 600;
            text-transform: uppercase;
            transition: all 0.8s ease;

            /* Default purple gaming theme */
            color: rgba(168, 85, 247, 0.6);
            text-shadow: 0 0 8px rgba(168, 85, 247, 0.3);
        }

        /* Color overrides for capacity levels */
        .capacity-low .progress-text {
            color: rgba(57, 255, 20, 0.6);
            text-shadow: 0 0 8px rgba(57, 255, 20, 0.3);
        }

        .capacity-medium .progress-text {
            color: rgba(255, 165, 0, 0.6);
            text-shadow: 0 0 8px rgba(255, 165, 0, 0.3);
        }

        .capacity-high .progress-text {
            color: rgba(255, 68, 68, 0.6);
            text-shadow: 0 0 8px rgba(255, 68, 68, 0.3);
        }

        /* Text changes color when liquid level reaches it */
        .progress-text.filled {
            color: #ffffff;
            text-shadow: 0 0 10px rgba(168, 85, 247, 0.8);
        }

        .capacity-low .progress-text.filled {
            text-shadow: 0 0 10px rgba(57, 255, 20, 0.8);
        }

        .capacity-medium .progress-text.filled {
            text-shadow: 0 0 10px rgba(255, 165, 0, 0.8);
        }

        .capacity-high .progress-text.filled {
            text-shadow: 0 0 10px rgba(255, 68, 68, 0.8);
        }

        .gaming-player-counter.game-offline .progress-text.filled {
            text-shadow: 0 0 10px rgba(255, 68, 68, 0.8);
        }

        .digit-row {
            position: relative;
            z-index: 3;
            display: flex;
            gap: 0.5rem;
            justify-content: center;
        }

        .digit {
            font-family: 'Orbitron', monospace;
            font-size: 3rem;
            font-weight: 700;
            min-width: 1.5ch;
            text-align: center;
            position: relative;
            transition: all 0.8s ease;

            /* Default empty state - gaming purple tint */
            color: rgba(168, 85, 247, 0.3);
            text-shadow: 0 0 5px rgba(168, 85, 247, 0.2);
        }

        /* Filled state - bright color when liquid reaches digit level */
        .digit.filled {
            color: #ffffff;
            text-shadow:
                0 0 10px rgba(168, 85, 247, 0.8),
                0 0 20px rgba(168, 85, 247, 0.6),
                0 0 30px rgba(168, 85, 247, 0.4);
            animation: digitGlow 2s ease-in-out infinite alternate;
        }

        /* Color overrides for capacity levels */
        .capacity-low .digit.filled {
            text-shadow:
                0 0 10px rgba(57, 255, 20, 0.8),
                0 0 20px rgba(57, 255, 20, 0.6),
                0 0 30px rgba(57, 255, 20, 0.4);
        }

        .capacity-medium .digit.filled {
            text-shadow:
                0 0 10px rgba(255, 165, 0, 0.8),
                0 0 20px rgba(255, 165, 0, 0.6),
                0 0 30px rgba(255, 165, 0, 0.4);
        }

        .capacity-high .digit.filled {
            text-shadow:
                0 0 10px rgba(255, 68, 68, 0.8),
                0 0 20px rgba(255, 68, 68, 0.6),
                0 0 30px rgba(255, 68, 68, 0.4);
        }

        .gaming-player-counter.game-offline .digit.filled {
            text-shadow:
                0 0 10px rgba(255, 68, 68, 0.8),
                0 0 20px rgba(255, 68, 68, 0.6),
                0 0 30px rgba(255, 68, 68, 0.4);
        }

        @keyframes digitGlow {
            0% {
                text-shadow:
                    0 0 10px rgba(168, 85, 247, 0.8),
                    0 0 20px rgba(168, 85, 247, 0.6),
                    0 0 30px rgba(168, 85, 247, 0.4);
            }
            100% {
                text-shadow:
                    0 0 15px rgba(168, 85, 247, 1),
                    0 0 25px rgba(168, 85, 247, 0.8),
                    0 0 35px rgba(168, 85, 247, 0.6);
            }
        }

        .counter-label h3 {
            font-family: 'Orbitron', sans-serif;
            font-size: 1.2rem;
            margin: 0;
            text-transform: uppercase;
            letter-spacing: 2px;
            transition: all 0.5s ease;

            /* Default purple gaming theme */
            color: rgba(168, 85, 247, 0.9);
            text-shadow: 0 0 8px rgba(168, 85, 247, 0.3);
        }

        /* Color overrides for capacity levels */
        .capacity-low .counter-label h3 {
            color: rgba(57, 255, 20, 0.9);
            text-shadow: 0 0 8px rgba(57, 255, 20, 0.3);
        }

        .capacity-medium .counter-label h3 {
            color: rgba(255, 165, 0, 0.9);
            text-shadow: 0 0 8px rgba(255, 165, 0, 0.3);
        }

        .capacity-high .counter-label h3 {
            color: rgba(255, 68, 68, 0.9);
            text-shadow: 0 0 8px rgba(255, 68, 68, 0.3);
        }

        .gaming-player-counter.game-offline .counter-label h3 {
            color: #ff6b6b;
        }

        .counter-label p {
            font-size: 0.9rem;
            margin: 0.5rem 0 0 0;
            text-transform: uppercase;
            transition: all 0.5s ease;

            /* Default purple gaming theme */
            color: rgba(168, 85, 247, 0.8);
            text-shadow: 0 0 5px rgba(168, 85, 247, 0.2);
        }

        /* Color overrides for capacity levels */
        .capacity-low .counter-label p {
            color: rgba(57, 255, 20, 0.8);
            text-shadow: 0 0 5px rgba(57, 255, 20, 0.2);
        }

        .capacity-medium .counter-label p {
            color: rgba(255, 165, 0, 0.8);
            text-shadow: 0 0 5px rgba(255, 165, 0, 0.2);
        }

        .capacity-high .counter-label p {
            color: rgba(255, 68, 68, 0.8);
            text-shadow: 0 0 5px rgba(255, 68, 68, 0.2);
        }

        .gaming-player-counter.game-offline .counter-label p {
            color: rgba(239, 68, 68, 0.8);
        }

        /* Simple Admin Panel */
        .admin-panel {
            position: absolute;
            top: 110%;
            left: 50%;
            transform: translateX(-50%);
            background: rgba(15, 23, 42, 0.95);
            border: 1px solid rgba(59, 130, 246, 0.3);
            border-radius: 10px;
            padding: 1rem;
            min-width: 200px;
            text-align: center;
        }

        .admin-panel button {
            background: #059669;
            color: white;
            border: none;
            padding: 0.5rem 1rem;
            border-radius: 5px;
            cursor: pointer;
            margin-bottom: 0.5rem;
        }

        .admin-panel div {
            color: #00ff88;
            font-size: 0.9rem;
            margin: 0.25rem 0;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .gaming-player-counter {
                padding: 1.5rem;
                min-width: 250px;
            }
            .digit {
                font-size: 2.5rem;
            }
        }

        /* RTL Support */
        [dir="rtl"] .admin-panel {
            left: auto;
            right: 50%;
            transform: translateX(50%);
        }
    </style>
</div>
