<?php

use Livewire\Volt\Component;
use Illuminate\Support\Facades\Auth;

new class extends Component {
    public $type = 'desktop';

    public $notifications = [];
    public $unreadCount = 0;
    public $adminMessages = [];
    public $adminMessagesUnreadCount = 0;
    public $charges = [];
    public $chargesUnreadCount = 0;
    public $currentLanguage = 'en';

    protected $listeners = [
        'language-changed' => 'updateLanguage',
        'refresh-notifications' => 'handleRefreshNotifications'
    ];

    public function mount()
    {
        try {
            $this->currentLanguage = app()->getLocale() ?: 'en';

            $this->loadNotifications();
            $this->loadAdminMessages();
            $this->loadCharges();

            // إرسال حدث لتحديث الـ badges عند التحميل الأولي فقط إذا كان هناك إشعارات
            $totalUnread = $this->unreadCount + $this->adminMessagesUnreadCount + $this->chargesUnreadCount;
            if ($totalUnread > 0) {
                $this->dispatch('notification-updated', [
                    'unreadCount' => $this->unreadCount,
                    'type' => $this->type
                ]);
            }
        } catch (\Exception $e) {
            \Log::error('Notification mount error: ' . $e->getMessage());
            $this->notifications = collect();
            $this->unreadCount = 0;
            $this->adminMessages = collect();
            $this->adminMessagesUnreadCount = 0;
            $this->charges = collect();
            $this->chargesUnreadCount = 0;
        }
    }

    public function loadNotifications()
    {
        try {
            if (!auth()->check()) {
                $this->notifications = collect();
                $this->unreadCount = 0;
                return;
            }

            // جلب الإشعارات النشطة (غير منتهية الصلاحية) من نوع notification
            $activeNotifications = \App\Models\Z_Website\NtfNotification::where('type', 'notification')
                ->where(function ($query) {
                    $query->whereNull('expires_at')
                          ->orWhere('expires_at', '>', now());
                })
                ->where(function ($query) {
                    $userId = auth()->user()->JID;
                    $query->whereNull('user_id') // إشعارات عامة للجميع
                          ->orWhere('user_id', $userId); // إشعارات خاصة بالمستخدم
                })
                ->get();

            // إضافة حالة القراءة وترتيب الإشعارات (غير المقروءة أولاً)
            $this->notifications = $activeNotifications->map(function ($notification) {
                $notification->isRead = false;
                if (auth()->check()) {
                    $notification->isRead = \App\Models\Z_Website\NtfNotificationUser::where('user_id', auth()->user()->JID)
                        ->where('notification_id', $notification->id)
                        ->exists();
                }

                // ترجمة الرسالة حسب اللغة الحالية
                $currentLanguage = $this->currentLanguage ?: 'en';
                $notification->translatedMessage = $notification->message[$currentLanguage] ?? $notification->message['en'] ?? $notification->message;
                $notification->translatedHead = $notification->head[$currentLanguage] ?? $notification->head['en'] ?? $notification->head;

                return $notification;
            })->sortBy('isRead');

            // حساب عدد الإشعارات غير المقروءة
            $this->unreadCount = $this->notifications->where('isRead', false)->count();
        } catch (\Exception $e) {
            // Log the error and set safe defaults
            \Log::error('Notification loadNotifications error: ' . $e->getMessage());
            $this->notifications = collect();
            $this->unreadCount = 0;
        }
    }

    public function loadAdminMessages()
    {
        try {
            if (!auth()->check()) {
                $this->adminMessages = collect();
                $this->adminMessagesUnreadCount = 0;
                return;
            }

            // جلب الإشعارات النشطة من نوع admin_message
            $activeAdminMessages = \App\Models\Z_Website\NtfNotification::where('type', 'admin_message')
                ->where(function ($query) {
                    $query->whereNull('expires_at')
                          ->orWhere('expires_at', '>', now());
                })
                ->where(function ($query) {
                    $userId = auth()->user()->JID;
                    $query->whereNull('user_id') // إشعارات عامة للجميع
                          ->orWhere('user_id', $userId); // إشعارات خاصة بالمستخدم
                })
                ->get();

            // إضافة حالة القراءة وترتيب الإشعارات
            $this->adminMessages = $activeAdminMessages->map(function ($notification) {
                $notification->isRead = false;
                if (auth()->check()) {
                    $notification->isRead = \App\Models\Z_Website\NtfNotificationUser::where('user_id', auth()->user()->JID)
                        ->where('notification_id', $notification->id)
                        ->exists();
                }

                // ترجمة الرسالة حسب اللغة الحالية
                $currentLanguage = $this->currentLanguage ?: 'en';
                $notification->translatedMessage = $notification->message[$currentLanguage] ?? $notification->message['en'] ?? $notification->message;
                $notification->translatedHead = $notification->head[$currentLanguage] ?? $notification->head['en'] ?? $notification->head;

                return $notification;
            })->sortBy('isRead');

            // حساب عدد الإشعارات غير المقروءة
            $this->adminMessagesUnreadCount = $this->adminMessages->where('isRead', false)->count();
        } catch (\Exception $e) {
            // Log the error and set safe defaults
            \Log::error('Notification loadAdminMessages error: ' . $e->getMessage());
            $this->adminMessages = collect();
            $this->adminMessagesUnreadCount = 0;
        }
    }

    public function loadCharges()
    {
        try {
            if (!auth()->check()) {
                $this->charges = collect();
                $this->chargesUnreadCount = 0;
                return;
            }

            // جلب الإشعارات النشطة من نوع charge
            $activeCharges = \App\Models\Z_Website\NtfNotification::where('type', 'charge')
                ->where(function ($query) {
                    $query->whereNull('expires_at')
                          ->orWhere('expires_at', '>', now());
                })
                ->where(function ($query) {
                    $userId = auth()->user()->JID;
                    $query->whereNull('user_id') // إشعارات عامة للجميع
                          ->orWhere('user_id', $userId); // إشعارات خاصة بالمستخدم
                })
                ->get();

            // إضافة حالة القراءة وترتيب الإشعارات
            $this->charges = $activeCharges->map(function ($notification) {
                $notification->isRead = false;
                if (auth()->check()) {
                    $notification->isRead = \App\Models\Z_Website\NtfNotificationUser::where('user_id', auth()->user()->JID)
                        ->where('notification_id', $notification->id)
                        ->exists();
                }

                // ترجمة الرسالة حسب اللغة الحالية
                $currentLanguage = $this->currentLanguage ?: 'en';
                $notification->translatedMessage = $notification->message[$currentLanguage] ?? $notification->message['en'] ?? $notification->message;
                $notification->translatedHead = $notification->head[$currentLanguage] ?? $notification->head['en'] ?? $notification->head;

                return $notification;
            })->sortBy('isRead');

            // حساب عدد الإشعارات غير المقروءة
            $this->chargesUnreadCount = $this->charges->where('isRead', false)->count();
        } catch (\Exception $e) {
            // Log the error and set safe defaults
            \Log::error('Notification loadCharges error: ' . $e->getMessage());
            $this->charges = collect();
            $this->chargesUnreadCount = 0;
        }
    }

    public function markAsRead($notificationId)
    {
        try {
            if (!auth()->check()) {
                return;
            }

            $userId = auth()->user()->JID;

            $record = \App\Models\Z_Website\NtfNotificationUser::firstOrCreate([
                'user_id' => $userId,
                'notification_id' => $notificationId,
            ]);

            if ($record->wasRecentlyCreated) {
                // تحديث العدادات حسب النوع
                $notification = \App\Models\Z_Website\NtfNotification::find($notificationId);
                if ($notification) {
                    if ($notification->type === 'notification') {
                        $this->unreadCount = max(0, $this->unreadCount - 1);
                    } elseif ($notification->type === 'admin_message') {
                        $this->adminMessagesUnreadCount = max(0, $this->adminMessagesUnreadCount - 1);
                    } elseif ($notification->type === 'charge') {
                        $this->chargesUnreadCount = max(0, $this->chargesUnreadCount - 1);
                    }
                }
            }

            // إعادة تحميل البيانات لتحديث حالة القراءة
            $this->loadNotifications();
            $this->loadAdminMessages();
            $this->loadCharges();

            // إرسال حدث لتحديث الـ badges في JavaScript فقط إذا كان هناك تغيير
            $totalUnread = $this->unreadCount + $this->adminMessagesUnreadCount + $this->chargesUnreadCount;
            $this->dispatch('notification-updated', [
                'unreadCount' => $this->unreadCount,
                'totalUnread' => $totalUnread,
                'type' => $this->type
            ]);
        } catch (\Exception $e) {
            // Log the error and prevent page expiration
            \Log::error('Notification markAsRead error: ' . $e->getMessage());
        }
    }

    public function updatedType()
    {
        // Removed to prevent page expiration issues
        // $this->type = '';
    }

    public function updateLanguage($lang)
    {
        $this->currentLanguage = $lang;
        $this->loadNotifications();
        $this->loadAdminMessages();
        $this->loadCharges();

        // إرسال حدث لتحديث الـ badges بعد تغيير اللغة فقط إذا كان هناك إشعارات
        $totalUnread = $this->unreadCount + $this->adminMessagesUnreadCount + $this->chargesUnreadCount;
        if ($totalUnread > 0) {
            $this->dispatch('notification-updated', [
                'unreadCount' => $this->unreadCount,
                'totalUnread' => $totalUnread,
                'type' => $this->type
            ]);
        }
    }

    //refresh method to reload all data
    public function refresh()
    {
        $this->loadNotifications();
        $this->loadAdminMessages();
        $this->loadCharges();
    }

    // التعامل مع حدث التحديث من الـ Model
    public function handleRefreshNotifications()
    {
        $this->refresh();

        // إرسال حدث لتحديث الـ badges
        $totalUnread = $this->unreadCount + $this->adminMessagesUnreadCount + $this->chargesUnreadCount;
        if ($totalUnread > 0) {
            $this->dispatch('notification-updated', [
                'unreadCount' => $this->unreadCount,
                'totalUnread' => $totalUnread,
                'type' => $this->type
            ]);
        }
    }
}; ?>
<div>
    @if ($type === 'desktop')
        <div class="notification-icons-desktop">
            <!-- Notifications -->
            <div class="notification-icon-container-desktop">
                <button class="notification-btn-desktop notifications" id="notificationsBtn">
                    <svg class="notification-icon-desktop-svg" width="16" height="16" viewBox="0 0 24 24"
                        fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path
                            d="M12 2C10.896 2 10 2.896 10 4V5.586L7.707 8.293C7.512 8.488 7.256 8.584 7 8.584V10H17V8.584C16.744 8.584 16.488 8.488 16.293 8.293L14 5.586V4C14 2.896 13.104 2 12 2Z"
                            fill="currentColor" />
                        <path
                            d="M18 11V12C18 15.314 15.314 18 12 18C8.686 18 6 15.314 6 12V11H4V12C4 16.418 7.582 20 12 20C16.418 20 20 16.418 20 12V11H18Z"
                            fill="currentColor" />
                        <circle cx="17" cy="4" r="3" fill="#ff4757" />
                    </svg>
                    <span class="notification-badge-desktop" id="notificationsBadge"
                        style="display: {{ $unreadCount > 0 ? 'block' : 'none' }};">{{ $unreadCount > 99 ? '99+' : $unreadCount }}</span>
                </button>
                <div class="notification-dropdown-desktop notifications" id="notificationsDropdown">
                    <div class="notification-header-desktop">
                        <h4 data-translate="notifications.title">Notifications</h4>
                    </div>
                    <div class="notification-list-desktop" id="notificationsList" >
                        @forelse($notifications as $notification)
                            <div  class="notification-item-desktop {{ $notification->isRead ? 'read' : 'unread' }}"
                                wire:click="markAsRead({{ $notification->id }})" style="cursor: pointer;">
                                <div class="notification-icon-desktop">
                                    <i
                                        class="fas fa-bell {{ $notification->isRead ? 'text-muted' : 'text-warning' }}"></i>
                                </div>
                                <div class="notification-content-desktop">
                                    <p class="notification-text-desktop">
                                        <strong>{{ $notification->translatedHead }}</strong><br>
                                        {!! $notification->translatedMessage !!}
                                        @if ($notification->link)
                                            <br><small><a href="{{ $notification->link }}" class="text-primary">View
                                                    More</a></small>
                                        @endif
                                    </p>
                                    <span class="notification-time-desktop">
                                        {{ $notification->created_at->diffForHumans() }}
                                        @if ($notification->expires_at)
                                            (Expires: {{ $notification->expires_at->diffForHumans() }})
                                        @endif
                                    </span>
                                </div>
                                @if (!$notification->isRead)
                                    <div class="notification-unread-indicator"></div>
                                @endif
                            </div>
                        @empty
                            <div class="notification-item-desktop">
                                <div class="notification-content-desktop">
                                    <p class="notification-text-desktop text-center text-muted">
                                        No new notifications
                                    </p>
                                </div>
                            </div>
                        @endforelse
                    </div>
                </div>
            </div>

            <!-- Admin Messages -->
            <div class="notification-icon-container-desktop">
                <button class="notification-btn-desktop admin-messages" id="adminMessagesBtn">
                    <svg class="notification-icon-desktop-svg" width="16" height="16" viewBox="0 0 24 24"
                        fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path
                            d="M20 4H4C2.896 4 2 4.896 2 6V18C2 19.104 2.896 20 4 20H20C21.104 20 22 19.104 22 18V6C22 4.896 21.104 4 20 4Z"
                            fill="currentColor" />
                        <path d="M22 6L12 13L2 6" stroke="currentColor" stroke-width="1.5" fill="none" />
                        <circle cx="19" cy="5" r="2.5" fill="#007bff" />
                        <path d="M19 3.5L20.5 5L19 6.5L17.5 5L19 3.5Z" fill="white" />
                    </svg>
                    <span class="notification-badge-desktop" id="adminMessagesBadge" style="display: {{ $adminMessagesUnreadCount > 0 ? 'block' : 'none' }};">{{ $adminMessagesUnreadCount > 99 ? '99+' : $adminMessagesUnreadCount }}</span>
                </button>
                <div class="notification-dropdown-desktop admin-messages" id="adminMessagesDropdown">
                    <div class="notification-header-desktop">
                        <h4 data-translate="adminMessages.title">Admin Messages</h4>
                    </div>
                    <div class="notification-list-desktop" id="adminMessagesList">
                        @forelse($adminMessages as $notification)
                            <div  class="notification-item-desktop {{ $notification->isRead ? 'read' : 'unread' }}"
                                wire:click="markAsRead({{ $notification->id }})" style="cursor: pointer;">
                                <div class="notification-icon-desktop">
                                    <i class="fas fa-shield-alt {{ $notification->isRead ? 'text-muted' : 'text-info' }}"></i>
                                </div>
                                <div class="notification-content-desktop">
                                    <p class="notification-text-desktop">
                                        <strong>{{ $notification->translatedHead }}</strong><br>
                                        {!! $notification->translatedMessage !!}
                                        @if ($notification->link)
                                            <br><small><a href="{{ $notification->link }}" class="text-primary">View
                                                    More</a></small>
                                        @endif
                                    </p>
                                    <span class="notification-time-desktop">
                                        {{ $notification->created_at->diffForHumans() }}
                                        @if ($notification->expires_at)
                                            (Expires: {{ $notification->expires_at->diffForHumans() }})
                                        @endif
                                    </span>
                                </div>
                                @if (!$notification->isRead)
                                    <div class="notification-unread-indicator"></div>
                                @endif
                            </div>
                        @empty
                            <div class="notification-item-desktop">
                                <div class="notification-content-desktop">
                                    <p class="notification-text-desktop text-center text-muted">
                                        No admin messages
                                    </p>
                                </div>
                            </div>
                        @endforelse
                    </div>
                </div>
            </div>

            <!-- Charges/Purchases -->
            <div class="notification-icon-container-desktop">
                <button class="notification-btn-desktop charges" id="chargesBtn">
                    <svg class="notification-icon-desktop-svg" width="16" height="16" viewBox="0 0 24 24"
                        fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path
                            d="M7 4V2C7 1.448 7.448 1 8 1H16C16.552 1 17 1.448 17 2V4H20C20.552 4 21 4.448 21 5S20.552 6 20 6H19V19C19 20.104 18.104 21 17 21H7C5.896 21 5 20.104 5 19V6H4C3.448 6 3 5.552 3 5S3.448 4 4 4H7Z"
                            fill="currentColor" />
                        <path d="M9 9H11V17H9V9Z" fill="currentColor" />
                        <path d="M13 9H15V17H13V9Z" fill="currentColor" />
                        <circle cx="19" cy="4" r="2.5" fill="#2ed573" />
                        <path d="M17.5 2.5L18.5 3.5L20.5 1.5" stroke="white" stroke-width="1.5" fill="none" />
                    </svg>
                    <span class="notification-badge-desktop" id="chargesBadge" style="display: {{ $chargesUnreadCount > 0 ? 'block' : 'none' }};">{{ $chargesUnreadCount > 99 ? '99+' : $chargesUnreadCount }}</span>
                </button>
                <div class="notification-dropdown-desktop charges" id="chargesDropdown">
                    <div class="notification-header-desktop">
                        <h4 data-translate="charges.title">Recent Purchases</h4>
                    </div>
                    <div class="notification-list-desktop" id="chargesList">
                        @forelse($charges as $notification)
                            <div  class="notification-item-desktop {{ $notification->isRead ? 'read' : 'unread' }}"
                                wire:click="markAsRead({{ $notification->id }})" style="cursor: pointer;">
                                <div class="notification-icon-desktop">
                                    <i class="fas fa-credit-card {{ $notification->isRead ? 'text-muted' : 'text-success' }}"></i>
                                </div>
                                <div class="notification-content-desktop">
                                    <p class="notification-text-desktop">
                                        <strong>{{ $notification->translatedHead }}</strong><br>
                                        {!! $notification->translatedMessage !!}
                                        @if ($notification->link)
                                            <br><small><a href="{{ $notification->link }}" class="text-primary">View
                                                    More</a></small>
                                        @endif
                                    </p>
                                    <span class="notification-time-desktop">
                                        {{ $notification->created_at->diffForHumans() }}
                                        @if ($notification->expires_at)
                                            (Expires: {{ $notification->expires_at->diffForHumans() }})
                                        @endif
                                    </span>
                                </div>
                                @if (!$notification->isRead)
                                    <div class="notification-unread-indicator"></div>
                                @endif
                            </div>
                        @empty
                            <div class="notification-item-desktop">
                                <div class="notification-content-desktop">
                                    <p class="notification-text-desktop text-center text-muted">
                                        No recent purchases
                                    </p>
                                </div>
                            </div>
                        @endforelse
                    </div>
                </div>
            </div>
        </div>
    @endif

    @if ($type === 'tablet')
        <div class="notification-icons-tablet">
            <!-- Notifications -->
            <div class="notification-icon-container-tablet">
                <button class="notification-btn-tablet notifications" id="tabletNotificationsBtn">
                    <svg class="notification-icon-tablet-svg" width="16" height="16" viewBox="0 0 24 24"
                        fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path
                            d="M12 2C10.896 2 10 2.896 10 4V5.586L7.707 8.293C7.512 8.488 7.256 8.584 7 8.584V10H17V8.584C16.744 8.584 16.488 8.488 16.293 8.293L14 5.586V4C14 2.896 13.104 2 12 2Z"
                            fill="currentColor" />
                        <path
                            d="M18 11V12C18 15.314 15.314 18 12 18C8.686 18 6 15.314 6 12V11H4V12C4 16.418 7.582 20 12 20C16.418 20 20 16.418 20 12V11H18Z"
                            fill="currentColor" />
                        <circle cx="17" cy="4" r="3" fill="#ff4757" />
                    </svg>
                    <span class="notification-badge-tablet" id="tabletNotificationsBadge"
                        style="display: {{ $unreadCount > 0 ? 'block' : 'none' }};">{{ $unreadCount > 99 ? '99+' : $unreadCount }}</span>
                </button>
                <div class="notification-dropdown-tablet notifications" id="tabletNotificationsDropdown">
                    <div class="notification-header-tablet">
                        <h4 data-translate="notifications.title">Notifications</h4>
                    </div>
                    <div class="notification-list-tablet" id="tabletNotificationsList">
                        @forelse($notifications as $notification)
                            <div class="notification-item-tablet {{ $notification->isRead ? 'read' : 'unread' }}"
                                wire:click="markAsRead({{ $notification->id }})" style="cursor: pointer;">
                                <div class="notification-icon-tablet">
                                    <i
                                        class="fas fa-bell {{ $notification->isRead ? 'text-muted' : ($notification->type === 'admin_message' ? 'text-info' : ($notification->type === 'charge' ? 'text-success' : 'text-warning')) }}"></i>
                                </div>
                                <div class="notification-content-tablet">
                                    <p class="notification-text-tablet">
                                        <strong>{{ $notification->translatedHead }}</strong><br>
                                        {!! $notification->translatedMessage !!}
                                        @if ($notification->link)
                                            <br><small><a href="{{ $notification->link }}" class="text-primary">View
                                                    More</a></small>
                                        @endif
                                    </p>
                                    <span class="notification-time-tablet">
                                        {{ $notification->created_at->diffForHumans() }}
                                        @if ($notification->expires_at)
                                            (Expires: {{ $notification->expires_at->diffForHumans() }})
                                        @endif
                                    </span>
                                </div>
                                @if (!$notification->isRead)
                                    <div class="notification-unread-indicator"></div>
                                @endif
                            </div>
                        @empty
                            <div class="notification-item-tablet">
                                <div class="notification-content-tablet">
                                    <p class="notification-text-tablet text-center text-muted">
                                        No new notifications
                                    </p>
                                </div>
                            </div>
                        @endforelse
                    </div>
                </div>
            </div>

            <!-- Admin Messages -->
            <div class="notification-icon-container-tablet">
                <button class="notification-btn-tablet admin-messages" id="tabletAdminMessagesBtn">
                    <svg class="notification-icon-tablet-svg" width="16" height="16" viewBox="0 0 24 24"
                        fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path
                            d="M20 4H4C2.896 4 2 4.896 2 6V18C2 19.104 2.896 20 4 20H20C21.104 20 22 19.104 22 18V6C22 4.896 21.104 4 20 4Z"
                            fill="currentColor" />
                        <path d="M22 6L12 13L2 6" stroke="currentColor" stroke-width="1.5" fill="none" />
                        <circle cx="19" cy="5" r="2.5" fill="#007bff" />
                        <path d="M19 3.5L20.5 5L19 6.5L17.5 5L19 3.5Z" fill="white" />
                    </svg>
                    <span class="notification-badge-tablet" id="tabletAdminMessagesBadge"
                        style="display: {{ $adminMessagesUnreadCount > 0 ? 'block' : 'none' }};">{{ $adminMessagesUnreadCount > 99 ? '99+' : $adminMessagesUnreadCount }}</span>
                </button>
                <div class="notification-dropdown-tablet admin-messages" id="tabletAdminMessagesDropdown">
                    <div class="notification-header-tablet">
                        <h4 data-translate="adminMessages.title">Admin Messages</h4>
                    </div>
                    <div class="notification-list-tablet" id="tabletAdminMessagesList">
                        @forelse($adminMessages as $notification)
                            <div class="notification-item-tablet {{ $notification->isRead ? 'read' : 'unread' }}"
                                wire:click="markAsRead({{ $notification->id }})" style="cursor: pointer;">
                                <div class="notification-icon-tablet">
                                    <i class="fas fa-shield-alt {{ $notification->isRead ? 'text-muted' : 'text-info' }}"></i>
                                </div>
                                <div class="notification-content-tablet">
                                    <p class="notification-text-tablet">
                                        <strong>{{ $notification->translatedHead }}</strong><br>
                                        {!! $notification->translatedMessage !!}
                                        @if ($notification->link)
                                            <br><small><a href="{{ $notification->link }}" class="text-primary">View
                                                    More</a></small>
                                        @endif
                                    </p>
                                    <span class="notification-time-tablet">
                                        {{ $notification->created_at->diffForHumans() }}
                                        @if ($notification->expires_at)
                                            (Expires: {{ $notification->expires_at->diffForHumans() }})
                                        @endif
                                    </span>
                                </div>
                                @if (!$notification->isRead)
                                    <div class="notification-unread-indicator"></div>
                                @endif
                            </div>
                        @empty
                            <div class="notification-item-tablet">
                                <div class="notification-content-tablet">
                                    <p class="notification-text-tablet text-center text-muted">
                                        No admin messages
                                    </p>
                                </div>
                            </div>
                        @endforelse
                    </div>
                </div>
            </div>

            <!-- Charges/Purchases -->
            <div class="notification-icon-container-tablet">
                <button class="notification-btn-tablet charges" id="tabletChargesBtn">
                    <svg class="notification-icon-tablet-svg" width="16" height="16" viewBox="0 0 24 24"
                        fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path
                            d="M7 4V2C7 1.448 7.448 1 8 1H16C16.552 1 17 1.448 17 2V4H20C20.552 4 21 4.448 21 5S20.552 6 20 6H19V19C19 20.104 18.104 21 17 21H7C5.896 21 5 20.104 5 19V6H4C3.448 6 3 5.552 3 5S3.448 4 4 4H7Z"
                            fill="currentColor" />
                        <path d="M9 9H11V17H9V9Z" fill="currentColor" />
                        <path d="M13 9H15V17H13V9Z" fill="currentColor" />
                        <circle cx="19" cy="4" r="2.5" fill="#2ed573" />
                        <path d="M17.5 2.5L18.5 3.5L20.5 1.5" stroke="white" stroke-width="1.5" fill="none" />
                    </svg>
                    <span class="notification-badge-tablet" id="tabletChargesBadge" style="display: {{ $chargesUnreadCount > 0 ? 'block' : 'none' }};">{{ $chargesUnreadCount > 99 ? '99+' : $chargesUnreadCount }}</span>
                </button>
                <div class="notification-dropdown-tablet charges" id="tabletChargesDropdown">
                    <div class="notification-header-tablet">
                        <h4 data-translate="charges.title">Recent Purchases</h4>
                    </div>
                    <div class="notification-list-tablet" id="tabletChargesList">
                        @forelse($charges as $notification)
                            <div class="notification-item-tablet {{ $notification->isRead ? 'read' : 'unread' }}"
                                wire:click="markAsRead({{ $notification->id }})" style="cursor: pointer;">
                                <div class="notification-icon-tablet">
                                    <i class="fas fa-credit-card {{ $notification->isRead ? 'text-muted' : 'text-success' }}"></i>
                                </div>
                                <div class="notification-content-tablet">
                                    <p class="notification-text-tablet">Purchase of 1000 Silk completed
                                        successfully.</p>
                                    <span class="notification-time-tablet">3 days ago</span>
                                </div>
                            </div>
                        @empty
                            <div class="notification-item-tablet">
                                <div class="notification-content-tablet">
                                    <p class="notification-text-tablet text-center text-muted">
                                        No recent purchases
                                    </p>
                                </div>
                            </div>
                        @endforelse
                    </div>
                </div>
            </div>
        </div>
    @endif

    @if ($type === 'mobile')
        <div class="mobile-header-notification-icons">
            <!-- Mobile Header Notifications -->
            <div class="mobile-header-notification-icon-container">
                <button class="mobile-header-notification-btn notifications" id="mobileHeaderNotificationsBtn">
                    <svg class="mobile-notification-icon" width="16" height="16" viewBox="0 0 24 24"
                        fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path
                            d="M12 2C10.896 2 10 2.896 10 4V5.586L7.707 8.293C7.512 8.488 7.256 8.584 7 8.584V10H17V8.584C16.744 8.584 16.488 8.488 16.293 8.293L14 5.586V4C14 2.896 13.104 2 12 2Z"
                            fill="currentColor" />
                        <path
                            d="M18 11V12C18 15.314 15.314 18 12 18C8.686 18 6 15.314 6 12V11H4V12C4 16.418 7.582 20 12 20C16.418 20 20 16.418 20 12V11H18Z"
                            fill="currentColor" />
                        <circle cx="17" cy="4" r="3" fill="#ff4757" />
                    </svg>
                    <span class="mobile-header-notification-badge" id="mobileHeaderNotificationsBadge"
                        style="display: {{ $unreadCount > 0 ? 'block' : 'none' }};">{{ $unreadCount > 99 ? '99+' : $unreadCount }}</span>
                </button>
                <!-- Mobile Header Notifications Dropdown -->
                <div class="mobile-header-notification-dropdown notifications" id="mobileHeaderNotificationsDropdown">
                    <div class="mobile-header-notification-header">
                        <h4 data-translate="notifications.title">Notifications</h4>
                    </div>
                    <div class="mobile-header-notification-list" id="mobileHeaderNotificationsList">
                        @forelse($notifications as $notification)
                            <div class="mobile-header-notification-item {{ $notification->isRead ? 'read' : 'unread' }}"
                                wire:click="markAsRead({{ $notification->id }})" style="cursor: pointer;">
                                <div class="mobile-header-notification-icon">
                                    <i
                                        class="fas fa-bell {{ $notification->isRead ? 'text-muted' : ($notification->type === 'admin_message' ? 'text-info' : ($notification->type === 'charge' ? 'text-success' : 'text-warning')) }}"></i>
                                </div>
                                <div class="mobile-header-notification-content">
                                    <p class="mobile-header-notification-text">
                                        <strong>{{ $notification->translatedHead }}</strong><br>
                                        {!! $notification->translatedMessage !!}
                                        @if ($notification->link)
                                            <br><small><a href="{{ $notification->link }}" class="text-primary">View
                                                    More</a></small>
                                        @endif
                                    </p>
                                    <span class="mobile-header-notification-time">
                                        {{ $notification->created_at->diffForHumans() }}
                                        @if ($notification->expires_at)
                                            (Expires: {{ $notification->expires_at->diffForHumans() }})
                                        @endif
                                    </span>
                                </div>
                                @if (!$notification->isRead)
                                    <div class="notification-unread-indicator"></div>
                                @endif
                            </div>
                        @empty
                            <div class="mobile-header-notification-item">
                                <div class="mobile-header-notification-content">
                                    <p class="mobile-header-notification-text text-center text-muted">
                                        No new notifications
                                    </p>
                                </div>
                            </div>
                        @endforelse
                    </div>
                </div>
            </div>

            <!-- Mobile Header Admin Messages -->
            <div class="mobile-header-notification-icon-container">
                <button class="mobile-header-notification-btn admin-messages" id="mobileHeaderAdminMessagesBtn">
                    <svg class="mobile-notification-icon" width="16" height="16" viewBox="0 0 24 24"
                        fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path
                            d="M20 4H4C2.896 4 2 4.896 2 6V18C2 19.104 2.896 20 4 20H20C21.104 20 22 19.104 22 18V6C22 4.896 21.104 4 20 4Z"
                            fill="currentColor" />
                        <path d="M22 6L12 13L2 6" stroke="currentColor" stroke-width="1.5" fill="none" />
                        <circle cx="19" cy="5" r="2.5" fill="#007bff" />
                        <path d="M19 3.5L20.5 5L19 6.5L17.5 5L19 3.5Z" fill="white" />
                    </svg>
                    <span class="mobile-header-notification-badge" id="mobileHeaderAdminMessagesBadge"
                        style="display: {{ $adminMessagesUnreadCount > 0 ? 'block' : 'none' }};">{{ $adminMessagesUnreadCount > 99 ? '99+' : $adminMessagesUnreadCount }}</span>
                </button>
                <!-- Mobile Header Admin Messages Dropdown -->
                <div class="mobile-header-notification-dropdown admin-messages"
                    id="mobileHeaderAdminMessagesDropdown">
                    <div class="mobile-header-notification-header">
                        <h4 data-translate="adminMessages.title">Admin Messages</h4>
                    </div>
                    <div class="mobile-header-notification-list" id="mobileHeaderAdminMessagesList">
                        @forelse($adminMessages as $notification)
                            <div class="mobile-header-notification-item {{ $notification->isRead ? 'read' : 'unread' }}"
                                wire:click="markAsRead({{ $notification->id }})" style="cursor: pointer;">
                                <div class="mobile-header-notification-icon">
                                    <i class="fas fa-shield-alt {{ $notification->isRead ? 'text-muted' : 'text-info' }}"></i>
                                </div>
                                <div class="mobile-header-notification-content">
                                    <p class="mobile-header-notification-text">
                                        <strong>{{ $notification->translatedHead }}</strong><br>
                                        {!! $notification->translatedMessage !!}
                                        @if ($notification->link)
                                            <br><small><a href="{{ $notification->link }}" class="text-primary">View
                                                    More</a></small>
                                        @endif
                                    </p>
                                    <span class="mobile-header-notification-time">
                                        {{ $notification->created_at->diffForHumans() }}
                                        @if ($notification->expires_at)
                                            (Expires: {{ $notification->expires_at->diffForHumans() }})
                                        @endif
                                    </span>
                                </div>
                                @if (!$notification->isRead)
                                    <div class="notification-unread-indicator"></div>
                                @endif
                            </div>
                        @empty
                            <div class="mobile-header-notification-item">
                                <div class="mobile-header-notification-content">
                                    <p class="mobile-header-notification-text text-center text-muted">
                                        No admin messages
                                    </p>
                                </div>
                            </div>
                        @endforelse
                    </div>
                </div>
            </div>

            <!-- Mobile Header Charges -->
            <div class="mobile-header-notification-icon-container">
                <button class="mobile-header-notification-btn charges" id="mobileHeaderChargesBtn">
                    <svg class="mobile-notification-icon" width="16" height="16" viewBox="0 0 24 24"
                        fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path
                            d="M7 4V2C7 1.448 7.448 1 8 1H16C16.552 1 17 1.448 17 2V4H20C20.552 4 21 4.448 21 5S20.552 6 20 6H19V19C19 20.104 18.104 21 17 21H7C5.896 21 5 20.104 5 19V6H4C3.448 6 3 5.552 3 5S3.448 4 4 4H7Z"
                            fill="currentColor" />
                        <path d="M9 9H11V17H9V9Z" fill="currentColor" />
                        <path d="M13 9H15V17H13V9Z" fill="currentColor" />
                        <circle cx="19" cy="4" r="2.5" fill="#2ed573" />
                        <path d="M17.5 2.5L18.5 3.5L20.5 1.5" stroke="white" stroke-width="1.5" fill="none" />
                    </svg>
                    <span class="mobile-header-notification-badge" id="mobileHeaderChargesBadge"
                        style="display: {{ $chargesUnreadCount > 0 ? 'block' : 'none' }};">{{ $chargesUnreadCount > 99 ? '99+' : $chargesUnreadCount }}</span>
                </button>
                <!-- Mobile Header Charges Dropdown -->
                <div class="mobile-header-notification-dropdown charges" id="mobileHeaderChargesDropdown">
                    <div class="mobile-header-notification-header">
                        <h4 data-translate="charges.title">Recent Purchases</h4>
                    </div>
                    <div class="mobile-header-notification-list" id="mobileHeaderChargesList">
                        @forelse($charges as $notification)
                            <div class="mobile-header-notification-item {{ $notification->isRead ? 'read' : 'unread' }}"
                                wire:click="markAsRead({{ $notification->id }})" style="cursor: pointer;">
                                <div class="mobile-header-notification-icon">
                                    <i class="fas fa-shopping-cart {{ $notification->isRead ? 'text-muted' : 'text-success' }}"></i>
                                </div>
                                <div class="mobile-header-notification-content">
                                    <p class="mobile-header-notification-text">
                                        <strong>{{ $notification->translatedHead }}</strong><br>
                                        {!! $notification->translatedMessage !!}
                                        @if ($notification->link)
                                            <br><small><a href="{{ $notification->link }}" class="text-primary">View
                                                    More</a></small>
                                        @endif
                                    </p>
                                    <span class="mobile-header-notification-time">
                                        {{ $notification->created_at->diffForHumans() }}
                                        @if ($notification->expires_at)
                                            (Expires: {{ $notification->expires_at->diffForHumans() }})
                                        @endif
                                    </span>
                                </div>
                                @if (!$notification->isRead)
                                    <div class="notification-unread-indicator"></div>
                                @endif
                            </div>
                        @empty
                            <div class="mobile-header-notification-item">
                                <div class="mobile-header-notification-content">
                                    <p class="mobile-header-notification-text text-center text-muted">
                                        No recent purchases
                                    </p>
                                </div>
                            </div>
                        @endforelse
                    </div>
                </div>
            </div>
        </div>
    @endif
</div>

<script>
    // استقبال أحداث التحديث من Livewire
    document.addEventListener('livewire:initialized', function () {
        // التحديث التلقائي كل 30 ثانية للإشعارات الجديدة
        setInterval(function() {
            if (typeof @this !== 'undefined') {
                @this.refresh();
            }
        }, 30000);

        // استقبال الأحداث المخصصة للتحديث
        Livewire.on('refresh-notifications', function() {
            if (typeof @this !== 'undefined') {
                @this.refresh();
            }
        });
    });

    // استقبال الأحداث من الخارج (من الـ Model)
    window.addEventListener('livewire:refresh-notifications', function(event) {
        if (typeof Livewire !== 'undefined') {
            Livewire.dispatch('refresh-notifications');
        }
    });
</script>
