<?php

use Livewire\Volt\Component;
use App\Models\Z_Website\News;
use Illuminate\Support\Facades\Storage;

new class extends Component {
    public $newsId = null;
    public $currentNews = null;

    public function mount($id = null)
    {
        $this->newsId = $id;

        if ($this->newsId) {
            $this->loadNewsDetail();
        }
    }

    public function loadNewsDetail()
    {
        try {
            $news = News::with('image')
                ->where('id', $this->newsId)
                ->whereNotNull('published_at')
                ->where('published_at', '<=', now())
                ->first();

            if ($news) {
                $imagePath = null;
                if ($news->image && $news->image->filename) {
                    $imagePath = asset('storage/web/images/' . $news->image->filename);
                }

                $this->currentNews = [
                    'id' => $news->id,
                    'title' => $news->title ?? 'Untitled News',
                    'body' => $news->body ?? '',
                    'published_at' => $news->published_at,
                    'image' => $imagePath,
                ];
            } else {
                // News not found or not published, redirect to news page
                return redirect()->route('news');
            }
        } catch (\Exception $e) {
            \Log::error('Error loading news detail: ' . $e->getMessage());
            return redirect()->route('news');
        }
    }

    public function loadNews()
    {
        try {
            return News::with('image')
                ->whereNotNull('published_at')
                ->where('published_at', '<=', now())
                ->orderBy('published_at', 'desc')
                ->get()
                ->map(function ($news) {
                    $imagePath = null;
                    if ($news->image && $news->image->filename) {
                        $imagePath = asset('storage/web/images/' . $news->image->filename);
                    }

                    return [
                        'id' => $news->id,
                        'title' => $news->title ?? 'Untitled News',
                        'body' => $news->body ?? '',
                        'published_at' => $news->published_at,
                        'image' => $imagePath,
                    ];
                });
        } catch (\Exception $e) {
            \Log::error('Error loading news: ' . $e->getMessage());
            return collect();
        }
    }

    public function getNewsProperty()
    {
        $news = $this->loadNews();
        return $news ?: collect();
    }
}; ?>
<div>
@if($newsId && $currentNews)
    <!-- News Detail View -->
    <div class="min-h-screen bg-gradient-to-br from-slate-900 via-purple-900 to-slate-900">
        <div class="container mx-auto px-6 py-16">
            <div class="max-w-4xl mx-auto">
                <!-- Back Button -->
                <div class="mb-8">
                    <a href="{{ route('news') }}" class="inline-flex items-center gap-2 px-6 py-3 bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white rounded-lg transition-all duration-300 font-medium shadow-lg hover:shadow-xl transform hover:scale-105">
                        ← <span data-translate="news.backToNews">Back to News</span>
                    </a>
                </div>

                <!-- News Article -->
                <article class="bg-gradient-to-br from-slate-800/50 to-purple-900/30 backdrop-blur-lg border border-purple-500/30 rounded-2xl overflow-hidden shadow-2xl hover:border-purple-400/50 transition-all duration-500 transform hover:scale-[1.02] animate-fade-in">
                    @if($currentNews['image'])
                    <div class="h-64 md:h-96 bg-cover bg-center relative" style="background-image: url('{{ $currentNews['image'] }}')">
                        <div class="absolute inset-0 bg-gradient-to-t from-purple-900/70 to-transparent"></div>
                    </div>
                    @else
                    <div class="h-64 md:h-96 bg-gradient-to-br from-purple-600 to-blue-600 flex items-center justify-center relative">
                        <div class="text-8xl">📰</div>
                        <div class="absolute inset-0 bg-gradient-to-t from-purple-900/30 to-transparent"></div>
                    </div>
                    @endif

                    <div class="p-8 md:p-12">
                        <div class="flex items-center gap-2 text-purple-300 text-sm mb-6">
                            <span>📅</span>
                            <span>{{ $currentNews['published_at'] ? \Carbon\Carbon::parse($currentNews['published_at'])->format('F d, Y \a\t H:i') : 'N/A' }}</span>
                        </div>

                        <div class="flex items-center gap-3 mb-8">
                            <div class="text-4xl">📰</div>
                            <h1 class="text-3xl md:text-4xl lg:text-5xl font-bold text-white leading-tight">
                                {{ $currentNews['title'] }}
                            </h1>
                        </div>

                        <div class="prose prose-lg prose-invert max-w-none">
                            <div class="text-purple-100 leading-relaxed text-lg md:text-xl whitespace-pre-line border-l-4 border-purple-500 pl-6 md:pl-8">
                                {!! ($currentNews['body'] ?? '') !!}
                            </div>
                        </div>

                        <!-- Back Button at the end -->
                        <div class="mt-12 text-center">
                            <a href="{{ route('news') }}" class="inline-flex items-center gap-2 px-8 py-4 bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white rounded-lg transition-all duration-300 font-medium text-lg shadow-lg hover:shadow-xl transform hover:scale-105">
                                ← <span data-translate="news.backToNews">Back to All News</span>
                            </a>
                        </div>
                    </div>
                </article>
            </div>
        </div>
    </div>
@else
    <!-- News List View -->
    <div class="min-h-screen bg-gradient-to-br from-slate-900 via-purple-900 to-slate-900">
        <!-- Latest News Section -->
        <div class="container mx-auto px-6 py-16">
            <div class="text-center mb-12">
                <h1 class="text-5xl font-bold text-white mb-4" data-translate="news.allNews">All News & Updates</h1>
                <div class="w-24 h-1 bg-gradient-to-r from-purple-500 to-cyan-500 mx-auto"></div>
            </div>

            <!-- News Grid -->
            <div class="space-y-12">
                @if($this->news && $this->news->count() > 0)
                @foreach($this->news as $news)
                @if($news && is_array($news))
                <article class="bg-gradient-to-br from-slate-800/50 to-purple-900/30 backdrop-blur-lg border border-purple-500/30 rounded-2xl overflow-hidden shadow-2xl hover:border-purple-400/50 transition-all duration-500 transform hover:scale-[1.02]">
                    @if($news['image'])
                    <div class="h-64 md:h-96 bg-cover bg-center relative" style="background-image: url('{{ $news['image'] }}')">
                        <div class="absolute inset-0 bg-gradient-to-t from-purple-900/70 to-transparent"></div>
                    </div>
                    @else
                    <div class="h-64 md:h-96 bg-gradient-to-br from-purple-600 to-blue-600 flex items-center justify-center relative">
                        <div class="text-8xl">📰</div>
                        <div class="absolute inset-0 bg-gradient-to-t from-purple-900/30 to-transparent"></div>
                    </div>
                    @endif

                    <div class="p-8 md:p-12">
                        <div class="flex items-center gap-2 text-purple-300 text-sm mb-6">
                            <span>📅</span>
                            <span>{{ $news['published_at'] ? \Carbon\Carbon::parse($news['published_at'])->format('F d, Y \a\t H:i') : 'N/A' }}</span>
                        </div>

                        <div class="flex items-center gap-3 mb-8">
                            <div class="text-4xl">📰</div>
                            <h2 class="text-3xl md:text-4xl font-bold text-white leading-tight">
                                {{ $news['title'] ?? 'Untitled News' }}
                            </h2>
                        </div>

                        <div class="prose prose-lg prose-invert max-w-none">
                            <div class="text-purple-100 leading-relaxed text-lg whitespace-pre-line border-l-4 border-purple-500 pl-6">
                                {!! nl2br(e($news['body'])) !!}
                            </div>
                        </div>

                        <div class="mt-8 text-center">
                            <a href="{{ route('news', ['id' => $news['id']]) }}" class="inline-flex items-center gap-2 px-6 py-3 bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white rounded-lg transition-all duration-300 font-medium shadow-lg hover:shadow-xl transform hover:scale-105">
                                <span data-translate="news.readMore">Read Full Article</span>
                                <span>→</span>
                            </a>
                        </div>
                    </div>
                </article>
                @endif
                @endforeach
                @else
                <div class="col-span-full text-center py-12">
                    <div class="text-6xl mb-4">📰</div>
                    <h3 class="text-xl font-bold text-white mb-2" data-translate="news.noNews">No News Available</h3>
                    <p class="text-purple-300" data-translate="news.checkBackLater">Please check back later for updates.</p>
                </div>
                @endif
            </div>

            <!-- Back to Home Button -->
            <div class="text-center">
                <a href="{{ route('home') }}" class="px-8 py-4 bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white rounded-lg transition-all duration-300 font-medium text-lg shadow-lg hover:shadow-xl">
                    ← <span data-translate="news.backToHome">Back to Home</span>
                </a>
            </div>
        </div>
    </div>
@endif

<style>
.line-clamp-2 {
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
}

.line-clamp-3 {
    display: -webkit-box;
    -webkit-line-clamp: 3;
    -webkit-box-orient: vertical;
    overflow: hidden;
}

.prose {
    color: rgb(241 245 249);
}

.prose strong {
    color: rgb(196 181 253);
    font-weight: 600;
}

.prose a {
    color: rgb(147 197 253);
    text-decoration: underline;
}

.prose a:hover {
    color: rgb(96 165 250);
}

/* Fade-in animation */
.animate-fade-in {
    animation: fadeIn 1s ease-in-out;
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Language switcher loading states */
.lang-loading {
    opacity: 0.6;
    pointer-events: none;
}

.lang-success {
    transform: scale(1.05);
    transition: transform 0.3s ease;
}

/* Direction transition */
.direction-transition {
    transition: all 0.6s cubic-bezier(0.4, 0, 0.2, 1);
}

/* RTL support */
[dir="rtl"] {
    text-align: right;
}

[dir="rtl"] .flex {
    flex-direction: row-reverse;
}

[dir="rtl"] .justify-between {
    flex-direction: row-reverse;
}
</style>
</div>
