<?php

use Livewire\Volt\Component;
use App\Models\Z_Website\News;

new class extends Component {

    public function mount()
    {
        // No need for mount since we're using pagination
    }

    public function loadNews()
    {
        try {
            return News::with('image')
                ->whereNotNull('published_at')
                ->where('published_at', '<=', now())
                ->orderBy('published_at', 'desc')
                ->get()
                ->map(function ($news) {
                    $imagePath = null;
                    if ($news->image && $news->image->filename && file_exists(public_path('storage/web/images/' . $news->image->filename))) {
                        $imagePath = asset('storage/web/images/' . $news->image->filename);
                    }

                    return [
                        'id' => $news->id,
                        'title' => $news->title ?? 'Untitled News',
                        'body' => strip_tags($news->body ?? ''),
                        'excerpt' => \Str::limit(strip_tags($news->body ?? ''), 120),
                        'published_at' => $news->published_at,
                        'image' => $imagePath,
                    ];
                });
        } catch (\Exception $e) {
            \Log::error('Error loading news: ' . $e->getMessage());
            return collect();
        }
    }

    public function getNewsProperty()
    {
        $news = $this->loadNews();
        return $news ?: collect();
    }
};
?>

<div class="min-h-screen bg-gradient-to-br from-slate-900 via-purple-900 to-slate-900">

<!-- Latest News Section -->
<div class="container mx-auto px-6 py-16">
    <div class="text-center mb-12">
        <h1 class="text-5xl font-bold text-white mb-4" data-translate="news.allNews">All News & Updates</h1>
        <div class="w-24 h-1 bg-gradient-to-r from-purple-500 to-cyan-500 mx-auto"></div>
    </div>

    <!-- News Grid -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mb-12">
        @if($this->news && $this->news->count() > 0)
        @foreach($this->news as $news)
        @if($news && is_array($news))
        <article class="bg-gradient-to-br from-slate-800/50 to-purple-900/30 backdrop-blur-lg border border-purple-500/30 rounded-2xl overflow-hidden shadow-2xl hover:border-purple-400/50 transition-all duration-300 hover:scale-105">
            @if($news['image'])
            <div class="h-48 bg-cover bg-center relative" style="background-image: url('{{ $news['image'] }}')">
                <div class="absolute inset-0 bg-gradient-to-t from-purple-900/50 to-transparent"></div>
            </div>
            @else
            <div class="h-48 bg-gradient-to-br from-purple-600 to-blue-600 flex items-center justify-center relative">
                <div class="text-6xl">📰</div>
                <div class="absolute inset-0 bg-gradient-to-t from-purple-900/30 to-transparent"></div>
            </div>
            @endif

            <div class="p-6">
                <div class="flex items-center gap-2 text-purple-300 text-sm mb-3">
                    <span>📅</span>
                    <span>{{ $news['published_at'] ? \Carbon\Carbon::parse($news['published_at'])->format('M d, Y') : 'N/A' }}</span>
                </div>

                <h3 class="text-xl font-bold text-white mb-3 line-clamp-2">
                    {{ $news['title'] ?? 'Untitled News' }}
                </h3>

                <p class="text-purple-200 text-sm leading-relaxed mb-4 line-clamp-3">
                    {{ $news['excerpt'] ?? 'No content available.' }}
                </p>

                <a href="{{ route('news', ['id' => $news['id']]) }}" class="inline-flex items-center gap-2 px-4 py-2 bg-purple-600 hover:bg-purple-700 text-white rounded-lg transition-colors text-sm font-medium">
                    <span data-translate="news.readMore">Read More</span>
                    <span>→</span>
                </a>
            </div>
        </article>
        @endif
        @endforeach
        @else
        <div class="col-span-full text-center py-12">
            <div class="text-6xl mb-4">📰</div>
            <h3 class="text-xl font-bold text-white mb-2" data-translate="news.noNews">No News Available</h3>
            <p class="text-purple-300" data-translate="news.checkBackLater">Please check back later for updates.</p>
        </div>
        @endif
    </div>

    <!-- Back to Home Button -->
    <div class="text-center">
        <a href="{{ route('home') }}" class="px-8 py-4 bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white rounded-lg transition-all duration-300 font-medium text-lg shadow-lg hover:shadow-xl">
            ← <span data-translate="news.backToHome">Back to Home</span>
        </a>
    </div>
</div>


</div>


