<?php

use Livewire\Volt\Component;
use App\Models\SRO_VT_PROXY\_Players;
use App\Models\Z_Website\SiteSettings;
use Illuminate\Support\Facades\Cache;

new class extends Component {
    public $onlinePlayers = 0;
    public $totalPlayers = 0;
    public $fakeCount = 0;
    public $maxCapacity = 5000;
    public $isLoading = true;

    public function mount()
    {
        $this->updatePlayerCount();
        $this->isLoading = false;
    }

    public function updatePlayerCount()
    {
        try {
            // Get settings including max capacity
            $settings = Cache::remember('site_settings', 10, function () {
                return SiteSettings::first();
            });

            // Get max capacity from settings
            if ($settings && isset($settings->settings['sro_max_server'])) {
                $this->maxCapacity = (int) $settings->settings['sro_max_server'];
            }

            // Get online players count with shorter cache for more real-time feel
            $this->onlinePlayers = Cache::remember('online_players_count', 15, function () {
                return _Players::where('cur_status', 1)->count();
            });

            // Get fake count from site settings with much shorter cache (5 seconds for more responsive updates)
            $this->fakeCount = Cache::remember('fake_players_count', 5, function () use ($settings) {
                if (!$settings) {
                    $settings = SiteSettings::first();
                }
                return $settings ? (int) $settings->fake : 0;
            });

            // Calculate total displayed players
            $this->totalPlayers = $this->onlinePlayers + $this->fakeCount;
        } catch (\Exception $e) {
            // Fallback values if database connection fails
            $this->onlinePlayers = 0;
            $this->fakeCount = rand(50, 150);
            $this->totalPlayers = $this->fakeCount;
            $this->maxCapacity = 5000; // fallback capacity
        }
    }

    // Method to force refresh the cache (useful for admin updates)
    public function forceRefresh()
    {
        Cache::forget('fake_players_count');
        Cache::forget('online_players_count');
        Cache::forget('site_settings');
        $this->updatePlayerCount();
    }

    public function getFormattedTotalPlayersProperty()
    {
        return number_format($this->totalPlayers);
    }
}; ?>
<div>
<div class="mini-gaming-player-counter @if($totalPlayers == 0) game-offline @endif">
    <!-- Mini Counter Display -->
    <div class="mini-counter-display">
        <div class="mini-digital-counter">
            @php
                $percentage = $maxCapacity > 0 ? min(($totalPlayers / $maxCapacity) * 100, 100) : 0;
            @endphp

            <!-- Mini Progress Border -->
            <div class="mini-progress-border capacity-@if($percentage < 70)low @elseif($percentage < 99)medium @else high @endif" style="--progress: {{ $percentage }}">
                <div class="mini-digit-row">
                    @php
                        $digits = str_split(str_pad($totalPlayers, 3, '0', STR_PAD_LEFT));
                    @endphp
                    @foreach($digits as $digit)
                        <div class="mini-digit @if($percentage > 25) filled @endif">{{ $digit }}</div>
                    @endforeach
                </div>
            </div>
        </div>

        <!-- Mini Counter Label -->
        <div class="mini-counter-label">
            @if($totalPlayers > 0)
                <h3  wire:ignore  data-translate="hero.gameOnline" >GAME ONLINE</h3>
            @else
                <h3 wire:ignore data-translate="hero.gameOffline" >GAME OFFLINE</h3>
            @endif
        </div>
    </div>

</div>



    <style>
        /* Beautiful Mini Gaming Player Counter */
        .mini-gaming-player-counter {
            background: linear-gradient(145deg,
                rgba(10, 10, 15, 0.98),
                rgba(26, 26, 46, 0.95),
                rgba(59, 130, 246, 0.05));
            border: 2px solid rgba(168, 85, 247, 0.4);
            border-radius: 15px;
            padding: 1rem;
            backdrop-filter: blur(15px);
            box-shadow:
                0 15px 35px rgba(0, 0, 0, 0.4),
                0 0 25px rgba(168, 85, 247, 0.15),
                inset 0 1px 0 rgba(255, 255, 255, 0.1);
            min-width: 200px;
            text-align: center;
            position: relative;
            overflow: hidden;
            transition: all 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
        }

        .mini-gaming-player-counter::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: conic-gradient(
                from 0deg,
                transparent,
                rgba(59, 130, 246, 0.1),
                transparent,
                rgba(147, 51, 234, 0.1),
                transparent
            );
            animation: miniRotate 20s linear infinite;
            z-index: -1;
        }

        .mini-gaming-player-counter:hover {
            transform: translateY(-3px) scale(1.02);
            box-shadow:
                0 20px 45px rgba(0, 0, 0, 0.5),
                0 0 35px rgba(168, 85, 247, 0.25),
                inset 0 1px 0 rgba(255, 255, 255, 0.15);
        }

        .mini-gaming-player-counter.game-offline {
            border-color: rgba(239, 68, 68, 0.4);
            background: linear-gradient(145deg,
                rgba(42, 15, 15, 0.98),
                rgba(59, 30, 30, 0.95),
                rgba(239, 68, 68, 0.05));
            box-shadow:
                0 15px 35px rgba(0, 0, 0, 0.4),
                0 0 25px rgba(239, 68, 68, 0.15),
                inset 0 1px 0 rgba(255, 255, 255, 0.1);
        }

        .mini-gaming-player-counter.game-offline::before {
            background: conic-gradient(
                from 0deg,
                transparent,
                rgba(239, 68, 68, 0.1),
                transparent,
                rgba(239, 68, 68, 0.1),
                transparent
            );
        }

        .mini-counter-display {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 0.75rem;
        }

        .mini-digital-counter {
            background: linear-gradient(135deg,
                rgba(10, 10, 15, 0.9),
                rgba(26, 26, 46, 0.9),
                rgba(59, 130, 246, 0.05));
            border-radius: 12px;
            padding: 0.75rem;
            direction: ltr;
            position: relative;
            overflow: hidden;
            box-shadow:
                inset 0 2px 10px rgba(0, 0, 0, 0.4),
                0 0 15px rgba(168, 85, 247, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.05);
        }

        .mini-progress-border {
            position: relative;
            border: 2px solid rgba(168, 85, 247, 0.5);
            border-radius: 10px;
            padding: 0.75rem;
            background: rgba(10, 10, 15, 0.2);
            backdrop-filter: blur(10px);
            transition: all 0.8s cubic-bezier(0.4, 0, 0.2, 1);
            overflow: hidden;
            box-shadow:
                inset 0 2px 10px rgba(0, 0, 0, 0.3),
                0 0 20px rgba(168, 85, 247, 0.15);
        }

        /* Capacity colors with enhanced effects */
        .mini-progress-border.capacity-low {
            border-color: rgba(57, 255, 20, 0.5);
            box-shadow:
                inset 0 2px 10px rgba(0, 0, 0, 0.3),
                0 0 20px rgba(57, 255, 20, 0.15);
        }

        .mini-progress-border.capacity-medium {
            border-color: rgba(255, 165, 0, 0.5);
            box-shadow:
                inset 0 2px 10px rgba(0, 0, 0, 0.3),
                0 0 20px rgba(255, 165, 0, 0.15);
        }

        .mini-progress-border.capacity-high {
            border-color: rgba(255, 68, 68, 0.5);
            box-shadow:
                inset 0 2px 10px rgba(0, 0, 0, 0.3),
                0 0 20px rgba(255, 68, 68, 0.15);
        }

        .mini-progress-border::before {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            height: calc(var(--progress) * 1%);
            background: linear-gradient(180deg,
                rgba(168, 85, 247, 0.7) 0%,
                rgba(168, 85, 247, 0.5) 50%,
                rgba(168, 85, 247, 0.8) 100%);
            border-radius: 0 0 8px 8px;
            transition: height 1.2s cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 1;
            animation: miniLiquidWave 4s ease-in-out infinite;
        }

        .mini-progress-border.capacity-low::before {
            background: linear-gradient(180deg,
                rgba(57, 255, 20, 0.7) 0%,
                rgba(57, 255, 20, 0.5) 50%,
                rgba(57, 255, 20, 0.8) 100%);
        }

        .mini-progress-border.capacity-medium::before {
            background: linear-gradient(180deg,
                rgba(255, 165, 0, 0.7) 0%,
                rgba(255, 165, 0, 0.5) 50%,
                rgba(255, 165, 0, 0.8) 100%);
        }

        .mini-progress-border.capacity-high::before {
            background: linear-gradient(180deg,
                rgba(255, 68, 68, 0.7) 0%,
                rgba(255, 68, 68, 0.5) 50%,
                rgba(255, 68, 68, 0.8) 100%);
        }

        .mini-digit-row {
            position: relative;
            z-index: 2;
            display: flex;
            gap: 0.3rem;
            justify-content: center;
        }

        .mini-digit {
            font-family: 'Orbitron', monospace;
            font-size: 2.2rem;
            font-weight: 800;
            min-width: 0.9ch;
            text-align: center;
            position: relative;
            transition: all 0.8s ease;
            color: rgba(168, 85, 247, 0.4);
            text-shadow: 0 0 5px rgba(168, 85, 247, 0.2);
            filter: drop-shadow(0 0 3px rgba(168, 85, 247, 0.1));
        }

        .mini-digit.filled {
            color: #ffffff;
            text-shadow:
                0 0 8px rgba(168, 85, 247, 0.8),
                0 0 15px rgba(168, 85, 247, 0.6),
                0 0 25px rgba(168, 85, 247, 0.4);
            animation: miniDigitGlow 3s ease-in-out infinite alternate;
        }

        .mini-progress-border.capacity-low .mini-digit.filled {
            text-shadow:
                0 0 8px rgba(57, 255, 20, 0.8),
                0 0 15px rgba(57, 255, 20, 0.6),
                0 0 25px rgba(57, 255, 20, 0.4);
        }

        .mini-progress-border.capacity-medium .mini-digit.filled {
            text-shadow:
                0 0 8px rgba(255, 165, 0, 0.8),
                0 0 15px rgba(255, 165, 0, 0.6),
                0 0 25px rgba(255, 165, 0, 0.4);
        }

        .mini-progress-border.capacity-high .mini-digit.filled {
            text-shadow:
                0 0 8px rgba(255, 68, 68, 0.8),
                0 0 15px rgba(255, 68, 68, 0.6),
                0 0 25px rgba(255, 68, 68, 0.4);
        }

        .mini-counter-label {
            font-size: 0.8rem;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 1.5px;
            color: rgba(168, 85, 247, 0.9);
            text-shadow: 0 0 5px rgba(168, 85, 247, 0.3);
            background: rgba(168, 85, 247, 0.1);
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            border: 1px solid rgba(168, 85, 247, 0.2);
            backdrop-filter: blur(5px);
            transition: all 0.3s ease;
        }

        .mini-counter-label:hover {
            background: rgba(168, 85, 247, 0.2);
            transform: scale(1.05);
        }

        .mini-gaming-player-counter.game-offline .mini-counter-label {
            color: rgba(239, 68, 68, 0.9);
            text-shadow: 0 0 5px rgba(239, 68, 68, 0.3);
            background: rgba(239, 68, 68, 0.1);
            border-color: rgba(239, 68, 68, 0.2);
        }

        /* Animations */
        @keyframes miniRotate {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        @keyframes miniLiquidWave {
            0%, 100% {
                transform: translateX(0px) scaleY(1);
                filter: hue-rotate(0deg);
            }
            25% {
                transform: translateX(-2px) scaleY(1.1);
                filter: hue-rotate(10deg);
            }
            50% {
                transform: translateX(0px) scaleY(1);
                filter: hue-rotate(20deg);
            }
            75% {
                transform: translateX(2px) scaleY(1.1);
                filter: hue-rotate(10deg);
            }
        }

        @keyframes miniDigitGlow {
            0% {
                text-shadow:
                    0 0 8px rgba(168, 85, 247, 0.8),
                    0 0 15px rgba(168, 85, 247, 0.6),
                    0 0 25px rgba(168, 85, 247, 0.4);
            }
            100% {
                text-shadow:
                    0 0 12px rgba(168, 85, 247, 1),
                    0 0 20px rgba(168, 85, 247, 0.8),
                    0 0 30px rgba(168, 85, 247, 0.6);
            }
        }

        /* Mobile Responsive */
        @media (max-width: 768px) {
            .mini-gaming-player-counter {
                padding: 0.75rem;
                min-width: 180px;
            }

            .mini-digit {
                font-size: 1.8rem;
            }

            .mini-counter-label {
                font-size: 0.7rem;
                padding: 0.2rem 0.6rem;
            }
        }

        @media (max-width: 480px) {
            .mini-gaming-player-counter {
                padding: 0.5rem;
                min-width: 160px;
            }

            .mini-digit {
                font-size: 1.6rem;
            }

            .mini-counter-label {
                font-size: 0.6rem;
                padding: 0.15rem 0.5rem;
            }
        }
    </style>

</div>
