<?php

use Livewire\Volt\Component;
use App\Models\GameInfo;
use Livewire\Attributes\Url;

new class extends Component {
    public $categories = [];
    public $categoryCounts = [];

    #[Url(as: 'category')]
    public $selectedCategory = null;

    public $gameInfos = [];

    #[Url(as: 'guide')]
    public $selectedGuide = null;

    #[Url(as: 'search')]
    public $searchTerm = '';

    public function mount()
    {
        $this->loadCategories();

        // Set default category if not set from URL
        if (!$this->selectedCategory) {
            $this->selectedCategory = 'All';
        }

        $this->loadAllGuides();
    }

    // Watch for URL parameter changes
    public function updatedSelectedCategory()
    {
        $this->selectedGuide = null;
        $this->searchTerm = '';
        $this->loadAllGuides();
    }

    public function updatedSelectedGuide()
    {
        // Guide is handled by property, no additional logic needed
    }

    public function loadCategories()
    {
        $this->categories = collect(['All'])->merge(GameInfo::getCategories());
        $this->categoryCounts = GameInfo::getCategoriesWithCounts();
        $this->categoryCounts['All'] = GameInfo::active()->where('id', '!=', 0)->count();
    }

    public function loadAllGuides()
    {
        $query = GameInfo::active()->where('id', '!=', 0)->orderBy('sort')->orderBy('created_at', 'desc');

        if ($this->selectedCategory && $this->selectedCategory !== 'All') {
            $query->byCategory($this->selectedCategory);
        }

        if ($this->searchTerm) {
            $query->where(function($q) {
                $q->where('title', 'like', '%' . $this->searchTerm . '%')
                  ->orWhere('body', 'like', '%' . $this->searchTerm . '%');
            });
        }

        $this->gameInfos = $query->get();
    }

    public function selectCategory($category)
    {
        $this->selectedCategory = $category;
        $this->selectedGuide = null;
        $this->searchTerm = '';
        $this->loadAllGuides();
    }

    public function viewGuide($guideId)
    {
        $guide = GameInfo::find($guideId);
        if ($guide) {
            $this->selectedGuide = $guideId; // Store ID instead of object for URL
        }
    }

    public function backToCategory()
    {
        $this->selectedGuide = null;
    }

    public function updatedSearchTerm()
    {
        $this->loadAllGuides();
    }

    // Get the selected guide object
    public function getSelectedGuideObjectProperty()
    {
        if ($this->selectedGuide) {
            return GameInfo::find($this->selectedGuide);
        }
        return null;
    }

    // Process content to handle img[link] and vid[link]
    public function processContent($content)
    {
        // Process img[link] patterns
        $content = preg_replace_callback('/img\[(.*?)\]/', function($matches) {
            $imageUrl = trim($matches[1]);
            return $this->generateImageHTML($imageUrl);
        }, $content);

        // Process vid[link] patterns
        $content = preg_replace_callback('/vid\[(.*?)\]/', function($matches) {
            $videoUrl = trim($matches[1]);
            return $this->generateVideoHTML($videoUrl);
        }, $content);

        return $content;
    }

    // Generate image HTML
    private function generateImageHTML($imageUrl)
    {
        if (empty($imageUrl)) {
            return '<p class="text-red-400 italic">⚠️ Image URL is empty</p>';
        }

        return '<div class="game-content-media-wrapper">
                    <div class="game-content-image-container">
                        <img src="' . htmlspecialchars($imageUrl) . '"  alt="Guide Image"  class="game-content-image"  loading="lazy" onerror="this.src=\'/assets/images/default-game-guide.png\';">
                        <div class="game-content-image-overlay">
                            <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-white">
                                <polyline points="15,3 21,3 21,9"/>
                                <polyline points="9,21 3,21 3,15"/>
                                <line x1="21" y1="3" x2="14" y2="10"/>
                                <line x1="3" y1="21" x2="10" y2="14"/>
                            </svg>
                        </div>
                    </div>
                </div>';
    }

    // Generate video HTML
    private function generateVideoHTML($videoUrl)
    {
        if (empty($videoUrl)) {
            return '<p class="text-red-400 italic">⚠️ Video URL is empty</p>';
        }

        // Check if it's a full iframe embed code from YouTube
        if (strpos($videoUrl, '<iframe') !== false && strpos($videoUrl, 'youtube.com/embed/') !== false) {
            if (preg_match('/src="([^"]*youtube\.com\/embed\/[^"]*)"/', $videoUrl, $iframeMatches)) {
                $videoUrl = $iframeMatches[1];
            }
        }

        // Process YouTube URLs
        if (strpos($videoUrl, 'youtube.com') !== false || strpos($videoUrl, 'youtu.be') !== false) {
            return $this->generateYouTubeHTML($videoUrl);
        }

        // Process Vimeo URLs
        if (strpos($videoUrl, 'vimeo.com') !== false) {
            return $this->generateVimeoHTML($videoUrl);
        }

        // Process other video formats
        return $this->generateGenericVideoHTML($videoUrl);
    }

    // Generate YouTube video HTML
    private function generateYouTubeHTML($videoUrl)
    {
        $videoId = null;
        $siParam = null;
        $originalUrl = $videoUrl;

        // Extract video ID and si parameter
        if (strpos($videoUrl, 'youtube.com/embed/') !== false) {
            // Embed URL
            if (preg_match('/youtube\.com\/embed\/([^?&\/]+)(?:.*?[\?&]si=([^&]+))?/', $videoUrl, $matches)) {
                $videoId = $matches[1];
                $siParam = isset($matches[2]) ? $matches[2] : null;
            }
        } elseif (preg_match('/(?:youtube\.com\/(?:watch\?v=|v\/)|youtu\.be\/)([^"&?\/\s]{11})/', $videoUrl, $matches)) {
            // Regular YouTube URL
            $videoId = $matches[1];
            if (preg_match('/[\?&]si=([^&]+)/', $videoUrl, $siMatches)) {
                $siParam = $siMatches[1];
            }
            $originalUrl = 'https://www.youtube.com/watch?v=' . $videoId;
        }

        if (!$videoId) {
            return '<p class="text-red-400 italic">⚠️ Invalid YouTube URL</p>';
        }

        // Build embed URL with parameters
        $params = [];
        if ($siParam) {
            $params[] = 'si=' . urlencode($siParam);
        }
        $params[] = 'rel=0';
        $params[] = 'modestbranding=1';
        $params[] = 'playsinline=1';

        $embedUrl = 'https://www.youtube.com/embed/' . $videoId . '?' . implode('&', $params);

        return $this->generateVideoContainer($embedUrl, $originalUrl, 'YouTube');
    }

    // Generate Vimeo video HTML
    private function generateVimeoHTML($videoUrl)
    {
        if (preg_match('/vimeo\.com\/(?:video\/)?(\d+)/', $videoUrl, $matches)) {
            $videoId = $matches[1];
            $embedUrl = 'https://player.vimeo.com/video/' . $videoId;
            return $this->generateVideoContainer($embedUrl, $videoUrl, 'Vimeo');
        }

        return '<p class="text-amber-400 italic">⚠️ Invalid Vimeo URL</p>';
    }

    // Generate generic video HTML
    private function generateGenericVideoHTML($videoUrl)
    {
        $fileExtension = strtolower(pathinfo($videoUrl, PATHINFO_EXTENSION));

        if (in_array($fileExtension, ['mp4', 'webm', 'ogg', 'avi', 'mov'])) {
            return '<div class="game-content-media-wrapper">
                        <div class="game-content-video-container">
                            <video class="game-content-video" controls preload="metadata">
                                <source src="' . htmlspecialchars($videoUrl) . '" type="video/' . $fileExtension . '">
                                Your browser does not support the video tag.
                            </video>
                        </div>
                    </div>';
        }

        return '<p class="text-amber-400 italic">⚠️ Unsupported video format. Supported: YouTube, Vimeo, MP4, WebM, OGG</p>';
    }

    // Generate video container with fallback
    private function generateVideoContainer($embedUrl, $originalUrl, $platform = 'YouTube')
    {
        $platformIcon = $platform === 'YouTube' ? '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                <path d="M23.498 6.186a3.016 3.016 0 0 0-2.122-2.136C19.505 3.545 12 3.545 12 3.545s-7.505 0-9.377.505A3.017 3.017 0 0 0 .502 6.186C0 8.07 0 12 0 12s0 3.93.502 5.814a3.016 3.016 0 0 0 2.122 2.136c1.871.505 9.376.505 9.376.505s7.505 0 9.377-.505a3.015 3.015 0 0 0 2.122-2.136C24 15.93 24 12 24 12s0-3.93-.502-5.814zM9.545 15.568V8.432L15.818 12l-6.273 3.568z"/>
            </svg>' : '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                <path d="M23.977 6.416c-.105 2.338-1.382 5.981-2.357 8.564L19.965 19c-.967.681-2.7.681-3.668 0l-.319-.319c-.681-.681-.681-1.794 0-2.475l1.334-1.334c.681-.681 1.794-.681 2.475 0 .681.681.681 1.794 0 2.475l-.637.637c-.681.681-1.794.681-2.475 0-.681-.681-.681-1.794 0-2.475L18.009 13.17c.681-.681 1.794-.681 2.475 0 .681.681.681 1.794 0 2.475l-.319.319c-.681.681-1.794.681-2.475 0-.681-.681-.681-1.794 0-2.475l1.016-1.016c.681-.681 1.794-.681 2.475 0 .681.681.681 1.794 0 2.475l-.319.319c-.681.681-1.794.681-2.475 0-.681-.681-.681-1.794 0-2.475l.319-.319c.681-.681 1.794-.681 2.475 0 .681.681.681 1.794 0 2.475l-.319.319c-.681.681-1.794.681-2.475 0-.681-.681-.681-1.794 0-2.475l.319-.319z"/>
            </svg>';
        $platformColor = $platform === 'YouTube' ? 'bg-red-600 hover:bg-red-500' : 'bg-blue-600 hover:bg-blue-500';

        return '<div class="game-content-media-wrapper">
                    <div class="game-content-video-container">
                        <iframe  class="game-content-video" width="560" height="315" src="' . htmlspecialchars($embedUrl) . '" title="' . $platform . ' video player" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" referrerpolicy="strict-origin-when-cross-origin" allowfullscreen></iframe>
                        <!-- Fallback link -->
                        <div class="video-fallback" style="display: none;">
                            <div class="text-center p-8 bg-gray-800/50 rounded-lg">
                                <i class="' . $platformIcon . ' text-red-500 text-4xl mb-4"></i>
                                <p class="text-white mb-4">Unable to load video</p>
                                <a href="' . htmlspecialchars($originalUrl) . '"
                                   target="_blank"
                                   rel="noopener noreferrer"
                                   class="inline-flex items-center px-4 py-2 ' . $platformColor . ' text-white rounded-lg transition-colors">
                                    <i class="' . $platformIcon . ' mr-2"></i>
                                    Watch on ' . $platform . '
                                </a>
                            </div>
                        </div>
                    </div>
                </div>';
    }
};
?>
<div>
<div class="game-info-container">
    <div class="container mx-auto px-6 py-8">

        @if (!$this->selectedGuideObject)
            <!-- Header Section -->
            <div class="mb-8 text-center">
                <div class="game-info-header rounded-2xl p-8 shadow-2xl">
                    <!-- Content -->
                    <div class="relative z-10">
                        <h1 class="game-info-title mb-4" data-translate="gameInfo.title">
                            🎮 Game Info & Guides
                        </h1>
                        <p class="text-xl text-gray-300 max-w-3xl mx-auto mb-6" data-translate="gameInfo.subtitle">
                            Discover comprehensive guides, tips, and essential information to master your gaming experience
                        </p>

                        <!-- Search Bar -->
                        <div class="game-info-search">
                            <input
                                type="text"
                                wire:model.live.debounce.300ms="searchTerm"
                                placeholder="Search guides..."
                                data-translate-placeholder="gameInfo.searchPlaceholder"
                            >
                            <svg class="game-info-search-icon w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                            </svg>

                            <!-- Loading indicator -->
                            <div wire:loading wire:target="searchTerm" class="absolute inset-y-0 right-12 flex items-center">
                                <div class="animate-spin h-4 w-4 border-2 border-blue-500 border-t-transparent rounded-full"></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Main Content Grid -->
            <div class="grid grid-cols-1 lg:grid-cols-4 gap-8">

                <!-- Sidebar - Categories -->
                <div class="lg:col-span-1">
                    <div class="game-info-sidebar">
                        <h3>
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-blue-400 inline">
                                <line x1="8" y1="6" x2="21" y2="6"/>
                                <line x1="8" y1="12" x2="21" y2="12"/>
                                <line x1="8" y1="18" x2="21" y2="18"/>
                                <line x1="3" y1="6" x2="3.01" y2="6"/>
                                <line x1="3" y1="12" x2="3.01" y2="12"/>
                                <line x1="3" y1="18" x2="3.01" y2="18"/>
                            </svg>
                            <span data-translate="gameInfo.categories">Categories</span>
                        </h3>

                        <div class="space-y-2">
                            @foreach ($categories as $category)
                                <button
                                    wire:click="selectCategory('{{ $category }}')"
                                    class="game-info-category-btn {{ $selectedCategory === $category ? 'active' : '' }}"
                                    data-category="{{ $category }}"
                                    wire:loading.class="opacity-50"
                                    wire:target="selectCategory('{{ $category }}')"
                                >
                                    <span>
                                        @if($category === 'All')
                                            🌟 <span data-translate="gameInfo.all">All</span>
                                        @else
                                            {{ $category }}
                                        @endif
                                    </span>
                                    <span class="game-info-category-count">
                                        {{ $categoryCounts[$category] ?? 0 }}
                                    </span>

                                    <!-- Loading indicator for category -->
                                    <div wire:loading wire:target="selectCategory('{{ $category }}')" class="absolute inset-0 flex items-center justify-center bg-gray-900/50 rounded-xl">
                                        <div class="animate-spin h-4 w-4 border-2 border-blue-500 border-t-transparent rounded-full"></div>
                                    </div>
                                </button>
                            @endforeach
                        </div>
                    </div>
                </div>

                <!-- Main Content - Guides List -->
                <div class="lg:col-span-3">
                    <!-- Loading overlay -->
                    <div wire:loading wire:target="selectCategory,searchTerm" class="absolute inset-0 bg-gray-900/50 backdrop-blur-sm rounded-2xl flex items-center justify-center z-50">
                        <div class="text-center">
                            <div class="animate-spin h-8 w-8 border-4 border-blue-500 border-t-transparent rounded-full mx-auto mb-3"></div>
                            <p class="text-white font-medium" data-translate="gameInfo.loading">Loading guides...</p>
                        </div>
                    </div>

                    @if ($gameInfos->isEmpty())
                        <!-- No Guides Found -->
                        <div class="game-info-empty">
                            <div class="game-info-empty-icon">
                                <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                            </div>
                            <h3 class="text-2xl font-bold mb-2" data-translate="gameInfo.noGuidesFound">No Guides Found</h3>
                            <p>
                                @if($searchTerm)
                                    <span data-translate="gameInfo.noSearchResults" data-translate-values='{"search": "{{ $searchTerm }}", "category": "{{ $selectedCategory === 'All' ? __('gameInfo.allCategories') : $selectedCategory }}"}'>
                                        No guides found for "{{ $searchTerm }}" in {{ $selectedCategory === 'All' ? 'all categories' : $selectedCategory }}
                                    </span>
                                @else
                                    <span data-translate="gameInfo.noGuidesMessage" data-translate-values='{"category": "{{ $selectedCategory === 'All' ? __('gameInfo.allCategories') : $selectedCategory }}"}'>
                                        No guides available in {{ $selectedCategory === 'All' ? 'any category' : $selectedCategory }}
                                    </span>
                                @endif
                            </p>

                            @if($searchTerm)
                                <button
                                    wire:click="$set('searchTerm', '')"
                                    class="mt-4 px-4 py-2 bg-blue-600 hover:bg-blue-500 text-white rounded-xl transition-colors duration-300"
                                    data-translate="gameInfo.clearSearch"
                                >
                                    Clear Search
                                </button>
                            @endif
                        </div>
                    @else
                        <!-- Guides Grid -->
                        <div class="game-info-grid">
                            @foreach ($gameInfos as $guide)
                                <div class="game-info-card">
                                    <!-- Guide Image -->
                                    <div class="game-info-card-image">
                                        <img
                                            src="{{ $guide->image_url }}"
                                            alt="{{ $guide->title }}"
                                        >
                                        <div class="game-info-card-overlay"></div>

                                        <!-- Category Badge -->
                                        <div class="game-info-card-badge">
                                            {{ $guide->category_name }}
                                        </div>
                                    </div>

                                    <!-- Guide Content -->
                                    <div class="game-info-card-content">
                                        <h3 class="game-info-card-title">
                                            {{ $guide->title }}
                                        </h3>

                                        <p class="game-info-card-excerpt">
                                            {{ Str::limit(strip_tags($guide->body), 120) }}
                                        </p>

                                        <!-- View More Button -->
                                        <button
                                            wire:click="viewGuide({{ $guide->id }})"
                                            class="game-info-card-btn"
                                        >
                                            <span data-translate="gameInfo.viewGuide">View Guide</span>
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                <line x1="5" y1="12" x2="19" y2="12"/>
                                                <polyline points="12,5 19,12 12,19"/>
                                            </svg>
                                        </button>
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    @endif
                </div>
            </div>

        @else
            <!-- Single Guide View -->
            @if($this->selectedGuideObject)
            <div class="game-info-single">

                <!-- Back Button -->
                <button
                    wire:click="backToCategory"
                    class="game-info-back-btn"
                >
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <line x1="19" y1="12" x2="5" y2="12"/>
                        <polyline points="12,19 5,12 12,5"/>
                    </svg>
                    <span data-translate="gameInfo.backTo" data-translate-values='{"category": "{{ $selectedCategory }}"}'>Back to {{ $selectedCategory }}</span>
                </button>

                <!-- Guide Content -->
                <article class="game-info-article">
                    <!-- Guide Header -->
                    <div class="game-info-article-header">
                        <img
                            src="{{ $this->selectedGuideObject->image_url }}"
                            alt="{{ $this->selectedGuideObject->title }}"
                        >
                        <div class="game-info-article-overlay"></div>

                        <!-- Title Overlay -->
                        <div class="game-info-article-meta">
                            <span class="game-info-article-category">
                                {{ $this->selectedGuideObject->category_name }}
                            </span>
                            <h1 class="game-info-article-title">
                                {{ $this->selectedGuideObject->title }}
                            </h1>
                        </div>
                    </div>

                    <!-- Guide Body -->
                    <div class="game-info-article-body">
                        <div class="game-content-processed">
                            {!! nl2br($this->processContent($this->selectedGuideObject->body)) !!}
                        </div>
                    </div>
                </article>
            </div>
            @else
                <!-- Guide not found -->
                <div class="text-center py-16">
                    <div class="text-red-400 text-6xl mb-4">
                        <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="mx-auto">
                            <path d="M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/>
                            <line x1="12" y1="9" x2="12" y2="13"/>
                            <line x1="12" y1="17" x2="12.01" y2="17"/>
                        </svg>
                    </div>
                    <h2 class="text-2xl font-bold text-white mb-4" data-translate="gameInfo.guideNotFound">Guide Not Found</h2>
                    <p class="text-gray-300 mb-6" data-translate="gameInfo.guideNotFoundMessage">The guide you're looking for doesn't exist or has been removed.</p>
                    <button
                        wire:click="backToCategory"
                        class="px-6 py-3 bg-blue-600 hover:bg-blue-700 text-white rounded-lg transition-colors"
                    >
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="mr-2">
                            <line x1="19" y1="12" x2="5" y2="12"/>
                            <polyline points="12,19 5,12 12,5"/>
                        </svg>
                        <span data-translate="gameInfo.backToGuides">Back to Guides</span>
                    </button>
                </div>
            @endif
        @endif

    </div>
</div>
</div>
