<?php

use Livewire\Volt\Component;
use App\Models\Z_Website\VoteSite;
use App\Models\Z_Website\UserVote;
use Carbon\Carbon;
use Livewire\Attributes\On;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

new class extends Component {
    public $votingSites = [];
    public $selectedEventId = null;

    public function mount($selectedEventId = null)
    {
        $this->selectedEventId = $selectedEventId;
        $this->loadVotingSites();
    }

    #[On('event-selected')]
    public function updateSelectedEvent($eventId)
    {
        $this->selectedEventId = $eventId;
        // Only load voting sites if this is event 21
        if ($eventId == 21) {
            $this->loadVotingSites();
        }
    }

    public function loadVotingSites()
    {
        try {
            // Only load if user is logged in
            if (!auth()->check()) {
                $this->votingSites = [];
                return;
            }

            $this->votingSites = VoteSite::where('active', 1)
                ->orderBy('Points', 'desc')
                ->get()
                ->map(function ($site) {
                    $userVote = null;
                    $canVote = true;
                    $nextVoteTime = null;

                    // Get user vote status
                    $userVote = UserVote::where('vote_site_id', $site->id)
                        ->where('user_id', auth()->id())
                        ->first();

                    if ($userVote && $userVote->next_vote_time) {
                        $nextVoteTime = Carbon::parse($userVote->next_vote_time);
                        $canVote = Carbon::now()->gte($nextVoteTime);
                    }

                    // Build vote link with StrUserID from session
                    $strUserID = session('StrUserID', auth()->user()->username ?? '');
                    $voteLink = str_replace('!username', $strUserID, $site->vote_link);

                    return [
                        'id' => $site->id,
                        'site_name' => $site->site_name,
                        'logo' => $site->logo,
                        'points' => $site->Points,
                        'vote_link' => $voteLink,
                        'vote_duration' => $site->vote_duration,
                        'can_vote' => $canVote,
                        'next_vote_time' => $nextVoteTime,
                        'time_remaining' => $nextVoteTime ? $this->getTimeRemaining($nextVoteTime) : null,
                    ];
                })
                ->toArray();
        } catch (\Exception $e) {
            $this->votingSites = [];
        }
    }
    private function getTimeRemaining($nextVoteTime)
    {
        if (!$nextVoteTime || Carbon::now()->gte($nextVoteTime)) {
            return null;
        }

        $diff = Carbon::now()->diff($nextVoteTime);

        if ($diff->h > 0) {
            return $diff->h . ' hours, ' . $diff->i . ' minutes';
        } elseif ($diff->i > 0) {
            return $diff->i . ' minutes, ' . $diff->s . ' seconds';
        } else {
            return $diff->s . ' seconds';
        }
    }

    public function openVoteModal($siteId)
    {
        // This function will be called to open the vote modal
        $this->dispatch('open-vote-modal', $siteId);
    }

    public function checkVoteStatus($siteId)
    {
        if (!auth()->check()) {
            return false;
        }

        try {
            $voteSite = VoteSite::find($siteId);
            if (!$voteSite || !$voteSite->active) {
                return false;
            }

            // Check if vote was recorded (the stored procedure should have updated the database)
            $userVote = UserVote::where('vote_site_id', $siteId)
                ->where('user_id', auth()->id())
                ->first();

            if ($userVote && $userVote->next_vote_time && Carbon::now()->lt(Carbon::parse($userVote->next_vote_time))) {
                // Vote confirmed - reload voting sites
                $this->loadVotingSites();
                session()->flash('success', 'Vote confirmed! You earned ' . $voteSite->Points . ' points!');
                return true;
            }

            return false;
        } catch (\Exception $e) {
            Log::error('Error checking vote status:', ['error' => $e->getMessage()]);
            return false;
        }
    }

    public function recordVote($siteId)
    {
        if (!auth()->check()) {
            session()->flash('error', 'You must be logged in to vote');
            return;
        }

        try {
            $voteSite = VoteSite::find($siteId);
            if (!$voteSite || !$voteSite->active) {
                session()->flash('error', 'Vote site is not available');
                return;
            }

            // Check if user can vote
            $userVote = UserVote::where('vote_site_id', $siteId)
                ->where('user_id', auth()->id())
                ->first();

            if ($userVote && $userVote->next_vote_time && Carbon::now()->lt(Carbon::parse($userVote->next_vote_time))) {
                session()->flash('error', 'Please wait for the cooldown period to end before voting again');
                return;
            }

            // Manual vote confirmation - call stored procedure directly
            try {
                // Get user JID for stored procedure
                $user = DB::table('Users')->where('JID', auth()->id())->select('Username')->first();
                if ($user) {
                    DB::statement("EXEC [__Vote_System] ?, ?, ?", [auth()->id(), $user->Username, 'manual']);
                    Log::info("Manual vote processed for user: " . $user->Username);
                } else {
                    throw new \Exception("User not found for JID: " . auth()->id());
                }
            } catch (\Exception $spException) {
                Log::error('Manual vote stored procedure failed:', ['error' => $spException->getMessage()]);
                session()->flash('error', 'Failed to process manual vote');
                return;
            }

            // Reload voting sites to update status
            $this->loadVotingSites();

            session()->flash('success', 'Vote recorded manually! You earned ' . $voteSite->Points . ' points');
        } catch (\Exception $e) {
            Log::error('Error recording manual vote:', ['error' => $e->getMessage()]);
            session()->flash('error', 'An error occurred while recording your vote');
        }
    }

    public function refreshVotingStatus()
    {
        $this->loadVotingSites();
    }

    #[On('refresh-events')]
    public function refreshVotingSystem()
    {
        if ($this->selectedEventId == 21 && auth()->check()) {
            $this->loadVotingSites();
        }
    }
};
?>

<div x-data="{
    showVoteModal: false,
    currentVoteUrl: '',
    currentSiteId: null,
    checkingVote: false,
    pollInterval: null,

    openVoteModal(voteUrl, siteId) {
        // Open in new window instead of modal due to X-Frame-Options restrictions
        window.open(voteUrl, '_blank', 'width=1200,height=800,scrollbars=yes,resizable=yes');

        // Show confirmation modal
        this.currentVoteUrl = voteUrl;
        this.currentSiteId = siteId;
        this.showVoteModal = true;
        this.checkingVote = true;

        // Start polling for vote completion
        this.startVotePolling();
    },

    closeVoteModal() {
        this.stopVotePolling();
        this.showVoteModal = false;
        this.currentVoteUrl = '';
        this.currentSiteId = null;
        this.checkingVote = false;
    },

    startVotePolling() {
        // Check every 2 seconds for postback confirmation (faster than before)
        this.pollInterval = setInterval(() => {
            if (this.currentSiteId) {
                $wire.checkVoteStatus(this.currentSiteId).then((result) => {
                    if (result) {
                        // Vote was confirmed via postback
                        this.closeVoteModal();
                    }
                });
            }
        }, 2000);

        // Stop polling after 3 minutes (shorter than before since postback should be immediate)
        setTimeout(() => {
            this.stopVotePolling();
            this.checkingVote = false;
        }, 180000);
    },    stopVotePolling() {
        if (this.pollInterval) {
            clearInterval(this.pollInterval);
            this.pollInterval = null;
        }
    },

    manualVoteComplete() {
        // Manual confirmation as fallback
        if (this.currentSiteId) {
            $wire.recordVote(this.currentSiteId);
            this.closeVoteModal();
        }
    }
}">
    @if ($selectedEventId == 21)
        <!-- Voting System Section -->
        <div class="mb-8">
            <div
                class="bg-gradient-to-r from-green-900/30 to-blue-900/30 backdrop-blur-lg border border-green-500/30 rounded-2xl p-6 shadow-2xl">
                <!-- Header -->
                <div class="text-center mb-8">
                    <div class="flex items-center justify-center gap-4 mb-4">
                        <div class="w-12 h-12 bg-green-500/20 rounded-xl flex items-center justify-center">
                            <svg class="w-6 h-6 text-green-400" viewBox="0 0 24 24" fill="currentColor">
                                <path
                                    d="M12 2C13.1 2 14 2.9 14 4V8C14 9.1 13.1 10 12 10S10 9.1 10 8V4C10 2.9 10.9 2 12 2M21 9V7L15 7V9C15 11.8 12.8 14 10 14V16L8 20V22H16V20L14 16V14C17.2 14 21 11.8 21 9Z" />
                            </svg>
                        </div>
                        <h3 class="text-3xl font-bold text-white">
                            <span data-translate="voting.voteForServer" wire:ignore>Vote for Our Server</span>
                        </h3>
                        <div class="w-12 h-12 bg-blue-500/20 rounded-xl flex items-center justify-center">
                            <svg class="w-6 h-6 text-blue-400" viewBox="0 0 24 24" fill="currentColor">
                                <path
                                    d="M5.5 7A4.5 4.5 0 1 1 10 11.5V16H6V18C6 19.1 6.9 20 8 20H16C17.1 20 18 19.1 18 18V16H14V11.5C16.5 10.8 18.5 8.6 18.5 6A4.5 4.5 0 0 1 23 10.5V13H21V10.5A2.5 2.5 0 0 0 18.5 8A2.5 2.5 0 0 0 16 10.5V16H20V18C20 20.2 18.2 22 16 22H8C5.8 22 4 20.2 4 18V16H8V10.5A2.5 2.5 0 0 0 5.5 8A2.5 2.5 0 0 0 3 10.5V13H1V10.5A4.5 4.5 0 0 1 5.5 6Z" />
                            </svg>
                        </div>
                    </div>
                    <p class="text-gray-300 text-lg" data-translate="voting.earnRewards" wire:ignore>
                        Vote daily and earn valuable rewards!
                    </p>
                </div>

                <!-- Flash Messages -->
                @if (session()->has('success'))
                    <div class="mb-6 p-4 bg-green-500/20 border border-green-500/30 rounded-lg text-green-300">
                        <svg class="w-4 h-4 inline mr-2" viewBox="0 0 24 24" fill="currentColor">
                            <path
                                d="M12 2C6.5 2 2 6.5 2 12S6.5 22 12 22 22 17.5 22 12 17.5 2 12 2M10 17L5 12L6.41 10.59L10 14.17L17.59 6.58L19 8L10 17Z" />
                        </svg>
                        {{ session('success') }}
                    </div>
                @endif

                @if (session()->has('error'))
                    <div class="mb-6 p-4 bg-red-500/20 border border-red-500/30 rounded-lg text-red-300">
                        <svg class="w-4 h-4 inline mr-2" viewBox="0 0 24 24" fill="currentColor">
                            <path
                                d="M12 2C6.5 2 2 6.5 2 12S6.5 22 12 22 22 17.5 22 12 17.5 2 12 2M13 17H11V15H13V17M13 13H11V7H13V13Z" />
                        </svg>
                        {{ session('error') }}
                    </div>
                @endif

                <!-- Voting Sites Grid -->
                @if (!empty($votingSites))
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        @foreach ($votingSites as $site)
                            <div class="vote-site-card group relative">
                                <div
                                    class="bg-gradient-to-br from-slate-800/80 to-slate-900/80 backdrop-blur border border-gray-600/30 rounded-xl p-6 transition-all duration-300 group-hover:border-green-400/50 group-hover:shadow-2xl group-hover:shadow-green-400/10">

                                    <!-- Site Logo -->
                                    <div class="text-center mb-4">
                                        @if ($site['logo'])
                                            <div
                                                class="w-16 h-16 mx-auto mb-3 rounded-xl overflow-hidden border-2 border-gray-600 group-hover:border-green-400 transition-colors duration-300">
                                                <img src="{{ $site['logo'] }}" alt="{{ $site['site_name'] }}"
                                                    class="w-full h-full object-cover">
                                            </div>
                                        @else
                                            <div
                                                class="w-16 h-16 mx-auto mb-3 rounded-xl bg-gradient-to-br from-green-500 to-blue-500 flex items-center justify-center">
                                                <svg class="w-8 h-8 text-white" viewBox="0 0 24 24" fill="currentColor">
                                                    <path
                                                        d="M12 2C13.1 2 14 2.9 14 4V8C14 9.1 13.1 10 12 10S10 9.1 10 8V4C10 2.9 10.9 2 12 2M21 9V7L15 7V9C15 11.8 12.8 14 10 14V16L8 20V22H16V20L14 16V14C17.2 14 21 11.8 21 9Z" />
                                                </svg>
                                            </div>
                                        @endif

                                        <h4
                                            class="text-lg font-bold text-white group-hover:text-green-300 transition-colors duration-300">
                                            {{ $site['site_name'] }}
                                        </h4>
                                    </div>

                                    <!-- Points Display -->
                                    <div class="text-center mb-4">
                                        <div
                                            class="inline-flex items-center px-4 py-2 bg-gradient-to-r from-yellow-500/20 to-orange-500/20 border border-yellow-500/30 rounded-full">
                                            <svg class="w-4 h-4 text-yellow-400 mr-2" viewBox="0 0 24 24"
                                                fill="currentColor">
                                                <path
                                                    d="M12 2C13.1 2 14 2.9 14 4V8C14 9.1 13.1 10 12 10S10 9.1 10 8V4C10 2.9 10.9 2 12 2M18 8V6C18 4.9 17.1 4 16 4S14 4.9 14 6V8C14 9.1 14.9 10 16 10S18 9.1 18 8M10 6V8C10 9.1 9.1 10 8 10S6 9.1 6 8V6C6 4.9 6.9 4 8 4S10 4.9 10 6M22 12C22 17.5 17.5 22 12 22S2 17.5 2 12 6.5 2 12 2 22 6.5 22 12M12 20C16.4 20 20 16.4 20 12S16.4 4 12 4 4 7.6 4 12 7.6 20 12 20Z" />
                                            </svg>
                                            <span
                                                class="text-yellow-300 font-bold text-lg">{{ number_format($site['points']) }}</span>
                                            <span class="text-gray-400 ml-1" data-translate="voting.points"
                                                wire:ignore>Points</span>
                                        </div>
                                    </div>

                                    <!-- Vote Status -->
                                    <div class="mb-4 text-center">
                                        @if ($site['can_vote'])
                                            <div class="flex items-center justify-center text-green-400 text-sm">
                                                <div class="w-2 h-2 bg-green-400 rounded-full animate-pulse mr-2"></div>
                                                <span data-translate="voting.readyToVote" wire:ignore>Ready to
                                                    Vote</span>
                                            </div>
                                        @else
                                            <div class="text-center">
                                                <div
                                                    class="flex items-center justify-center text-orange-400 text-sm mb-1">
                                                    <div class="w-2 h-2 bg-orange-400 rounded-full mr-2"></div>
                                                    <span data-translate="voting.waitingPeriod" wire:ignore>Waiting
                                                        Period</span>
                                                </div>
                                                @if ($site['time_remaining'])
                                                    <div class="text-xs text-gray-400">
                                                        <span data-translate="voting.nextVoteIn" wire:ignore>Next vote
                                                            in:</span>
                                                        {{ $site['time_remaining'] }}
                                                    </div>
                                                @endif
                                            </div>
                                        @endif
                                    </div>

                                    <!-- Vote Button -->
                                    <div class="text-center">
                                        @if ($site['can_vote'])
                                            <button @click="openVoteModal('{{ $site['vote_link'] }}', {{ $site['id'] }})"
                                                class="inline-flex items-center px-6 py-3 bg-gradient-to-r from-green-500 to-blue-500 hover:from-green-600 hover:to-blue-600 text-white font-bold rounded-xl transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                                <svg class="w-4 h-4 mr-2" viewBox="0 0 24 24" fill="currentColor">
                                                    <path
                                                        d="M14 3V5H17.59L7.76 14.83L9.17 16.24L19 6.41V10H21V3M19 19H5V5H12V3H5C3.9 3 3 3.9 3 5V19C3 20.1 3.9 21 5 21H19C20.1 21 21 20.1 21 19V12H19V19Z" />
                                                </svg>
                                                <span data-translate="voting.voteNow" wire:ignore>Vote Now</span>
                                            </button>
                                        @else
                                            <button disabled
                                                class="inline-flex items-center px-6 py-3 bg-gray-600 text-gray-400 font-bold rounded-xl cursor-not-allowed">
                                                <svg class="w-4 h-4 mr-2" viewBox="0 0 24 24" fill="currentColor">
                                                    <path
                                                        d="M12 2C6.5 2 2 6.5 2 12S6.5 22 12 22 22 17.5 22 12 17.5 2 12 2M15.5 7L14 8.5L15.5 10L14 11.5L15.5 13L14 14.5L15.5 16L17 14.5L15.5 13L17 11.5L15.5 10L17 8.5L15.5 7M9.5 7L8 8.5L9.5 10L8 11.5L9.5 13L8 14.5L9.5 16L11 14.5L9.5 13L11 11.5L9.5 10L11 8.5L9.5 7Z" />
                                                </svg>
                                                <span data-translate="voting.pleaseWait" wire:ignore>Please Wait</span>
                                            </button>
                                        @endif
                                    </div>

                                    <!-- Vote Duration Info -->
                                    <div class="mt-3 text-center text-xs text-gray-400">
                                        <svg class="w-3 h-3 inline mr-1" viewBox="0 0 24 24" fill="currentColor">
                                            <path
                                                d="M12 2C6.48 2 2 6.48 2 12S6.48 22 12 22 22 17.52 22 12 17.52 2 12 2M13 17H11V11H13V17M13 9H11V7H13V9Z" />
                                        </svg>
                                        <span data-translate="voting.voteCooldown" wire:ignore>Vote cooldown:</span>
                                        {{ $site['vote_duration'] }} <span data-translate="voting.hours"
                                            wire:ignore>hours</span>
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="text-center py-8">
                        <div class="text-gray-400 text-lg">
                            <svg class="w-16 h-16 mx-auto mb-4 text-gray-500" viewBox="0 0 24 24" fill="currentColor">
                                <path
                                    d="M12 2C13.1 2 14 2.9 14 4V8C14 9.1 13.1 10 12 10S10 9.1 10 8V4C10 2.9 10.9 2 12 2M21 9V7L15 7V9C15 11.8 12.8 14 10 14V16L8 20V22H16V20L14 16V14C17.2 14 21 11.8 21 9Z" />
                            </svg>
                            <p data-translate="voting.noVotingSites" wire:ignore>No voting sites available at the
                                moment.</p>
                        </div>
                    </div>
                @endif

                <!-- Refresh Button -->
                <div class="text-center mt-6">
                    <button wire:click="refreshVotingStatus"
                        class="inline-flex items-center px-4 py-2 bg-slate-600 hover:bg-slate-500 text-white rounded-lg transition-colors duration-300">
                        <svg class="w-4 h-4 mr-2" viewBox="0 0 24 24" fill="currentColor">
                            <path
                                d="M17.65 6.35C16.2 4.9 14.21 4 12 4C7.58 4 4 7.58 4 12S7.58 20 12 20C15.73 20 18.84 17.45 19.73 14H17.65C16.83 16.33 14.61 18 12 18C8.69 18 6 15.31 6 12S8.69 6 12 6C13.66 6 15.14 6.69 16.22 7.78L13 11H20V4L17.65 6.35Z" />
                        </svg>
                        <span data-translate="voting.refreshStatus" wire:ignore>Refresh Status</span>
                    </button>
                </div>

                <!-- Additional Info -->
                <div class="mt-8 text-center">
                    <div
                        class="bg-gradient-to-r from-blue-900/40 to-purple-900/40 border border-blue-500/30 rounded-lg p-4">
                        <h4 class="text-lg font-bold text-blue-300 mb-2">
                            <svg class="w-5 h-5 inline mr-2" viewBox="0 0 24 24" fill="currentColor">
                                <path
                                    d="M12 2C6.48 2 2 6.48 2 12S6.48 22 12 22 22 17.52 22 12 17.52 2 12 2M13 17H11V11H13V17M13 9H11V7H13V9Z" />
                            </svg>
                            <span data-translate="voting.votingTips" wire:ignore>Voting Tips</span>
                        </h4>
                        <ul class="text-sm text-gray-300 space-y-1">
                            <li data-translate="voting.tip1" wire:ignore>• Vote daily to maximize your rewards</li>
                            <li data-translate="voting.tip2" wire:ignore>• Each site has different cooldown periods</li>
                            <li data-translate="voting.tip3" wire:ignore>• Points are added automatically after voting
                            </li>
                            <li data-translate="voting.tip4" wire:ignore>• Help our server grow by voting regularly!
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>



        <!-- Vote Confirmation Modal -->
        <div x-show="showVoteModal" x-transition:enter="transition ease-out duration-300"
             x-transition:enter-start="opacity-0 transform scale-90"
             x-transition:enter-end="opacity-100 transform scale-100"
             x-transition:leave="transition ease-in duration-200"
             x-transition:leave-start="opacity-100 transform scale-100"
             x-transition:leave-end="opacity-0 transform scale-90"
             class="fixed inset-0 bg-black/60 backdrop-blur-sm z-50 flex items-center justify-center p-4"
             @click.self="closeVoteModal()">

            <div class="bg-gradient-to-br from-slate-800 to-slate-900 rounded-2xl p-6 max-w-md w-full mx-4 relative overflow-hidden border border-gray-600">
                <!-- Modal Header -->
                <div class="text-center mb-6">
                    <div class="w-16 h-16 bg-green-500/20 rounded-full flex items-center justify-center mx-auto mb-4" :class="checkingVote ? 'animate-pulse' : ''">
                        <svg class="w-8 h-8 text-green-400" viewBox="0 0 24 24" fill="currentColor" x-show="!checkingVote">
                            <path d="M12 2C13.1 2 14 2.9 14 4V8C14 9.1 13.1 10 12 10S10 9.1 10 8V4C10 2.9 10.9 2 12 2M21 9V7L15 7V9C15 11.8 12.8 14 10 14V16L8 20V22H16V20L14 16V14C17.2 14 21 11.8 21 9Z" />
                        </svg>
                        <svg class="w-8 h-8 text-green-400 animate-spin" viewBox="0 0 24 24" fill="currentColor" x-show="checkingVote">
                            <path d="M12 4V2A10 10 0 0 0 2 12H4A8 8 0 0 1 12 4Z" />
                        </svg>
                    </div>
                    <h3 class="text-xl font-bold text-white mb-2" data-translate="voting.voteOpened" wire:ignore>
                        Vote Site Opened
                    </h3>
                    <div x-show="checkingVote">
                        <p class="text-gray-400 mb-2" data-translate="voting.checkingVoteStatus" wire:ignore>
                            Checking vote status automatically...
                        </p>
                        <div class="flex items-center justify-center text-sm text-green-400">
                            <div class="w-2 h-2 bg-green-400 rounded-full animate-pulse mr-2"></div>
                            <span data-translate="voting.waitingForConfirmation" wire:ignore>Waiting for vote confirmation</span>
                        </div>
                    </div>
                    <p class="text-gray-400" data-translate="voting.voteOpenedDescription" wire:ignore x-show="!checkingVote">
                        The voting site has been opened in a new window. Please complete your vote.
                    </p>
                </div>

                <!-- Action Buttons -->
                <div class="flex gap-3">
                    <button @click="manualVoteComplete()" x-show="!checkingVote"
                            class="flex-1 bg-gradient-to-r from-green-500 to-green-600 hover:from-green-600 hover:to-green-700 text-white font-semibold py-3 px-4 rounded-lg transition-all duration-300 flex items-center justify-center">
                        <svg class="w-4 h-4 mr-2" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 2C6.5 2 2 6.5 2 12S6.5 22 12 22 22 17.5 22 12 17.5 2 12 2M10 17L5 12L6.41 10.59L10 14.17L17.59 6.58L19 8L10 17Z" />
                        </svg>
                        <span data-translate="voting.manualConfirm" wire:ignore>Manual Confirmation</span>
                    </button>
                    <button @click="closeVoteModal()"
                            class="flex-1 bg-gray-600 hover:bg-gray-700 text-white font-semibold py-3 px-4 rounded-lg transition-all duration-300">
                        <span data-translate="voting.cancel" wire:ignore>Cancel</span>
                    </button>
                </div>

                <!-- Additional Info -->
                <div class="mt-4 text-center">
                    <div class="text-sm text-gray-400 flex items-center justify-center" x-show="checkingVote">
                        <svg class="w-4 h-4 mr-2" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 2C6.48 2 2 6.48 2 12S6.48 22 12 22 22 17.52 22 12 17.52 2 12 2M13 17H11V11H13V17M13 9H11V7H13V9Z" />
                        </svg>
                        <span data-translate="voting.autoCheckInfo" wire:ignore>
                            We're automatically checking if your vote was completed
                        </span>
                    </div>
                    <div class="text-sm text-gray-400 flex items-center justify-center" x-show="!checkingVote">
                        <svg class="w-4 h-4 mr-2" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 2C6.48 2 2 6.48 2 12S6.48 22 12 22 22 17.52 22 12 17.52 2 12 2M13 17H11V11H13V17M13 9H11V7H13V9Z" />
                        </svg>
                        <span data-translate="voting.newWindowInfo" wire:ignore>
                            If the window didn't open, please check your popup blocker
                        </span>
                    </div>
                </div>
            </div>
        </div>

        <style>
            .vote-site-card {
                position: relative;
            }

            .vote-site-card::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: linear-gradient(45deg, transparent, rgba(34, 197, 94, 0.1), transparent);
                border-radius: 12px;
                opacity: 0;
                transition: opacity 0.3s ease;
                pointer-events: none;
            }

            .vote-site-card:hover::before {
                opacity: 1;
            }

            @keyframes votePulse {

                0%,
                100% {
                    box-shadow: 0 0 20px rgba(34, 197, 94, 0.3);
                }

                50% {
                    box-shadow: 0 0 30px rgba(34, 197, 94, 0.5);
                }
            }

            .vote-site-card:hover {
                animation: votePulse 2s infinite;
            }
        </style>

    @endif

</div>
