<?php

use Livewire\Volt\Component;
use App\Models\Z_A_BOTS_SYSTEMS\EventPlayersRank;
use Livewire\Attributes\On;

new class extends Component {
    public $topPlayers = [];
    public $selectedEventId = null;

    public function mount($selectedEventId = null)
    {
        if ($selectedEventId) {
            $this->selectedEventId = $selectedEventId;
            $this->loadPlayerRankings();
        }
    }

    #[On('event-selected')]
    public function loadTopPlayers($eventId)
    {
        $this->selectedEventId = $eventId;
        $this->loadPlayerRankings();
    }

    public function loadPlayerRankings()
    {
        try {
            // تحميل اللاعبين مع العلاقة الأساسية فقط
            $players = EventPlayersRank::where('EventID', $this->selectedEventId)
                ->with([
                    'char' => function ($query) {
                        $query->select('CharID', 'CharName16', 'RefObjID');
                    }
                ])
                ->orderBy('Points', 'desc')
                ->orderBy('LastPoint', 'asc')
                ->take(3)
                ->get();

            $this->topPlayers = $players
                ->map(function ($player, $index) {
                    // حساب الملكية بطريقة آمنة
                    $isOwned = $this->checkPlayerOwnership($player);

                    return [
                        'rank' => $index + 1,
                        'name' => $player->Charname ?? 'Unknown',
                        'charName' => optional($player->char)->CharName16 ?? 'Unknown',
                        'points' => $player->Points ?? 0,
                        'kills' => $player->Kills ?? 0,
                        'deaths' => $player->Death ?? 0,
                        'job_type' => $player->JobType ?? 0,
                        'char_data' => $player->char,
                        'avatar' => $this->getCharacterAvatar(optional($player->char)->RefObjID ?? 1907),
                        'refObjID' => optional($player->char)->RefObjID ?? 1907,
                        'race' => $this->getCharacterRace(optional($player->char)->RefObjID ?? 1907),
                        'gender' => $this->getCharacterGender(optional($player->char)->RefObjID ?? 1907),
                        'is_owned' => $isOwned,
                    ];
                })
                ->toArray();

            $this->dispatch('top-players-loaded', $this->topPlayers);
        } catch (\Exception $e) {
            $this->topPlayers = [];
            // Silently handle error
        }
    }

    /**
     * فحص ملكية اللاعب بطريقة آمنة ومنعزلة - محسنة
     */
    private function checkPlayerOwnership($player)
    {
        try {
            // إذا لم يكن المستخدم مسجل دخول، فلا توجد ملكية
            if (!auth()->check()) {
                return false;
            }

            // إذا لم تكن هناك بيانات للمستخدم المسجل، فلا توجد ملكية
            if (!auth()->user()) {
                return false;
            }

            // استخدام الدالة المحسنة في النموذج
            return $player->isOwnedSafely();

        } catch (\Exception $e) {
            // في حالة حدوث خطأ، نعيد false لتجنب المشاكل
            return false;
        }
    }

    public function refreshPlayerRankings()
    {
        $this->loadPlayerRankings();
        $this->dispatch('rankings-updated');
    }

    public function getJobTypeName($jobType)
    {
        $jobTypes = [
            0 => 'No Job',
            1 => 'Trader',
            2 => 'Thief',
            3 => 'Hunter',
        ];
        return $jobTypes[$jobType] ?? 'Unknown';
    }

    public function getJobBadgeClasses($jobType)
    {
        $badges = [
            0 => 'job-badge job-badge-sm job-no-job', // No Job
            1 => 'job-badge job-badge-sm job-trader', // Trader
            2 => 'job-badge job-badge-sm job-thief', // Thief
            3 => 'job-badge job-badge-sm job-hunter', // Hunter
        ];
        return $badges[$jobType] ?? 'job-badge job-badge-sm job-no-job';
    }

    public function getJobIcon($jobType)
    {
        $icons = [
            0 => 'fas fa-question-circle', // No Job
            1 => 'fas fa-coins', // Trader
            2 => 'fas fa-mask', // Thief
            3 => 'fas fa-crosshairs', // Hunter
        ];
        return $icons[$jobType] ?? 'fas fa-question-circle';
    }

    public function getCharacterAvatar($refObjID)
    {
        // Return avatar path directly using RefObjID
        $avatarPath = "/assets/images/avatars/{$refObjID}.gif";

        // Check if file exists, otherwise use default
        $fullPath = public_path("assets/images/avatars/{$refObjID}.gif");
        if (!file_exists($fullPath)) {
            return "/assets/images/avatars/1907.gif"; // Default avatar
        }

        return $avatarPath;
    }

    public function getCharacterRace($refObjID)
    {
        // Chinese characters (1907 - 14875)
        if ($refObjID >= 1907 && $refObjID <= 14875) {
            return 'chinese';
        }
        // European characters (14876 and above)
        else {
            return 'european';
        }
    }

    public function getCharacterGender($refObjID)
    {
        // Even RefObjID = Male, Odd RefObjID = Female
        return $refObjID % 2 == 0 ? 'male' : 'female';
    }

    #[On('refresh-events')]
    public function refreshTopPlayers()
    {
        if ($this->selectedEventId) {
            $this->loadTopPlayers($this->selectedEventId);
        }
    }
};
?>
<div>
        @if (!empty($topPlayers))
            <!-- Top 3 Players Podium -->
            <div class="top-players-section">
                <div class="flex flex-col items-center justify-center mb-8 gap-6">
                    <!-- Elegant Champions Title - Fully Centered -->
                    <div class="champions-title-container w-full flex justify-center">
                        <div class="relative max-w-4xl">
                            <!-- Background glow effect using site colors -->
                            <div class="absolute inset-0 bg-gradient-to-r from-purple-500/30 via-purple-600/30 to-purple-700/30 rounded-2xl blur-xl"></div>
                            <div class="absolute inset-0 bg-black/20 rounded-2xl"></div>

                            <!-- Main title container -->
                            <div class="relative bg-gradient-to-r from-slate-900/90 to-slate-800/90 backdrop-blur-md border border-purple-400/50 rounded-2xl px-6 sm:px-8 lg:px-12 py-4 sm:py-6 shadow-2xl">
                                <!-- Crown and sparkles -->
                                <div class="absolute -top-2 -right-2">
                                    <div class="w-6 h-6 sm:w-8 sm:h-8 bg-purple-400/20 rounded-full flex items-center justify-center animate-pulse">
                                        <span class="text-purple-300 text-sm sm:text-lg">✨</span>
                                    </div>
                                </div>
                                <div class="absolute -top-1 -left-1">
                                    <div class="w-5 h-5 sm:w-6 sm:h-6 bg-purple-500/20 rounded-full flex items-center justify-center animate-ping">
                                        <span class="text-purple-300 text-xs sm:text-sm">⭐</span>
                                    </div>
                                </div>

                                <!-- Title content - Always centered -->
                                <div class="flex flex-col sm:flex-row items-center justify-center gap-3 sm:gap-6">
                                    <!-- Animated crown -->
                                    <div class="relative">
                                        <div class="text-3xl sm:text-4xl lg:text-5xl animate-bounce" style="animation-duration: 2s;">👑</div>
                                        <div class="absolute inset-0 text-3xl sm:text-4xl lg:text-5xl animate-pulse opacity-50">👑</div>
                                    </div>

                                    <!-- Title text with site gradient colors -->
                                    <div class="text-center">
                                        <h3 class="text-3xl sm:text-4xl lg:text-5xl xl:text-6xl font-extrabold text-white mb-1 tracking-wide drop-shadow-2xl">
                                            <span class="bg-gradient-to-r from-purple-300 via-purple-400 to-purple-500 bg-clip-text text-transparent drop-shadow-lg" data-translate="rankings.topChampions" wire:ignore>TOP CHAMPIONS</span>
                                        </h3>
                                        <div class="text-sm sm:text-base font-medium text-purple-200 tracking-[0.15em] sm:tracking-[0.3em] uppercase drop-shadow-lg">
                                            Hall of Fame
                                        </div>
                                    </div>

                                    <!-- Decorative elements - Always visible and centered -->
                                    <div class="flex flex-col gap-2">
                                        <div class="w-10 sm:w-12 lg:w-16 h-1 bg-gradient-to-r from-purple-400 to-purple-600 rounded-full"></div>
                                        <div class="w-8 sm:w-10 lg:w-12 h-1 bg-gradient-to-r from-purple-500 to-purple-700 rounded-full mx-auto"></div>
                                        <div class="w-6 sm:w-8 lg:w-10 h-1 bg-gradient-to-r from-purple-600 to-purple-800 rounded-full mx-auto"></div>
                                    </div>
                                </div>

                                <!-- Subtle particles animation -->
                                <div class="absolute inset-0 overflow-hidden rounded-2xl">
                                    <div class="absolute w-1.5 h-1.5 sm:w-2 sm:h-2 bg-purple-400/30 rounded-full animate-ping top-3 sm:top-4 left-1/4" style="animation-delay: 0s; animation-duration: 3s;"></div>
                                    <div class="absolute w-1 h-1 bg-purple-500/40 rounded-full animate-ping top-6 sm:top-8 right-1/4" style="animation-delay: 1s; animation-duration: 4s;"></div>
                                    <div class="absolute w-1 h-1 sm:w-1.5 sm:h-1.5 bg-purple-600/30 rounded-full animate-ping bottom-4 sm:bottom-6 left-1/3" style="animation-delay: 2s; animation-duration: 3.5s;"></div>
                                    <div class="absolute w-1.5 h-1.5 bg-purple-400/25 rounded-full animate-ping bottom-6 right-1/3" style="animation-delay: 2.5s; animation-duration: 4.5s;"></div>
                                </div>
                            </div>
                        </div>
                    </div>

                </div>

                <div class="podium-container">
                    <!-- 2nd Place -->
                    @if (isset($topPlayers[1]))
                        <div class="podium-player podium-second group {{ ($topPlayers[1]['is_owned'] ?? false) ? 'player-owned' : '' }}"
                             data-player-id="{{ $topPlayers[1]['charName'] ?? 'unknown' }}"
                             data-rank="2"
                             data-points="{{ $topPlayers[1]['points'] ?? 0 }}">

                            @if(($topPlayers[1]['is_owned'] ?? false))
                                <div class="ownership-badge">
                                    <i class="fas fa-user-check"></i>
                                    <span>Your Character</span>
                                </div>
                            @endif                    <!-- Rank Badge -->
                    <div class="rank-badge rank-2">2</div>

                    <!-- Player Avatar -->
                    <div class="player-avatar-container">
                        <img src="{{ $topPlayers[1]['avatar'] ?? '/assets/images/avatars/default.gif' }}"
                            alt="{{ $topPlayers[1]['charName'] ?? $topPlayers[1]['name'] }}"
                            class="player-avatar player-avatar-clickable" loading="lazy">
                    </div>

                    <!-- Player Info -->
                    <h4 class="player-name player-name-clickable">
                        {{ $topPlayers[1]['charName'] ?? $topPlayers[1]['name'] }}</h4>
                    <div class="player-points">{{ number_format($topPlayers[1]['points']) }}
                    </div>

                    <!-- Player Stats -->
                    <div class="player-stats">
                        <div class="stat-item">
                            <span class="stat-value">{{ $topPlayers[1]['kills'] ?? 0 }}</span>
                            <span class="stat-label" data-translate="rankings.kills" wire:ignore>Kills</span>
                        </div>
                        <div class="stat-item">
                            <div class="job-badge-with-icon {{ $this->getJobBadgeClasses($topPlayers[1]['job_type'] ?? 0) }}">
                                <i class="{{ $this->getJobIcon($topPlayers[1]['job_type'] ?? 0) }}"></i>
                                {{ $this->getJobTypeName($topPlayers[1]['job_type'] ?? 0) }}
                            </div>
                        </div>
                    </div>
                </div>
            @endif

                    <!-- 1st Place -->
                    @if (isset($topPlayers[0]))
                        <div class="podium-player podium-first group {{ ($topPlayers[0]['is_owned'] ?? false) ? 'player-owned' : '' }}"
                             data-player-id="{{ $topPlayers[0]['charName'] ?? 'unknown' }}"
                             data-rank="1"
                             data-points="{{ $topPlayers[0]['points'] ?? 0 }}">

                            @if(($topPlayers[0]['is_owned'] ?? false))
                                <div class="ownership-badge ownership-champion">
                                    <i class="fas fa-crown"></i>
                                    <span>YOUR CHAMPION!</span>
                                </div>
                            @endif                    <!-- Crown -->
                    <div class="crown">👑</div>

                    <!-- Rank Badge -->
                    <div class="rank-badge rank-1">1</div>

                    <!-- Player Avatar -->
                    <div class="player-avatar-container">
                        <img src="{{ $topPlayers[0]['avatar'] ?? '/assets/images/avatars/default.gif' }}"
                            alt="{{ $topPlayers[0]['charName'] ?? $topPlayers[0]['name'] }}"
                            class="player-avatar player-avatar-clickable" loading="lazy">
                    </div>

                    <!-- Player Info -->
                    <h4 class="player-name player-name-clickable">
                        {{ $topPlayers[0]['charName'] ?? $topPlayers[0]['name'] }}</h4>
                    <div class="player-points">{{ number_format($topPlayers[0]['points']) }}
                    </div>

                    <!-- Player Stats -->
                    <div class="player-stats">
                        <div class="stat-item">
                            <span class="stat-value">{{ $topPlayers[0]['kills'] ?? 0 }}</span>
                            <span class="stat-label" data-translate="rankings.kills" wire:ignore>Kills</span>
                        </div>
                        <div class="stat-item">
                            <div class="job-badge-with-icon {{ $this->getJobBadgeClasses($topPlayers[0]['job_type'] ?? 0) }}">
                                <i class="{{ $this->getJobIcon($topPlayers[0]['job_type'] ?? 0) }}"></i>
                                {{ $this->getJobTypeName($topPlayers[0]['job_type'] ?? 0) }}
                            </div>
                        </div>
                    </div>
                </div>
            @endif

                    <!-- 3rd Place -->
                    @if (isset($topPlayers[2]))
                        <div class="podium-player podium-third group {{ ($topPlayers[2]['is_owned'] ?? false) ? 'player-owned' : '' }}"
                             data-player-id="{{ $topPlayers[2]['charName'] ?? 'unknown' }}"
                             data-rank="3"
                             data-points="{{ $topPlayers[2]['points'] ?? 0 }}">

                            @if(($topPlayers[2]['is_owned'] ?? false))
                                <div class="ownership-badge">
                                    <i class="fas fa-user-check"></i>
                                    <span>Your Character</span>
                                </div>
                            @endif                    <!-- Rank Badge -->
                    <div class="rank-badge rank-3">3</div>

                    <!-- Player Avatar -->
                    <div class="player-avatar-container">
                        <img src="{{ $topPlayers[2]['avatar'] ?? '/assets/images/avatars/default.gif' }}"
                            alt="{{ $topPlayers[2]['charName'] ?? $topPlayers[2]['name'] }}"
                            class="player-avatar player-avatar-clickable" loading="lazy">
                    </div>

                    <!-- Player Info -->
                    <h4 class="player-name player-name-clickable">
                        {{ $topPlayers[2]['charName'] ?? $topPlayers[2]['name'] }}</h4>
                    <div class="player-points">{{ number_format($topPlayers[2]['points']) }}
                    </div>

                    <!-- Player Stats -->
                    <div class="player-stats">
                        <div class="stat-item">
                            <span class="stat-value">{{ $topPlayers[2]['kills'] ?? 0 }}</span>
                            <span class="stat-label" data-translate="rankings.kills" wire:ignore>Kills</span>
                        </div>
                        <div class="stat-item">
                            <div class="job-badge-with-icon {{ $this->getJobBadgeClasses($topPlayers[2]['job_type'] ?? 0) }}">
                                <i class="{{ $this->getJobIcon($topPlayers[2]['job_type'] ?? 0) }}"></i>
                                {{ $this->getJobTypeName($topPlayers[2]['job_type'] ?? 0) }}
                            </div>
                        </div>
                    </div>
                </div>
            @endif
        </div>
    </div>
@else
    <!-- No Players - Show only when there are truly no players -->
    @if ($selectedEventId && empty($topPlayers))
        <div class="text-center py-16">
            <div
                class="bg-gradient-to-br from-slate-800/50 to-purple-900/30 backdrop-blur-lg border border-purple-500/30 rounded-2xl p-12 shadow-2xl">
                <i class="fas fa-users text-6xl text-purple-400 mb-6"></i>
                <h2 class="text-3xl font-bold text-white mb-4" data-translate="rankings.noPlayersYet" wire:ignore>
                    No Players Yet</h2>
                <p class="text-purple-200 text-lg" data-translate="rankings.noPlayersYetDesc" wire:ignore>Be the
                    first to join this event and claim the top spot!</p>
            </div>
        </div>
    @endif
@endif
</div>
