<?php

use Livewire\Volt\Component;
use App\Models\Z_A_BOTS_SYSTEMS\EventPlayersRank;
use Livewire\Attributes\On;

new class extends Component {
    public $otherPlayers = [];
    public $selectedEventId = null;

    public function mount($selectedEventId = null)
    {
        if ($selectedEventId) {
            $this->selectedEventId = $selectedEventId;
            $this->loadPlayerRankings();
        }
    }

    #[On('event-selected')]
    public function loadOtherPlayers($eventId)
    {
        $this->selectedEventId = $eventId;
        $this->loadPlayerRankings();
    }

    public function loadPlayerRankings()
    {
        try {
            // تحميل اللاعبين مع العلاقة الأساسية فقط
            $players = EventPlayersRank::where('EventID', $this->selectedEventId)
                ->with([
                    'char' => function ($query) {
                        $query->select('CharID', 'CharName16', 'RefObjID');
                    }
                ])
                ->orderBy('Points', 'desc')
                ->orderBy('LastPoint', 'asc')
                ->take(30)
                ->get();

            $this->otherPlayers = $players
                ->skip(3)
                ->values() // إعادة فهرسة المصفوفة
                ->map(function ($player, $index) {
                    // حساب الملكية بطريقة آمنة
                    $isOwned = $this->checkPlayerOwnership($player);

                    return [
                        'rank' => $index + 4, // الآن سيبدأ من 4، 5، 6، إلخ بشكل صحيح
                        'name' => $player->Charname ?? 'Unknown',
                        'charName' => optional($player->char)->CharName16 ?? 'Unknown',
                        'points' => $player->Points ?? 0,
                        'kills' => $player->Kills ?? 0,
                        'deaths' => $player->Death ?? 0,
                        'job_type' => $player->JobType ?? 0,
                        'char_data' => $player->char,
                        'avatar' => $this->getCharacterAvatar(optional($player->char)->RefObjID ?? 1907),
                        'refObjID' => optional($player->char)->RefObjID ?? 1907,
                        'is_owned' => $isOwned,
                    ];
                })
                ->toArray();

            $this->dispatch('other-players-loaded', $this->otherPlayers);
        } catch (\Exception $e) {
            $this->otherPlayers = [];
            // Silently handle error
        }
    }

    /**
     * فحص ملكية اللاعب بطريقة آمنة ومنعزلة - محسنة
     */
    private function checkPlayerOwnership($player)
    {
        try {
            // إذا لم يكن المستخدم مسجل دخول، فلا توجد ملكية
            if (!auth()->check()) {
                return false;
            }

            // إذا لم تكن هناك بيانات للمستخدم المسجل، فلا توجد ملكية
            if (!auth()->user()) {
                return false;
            }

            // استخدام الدالة المحسنة في النموذج
            return $player->isOwnedSafely();

        } catch (\Exception $e) {
            // في حالة حدوث خطأ، نعيد false لتجنب المشاكل
            return false;
        }
    }

    public function refreshPlayerRankings()
    {
        $this->loadPlayerRankings();
        $this->dispatch('rankings-updated');
    }

    public function getJobTypeName($jobType)
    {
        $jobTypes = [
            0 => 'No Job',
            1 => 'Trader',
            2 => 'Thief',
            3 => 'Hunter',
        ];
        return $jobTypes[$jobType] ?? 'Unknown';
    }

    public function getJobBadgeClasses($jobType)
    {
        $badges = [
            0 => 'job-badge job-badge-sm job-no-job', // No Job
            1 => 'job-badge job-badge-sm job-trader', // Trader
            2 => 'job-badge job-badge-sm job-thief', // Thief
            3 => 'job-badge job-badge-sm job-hunter', // Hunter
        ];
        return $badges[$jobType] ?? 'job-badge job-badge-sm job-no-job';
    }

    public function getJobIcon($jobType)
    {
        $icons = [
            0 => 'fas fa-question-circle', // No Job
            1 => 'fas fa-coins', // Trader
            2 => 'fas fa-mask', // Thief
            3 => 'fas fa-crosshairs', // Hunter
        ];
        return $icons[$jobType] ?? 'fas fa-question-circle';
    }

    public function getCharacterAvatar($refObjID)
    {
        // Return avatar path directly using RefObjID
        $avatarPath = "/assets/images/avatars/{$refObjID}.gif";

        // Check if file exists, otherwise use default
        $fullPath = public_path("assets/images/avatars/{$refObjID}.gif");
        if (!file_exists($fullPath)) {
            return "/assets/images/avatars/1907.gif"; // Default avatar
        }

        return $avatarPath;
    }

    #[On('refresh-events')]
    public function refreshRankingsTable()
    {
        if ($this->selectedEventId) {
            $this->loadOtherPlayers($this->selectedEventId);
        }
    }
};
?>
<div>
@if (!empty($otherPlayers))
    <!-- Other Players Table -->
    <div class="other-players-section">
        <div class="players-table-container">
            <div class="p-6 border-b border-purple-500/30">
                <div class="flex justify-between items-center">
                    <h3 class="text-2xl font-bold text-white flex items-center gap-3">
                        <i class="fas fa-list text-purple-400"></i>
                        <span data-translate="rankings.rankings4to30" wire:ignore>Rankings (4-30)</span>
                    </h3>
                   <!-- Live updates indicator - Centered -->
                    <div class="bg-gradient-to-r from-purple-600/20 to-purple-700/20 backdrop-blur-md border border-purple-400/30 rounded-xl px-4 sm:px-6 py-3 sm:py-4 shadow-lg">
                        <div class="text-sm sm:text-base text-purple-300 flex items-center gap-3 sm:gap-4">
                            <div class="relative">
                                <div class="w-3 h-3 sm:w-4 sm:h-4 bg-purple-400 rounded-full animate-ping"></div>
                                <div class="absolute inset-0 w-3 h-3 sm:w-4 sm:h-4 bg-purple-400 rounded-full animate-pulse"></div>
                            </div>
                            <span class="font-medium whitespace-nowrap">Live Updates Every 5s</span>
                        </div>
                    </div>
                </div>
            </div>

            <div class="overflow-x-auto" >
                <table class="players-table">
                    <thead class="table-header">
                        <tr>
                            <th class="table-header th" data-translate="rankings.rank" wire:ignore>Rank</th>
                            <th class="table-header th" data-translate="rankings.player" wire:ignore>Player</th>
                            <th class="table-header th" data-translate="rankings.job" wire:ignore>Job</th>
                            <th class="table-header th" data-translate="rankings.points" wire:ignore>Points</th>
                            <th class="table-header th" data-translate="rankings.kills" wire:ignore>Kills</th>
                            <th class="table-header th" data-translate="rankings.deaths" wire:ignore>Deaths</th>
                            <th class="table-header th" data-translate="rankings.kdRatio" wire:ignore>K/D</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach ($otherPlayers as $player)
                            <tr class="table-row {{ ($player['is_owned'] ?? false) ? 'table-row-owned' : '' }}"
                                data-player-id="{{ $player['charName'] ?? 'unknown' }}"
                                data-rank="{{ $player['rank'] ?? 0 }}"
                                data-points="{{ $player['points'] ?? 0 }}">

                                <td class="table-cell rank-cell">
                                    <div class="rank-wrapper">
                                        {{ $player['rank'] ?? 0 }}
                                        @if(($player['is_owned'] ?? false))
                                            <div class="owned-indicator">
                                                <i class="fas fa-user-check"></i>
                                            </div>
                                        @endif
                                    </div>
                                </td>

                                <td class="table-cell">
                                    <div class="player-info {{ ($player['is_owned'] ?? false) ? 'player-info-owned' : '' }}">
                                        <img src="{{ $player['avatar'] ?? '/assets/images/avatars/default.gif' }}"
                                            alt="{{ $player['charName'] ?? $player['name'] ?? 'Unknown' }}"
                                            class="player-small-avatar player-small-avatar-clickable {{ ($player['is_owned'] ?? false) ? 'avatar-owned' : '' }}" loading="lazy">
                                        <div class="player-name-wrapper">
                                            <div class="player-name-container player-name-clickable">
                                                <span class="text-white font-medium {{ ($player['is_owned'] ?? false) ? 'owned-player-name' : '' }}">
                                                    {{ $player['charName'] ?? $player['name'] ?? 'Unknown' }}
                                                    @if(($player['is_owned'] ?? false))
                                                        <i class="fas fa-crown text-amber-400 ml-1 text-xs"></i>
                                                    @endif
                                                </span>
                                                @if(($player['is_owned'] ?? false))
                                                    <div class="your-character-badge">
                                                        <i class="fas fa-user-shield"></i>
                                                        <span data-translate="rankings.yourCharacter" wire:ignore>Your Character</span>
                                                    </div>
                                                @endif
                                            </div>
                                        </div>
                                    </div>
                                </td>
                                <td class="table-cell">
                                    <div class="job-badge-with-icon {{ $this->getJobBadgeClasses($player['job_type'] ?? 0) }}">
                                        <i class="{{ $this->getJobIcon($player['job_type'] ?? 0) }}"></i>
                                        {{ $this->getJobTypeName($player['job_type'] ?? 0) }}
                                    </div>
                                </td>
                                <td class="table-cell points-cell">
                                    {{ number_format($player['points']) }}
                                </td>
                                <td class="table-cell text-center">
                                    <span class="text-green-400">{{ $player['kills'] ?? 0 }}</span>
                                </td>
                                <td class="table-cell text-center">
                                    <span class="text-red-400">{{ $player['deaths'] ?? 0 }}</span>
                                </td>
                                <td class="table-cell text-center">
                                    <span class="text-cyan-400">
                                        {{ ($player['deaths'] ?? 0) > 0 ? number_format(($player['kills'] ?? 0) / $player['deaths'], 2) : $player['kills'] ?? 0 }}
                                    </span>
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
        </div>
    </div>
@endif
</div>
