<?php

use Livewire\Volt\Component;
use Livewire\Attributes\On;
use App\Helpers\InventoryHelper;
use App\Models\SRO_VT_SHARD\Char;

new class extends Component {
    public $showPlayerModal = false;
    public $selectedPlayer = null;
    public $playerEquipment = [];

    protected $listeners = [
        'player-modal-open' => 'openPlayerModal',
        'player-modal-close' => 'closePlayerModal',
    ];

    public function openPlayerModal($playerData)
    {
        $this->selectedPlayer = $playerData;
        $this->showPlayerModal = true;

        // Load player equipment
        $this->loadPlayerEquipment();

        // Dispatch event to notify parent component
        $this->dispatch('modal-opened', $playerData);
    }

    public function loadPlayerEquipment()
    {
        $this->playerEquipment = [];

        if (isset($this->selectedPlayer['charName'])) {
            try {
                // Find character by name
                $character = Char::where('CharName16', $this->selectedPlayer['charName'])->first();

                if ($character) {
                    // Get enhanced equipment data with proper opt_level values
                    $this->playerEquipment = InventoryHelper::getEnhancedCharacterEquipment($character->CharID);
                }
            } catch (\Exception $e) {
                // Silently handle equipment loading error
                $this->playerEquipment = [];
            }
        }
    }

    public function closePlayerModal()
    {
        $this->showPlayerModal = false;
        $this->selectedPlayer = null;
        $this->playerEquipment = [];

        // Dispatch event to notify parent component
        $this->dispatch('modal-closed');
    }

    // Additional helper methods for future features
    public function getPlayerJobType($jobType)
    {
        $jobTypes = [
            0 => 'No Job',
            1 => 'Trader',
            2 => 'Thief',
            3 => 'Hunter',
        ];
        return $jobTypes[$jobType] ?? 'Unknown';
    }

    public function getJobBadgeClasses($jobType)
    {
        $badges = [
            0 => 'job-badge job-badge-sm job-no-job', // No Job
            1 => 'job-badge job-badge-sm job-trader', // Trader
            2 => 'job-badge job-badge-sm job-thief', // Thief
            3 => 'job-badge job-badge-sm job-hunter', // Hunter
        ];
        return $badges[$jobType] ?? 'job-badge job-badge-sm job-no-job';
    }

    public function getJobIcon($jobType)
    {
        $icons = [
            0 => 'fas fa-question-circle', // No Job
            1 => 'fas fa-coins', // Trader
            2 => 'fas fa-mask', // Thief
            3 => 'fas fa-crosshairs', // Hunter
        ];
        return $icons[$jobType] ?? 'fas fa-question-circle';
    }

    public function getPlayerRace($refObjID)
    {
        // Chinese characters (1907 - 1932)
        if ($refObjID >= 1907 && $refObjID <= 1932) {
            return 'Chinese';
        }
        // European characters (all others)
        else {
            return 'European';
        }
    }

    public function getPlayerGender($refObjID)
    {
        // Male characters: 1907-1919 or 14875-14887
        if (($refObjID >= 1907 && $refObjID <= 1919) || ($refObjID >= 14875 && $refObjID <= 14887)) {
            return 'Male';
        }
        // Female characters (all others)
        else {
            return 'Female';
        }
    }

    public function calculateKDRatio($kills, $deaths)
    {
        if ($deaths > 0) {
            return number_format($kills / $deaths, 2);
        }
        return $kills;
    }

    #[On('refresh-events')]
    public function refreshPlayerModal()
    {
        // Player modal data will be refreshed when needed
        if ($this->selectedPlayer && $this->showPlayerModal) {
            // Re-query player data if modal is currently open
            $this->dispatch('modal-data-refreshed');
        }
    }
};
?>
<div>
    <!-- Player Details Modal -->
    @if ($showPlayerModal && $selectedPlayer)
        <div class="fixed inset-0 z-50 overflow-y-auto" x-data="{ show: @entangle('showPlayerModal') }" x-show="show"
            x-transition:enter="ease-out duration-300" x-transition:enter-start="opacity-0"
            x-transition:enter-end="opacity-100" x-transition:leave="ease-in duration-200"
            x-transition:leave-start="opacity-100" x-transition:leave-end="opacity-0"
            @keydown.escape="$wire.closePlayerModal()">

            <!-- Background overlay -->
            <div class="fixed inset-0 player-modal-overlay" wire:click="closePlayerModal"></div>

            <!-- Modal content -->
            <div class="flex items-start justify-center min-h-screen p-2 sm:p-4 pt-4 sm:pt-8">
                <div class="relative player-modal-content rounded-2xl shadow-2xl w-full max-w-4xl mx-auto"
                    x-transition:enter="ease-out duration-300" x-transition:enter-start="opacity-0 transform scale-95"
                    x-transition:enter-end="opacity-100 transform scale-100" x-transition:leave="ease-in duration-200"
                    x-transition:leave-start="opacity-100 transform scale-100"
                    x-transition:leave-end="opacity-0 transform scale-95">

                    <!-- Close button with RTL/LTR support -->
                    <x-modal-close-button action="closePlayerModal" />

                    <!-- Modal header -->
                    <div class="p-4 sm:p-6 border-b border-purple-500/30">
                        <h3 class="text-xl sm:text-2xl font-bold text-white text-center">
                            <i class="fas fa-user text-purple-400 mr-2"></i>
                            <span data-translate="rankings.playerDetails" wire:ignore>Player Details</span>
                        </h3>
                    </div>

                    <!-- Modal body with responsive layout -->
                    <div class="flex flex-col lg:flex-row min-h-[500px]">
                        <!-- Left Panel - Player Info (طولي على الموبايل، يسار على الديسكتوب) -->
                        <div class="w-full lg:w-2/5 p-4 sm:p-6 border-b lg:border-b-0 lg:border-r border-purple-500/30">
                            <!-- Player Avatar Section -->
                            <div class="text-center mb-6">
                                <div class="relative inline-block mb-4">
                                    <img src="{{ $selectedPlayer['avatar'] ?? '/assets/images/avatars/default.gif' }}"
                                        alt="{{ $selectedPlayer['charName'] ?? 'Unknown' }}"
                                        class="w-20 h-20 sm:w-24 sm:h-24 lg:w-28 lg:h-28 mx-auto rounded-full player-modal-avatar">

                                    <!-- Rank badge overlay -->
                                    @if (isset($selectedPlayer['rank']) && $selectedPlayer['rank'] <= 3)
                                        <div
                                            class="absolute -top-2 -right-2 w-6 h-6 sm:w-8 sm:h-8 rounded-full flex items-center justify-center text-xs font-bold
                                        {{ $selectedPlayer['rank'] == 1 ? 'bg-gradient-to-r from-yellow-400 to-yellow-600 text-yellow-900' : '' }}
                                        {{ $selectedPlayer['rank'] == 2 ? 'bg-gradient-to-r from-gray-300 to-gray-500 text-gray-900' : '' }}
                                        {{ $selectedPlayer['rank'] == 3 ? 'bg-gradient-to-r from-amber-600 to-amber-800 text-amber-100' : '' }}">
                                            {{ $selectedPlayer['rank'] }}
                                        </div>
                                    @endif
                                </div>

                                <!-- Player Name -->
                                <h4 class="text-xl sm:text-2xl font-bold text-white mb-3">
                                    {{ $selectedPlayer['charName'] ?? ($selectedPlayer['name'] ?? 'Unknown') }}
                                    @if ($selectedPlayer['is_owned'] ?? false)
                                        <i class="fas fa-crown text-amber-400 ml-2"></i>
                                    @endif
                                </h4>

                                <!-- Player Rank Badge -->
                                @if (isset($selectedPlayer['rank']))
                                    <div
                                        class="inline-flex items-center px-3 py-2 sm:px-4 sm:py-2 bg-gradient-to-r from-purple-600 to-blue-600 rounded-full text-white font-semibold mb-4 text-sm sm:text-base">
                                        <i class="fas fa-trophy mr-2"></i>
                                        <span data-translate="rankings.rank" wire:ignore>Rank</span>:
                                        #{{ $selectedPlayer['rank'] }}
                                    </div>
                                @endif

                                <!-- Ownership Badge -->
                                @if ($selectedPlayer['is_owned'] ?? false)
                                    <div
                                        class="inline-flex items-center px-2 py-1 sm:px-3 sm:py-1 bg-gradient-to-r from-amber-500 to-yellow-500 rounded-full text-black font-semibold text-xs sm:text-sm mb-4">
                                        <i class="fas fa-user-check mr-1"></i>
                                        <span data-translate="rankings.yourCharacter" wire:ignore>Your Character</span>
                                    </div>
                                @endif
                            </div>

                            <!-- Player Stats Grid -->
                            <div class="grid grid-cols-2 gap-2 sm:gap-3 lg:gap-4 mb-6">
                                <!-- Points -->
                                @if (isset($selectedPlayer['points']))
                                    <div
                                        class="bg-gradient-to-r from-blue-600/20 to-purple-600/20 rounded-lg sm:rounded-xl p-2 sm:p-3 lg:p-4 border border-blue-500/30">
                                        <div class="text-lg sm:text-xl lg:text-2xl font-bold text-blue-400 mb-1">
                                            {{ number_format($selectedPlayer['points']) }}
                                        </div>
                                        <div class="text-blue-200 text-xs sm:text-sm" data-translate="rankings.points" wire:ignore>
                                            Points
                                        </div>
                                    </div>
                                @endif

                                <!-- Kills -->
                                @if (isset($selectedPlayer['kills']))
                                    <div
                                        class="bg-gradient-to-r from-green-600/20 to-emerald-600/20 rounded-lg sm:rounded-xl p-2 sm:p-3 lg:p-4 border border-green-500/30">
                                        <div class="text-lg sm:text-xl lg:text-2xl font-bold text-green-400 mb-1">
                                            {{ number_format($selectedPlayer['kills']) }}
                                        </div>
                                        <div class="text-green-200 text-xs sm:text-sm" data-translate="rankings.kills" wire:ignore>
                                            Kills
                                        </div>
                                    </div>
                                @endif

                                <!-- Deaths -->
                                @if (isset($selectedPlayer['deaths']))
                                    <div
                                        class="bg-gradient-to-r from-red-600/20 to-rose-600/20 rounded-lg sm:rounded-xl p-2 sm:p-3 lg:p-4 border border-red-500/30">
                                        <div class="text-lg sm:text-xl lg:text-2xl font-bold text-red-400 mb-1">
                                            {{ number_format($selectedPlayer['deaths']) }}
                                        </div>
                                        <div class="text-red-200 text-xs sm:text-sm" data-translate="rankings.deaths" wire:ignore>
                                            Deaths
                                        </div>
                                    </div>
                                @endif

                                <!-- K/D Ratio -->
                                @if (isset($selectedPlayer['kills']) && isset($selectedPlayer['deaths']))
                                    <div
                                        class="bg-gradient-to-r from-cyan-600/20 to-blue-600/20 rounded-lg sm:rounded-xl p-2 sm:p-3 lg:p-4 border border-cyan-500/30">
                                        <div class="text-lg sm:text-xl lg:text-2xl font-bold text-cyan-400 mb-1">
                                            {{ $this->calculateKDRatio($selectedPlayer['kills'], $selectedPlayer['deaths']) }}
                                        </div>
                                        <div class="text-cyan-200 text-xs sm:text-sm" data-translate="rankings.kdRatio"
                                            wire:ignore>
                                            K/D Ratio
                                        </div>
                                    </div>
                                @endif
                            </div>

                            <!-- Character Info -->
                            @if (isset($selectedPlayer['job_type']) || isset($selectedPlayer['refObjID']))
                                <div class="bg-slate-800/50 rounded-lg border border-slate-600/30 p-3 sm:p-4">
                                    <h5 class="text-base sm:text-lg font-semibold text-white mb-3">
                                        <i class="fas fa-info-circle text-slate-400 mr-2"></i>
                                        <span data-translate="rankings.characterInfo" wire:ignore>Character Info</span>
                                    </h5>

                                    <div class="grid grid-cols-1 gap-2 sm:gap-3 text-xs sm:text-sm">
                                        @if (isset($selectedPlayer['job_type']))
                                            <div class="flex justify-between items-center">
                                                <span class="text-slate-400" data-translate="rankings.job"
                                                    wire:ignore>Job:</span>
                                                <div class="job-badge-with-icon {{ $this->getJobBadgeClasses($selectedPlayer['job_type']) }}">
                                                    <i class="{{ $this->getJobIcon($selectedPlayer['job_type']) }}"></i>
                                                    {{ $this->getPlayerJobType($selectedPlayer['job_type']) }}
                                                </div>
                                            </div>
                                        @endif

                                        @if (isset($selectedPlayer['refObjID']))
                                            <div class="flex justify-between items-center">
                                                <span class="text-slate-400" data-translate="rankings.race"
                                                    wire:ignore>Race:</span>
                                                <span
                                                    class="text-white font-medium">{{ $this->getPlayerRace($selectedPlayer['refObjID']) }}</span>
                                            </div>

                                            <div class="flex justify-between items-center">
                                                <span class="text-slate-400" data-translate="rankings.gender"
                                                    wire:ignore>Gender:</span>
                                                <span
                                                    class="text-white font-medium">{{ $this->getPlayerGender($selectedPlayer['refObjID']) }}</span>
                                            </div>
                                        @endif
                                    </div>
                                </div>
                            @endif
                        </div>

                        <!-- Right Panel - Inventory Grid (أسفل على الموبايل، يمين على الديسكتوب) -->
                        <div class="w-full lg:w-3/5 p-4 sm:p-6">
                            <h5 class="text-base sm:text-lg font-semibold text-white mb-4 text-center lg:text-left">
                                <i class="fas fa-boxes text-purple-400 mr-2"></i>
                                <span data-translate="rankings.playerInventory" wire:ignore>Player Equipment</span>
                            </h5>

                            <!-- Silkroad Online Style Inventory Component -->
                            <x-silkroad-inventory
                                :char-id="$selectedPlayer['charName'] ?? null"
                                :items="$playerEquipment"
                            />
                        </div>
                    </div>
                </div>
            </div>
        </div>
    @endif
</div>
