<?php

use Livewire\Volt\Component;
use App\Models\Z_A_BOTS_SYSTEMS\EventBasicsInfo;
use App\Models\Z_A_BOTS_SYSTEMS\EventBasicsTimer;
use Carbon\Carbon;
use Livewire\Attributes\On;

new class extends Component {
    public $selectedEventId = null;
    public $eventName = '';
    public $isEventActive = false;
    public $formattedEndDate = '';
    public $formattedEndTime = '';
    public $dayName = '';
    public $timeRemaining = '';
    public $timeStatus = ''; // 'remaining' or 'expired'
    public $timeDifference = [];

    public function mount($selectedEventId = null)
    {
        if ($selectedEventId) {
            $this->loadEventEndTime($selectedEventId);
        }
    }

    #[On('event-selected')]
    public function loadEventEndTime($eventId)
    {
        $this->selectedEventId = $eventId;

        try {
            // Load event basic info
            $event = EventBasicsInfo::find($eventId);
            if ($event) {
                $this->eventName = $event->EventName ?? '';
                $this->isEventActive = $event->Active ?? false;

                // Get current active timer using the new relationship
                $timer = $event->getCurrentActiveTimer();
            } else {
                $timer = null;
            }

            if ($timer && $timer->End_Time) {
                $endDateTime = Carbon::parse($timer->End_Time);

                // Format for display
                $this->formattedEndDate = $endDateTime->format('Y-m-d');
                $this->formattedEndTime = $endDateTime->format('g:i A'); // 12-hour format with AM/PM
                $this->dayName = $endDateTime->format('l'); // Day name like "Monday"

                // Calculate time difference
                $this->calculateTimeDifference($endDateTime);
            } else {
                $this->formattedEndDate = '';
                $this->formattedEndTime = '';
                $this->dayName = '';
                $this->timeRemaining = '';
                $this->timeStatus = '';
                $this->timeDifference = [];
            }
        } catch (\Exception $e) {
            // Silently handle error
            $this->formattedEndDate = '';
            $this->formattedEndTime = '';
            $this->dayName = '';
            $this->timeRemaining = '';
            $this->timeStatus = '';
            $this->timeDifference = [];
        }
    }

    private function calculateTimeDifference($endDateTime)
    {
        $now = Carbon::now();

        if ($now->lt($endDateTime)) {
            // Event hasn't ended yet
            $this->timeStatus = 'remaining';
            $diff = $now->diff($endDateTime);

            $this->timeDifference = [
                'days' => $diff->days,
                'hours' => $diff->h,
                'minutes' => $diff->i,
                'seconds' => $diff->s
            ];

            if ($diff->days > 0) {
                $this->timeRemaining = $diff->days . ' days, ' . $diff->h . ' hours';
            } elseif ($diff->h > 0) {
                $this->timeRemaining = $diff->h . ' hours, ' . $diff->i . ' minutes';
            } elseif ($diff->i > 0) {
                $this->timeRemaining = $diff->i . ' minutes, ' . $diff->s . ' seconds';
            } else {
                $this->timeRemaining = $diff->s . ' seconds';
            }
        } else {
            // Event has ended
            $this->timeStatus = 'expired';
            $diff = $endDateTime->diff($now);

            $this->timeDifference = [
                'days' => $diff->days,
                'hours' => $diff->h,
                'minutes' => $diff->i,
                'seconds' => $diff->s
            ];

            if ($diff->days > 0) {
                $this->timeRemaining = $diff->days . ' days, ' . $diff->h . ' hours ago';
            } elseif ($diff->h > 0) {
                $this->timeRemaining = $diff->h . ' hours, ' . $diff->i . ' minutes ago';
            } elseif ($diff->i > 0) {
                $this->timeRemaining = $diff->i . ' minutes ago';
            } else {
                $this->timeRemaining = $diff->s . ' seconds ago';
            }
        }
    }

    public function refreshTimer()
    {
        if ($this->selectedEventId) {
            $this->loadEventEndTime($this->selectedEventId);
        }
    }

    #[On('refresh-events')]
    public function refreshEventTimer()
    {
        if ($this->selectedEventId) {
            $this->loadEventEndTime($this->selectedEventId);
        }
    }
};
?>

<div>
    @if ($isEventActive && $formattedEndDate && $formattedEndTime)
        <div class="event-info-card">
            @if($timeStatus === 'remaining')
                <!-- Show timer interface only if event is still running -->
                <div class="flex items-center justify-between mb-6">
                    <h3 class="text-2xl font-bold text-white flex items-center gap-3">
                        ⏰ <span data-translate="rankings.eventEndTime" wire:ignore>Event Timer</span>
                    </h3>
                </div>
            @else
                <!-- Simple header for ended events -->
                <div class="mb-6">
                    <h3 class="text-2xl font-bold text-white flex items-center gap-3">
                        ⏰ <span data-translate="rankings.eventEndTime" wire:ignore>Event Information</span>
                    </h3>
                </div>
            @endif

            <div class="bg-gradient-to-r from-purple-900/30 to-pink-900/30 backdrop-blur-lg border border-purple-500/30 rounded-2xl p-6">

                @if($timeStatus === 'remaining')
                    <!-- Time Status Display - only for active events -->
                    @if($timeRemaining)
                        <div class="mb-6 text-center">
                            <div class="inline-flex items-center px-6 py-3 bg-green-500/20 border-green-500/40 border rounded-2xl">
                                <div class="flex items-center gap-3">
                                    <div class="w-3 h-3 bg-green-400 rounded-full animate-pulse"></div>
                                    <span class="text-green-300 font-bold text-lg">⏳ Time Remaining</span>
                                </div>
                            </div>

                        </div>

                        <!-- Detailed Time Breakdown - only for active events -->
                        @if(!empty($timeDifference))
                            <div class="grid grid-cols-4 gap-3 mb-6">
                                <div class="bg-slate-800/50 rounded-xl p-3 text-center border border-slate-600/30">
                                    <div class="text-2xl font-bold text-white">{{ $timeDifference['days'] ?? 0 }}</div>
                                    <div class="text-xs text-gray-400 uppercase">Days</div>
                                </div>
                                <div class="bg-slate-800/50 rounded-xl p-3 text-center border border-slate-600/30">
                                    <div class="text-2xl font-bold text-white">{{ $timeDifference['hours'] ?? 0 }}</div>
                                    <div class="text-xs text-gray-400 uppercase">Hours</div>
                                </div>
                                <div class="bg-slate-800/50 rounded-xl p-3 text-center border border-slate-600/30">
                                    <div class="text-2xl font-bold text-white">{{ $timeDifference['minutes'] ?? 0 }}</div>
                                    <div class="text-xs text-gray-400 uppercase">Minutes</div>
                                </div>
                                <div class="bg-slate-800/50 rounded-xl p-3 text-center border border-slate-600/30">
                                    <div class="text-2xl font-bold text-white">{{ $timeDifference['seconds'] ?? 0 }}</div>
                                    <div class="text-xs text-gray-400 uppercase">Seconds</div>
                                </div>
                            </div>
                        @endif
                    @endif
                @else
                    <!-- Event Ended Message - simple and clean -->
                    <div class="mb-6 text-center">
                        <div class="inline-flex items-center px-6 py-3 bg-gray-500/20 border-gray-500/40 border rounded-2xl">
                            <div class="flex items-center gap-3">
                                <div class="w-3 h-3 bg-gray-400 rounded-full"></div>
                                <span class="text-gray-300 font-bold text-lg">🏁 Event Ended</span>
                            </div>
                        </div>
                    </div>
                @endif

                <!-- Event End Date & Time Display - always shown -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <!-- End Date -->
                    <div class="bg-slate-900/40 rounded-xl p-4 border border-purple-500/20">
                        <div class="flex items-center space-x-3">
                            <div class="p-2 bg-purple-500/20 rounded-lg">
                                <span class="text-purple-400 text-xl">📅</span>
                            </div>
                            <div>
                                <p class="text-purple-300 font-medium text-sm" data-translate="rankings.endDate" wire:ignore>End Date</p>
                                <p class="text-white font-bold text-lg">{{ $formattedEndDate }}</p>
                                @if($dayName)
                                    <p class="text-gray-400 text-sm">{{ $dayName }}</p>
                                @endif
                            </div>
                        </div>
                    </div>

                    <!-- End Time -->
                    <div class="bg-slate-900/40 rounded-xl p-4 border border-pink-500/20">
                        <div class="flex items-center space-x-3">
                            <div class="p-2 bg-pink-500/20 rounded-lg">
                                <span class="text-pink-400 text-xl">🕐</span>
                            </div>
                            <div>
                                <p class="text-pink-300 font-medium text-sm" data-translate="rankings.endTime" wire:ignore>End Time</p>
                                <p class="text-white font-bold text-lg">{{ $formattedEndTime }}</p>
                                <p class="text-gray-400 text-sm" data-translate="rankings.serverTime" wire:ignore>Server Time</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    @endif
</div>
