<?php

use Livewire\Volt\Component;
use App\Models\Z_A_BOTS_SYSTEMS\EventRulesBasic;
use Livewire\Attributes\On;

new class extends Component {
    public $eventRules = null;
    public $selectedEventId = null;
    public $activeRules = [];

    public function mount($selectedEventId = null)
    {
        if ($selectedEventId) {
            $this->loadEventRules($selectedEventId);
        }
    }

    #[On('event-selected')]
    public function loadEventRules($eventId)
    {
        $this->selectedEventId = $eventId;
        try {
            $this->eventRules = EventRulesBasic::where('EventID', $eventId)->first();
            $this->buildActiveRules();
        } catch (\Exception $e) {
            $this->eventRules = null;
            $this->activeRules = [];
        }
    }

    #[On('refresh-events')]
    public function refreshEventInfo()
    {
        if ($this->selectedEventId) {
            $this->loadEventRules($this->selectedEventId);
        }
    }

    private function buildActiveRules()
    {
        $this->activeRules = [];

        if (!$this->eventRules) {
            return;
        }

        // Always show level range if any value exists
        if ($this->eventRules->Cur_Level_Start || $this->eventRules->Cur_Level_End) {
            $this->activeRules[] = [
                'type' => 'level_range',
                'icon' => '🔢',
                'color' => 'emerald',
                'label' => 'Level Range',
                'value' => ($this->eventRules->Cur_Level_Start ?: '?') . ' - ' . ($this->eventRules->Cur_Level_End ?: '?'),
                'animation' => 'fade-slide-up'
            ];
        }

        // Show PC Limit if exists
        if ($this->eventRules->LimitPC) {
            $this->activeRules[] = [
                'type' => 'pc_limit',
                'icon' => '💻',
                'color' => 'blue',
                'label' => 'PC Limit',
                'value' => $this->eventRules->LimitPC,
                'animation' => 'fade-slide-right'
            ];
        }

        // Show Job System if enabled
        if ($this->eventRules->Job) {
            $this->activeRules[] = [
                'type' => 'job_system',
                'icon' => '💼',
                'color' => 'cyan',
                'label' => 'Job System',
                'value' => 'Active',
                'animation' => 'fade-pulse'
            ];
        }

        // Show jobs if any are enabled
        $jobs = [];
        if ($this->eventRules->Trader) $jobs[] = ['name' => 'Trader', 'color' => 'text-green-300', 'icon' => '💰'];
        if ($this->eventRules->Hunter) $jobs[] = ['name' => 'Hunter', 'color' => 'text-red-300', 'icon' => '🏹'];
        if ($this->eventRules->Thief) $jobs[] = ['name' => 'Thief', 'color' => 'text-purple-300', 'icon' => '🗡️'];

        if (!empty($jobs)) {
            $this->activeRules[] = [
                'type' => 'jobs_allowed',
                'icon' => '👥',
                'color' => 'indigo',
                'label' => 'Jobs Allowed',
                'value' => $jobs,
                'animation' => 'fade-scale'
            ];
        }

        // Show CH/EU if enabled
        if ($this->eventRules->CH_EU) {
            $this->activeRules[] = [
                'type' => 'ch_eu',
                'icon' => '🌍',
                'color' => 'yellow',
                'label' => 'CH/EU',
                'value' => 'Allowed',
                'animation' => 'fade-bounce'
            ];
        }

        // Show Strength if exists
        if ($this->eventRules->Strength_Start || $this->eventRules->Strength_End) {
            $this->activeRules[] = [
                'type' => 'strength_range',
                'icon' => '�',
                'color' => 'red',
                'label' => 'Strength Range',
                'value' => ($this->eventRules->Strength_Start ?: '?') . ' - ' . ($this->eventRules->Strength_End ?: '?'),
                'animation' => 'fade-slide-left'
            ];
        }

        // Show Intellect if exists
        if ($this->eventRules->Intellect_Start || $this->eventRules->Intellect_End) {
            $this->activeRules[] = [
                'type' => 'intellect_range',
                'icon' => '🧠',
                'color' => 'purple',
                'label' => 'Intellect Range',
                'value' => ($this->eventRules->Intellect_Start ?: '?') . ' - ' . ($this->eventRules->Intellect_End ?: '?'),
                'animation' => 'fade-slide-down'
            ];
        }

        // Show Reset System if enabled
        if ($this->eventRules->Reset) {
            $this->activeRules[] = [
                'type' => 'reset_system',
                'icon' => '🔄',
                'color' => 'orange',
                'label' => 'Reset System',
                'value' => 'Allowed',
                'animation' => 'fade-rotate'
            ];
        }
    }
};
?>
<div>
 @if (!empty($activeRules))
    <div class="event-info-card">
        <h3 class="text-2xl font-bold text-white mb-6 flex items-center gap-3">
            📜 <span data-translate="rankings.eventRules" wire:ignore>Event Rules</span>
        </h3>

        <!-- Debug Info -->
        <div class="mb-6 p-4 bg-slate-900/50 rounded-xl border border-slate-600/30 text-sm">
            <div class="grid grid-cols-2 md:grid-cols-3 gap-4">
                <div>
                    <span class="text-gray-400">Active Rules:</span>
                    <span class="text-blue-400 font-bold ml-2">{{ count($activeRules) }}</span>
                </div>
                <div>
                    <span class="text-gray-400">Job System:</span>
                    <span class="text-{{ $eventRules->Job ? 'green' : 'gray' }}-400 ml-2">{{ $eventRules->Job ? 'Active' : 'Inactive' }}</span>
                </div>
                <div>
                    <span class="text-gray-400">Reset Allowed:</span>
                    <span class="text-{{ $eventRules->Reset ? 'green' : 'gray' }}-400 ml-2">{{ $eventRules->Reset ? 'Yes' : 'No' }}</span>
                </div>
                <div>
                    <span class="text-gray-400">CH/EU:</span>
                    <span class="text-{{ $eventRules->CH_EU ? 'green' : 'gray' }}-400 ml-2">{{ $eventRules->CH_EU ? 'Allowed' : 'Not Allowed' }}</span>
                </div>
            </div>
        </div>

        @if (!empty($activeRules))
            <div class="grid gap-4">
                @foreach ($activeRules as $index => $rule)
                    <div class="rule-card group relative overflow-hidden bg-gradient-to-r from-slate-800/50 to-slate-700/30 backdrop-blur-md rounded-2xl border border-white/10 p-5 hover:border-{{ $rule['color'] }}-400/50 hover:shadow-xl hover:shadow-{{ $rule['color'] }}-500/20 transition-all duration-500 hover:scale-[1.02]">
                        <!-- Background Pattern -->
                        <div class="absolute inset-0 opacity-5">
                            <div class="absolute inset-0 bg-gradient-to-br from-{{ $rule['color'] }}-500/20 via-transparent to-{{ $rule['color'] }}-600/10"></div>
                        </div>

                        <!-- Content -->
                        <div class="relative flex items-center space-x-4">
                            <!-- Icon -->
                            <div class="flex-shrink-0 w-14 h-14 bg-{{ $rule['color'] }}-500/20 rounded-xl flex items-center justify-center border border-{{ $rule['color'] }}-400/30 group-hover:bg-{{ $rule['color'] }}-500/30 transition-all duration-300">
                                <span class="text-2xl animate-pulse">{{ $rule['icon'] }}</span>
                            </div>

                            <!-- Text Content -->
                            <div class="flex-1 min-w-0">
                                <h4 class="text-{{ $rule['color'] }}-300 font-bold text-lg mb-1 group-hover:text-{{ $rule['color'] }}-200 transition-colors duration-300">
                                    {{ $rule['label'] }}
                                </h4>

                                <div class="text-white/90 font-medium">
                                    @if (is_array($rule['value']))
                                        <div class="flex flex-wrap gap-2">
                                            @foreach ($rule['value'] as $jobIndex => $job)
                                                <span class="inline-flex items-center px-3 py-1 rounded-full text-sm {{ $job['color'] }} bg-white/10 border border-white/20">
                                                    @if(isset($job['icon']))
                                                        <span class="mr-1">{{ $job['icon'] }}</span>
                                                    @endif
                                                    {{ $job['name'] }}
                                                </span>
                                            @endforeach
                                        </div>
                                    @else
                                        <span class="text-white font-semibold text-lg">{{ $rule['value'] }}</span>
                                    @endif
                                </div>
                            </div>

                            <!-- Status Indicator -->
                            <div class="flex-shrink-0">
                                <div class="w-3 h-3 bg-{{ $rule['color'] }}-400 rounded-full animate-ping"></div>
                            </div>
                        </div>

                        <!-- Subtle glow effect -->
                        <div class="absolute -inset-1 bg-gradient-to-r from-{{ $rule['color'] }}-500/0 via-{{ $rule['color'] }}-500/5 to-{{ $rule['color'] }}-500/0 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-500 -z-10"></div>
                    </div>
                @endforeach
            </div>
        @else
            <div class="text-center py-12 bg-gradient-to-br from-slate-800/40 to-slate-700/20 rounded-2xl border-2 border-dashed border-gray-500/30">
                <div class="mb-4">
                    <div class="w-20 h-20 mx-auto bg-gray-600/20 rounded-full flex items-center justify-center mb-4">
                        <span class="text-4xl text-gray-400">📋</span>
                    </div>
                    <h3 class="text-gray-300 text-xl font-semibold mb-2">No Active Rules</h3>
                    <p class="text-gray-500 text-sm max-w-md mx-auto">
                        This event doesn't have any specific restrictions or requirements.
                        All players can participate freely!
                    </p>
                </div>
                <div class="text-xs text-gray-600 bg-gray-800/30 inline-block px-4 py-2 rounded-full">
                    Event ID: {{ $selectedEventId }} • Rules Processed: {{ count($activeRules) }}
                </div>
            </div>
        @endif
    </div>
@else
    <div class="event-info-card">
                <h3 class="text-2xl font-bold text-white mb-6">📜 Event Rules</h3>
        <div class="text-center text-gray-400 py-8">
            <p>No event rules found</p>
            <p>Event ID: {{ $selectedEventId ?? 'Not selected' }}</p>
        </div>
    </div>
@endif
</div>
