<?php

use Livewire\Volt\Component;
use App\Models\Z_Website\Download;
use App\Models\Z_Website\SiteSettings;
use App\Helpers\SiteSettingsHelper;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;

new class extends Component {
    public $downloads = [];
    public $serverSettings = [];
    public $systemRequirements = [];
    public $downloadStats = [];
    public $activeTab = 'client';

    public function mount()
    {
        $this->loadDownloads();
        $this->loadServerSettings();
        $this->setupSystemRequirements();
        $this->loadDownloadStats();
    }

    public function loadDownloads()
    {
        try {
            $this->downloads = Cache::remember('game_downloads', 1800, function () {
                return Download::active()
                    ->ordered()
                    ->get()
                    ->map(function ($download) {
                        return [
                            'id' => $download->id,
                            'name' => $download->name,
                            'description' => $download->description,
                            'link' => $download->link,
                            'file_size' => $download->formatted_file_size,
                            'version' => $download->version,
                            'category' => $download->category,
                            'download_count' => $download->download_count,
                            'icon' => $download->icon,
                            'created_at' => $download->created_at,
                        ];
                    });
            });
        } catch (\Exception $e) {
            $this->downloads = [];
        }
    }

    public function loadServerSettings()
    {
        try {
            $settings = Cache::remember('server_settings', 300, function () {
                return SiteSettings::first();
            });

            if ($settings && $settings->settings) {
                $this->serverSettings = $settings->settings;
            }
        } catch (\Exception $e) {
            $this->serverSettings = [];
        }
    }

    public function setupSystemRequirements()
    {
        $this->systemRequirements = [
            'minimum' => [
                'os' => 'Windows 10 (64-bit)',
                'processor' => 'Intel Core i3-2100 / AMD FX-6100',
                'memory' => '4 GB RAM',
                'graphics' => 'NVIDIA GTX 750 Ti / AMD Radeon R7 260X',
                'directx' => 'Version 11',
                'network' => 'Broadband Internet connection',
                'storage' => '15 GB available space',
                'additional' => '.NET Framework 4.8'
            ],
            'recommended' => [
                'os' => 'Windows 11 (64-bit)',
                'processor' => 'Intel Core i5-8400 / AMD Ryzen 5 2600',
                'memory' => '8 GB RAM',
                'graphics' => 'NVIDIA GTX 1060 / AMD RX 580',
                'directx' => 'Version 12',
                'network' => 'Broadband Internet connection',
                'storage' => '20 GB available space (SSD recommended)',
                'additional' => '.NET Framework 4.8, Visual C++ Redistributable'
            ]
        ];
    }

    public function loadDownloadStats()
    {
        // استخدام الدوال المساعدة الجديدة
        $social = socialLinks();
        $rates = serverRates();
        $limits = serverLimits();

        $this->downloadStats = [
            'total_downloads' => '50,000+',
            'server_version' => $this->serverSettings['server_version'] ?? '1.500',
            'last_update' => Carbon::now()->subDays(7)->format('Y-m-d'),
            'client_size' => '4.5 GB',
            // الروابط الاجتماعية
            'facebook_url' => $social['facebook'],
            'youtube_url' => $social['youtube'],
            'contact_email' => $social['contact_email'],
            'discord_invite' => $social['discord_invite'],
            // معدلات الخادم
            'exp_rate' => $rates['exp'],
            'gold_rate' => $rates['gold'],
            'drop_rate' => $rates['drop'],
            'job_rate' => $rates['job'],
            'party_rate' => $rates['party'],
            // حدود الخادم
            'max_server' => $limits['max_server'],
            'level_cap' => $limits['level_cap'],
            'ip_limit' => $limits['ip_limit'],
            'hwid_limit' => $limits['hwid_limit']
        ];
    }

    public function setActiveTab($tab)
    {
        $this->activeTab = $tab;
    }

    public function trackDownload($downloadId, $downloadName)
    {
        try {
            // Increment download count
            $download = Download::find($downloadId);
            if ($download) {
                $download->incrementDownloadCount();

                // تحديث الكاش فوراً لإظهار العدد الجديد
                Cache::forget('game_downloads');
                $this->loadDownloads();
            }

            // Log download activity (optional)
            \Log::info('Download tracked', [
                'download_id' => $downloadId,
                'download_name' => $downloadName,
                'user_ip' => request()->ip(),
                'user_agent' => request()->userAgent(),
                'timestamp' => now()
            ]);

            $this->dispatch('notify', [
                'type' => 'success',
                'message' => "Starting download {$downloadName}..."
            ]);
        } catch (\Exception $e) {
            $this->dispatch('notify', [
                'type' => 'error',
                'message' => 'Download tracking error'
            ]);
        }
    }

    public function refreshDownloads()
    {
        Cache::forget('game_downloads');
        Cache::forget('server_settings');

        $this->loadDownloads();
        $this->loadServerSettings();
        $this->loadDownloadStats();

        $this->dispatch('notify', [
            'type' => 'success',
            'message' => 'Download list updated successfully!'
        ]);
    }

    /**
     * Helper method to get site setting value
     */
    public function getSiteSetting($key, $default = null)
    {
        return $this->serverSettings[$key] ?? $default;
    }

    /**
     * Helper method to get formatted site settings for display
     */
    public function getFormattedSiteSettings()
    {
        return [
            'facebook_url' => $this->getSiteSetting('facebook_url', '#'),
            'youtube_url' => $this->getSiteSetting('youtube_url', '#'),
            'contact_email' => $this->getSiteSetting('contact_email', 'support@4tressro.com'),
            'discord_invite' => $this->getSiteSetting('discord_invite', ''),
            'server_rates' => [
                'exp' => $this->getSiteSetting('sro_exp', '45') . 'x',
                'gold' => $this->getSiteSetting('sro_exp_gold', '45') . 'x',
                'drop' => $this->getSiteSetting('sro_exp_drop', '45') . 'x',
                'job' => $this->getSiteSetting('sro_exp_job', '40') . 'x',
                'party' => $this->getSiteSetting('sro_exp_party', '55') . 'x',
            ],
            'server_limits' => [
                'max_players' => $this->getSiteSetting('sro_max_server', '1000'),
                'level_cap' => $this->getSiteSetting('sro_cap', '130'),
                'ip_limit' => $this->getSiteSetting('sro_ip_limit', '99'),
                'hwid_limit' => $this->getSiteSetting('sro_hwid_limit', '8'),
            ],
            'currency' => [
                'silk_name' => $this->getSiteSetting('sro_silk_name', 'Silk'),
                'silk_gift_name' => $this->getSiteSetting('sro_silk_gift_name', '4T Donate'),
            ]
        ];
    }
}; ?>

<div class="min-h-screen bg-gradient-to-br from-slate-900 via-purple-900 to-slate-900">

    <!-- Hero Section -->
    <div class="relative overflow-hidden">
        <!-- Background Effects -->
        <div class="absolute inset-0 bg-gradient-to-br from-purple-900/20 to-cyan-900/20"></div>
        <div class="absolute inset-0 bg-[url('/assets/silkroad-bg.jpg')] bg-cover bg-center opacity-10"></div>

        <!-- Animated Background Elements -->
        <div class="absolute inset-0 overflow-hidden pointer-events-none">
            <div class="absolute top-20 left-10 w-64 h-64 bg-purple-500/10 rounded-full blur-3xl animate-pulse"></div>
            <div class="absolute bottom-20 right-10 w-96 h-96 bg-cyan-500/10 rounded-full blur-3xl animate-pulse delay-1000"></div>
            <div class="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-128 h-128 bg-purple-600/5 rounded-full blur-3xl animate-pulse delay-500"></div>
        </div>

        <div class="relative container mx-auto px-6 py-24">
            <div class="text-center">
                <!-- Download Icon -->
                <div class="inline-flex items-center justify-center w-32 h-32 mb-8 bg-gradient-to-br from-purple-500 to-cyan-500 rounded-full shadow-2xl">
                    <svg class="w-16 h-16 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3M3 17V7a2 2 0 012-2h6l2 2h6a2 2 0 012 2v10a2 2 0 01-2 2H5a2 2 0 01-2-2z"></path>
                    </svg>
                </div>

                <h1 class="text-6xl font-bold mb-6 bg-gradient-to-r from-purple-400 to-cyan-400 bg-clip-text text-transparent">
                    <span data-translate="download.title">Download Game</span>
                </h1>
                <p class="text-xl text-purple-100 mb-8 max-w-3xl mx-auto leading-relaxed">
                    <span data-translate="download.subtitle">Download the complete 4TresSRO Online client and start your epic adventure in the ancient 4TresSro world</span>
                </p>

                <!-- Quick Stats -->
                <div class="flex flex-wrap justify-center gap-8 text-white/80 mb-8">
                    <div class="flex items-center gap-2">
                        <span class="w-2 h-2 bg-green-400 rounded-full animate-pulse"></span>
                        <span data-translate="download.serverOnline">Server Online</span>
                    </div>
                    <div class="flex items-center gap-2">
                        <span>📦</span>
                        <span>{{ $downloadStats['client_size'] }}</span>
                    </div>
                    <div class="flex items-center gap-2">
                        <span>🔄</span>
                        <span data-translate="download.lastUpdated">Last Updated</span>: {{ $downloadStats['last_update'] }}
                    </div>
                </div>

                <!-- Server Information -->
                <div class="bg-slate-800/50 backdrop-blur-lg border border-purple-500/30 rounded-2xl p-6 max-w-4xl mx-auto">
                    <div class="grid grid-cols-2 md:grid-cols-4 gap-4 text-center">
                        <div class="bg-gradient-to-br from-purple-600/20 to-cyan-600/20 rounded-xl p-4">
                            <div class="text-2xl font-bold text-white">{{ $downloadStats['level_cap'] }}</div>
                            <div class="text-sm text-purple-300" data-translate="welcome.levelCap">Level Cap</div>
                        </div>
                        <div class="bg-gradient-to-br from-purple-600/20 to-cyan-600/20 rounded-xl p-4">
                            <div class="text-2xl font-bold text-white">{{ $downloadStats['exp_rate'] }}x</div>
                            <div class="text-sm text-purple-300" data-translate="welcome.expRate">EXP Rate</div>
                        </div>
                        <div class="bg-gradient-to-br from-purple-600/20 to-cyan-600/20 rounded-xl p-4">
                            <div class="text-2xl font-bold text-white">{{ $downloadStats['drop_rate'] }}x</div>
                            <div class="text-sm text-purple-300" data-translate="welcome.dropRate">Drop Rate</div>
                        </div>
                        <div class="bg-gradient-to-br from-purple-600/20 to-cyan-600/20 rounded-xl p-4">
                            <div class="text-2xl font-bold text-white">{{ $downloadStats['max_server'] }}</div>
                            <div class="text-sm text-purple-300" data-translate="download.maxPlayers">Max Players</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Download Tabs Section -->
    <div class="container mx-auto px-6 py-16">
        <!-- Tab Navigation -->
        <div class="flex flex-wrap justify-center gap-4 mb-12">
            @foreach([
                ['key' => 'client', 'label' => 'Game Client', 'icon' => '🎮'],
                ['key' => 'tools', 'label' => 'Tools & Utilities', 'icon' => '🔧'],
                ['key' => 'requirements', 'label' => 'System Requirements', 'icon' => '💻']
            ] as $tab)
                <button
                    wire:click="setActiveTab('{{ $tab['key'] }}')"
                    class="flex items-center gap-3 px-6 py-3 rounded-xl transition-all duration-300 {{ $activeTab === $tab['key'] ? 'bg-gradient-to-r from-purple-500 to-cyan-500 text-white shadow-lg' : 'bg-slate-700/50 text-purple-200 hover:bg-slate-600/50' }}">
                    <span class="text-xl">{{ $tab['icon'] }}</span>
                    <span data-translate="download.{{ $tab['key'] }}">{{ $tab['label'] }}</span>
                </button>
            @endforeach
        </div>

        <!-- Tab Content -->
        <div class="transition-all duration-500">

            <!-- Game Client Tab -->
            @if($activeTab === 'client')
                <div class="space-y-8">
                    <!-- Main Download Card -->
                    <div class="bg-gradient-to-br from-slate-800/50 to-purple-900/30 backdrop-blur-lg border border-purple-500/30 rounded-2xl p-8 shadow-2xl">
                        <div class="flex flex-col lg:flex-row items-center gap-8">
                            <!-- Game Logo/Image -->
                            <div class="flex-shrink-0">
                                <div class="w-32 h-32 bg-gradient-to-br from-purple-500 to-cyan-500 rounded-2xl flex items-center justify-center shadow-xl p-4">
                                    <img src="/assets/logo-small.png" alt="4TresSRO Logo" class="w-full h-full object-contain">
                                </div>
                            </div>

                            <!-- Download Info -->
                            <div class="flex-grow text-center lg:text-left">
                                <h3 class="text-3xl font-bold text-white mb-4">
                                    <span data-translate="download.fullClient">4TresSRO Online - Full Client</span>
                                </h3>
                                <p class="text-purple-200 mb-6 text-lg">
                                    <span data-translate="download.clientDescription">Complete game client with all latest updates and features</span>
                                </p>

                                <!-- Download Stats -->
                                <div class="flex flex-wrap justify-center lg:justify-start gap-6 text-sm text-purple-300 mb-6">
                                    <div class="flex items-center gap-2">
                                        <span>📊</span>
                                        <span data-translate="download.version">Version</span>: {{ $downloadStats['server_version'] }}
                                    </div>
                                    <div class="flex items-center gap-2">
                                        <span>💾</span>
                                        <span data-translate="download.size">Size</span>: {{ $downloadStats['client_size'] }}
                                    </div>
                                    <div class="flex items-center gap-2">
                                        <span>⬇️</span>
                                        <span data-translate="download.downloads">Downloads</span>: {{ $downloadStats['total_downloads'] }}
                                    </div>
                                </div>

                                <!-- Download Options -->
                    @php
                        $clientDownloads = collect($downloads)->where('category', 'client');
                    @endphp

                    @if($clientDownloads->count() > 0)
                        <!-- All Download Options Grid -->
                        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                            @foreach($clientDownloads as $download)
                                <div class="bg-slate-700/50 backdrop-blur-lg border border-purple-500/20 rounded-xl p-6 hover:border-purple-400/40 transition-all duration-300 hover:transform hover:scale-105 group">
                                    <!-- Platform Logo -->
                                    <div class="text-center mb-4">
                                        @if($download['icon'])
                                            <!-- عرض الأيقونة من قاعدة البيانات -->
                                            @if(str_starts_with($download['icon'], '<svg'))
                                                <!-- SVG Icon من قاعدة البيانات -->
                                                <div class="w-16 h-16 mx-auto bg-white rounded-full flex items-center justify-center shadow-lg">
                                                    {!! $download['icon'] !!}
                                                </div>
                                            @else
                                                <!-- مسار صورة من قاعدة البيانات -->
                                                <div class="w-16 h-16 mx-auto rounded-full flex items-center justify-center shadow-lg overflow-hidden">
                                                    <img src="{{ $download['icon'] }}" alt="{{ $download['name'] }}" class="w-full h-full object-cover">
                                                </div>
                                            @endif
                                            <h4 class="text-lg font-bold text-white mt-2">{{ $download['name'] }}</h4>
                                        @else
                                            <!-- أيقونة افتراضية -->
                                            <div class="w-16 h-16 mx-auto bg-purple-600 rounded-full flex items-center justify-center shadow-lg">
                                                <svg class="w-10 h-10 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3"></path>
                                                </svg>
                                            </div>
                                            <h4 class="text-lg font-bold text-white mt-2">{{ $download['name'] }}</h4>
                                        @endif
                                    </div>

                                    <!-- Download Info -->
                                    <div class="text-center mb-4">
                                        @if($download['description'])
                                            <p class="text-purple-200 text-sm mb-3">{{ \Str::limit($download['description'], 60) }}</p>
                                        @endif

                                        <div class="flex justify-between items-center text-xs text-purple-300 mb-4">
                                            <span>{{ $download['file_size'] }}</span>
                                            @if($download['version'])
                                                <span class="bg-purple-600 text-white px-2 py-1 rounded">
                                                    <span data-translate="download.versionPrefix">v</span>{{ $download['version'] }}
                                                </span>
                                            @endif
                                        </div>
                                    </div>

                                    <!-- Download Button -->
                                    <a href="{{ $download['link'] }}"
                                       target="_blank"
                                       wire:click="trackDownload({{ $download['id'] }}, '{{ $download['name'] }}')"
                                       class="block w-full text-center px-4 py-3 bg-gradient-to-r from-purple-600 to-cyan-600 hover:from-purple-700 hover:to-cyan-700 text-white rounded-lg font-bold transition-all duration-300 group-hover:shadow-xl">
                                        <div class="flex items-center justify-center gap-2">
                                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3"></path>
                                            </svg>
                                            <span data-translate="download.download">Download</span>
                                        </div>
                                    </a>

                                    <!-- Download Count -->
                                    <div class="text-center mt-3 text-xs text-purple-400">
                                        <span>⬇️ {{ number_format($download['download_count']) }} <span data-translate="download.downloads">downloads</span></span>
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    @else
                        <div class="text-center">
                            <div class="inline-flex items-center gap-3 px-8 py-4 bg-slate-600 text-slate-300 rounded-xl font-bold text-lg cursor-not-allowed">
                                <span data-translate="download.comingSoon">Coming Soon</span>
                            </div>
                        </div>
                    @endif
                </div>
            </div>
        </div>
                </div>
            @endif

            <!-- Tools & Utilities Tab -->
            @if($activeTab === 'tools')
                @php
                    $toolDownloads = collect($downloads)->where('category', 'tool');
                @endphp

                @if($toolDownloads->count() > 0)
                    <div class="space-y-4">
                        @foreach($toolDownloads as $tool)
                            <div class="bg-slate-700/50 backdrop-blur-lg border border-purple-500/20 rounded-xl p-6 hover:border-purple-400/40 transition-all duration-300 hover:transform hover:scale-[1.02]">
                                <div class="flex items-center gap-6">
                                    <!-- Icon Column -->
                                    <div class="flex-shrink-0">
                                        @if($tool['icon'])
                                            <!-- عرض الأيقونة من قاعدة البيانات -->
                                            @if(str_starts_with($tool['icon'], '<svg'))
                                                <!-- SVG Icon من قاعدة البيانات -->
                                                <div class="w-16 h-16 bg-gradient-to-br from-purple-500 to-pink-500 rounded-xl flex items-center justify-center shadow-lg">
                                                    {!! $tool['icon'] !!}
                                                </div>
                                            @else
                                                <!-- مسار صورة من قاعدة البيانات -->
                                                <div class="w-16 h-16 rounded-xl flex items-center justify-center shadow-lg overflow-hidden">
                                                    <img src="{{ $tool['icon'] }}" alt="{{ $tool['name'] }}" class="w-full h-full object-cover">
                                                </div>
                                            @endif
                                        @else
                                            <!-- أيقونة افتراضية -->
                                            <div class="w-16 h-16 bg-gradient-to-br from-gray-500 to-slate-500 rounded-xl flex items-center justify-center shadow-lg">
                                                <svg class="w-10 h-10 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3"></path>
                                                </svg>
                                            </div>
                                        @endif
                                    </div>

                                    <!-- Content Column -->
                                    <div class="flex-grow">
                                        <div class="flex justify-between items-start mb-2">
                                            <h4 class="text-xl font-bold text-white">{{ $tool['name'] }}</h4>
                                            @if($tool['version'])
                                                <span class="text-xs bg-purple-600 text-white px-2 py-1 rounded ml-2">
                                                    <span data-translate="download.versionPrefix">v</span>{{ $tool['version'] }}
                                                </span>
                                            @endif
                                        </div>

                                        @if($tool['description'])
                                            <p class="text-purple-200 text-sm mb-3">{{ $tool['description'] }}</p>
                                        @endif

                                        <div class="flex items-center justify-between">
                                            <div class="flex items-center gap-4 text-sm text-purple-300">
                                                <span class="flex items-center gap-1">
                                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-8l-4-4m0 0L8 8m4-4v12"></path>
                                                    </svg>
                                                    {{ $tool['file_size'] }}
                                                </span>
                                                <span class="flex items-center gap-1">
                                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3"></path>
                                                    </svg>
                                                    {{ number_format($tool['download_count']) }} <span data-translate="download.downloads">downloads</span>
                                                </span>
                                            </div>

                                            <a href="{{ $tool['link'] }}"
                                               target="_blank"
                                               wire:click="trackDownload({{ $tool['id'] }}, '{{ $tool['name'] }}')"
                                               class="inline-flex items-center gap-2 px-6 py-2 bg-gradient-to-r from-purple-600 to-cyan-600 hover:from-purple-700 hover:to-cyan-700 text-white rounded-lg font-medium transition-all duration-300 shadow-md hover:shadow-lg">
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3"></path>
                                                </svg>
                                                <span data-translate="download.download">Download</span>
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <!-- Default Tools when no database tools exist -->
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        @foreach([
                            ['name' => '.NET Framework 4.8', 'name_key' => 'download.netFramework', 'description' => 'Required runtime for the game', 'desc_key' => 'download.netFrameworkDesc', 'icon' => '⚙️', 'url' => 'https://dotnet.microsoft.com/download/dotnet-framework/net48'],
                            ['name' => 'Visual C++ Redistributable', 'name_key' => 'download.visualCpp', 'description' => 'Microsoft Visual C++ runtime libraries', 'desc_key' => 'download.visualCppDesc', 'icon' => '🔧', 'url' => 'https://support.microsoft.com/help/2977003/the-latest-supported-visual-c-downloads'],
                            ['name' => 'DirectX End-User Runtime', 'name_key' => 'download.directx', 'description' => 'DirectX runtime for enhanced graphics', 'desc_key' => 'download.directxDesc', 'icon' => '🎮', 'url' => 'https://www.microsoft.com/download/details.aspx?id=35'],
                            ['name' => 'Game Guide PDF', 'name_key' => 'download.gameGuidePdf', 'description' => 'Complete beginner guide', 'desc_key' => 'download.gameGuideDesc', 'icon' => '📖', 'url' => '#'],
                            ['name' => 'Character Creation Tool', 'name_key' => 'download.characterCreationTool', 'description' => 'Plan your character build', 'desc_key' => 'download.characterToolDesc', 'icon' => '👤', 'url' => '#'],
                            ['name' => 'Trade Calculator', 'name_key' => 'download.tradeCalculator', 'description' => 'Calculate trade profits', 'desc_key' => 'download.tradeCalcDesc', 'icon' => '💰', 'url' => '#']
                        ] as $tool)
                            <div class="bg-slate-700/50 backdrop-blur-lg border border-purple-500/20 rounded-xl p-6 hover:border-purple-400/40 transition-all duration-300 hover:transform hover:scale-105">
                                <div class="text-4xl mb-4">{{ $tool['icon'] }}</div>
                                <h4 class="text-xl font-bold text-white mb-2">
                                    <span data-translate="{{ $tool['name_key'] }}">{{ $tool['name'] }}</span>
                                </h4>
                                <p class="text-purple-200 text-sm mb-4">
                                    <span data-translate="{{ $tool['desc_key'] }}">{{ $tool['description'] }}</span>
                                </p>
                                <a href="{{ $tool['url'] }}"
                                   target="_blank"
                                   class="inline-flex items-center gap-2 px-4 py-2 bg-purple-600 hover:bg-purple-700 text-white rounded-lg transition-colors text-sm">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3"></path>
                                    </svg>
                                    <span data-translate="download.download">Download</span>
                                </a>
                            </div>
                        @endforeach
                    </div>
                @endif
            @endif

            <!-- System Requirements Tab -->
            @if($activeTab === 'requirements')
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
                    <!-- Minimum Requirements -->
                    <div class="bg-gradient-to-br from-slate-800/50 to-orange-900/30 backdrop-blur-lg border border-orange-500/30 rounded-2xl p-8 shadow-2xl">
                        <div class="flex items-center gap-3 mb-6">
                            <div class="w-12 h-12 bg-orange-500 rounded-xl flex items-center justify-center">
                                <span class="text-2xl">⚠️</span>
                            </div>
                            <h3 class="text-2xl font-bold text-white">
                                <span data-translate="download.minimumRequirements">Minimum Requirements</span>
                            </h3>
                        </div>

                        <div class="space-y-4">
                            @foreach($systemRequirements['minimum'] as $key => $value)
                                <div class="flex justify-between items-center py-2 border-b border-orange-500/20">
                                    <span class="text-orange-200 font-medium" data-translate="requirements.{{ $key }}">
                                        {{ ucfirst(str_replace('_', ' ', $key)) }}:
                                    </span>
                                    <span class="text-white text-sm">{{ $value }}</span>
                                </div>
                            @endforeach
                        </div>
                    </div>

                    <!-- Recommended Requirements -->
                    <div class="bg-gradient-to-br from-slate-800/50 to-green-900/30 backdrop-blur-lg border border-green-500/30 rounded-2xl p-8 shadow-2xl">
                        <div class="flex items-center gap-3 mb-6">
                            <div class="w-12 h-12 bg-green-500 rounded-xl flex items-center justify-center">
                                <span class="text-2xl">✅</span>
                            </div>
                            <h3 class="text-2xl font-bold text-white">
                                <span data-translate="download.recommendedRequirements">Recommended Requirements</span>
                            </h3>
                        </div>

                        <div class="space-y-4">
                            @foreach($systemRequirements['recommended'] as $key => $value)
                                <div class="flex justify-between items-center py-2 border-b border-green-500/20">
                                    <span class="text-green-200 font-medium" data-translate="requirements.{{ $key }}">
                                        {{ ucfirst(str_replace('_', ' ', $key)) }}:
                                    </span>
                                    <span class="text-white text-sm">{{ $value }}</span>
                                </div>
                            @endforeach
                        </div>
                    </div>
                </div>

                <!-- Installation Instructions -->
                <div class="mt-12 bg-slate-700/50 backdrop-blur-lg border border-purple-500/20 rounded-2xl p-8">
                    <h3 class="text-2xl font-bold text-white mb-6 flex items-center gap-3">
                        <span class="text-3xl">📋</span>
                        <span data-translate="download.installationGuide">Installation Guide</span>
                    </h3>

                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                        @foreach([
                            ['step' => '1', 'title' => 'Download Prerequisites', 'desc' => 'Install .NET Framework 4.8 and Visual C++ Redistributable'],
                            ['step' => '2', 'title' => 'Download Client', 'desc' => 'Download the full game client using the link above'],
                            ['step' => '3', 'title' => 'Extract & Install', 'desc' => 'Extract the downloaded file and run the installer'],
                            ['step' => '4', 'title' => 'Create Account', 'desc' => 'Register an account and start playing']
                        ] as $instruction)
                            <div class="text-center">
                                <div class="w-16 h-16 bg-gradient-to-br from-purple-500 to-cyan-500 rounded-full flex items-center justify-center text-white text-2xl font-bold mx-auto mb-4">
                                    {{ $instruction['step'] }}
                                </div>
                                <h4 class="text-lg font-bold text-white mb-2" data-translate="install.step{{ $instruction['step'] }}Title">
                                    {{ $instruction['title'] }}
                                </h4>
                                <p class="text-purple-200 text-sm" data-translate="install.step{{ $instruction['step'] }}Desc">
                                    {{ $instruction['desc'] }}
                                </p>
                            </div>
                        @endforeach
                    </div>
                </div>
            @endif
        </div>
    </div>

    <!-- Support Section -->
    <div class="container mx-auto px-6 py-16">
        <div class="bg-gradient-to-br from-slate-800/50 to-cyan-900/30 backdrop-blur-lg border border-cyan-500/30 rounded-2xl p-8 shadow-2xl">
            <div class="text-center">
                <h3 class="text-3xl font-bold text-white mb-4">
                    <span data-translate="download.needHelp">Need Help?</span>
                </h3>
                <p class="text-cyan-200 mb-8 max-w-2xl mx-auto">
                    <span data-translate="download.supportDescription">Having trouble downloading or installing? Our support team is here to help you get started on your adventure.</span>
                </p>

                <div class="flex flex-wrap justify-center gap-4">
                    <!-- Discord Live Chat -->
                    @if($downloadStats['discord_invite'])
                        <a href="{{ $downloadStats['discord_invite'] }}"
                           target="_blank"
                           class="flex items-center gap-2 px-6 py-3 bg-cyan-600 hover:bg-cyan-700 text-white rounded-xl transition-colors">
                            <span>💬</span>
                            <span data-translate="download.liveChat">Live Chat</span>
                        </a>
                    @else
                        <a href="#" class="flex items-center gap-2 px-6 py-3 bg-cyan-600 hover:bg-cyan-700 text-white rounded-xl transition-colors">
                            <span>💬</span>
                            <span data-translate="download.liveChat">Live Chat</span>
                        </a>
                    @endif

                    <!-- Email Support -->
                    <a href="mailto:{{ $downloadStats['contact_email'] }}"
                       class="flex items-center gap-2 px-6 py-3 bg-purple-600 hover:bg-purple-700 text-white rounded-xl transition-colors">
                        <span>📧</span>
                        <span data-translate="download.emailSupport">Email Support</span>
                    </a>

                    <!-- Facebook Page -->
                    @if($downloadStats['facebook_url'] !== '#')
                        <a href="{{ $downloadStats['facebook_url'] }}"
                           target="_blank"
                           class="flex items-center gap-2 px-6 py-3 bg-blue-600 hover:bg-blue-700 text-white rounded-xl transition-colors">
                            <span>📘</span>
                            <span data-translate="social.facebook">Facebook</span>
                        </a>
                    @endif

                    <!-- YouTube Channel -->
                    @if($downloadStats['youtube_url'] !== '#')
                        <a href="{{ $downloadStats['youtube_url'] }}"
                           target="_blank"
                           class="flex items-center gap-2 px-6 py-3 bg-red-600 hover:bg-red-700 text-white rounded-xl transition-colors">
                            <span>📺</span>
                            <span data-translate="social.youtube">YouTube</span>
                        </a>
                    @endif

                    <!-- FAQ -->
                    <a href="#" class="flex items-center gap-2 px-6 py-3 bg-slate-600 hover:bg-slate-700 text-white rounded-xl transition-colors">
                        <span>❓</span>
                        <span data-translate="download.faq">FAQ</span>
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Refresh Button -->
    <div class="fixed bottom-6 left-6 z-50">
        <button wire:click="refreshDownloads"
                class="flex items-center gap-2 px-4 py-2 bg-purple-600 hover:bg-purple-700 text-white rounded-lg shadow-lg transition-colors">
            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
            </svg>
            <span data-translate="download.refresh">Refresh</span>
        </button>
    </div>
</div>
