<?php

use Livewire\Volt\Component;
use App\Models\Z_Website\SiteSettings;
use App\Models\GameInfo;
use App\Models\SRO_VT_PROXY\_Players;
use Illuminate\Support\Facades\Cache;

new class extends Component {
    public $serverSettings = [];
    public $serverStats = [];
    public $onlinePlayers = 0;
    public $isServerOnline = true;
    public $aboutContent = null;

    public function mount()
    {
        $this->loadServerSettings();
        $this->loadServerStats();
        $this->loadAboutContent();
    }

    public function loadServerSettings()
    {
        try {
            $settings = Cache::remember('server_settings_about', 300, function () {
                return SiteSettings::first();
            });

            if ($settings && $settings->settings) {
                $this->serverSettings = $settings->settings;
                $this->isServerOnline = !($settings->settings['registration_close'] ?? false);
            }
        } catch (\Exception $e) {
            $this->serverSettings = [];
        }
    }

    public function loadServerStats()
    {
        try {
            $this->onlinePlayers = Cache::remember('online_players_about', 60, function () {
                return _Players::where('cur_status', 1)->count();
            });

            $this->serverStats = [
                'online_players' => $this->onlinePlayers,
                'max_players' => $this->serverSettings['sro_max_server'] ?? '1000',
                'level_cap' => $this->serverSettings['sro_cap'] ?? '130',
                'exp_rate' => $this->serverSettings['sro_exp'] ?? '45',
                'gold_rate' => $this->serverSettings['sro_exp_gold'] ?? '45',
                'drop_rate' => $this->serverSettings['sro_exp_drop'] ?? '45',
                'job_rate' => $this->serverSettings['sro_exp_job'] ?? '40',
                'party_rate' => $this->serverSettings['sro_exp_party'] ?? '55',
                'ip_limit' => $this->serverSettings['sro_ip_limit'] ?? '99',
                'hwid_limit' => $this->serverSettings['sro_hwid_limit'] ?? '8',
            ];
        } catch (\Exception $e) {
            $this->serverStats = [];
        }
    }

    public function loadAboutContent()
    {
        try {
            $this->aboutContent = Cache::remember('about_game_content', 600, function () {
                return GameInfo::where('id', 0)->first();
            });
        } catch (\Exception $e) {
            $this->aboutContent = null;
        }
    }

    public function refreshData()
    {
        Cache::forget('server_settings_about');
        Cache::forget('online_players_about');
        Cache::forget('about_game_content');

        $this->loadServerSettings();
        $this->loadServerStats();
        $this->loadAboutContent();

        $this->dispatch('notify', 'Data updated successfully!');
    }

    public function getServerStatusProperty()
    {
        return $this->isServerOnline ? 'ONLINE' : 'MAINTENANCE';
    }

    // Process content to handle img[link] and vid[link] from GameInfo
    // Process content to handle img[link] and vid[link]
    public function processContent($content)
    {
        // Process img[link] patterns
        $content = preg_replace_callback('/img\[(.*?)\]/', function($matches) {
            $imageUrl = trim($matches[1]);
            return $this->generateImageHTML($imageUrl);
        }, $content);

        // Process vid[link] patterns
        $content = preg_replace_callback('/vid\[(.*?)\]/', function($matches) {
            $videoUrl = trim($matches[1]);
            return $this->generateVideoHTML($videoUrl);
        }, $content);

        return $content;
    }

    // Generate image HTML
    private function generateImageHTML($imageUrl)
    {
        if (empty($imageUrl)) {
            return '<p class="text-red-400 italic">⚠️ Image URL is empty</p>';
        }

        return '<div class="game-content-media-wrapper">
                    <div class="game-content-image-container">
                        <img src="' . htmlspecialchars($imageUrl) . '"  alt="Guide Image"  class="game-content-image"  loading="lazy" onerror="this.src=\'/assets/images/default-game-guide.png\';">
                        <div class="game-content-image-overlay">
                            <i class="fas fa-expand text-white text-2xl"></i>
                        </div>
                    </div>
                </div>';
    }

    // Generate video HTML
    private function generateVideoHTML($videoUrl)
    {
        if (empty($videoUrl)) {
            return '<p class="text-red-400 italic">⚠️ Video URL is empty</p>';
        }

        // Check if it's a full iframe embed code from YouTube
        if (strpos($videoUrl, '<iframe') !== false && strpos($videoUrl, 'youtube.com/embed/') !== false) {
            if (preg_match('/src="([^"]*youtube\.com\/embed\/[^"]*)"/', $videoUrl, $iframeMatches)) {
                $videoUrl = $iframeMatches[1];
            }
        }

        // Process YouTube URLs
        if (strpos($videoUrl, 'youtube.com') !== false || strpos($videoUrl, 'youtu.be') !== false) {
            return $this->generateYouTubeHTML($videoUrl);
        }

        // Process Vimeo URLs
        if (strpos($videoUrl, 'vimeo.com') !== false) {
            return $this->generateVimeoHTML($videoUrl);
        }

        // Process other video formats
        return $this->generateGenericVideoHTML($videoUrl);
    }

    // Generate YouTube video HTML
    private function generateYouTubeHTML($videoUrl)
    {
        $videoId = null;
        $siParam = null;
        $originalUrl = $videoUrl;

        // Extract video ID and si parameter
        if (strpos($videoUrl, 'youtube.com/embed/') !== false) {
            // Embed URL
            if (preg_match('/youtube\.com\/embed\/([^?&\/]+)(?:.*?[\?&]si=([^&]+))?/', $videoUrl, $matches)) {
                $videoId = $matches[1];
                $siParam = isset($matches[2]) ? $matches[2] : null;
            }
        } elseif (preg_match('/(?:youtube\.com\/(?:watch\?v=|v\/)|youtu\.be\/)([^"&?\/\s]{11})/', $videoUrl, $matches)) {
            // Regular YouTube URL
            $videoId = $matches[1];
            if (preg_match('/[\?&]si=([^&]+)/', $videoUrl, $siMatches)) {
                $siParam = $siMatches[1];
            }
            $originalUrl = 'https://www.youtube.com/watch?v=' . $videoId;
        }

        if (!$videoId) {
            return '<p class="text-red-400 italic">⚠️ Invalid YouTube URL</p>';
        }

        // Build embed URL with parameters
        $params = [];
        if ($siParam) {
            $params[] = 'si=' . urlencode($siParam);
        }
        $params[] = 'rel=0';
        $params[] = 'modestbranding=1';
        $params[] = 'playsinline=1';

        $embedUrl = 'https://www.youtube.com/embed/' . $videoId . '?' . implode('&', $params);

        return $this->generateVideoContainer($embedUrl, $originalUrl, 'YouTube');
    }

    // Generate Vimeo video HTML
    private function generateVimeoHTML($videoUrl)
    {
        if (preg_match('/vimeo\.com\/(?:video\/)?(\d+)/', $videoUrl, $matches)) {
            $videoId = $matches[1];
            $embedUrl = 'https://player.vimeo.com/video/' . $videoId;
            return $this->generateVideoContainer($embedUrl, $videoUrl, 'Vimeo');
        }

        return '<p class="text-amber-400 italic">⚠️ Invalid Vimeo URL</p>';
    }

    // Generate generic video HTML
    private function generateGenericVideoHTML($videoUrl)
    {
        $fileExtension = strtolower(pathinfo($videoUrl, PATHINFO_EXTENSION));

        if (in_array($fileExtension, ['mp4', 'webm', 'ogg', 'avi', 'mov'])) {
            return '<div class="game-content-media-wrapper">
                        <div class="game-content-video-container">
                            <video class="game-content-video" controls preload="metadata">
                                <source src="' . htmlspecialchars($videoUrl) . '" type="video/' . $fileExtension . '">
                                Your browser does not support the video tag.
                            </video>
                        </div>
                    </div>';
        }

        return '<p class="text-amber-400 italic">⚠️ Unsupported video format. Supported: YouTube, Vimeo, MP4, WebM, OGG</p>';
    }

    // Generate video container with fallback
    private function generateVideoContainer($embedUrl, $originalUrl, $platform = 'YouTube')
    {
        $platformIcon = $platform === 'YouTube' ? 'fab fa-youtube' : 'fab fa-vimeo-v';
        $platformColor = $platform === 'YouTube' ? 'bg-red-600 hover:bg-red-500' : 'bg-blue-600 hover:bg-blue-500';

        return '<div class="game-content-media-wrapper">
                    <div class="game-content-video-container">
                        <iframe  class="game-content-video" width="560" height="315" src="' . htmlspecialchars($embedUrl) . '" title="' . $platform . ' video player" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" referrerpolicy="strict-origin-when-cross-origin" allowfullscreen></iframe>
                        <!-- Fallback link -->
                        <div class="video-fallback" style="display: none;">
                            <div class="text-center p-8 bg-gray-800/50 rounded-lg">
                                <i class="' . $platformIcon . ' text-red-500 text-4xl mb-4"></i>
                                <p class="text-white mb-4">Unable to load video</p>
                                <a href="' . htmlspecialchars($originalUrl) . '"
                                   target="_blank"
                                   rel="noopener noreferrer"
                                   class="inline-flex items-center px-4 py-2 ' . $platformColor . ' text-white rounded-lg transition-colors">
                                    <i class="' . $platformIcon . ' mr-2"></i>
                                    Watch on ' . $platform . '
                                </a>
                            </div>
                        </div>
                    </div>
                </div>';
    }
};
?>
<div>
<div class="min-h-screen bg-gradient-to-br from-slate-900 via-purple-900 to-slate-900">
    <!-- Hero Section -->
    <div class="relative overflow-hidden">
        <div class="absolute inset-0 bg-gradient-to-br from-purple-900/50 via-blue-900/30 to-slate-900/50"></div>
        <div class="container mx-auto px-6 py-24 relative z-10">
            <div class="text-center">
                <div class="inline-flex items-center gap-3 px-6 py-3 bg-purple-600/20 backdrop-blur-sm rounded-full border border-purple-500/30 mb-8">
                    <span class="w-2 h-2 bg-green-400 rounded-full animate-pulse"></span>
                    <span class="text-green-300 font-medium">{{ $this->serverStatus }}</span>
                    <span class="text-purple-300">•</span>
                    <span class="text-white">{{ $serverStats['online_players'] ?? 0 }}/{{ $serverStats['max_players'] ?? 1000 }} <span data-translate="about.players">Players</span></span>
                </div>

                <h1 class="text-5xl md:text-7xl font-bold mb-6 bg-gradient-to-r from-purple-400 via-blue-300 to-cyan-300 bg-clip-text text-transparent" data-translate="about.aboutOurGame">
                    About Our Game
                </h1>
                <p class="text-xl text-purple-200 max-w-3xl mx-auto mb-8" data-translate="about.discoverEpicWorld">
                    Discover the epic world of Silkroad Online and learn about our unique server features
                </p>

                <button wire:click="refreshData" class="inline-flex items-center gap-2 px-6 py-3 bg-purple-600 hover:bg-purple-700 text-white rounded-lg transition-colors shadow-lg hover:shadow-xl">
                    <i class="fas fa-sync-alt"></i>
                    <span data-translate="about.refreshData">Refresh Data</span>
                </button>
            </div>
        </div>
    </div>

    <!-- Welcome Display Section with Animated Border -->
    <div class="container mx-auto px-6 py-16">
        <div x-data="{
            currentMessage: 0,
            isTransitioning: false,
            progress: 0,
            messages: [
                {
                    titleKey: 'about.welcome.legends.title',
                    subtitleKey: 'about.welcome.legends.subtitle',
                    contentKey: 'about.welcome.legends.content',
                    titleText: 'Welcome to the World of Legends',
                    subtitleText: 'Welcome to the World of Legends',
                    contentText: 'Embark on an epic adventure through the ancient Silk Road',
                    icon: '⚔️'
                },
                {
                    titleKey: 'about.welcome.endless.title',
                    subtitleKey: 'about.welcome.endless.subtitle',
                    contentKey: 'about.welcome.endless.content',
                    titleText: 'Discover an Endless World',
                    subtitleText: 'Discover an Endless World',
                    contentText: 'Explore vast lands filled with treasures and challenges',
                    icon: '🗺️'
                },
                {
                    titleKey: 'about.welcome.warriors.title',
                    subtitleKey: 'about.welcome.warriors.subtitle',
                    contentKey: 'about.welcome.warriors.content',
                    titleText: 'Join the Brave Warriors',
                    subtitleText: 'Join the Brave Warriors',
                    contentText: 'Form powerful alliances and engage in epic battles',
                    icon: '🛡️'
                },
                {
                    titleKey: 'about.welcome.legend.title',
                    subtitleKey: 'about.welcome.legend.subtitle',
                    contentKey: 'about.welcome.legend.content',
                    titleText: 'Write Your Own Legend',
                    subtitleText: 'Write Your Own Legend',
                    contentText: 'Every battle, every victory, every step creates your history',
                    icon: '👑'
                }
            ],
            borderAnimation: 'glow-1',
            init() {
                // Start progress animation
                this.startProgressAnimation();

                // Cycle through messages every 7 seconds
                setInterval(() => {
                    this.nextMessage();
                }, 2500);

                // Change border animation every 3 seconds
                setInterval(() => {
                    this.updateBorderAnimation();
                }, 1500);
            },
            nextMessage() {
                if (this.isTransitioning) return;

                this.isTransitioning = true;
                this.progress = 0;

                // Wait for transition to complete
                setTimeout(() => {
                    this.currentMessage = (this.currentMessage + 1) % this.messages.length;
                    this.updateBorderAnimation();
                    this.isTransitioning = false;

                    // Trigger translation update
                    this.$nextTick(() => {
                        if (window.translatePage) {
                            window.translatePage();
                        }
                    });
                }, 300);
            },
            startProgressAnimation() {
                setInterval(() => {
                    if (!this.isTransitioning) {
                        this.progress = (this.progress + 1.43) % 100; // 7 seconds = 100%
                    }
                }, 100);
            },
            updateBorderAnimation() {
                const animations = ['glow-1', 'glow-2', 'glow-3', 'sweep-1', 'sweep-2', 'pulse-1', 'pulse-2'];
                this.borderAnimation = animations[Math.floor(Math.random() * animations.length)];
            }
        }"
        x-init="init()"
        class="mb-16">

            <!-- Animated Welcome Display -->
            <div class="relative max-w-4xl mx-auto">
                <!-- Animated Border Container -->
                <div class="relative p-1 rounded-3xl overflow-hidden" :class="borderAnimation">
                    <!-- Background Gradient -->
                    <div class="absolute inset-0 bg-gradient-to-r from-purple-600 to-cyan-500 animate-gradient-x"></div>

                    <!-- Content Container -->
                    <div class="relative bg-gradient-to-br from-slate-900/95 via-purple-900/90 to-slate-900/95 backdrop-blur-xl rounded-3xl p-8 md:p-12">
                        <!-- Decorative Elements -->
                        <div class="absolute top-4 left-4 text-4xl opacity-20">✨</div>
                        <div class="absolute top-4 right-4 text-4xl opacity-20">⚡</div>
                        <div class="absolute bottom-4 left-4 text-4xl opacity-20">🌟</div>
                        <div class="absolute bottom-4 right-4 text-4xl opacity-20">💎</div>

                        <!-- Main Content -->
                        <div class="text-center relative z-10">
                            <!-- Icon with Animation -->
                            <div class="mb-6">
                                <span x-text="messages[currentMessage].icon"
                                      class="inline-block text-6xl md:text-8xl animate-bounce"
                                      x-transition:enter="transition ease-out duration-1000"
                                      x-transition:enter-start="opacity-0 scale-50 rotate-180"
                                      x-transition:enter-end="opacity-100 scale-100 rotate-0"
                                      x-transition:leave="transition ease-in duration-300"
                                      x-transition:leave-start="opacity-100 scale-100"
                                      x-transition:leave-end="opacity-0 scale-75"></span>
                            </div>

                            <!-- Main Title -->
                            <h2 class="text-3xl md:text-5xl font-bold mb-3 text-shimmer"
                                x-transition:enter="transition ease-out duration-1000 delay-200"
                                x-transition:enter-start="opacity-0 translate-y-6"
                                x-transition:enter-end="opacity-100 translate-y-0"
                                x-transition:leave="transition ease-in duration-300"
                                x-transition:leave-start="opacity-100 translate-y-0"
                                x-transition:leave-end="opacity-0 translate-y-4"
                                :data-translate="messages[currentMessage].titleKey"
                                x-text="messages[currentMessage].titleText">
                            </h2>

                            <!-- Subtitle -->
                            <h3 class="text-xl md:text-2xl font-semibold mb-6 text-cyan-300"
                                x-transition:enter="transition ease-out duration-1000 delay-400"
                                x-transition:enter-start="opacity-0 translate-y-6"
                                x-transition:enter-end="opacity-100 translate-y-0"
                                x-transition:leave="transition ease-in duration-300 delay-100"
                                x-transition:leave-start="opacity-100 translate-y-0"
                                x-transition:leave-end="opacity-0 translate-y-4"
                                :data-translate="messages[currentMessage].subtitleKey"
                                x-text="messages[currentMessage].subtitleText">
                            </h3>

                            <!-- Content Message -->
                            <p class="text-lg md:text-xl text-purple-200 leading-relaxed mb-8 max-w-2xl mx-auto"
                               x-transition:enter="transition ease-out duration-1000 delay-600"
                               x-transition:enter-start="opacity-0 translate-y-6"
                               x-transition:enter-end="opacity-100 translate-y-0"
                               x-transition:leave="transition ease-in duration-300 delay-200"
                               x-transition:leave-start="opacity-100 translate-y-0"
                               x-transition:leave-end="opacity-0 translate-y-4"
                               :data-translate="messages[currentMessage].contentKey"
                               x-text="messages[currentMessage].contentText">
                            </p>                            <!-- Progress Indicators with Progress Bars -->
                            <div class="flex justify-center gap-3 mb-4">
                                <template x-for="(message, index) in messages" :key="index">
                                    <div class="relative">
                                        <button @click="currentMessage = index; progress = 0"
                                                :class="currentMessage === index ? 'bg-gradient-to-r from-purple-500 to-cyan-500 scale-125' : 'bg-white/30 hover:bg-white/50'"
                                                class="w-3 h-3 rounded-full transition-all duration-300 hover:scale-110 relative overflow-hidden">
                                            <!-- Progress overlay for active indicator -->
                                            <div x-show="currentMessage === index"
                                                 class="absolute inset-0 bg-white/20 rounded-full origin-left transform"
                                                 :style="`transform: scaleX(${progress / 100})`"></div>
                                        </button>
                                    </div>
                                </template>
                            </div>


                            <!-- Call to Action Button -->
                            <div class="animate-pulse">
                                <button @click="$el.scrollIntoView({behavior: 'smooth', block: 'nearest'})"
                                        class="inline-flex items-center gap-3 px-8 py-4 bg-gradient-to-r from-purple-600 via-blue-600 to-cyan-600 hover:from-purple-700 hover:via-blue-700 hover:to-cyan-700 text-white rounded-2xl transition-all duration-300 font-medium text-lg shadow-2xl hover:shadow-purple-500/25 hover:scale-105 transform">
                                    <i class="fas fa-rocket"></i>
                                    <span data-translate="about.startAdventure">Start Your Adventure Now</span>
                                    <span class="text-sm opacity-90" data-translate="about.startAdventureEn">Start Your Adventure</span>
                                </button>
                            </div>
                        </div>

                        <!-- Floating Particles -->
                        <div class="absolute inset-0 overflow-hidden pointer-events-none">
                            <div class="absolute top-1/4 left-1/4 w-2 h-2 bg-purple-400 rounded-full opacity-60 animate-float-1"></div>
                            <div class="absolute top-1/3 right-1/3 w-1 h-1 bg-cyan-400 rounded-full opacity-80 animate-float-2"></div>
                            <div class="absolute bottom-1/4 left-1/3 w-1.5 h-1.5 bg-yellow-400 rounded-full opacity-70 animate-float-3"></div>
                            <div class="absolute bottom-1/3 right-1/4 w-1 h-1 bg-pink-400 rounded-full opacity-60 animate-float-4"></div>
                            <div class="absolute top-1/2 left-1/6 w-1 h-1 bg-green-400 rounded-full opacity-50 animate-float-5"></div>
                            <div class="absolute top-2/3 right-1/6 w-1.5 h-1.5 bg-blue-400 rounded-full opacity-70 animate-float-6"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Server Information Section -->
    <div class="container mx-auto px-6 py-16">
        <div class="text-center mb-16">
            <h2 class="text-4xl font-bold text-white mb-4">
                <i class="fas fa-server text-purple-400 mr-3"></i>
                <span data-translate="about.serverInformation">Server Information</span>
            </h2>
            <div class="w-24 h-1 bg-gradient-to-r from-purple-500 to-cyan-500 mx-auto"></div>
        </div>

        <!-- Server Status Overview -->
        <div class="bg-gradient-to-br from-slate-800/60 to-purple-900/40 backdrop-blur-lg border border-purple-500/30 rounded-3xl p-8 mb-12 shadow-2xl">
            <div class="grid grid-cols-1 md:grid-cols-3 gap-8 mb-8">
                <!-- Server Status -->
                <div class="text-center">
                    <div class="w-16 h-16 mx-auto mb-4 rounded-full {{ $isServerOnline ? 'bg-green-500' : 'bg-red-500' }} flex items-center justify-center animate-pulse">
                        <i class="fas {{ $isServerOnline ? 'fa-check' : 'fa-times' }} text-white text-2xl"></i>
                    </div>
                    <h3 class="text-2xl font-bold text-white mb-2" data-translate="about.serverStatus">Server Status</h3>
                    <p class="text-{{ $isServerOnline ? 'green' : 'red' }}-300 text-xl font-semibold">{{ $this->serverStatus }}</p>
                </div>

                <!-- Online Players -->
                <div class="text-center">
                    <div class="w-16 h-16 mx-auto mb-4 rounded-full bg-blue-500 flex items-center justify-center">
                        <i class="fas fa-users text-white text-2xl"></i>
                    </div>
                    <h3 class="text-2xl font-bold text-white mb-2" data-translate="about.playersOnline">Players Online</h3>
                    <p class="text-blue-300 text-xl font-semibold">{{ $serverStats['online_players'] ?? 0 }}</p>
                </div>

                <!-- Server Capacity -->
                <div class="text-center">
                    <div class="w-16 h-16 mx-auto mb-4 rounded-full bg-purple-500 flex items-center justify-center">
                        <i class="fas fa-database text-white text-2xl"></i>
                    </div>
                    <h3 class="text-2xl font-bold text-white mb-2" data-translate="about.maxCapacity">Max Capacity</h3>
                    <p class="text-purple-300 text-xl font-semibold">{{ $serverStats['max_players'] ?? 1000 }}</p>
                </div>
            </div>

            <!-- Server Stats Grid -->
            <div class="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-8 gap-4">
                @foreach([
                    ['key' => 'levelCap', 'label' => 'Level Cap', 'value' => $serverStats['level_cap'] ?? 'N/A', 'icon' => '⭐', 'color' => 'yellow'],
                    ['key' => 'expRate', 'label' => 'EXP Rate', 'value' => ($serverStats['exp_rate'] ?? 'N/A') . 'x', 'icon' => '⚡', 'color' => 'blue'],
                    ['key' => 'goldRate', 'label' => 'Gold Rate', 'value' => ($serverStats['gold_rate'] ?? 'N/A') . 'x', 'icon' => '💰', 'color' => 'yellow'],
                    ['key' => 'dropRate', 'label' => 'Drop Rate', 'value' => ($serverStats['drop_rate'] ?? 'N/A') . 'x', 'icon' => '🎁', 'color' => 'green'],
                    ['key' => 'jobRate', 'label' => 'Job Rate', 'value' => ($serverStats['job_rate'] ?? 'N/A') . 'x', 'icon' => '🔨', 'color' => 'orange'],
                    ['key' => 'partyRate', 'label' => 'Party Rate', 'value' => ($serverStats['party_rate'] ?? 'N/A') . 'x', 'icon' => '🤝', 'color' => 'purple'],
                    ['key' => 'ipLimit', 'label' => 'IP Limit', 'value' => $serverStats['ip_limit'] ?? 'N/A', 'icon' => '🌐', 'color' => 'cyan'],
                    ['key' => 'hwidLimit', 'label' => 'HWID Limit', 'value' => $serverStats['hwid_limit'] ?? 'N/A', 'icon' => '💻', 'color' => 'indigo']
                ] as $stat)
                    <div class="bg-slate-700/60 rounded-2xl p-4 border border-{{ $stat['color'] }}-500/20 hover:border-{{ $stat['color'] }}-400/40 transition-all duration-300 hover:scale-105">
                        <div class="text-3xl mb-2 text-center">{{ $stat['icon'] }}</div>
                        <div class="text-{{ $stat['color'] }}-300 text-xs mb-1 text-center font-medium" data-translate="welcome.{{ $stat['key'] }}">{{ $stat['label'] }}</div>
                        <div class="text-white text-lg font-bold text-center">{{ $stat['value'] }}</div>
                    </div>
                @endforeach
            </div>
        </div>

        <!-- Server Features -->
        @if(!empty($serverSettings))
        <div class="bg-gradient-to-br from-slate-800/60 to-blue-900/40 backdrop-blur-lg border border-blue-500/30 rounded-3xl p-8 shadow-2xl">
            <h3 class="text-3xl font-bold text-white mb-8 text-center flex items-center justify-center gap-3">
                <i class="fas fa-star text-blue-400"></i>
                <span data-translate="welcome.serverFeatures">Server Features</span>
            </h3>

            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                @foreach([
                    ['key' => 'jangan_fortress', 'label' => 'Jangan Fortress', 'icon' => '🏰', 'desc' => 'Epic fortress battles'],
                    ['key' => 'bandit_fortress', 'label' => 'Bandit Fortress', 'icon' => '🗡️', 'desc' => 'Dangerous adventures'],
                    ['key' => 'hotan_fortress', 'label' => 'Hotan Fortress', 'icon' => '🏛️', 'desc' => 'Ancient mysteries'],
                    ['key' => 'constantinople_fortress', 'label' => 'Constantinople Fortress', 'icon' => '🛡️', 'desc' => 'Ultimate challenges'],
                    ['key' => 'char_ranking', 'label' => 'Character Ranking', 'icon' => '🏆', 'desc' => 'Compete for glory'],
                    ['key' => 'guild_ranking', 'label' => 'Guild Ranking', 'icon' => '👑', 'desc' => 'Guild supremacy'],
                    ['key' => 'job_ranking', 'label' => 'Job Ranking', 'icon' => '🔧', 'desc' => 'Master your trade'],
                    ['key' => 'trader_ranking', 'label' => 'Trader Ranking', 'icon' => '💼', 'desc' => 'Trading empire'],
                    ['key' => 'hunter_ranking', 'label' => 'Hunter Ranking', 'icon' => '🏹', 'desc' => 'Hunt for fame'],
                    ['key' => 'thief_ranking', 'label' => 'Thief Ranking', 'icon' => '🗡️', 'desc' => 'Shadow mastery'],
                    ['key' => 'unique_ranking', 'label' => 'Unique Ranking', 'icon' => '💎', 'desc' => 'Rare achievements'],
                    ['key' => 'free_pvp_ranking', 'label' => 'Free PvP Ranking', 'icon' => '⚔️', 'desc' => 'Combat excellence']
                ] as $feature)
                    @php $isEnabled = !($serverSettings[$feature['key']] ?? true); @endphp
                    <div class="relative group">
                        <div class="absolute inset-0 {{ $isEnabled ? 'bg-gradient-to-br from-green-500/20 to-emerald-500/20' : 'bg-gradient-to-br from-red-500/20 to-rose-500/20' }} rounded-2xl blur opacity-75 group-hover:opacity-100 transition duration-300"></div>
                        <div class="relative bg-slate-800/60 backdrop-blur-sm rounded-2xl p-6 border {{ $isEnabled ? 'border-green-500/30 hover:border-green-400/50' : 'border-red-500/30 hover:border-red-400/50' }} transition-all duration-300 hover:scale-105">
                            <div class="flex items-center gap-4 mb-3">
                                <span class="text-3xl">{{ $feature['icon'] }}</span>
                                <div class="flex-1">
                                    <h4 class="text-white font-bold text-lg" data-translate="welcome.fortress.jangan">{{ $feature['label'] }}</h4>
                                    <p class="text-gray-400 text-sm">{{ $feature['desc'] }}</p>
                                </div>
                                <div class="text-2xl">
                                    {{ $isEnabled ? '✅' : '❌' }}
                                </div>
                            </div>
                            <div class="text-center">
                                <span class="px-3 py-1 rounded-full text-xs font-semibold {{ $isEnabled ? 'bg-green-500/20 text-green-300 border border-green-500/30' : 'bg-red-500/20 text-red-300 border border-red-500/30' }}">
                                    <span data-translate="welcome.{{ $isEnabled ? 'enabled' : 'disabled' }}">{{ $isEnabled ? 'Enabled' : 'Disabled' }}</span>
                                </span>
                            </div>
                        </div>
                    </div>
                @endforeach
            </div>
        </div>
        @endif
    </div>

    <!-- Game Information Content from GameInfo ID 0 -->
    @if($aboutContent)
    <div class="container mx-auto px-6 py-16">
        <div class="bg-gradient-to-br from-slate-800/60 to-indigo-900/40 backdrop-blur-lg border border-indigo-500/30 rounded-3xl p-8 shadow-2xl">
            <!-- Content Header -->
            <div class="text-center mb-12">
                <h2 class="text-4xl font-bold text-white mb-4">
                    <i class="fas fa-gamepad text-indigo-400 mr-3"></i>
                    {{ $aboutContent->title }}
                </h2>
                <div class="w-24 h-1 bg-gradient-to-r from-indigo-500 to-purple-500 mx-auto"></div>
            </div>

            <!-- Content Body -->
            <div class="about-game-content prose prose-invert max-w-none">
                <div class="text-lg leading-relaxed text-gray-200">
                    {!! nl2br($this->processContent($aboutContent->body)) !!}
                </div>
            </div>
        </div>
    </div>
    @endif

    <!-- Call to Action -->
    <div class="container mx-auto px-6 py-16">
        <div class="text-center">
            <h2 class="text-4xl font-bold text-white mb-6" data-translate="about.readyToBegin">Ready to Begin Your Journey?</h2>
            <p class="text-xl text-purple-200 mb-8 max-w-2xl mx-auto" data-translate="about.joinThousands">
                Join thousands of players in the ultimate Silkroad Online experience
            </p>
            <div class="flex flex-col sm:flex-row gap-4 justify-center">
                <a href="{{ route('register') }}" class="px-8 py-4 bg-gradient-to-r from-purple-600 to-pink-600 hover:from-purple-700 hover:to-pink-700 text-white rounded-xl transition-all duration-300 font-medium text-lg shadow-lg hover:shadow-xl hover:scale-105">
                    <i class="fas fa-user-plus mr-2"></i>
                    <span data-translate="welcome.createAccount">Create Account</span>
                </a>
                <a href="{{ route('download') }}" class="px-8 py-4 bg-gradient-to-r from-blue-600 to-cyan-600 hover:from-blue-700 hover:to-cyan-700 text-white rounded-xl transition-all duration-300 font-medium text-lg shadow-lg hover:shadow-xl hover:scale-105">
                    <i class="fas fa-download mr-2"></i>
                    <span data-translate="welcome.downloadClient">Download Client</span>
                </a>
            </div>
        </div>
    </div>
</div>

<style>
/* About Game Media Styles */
.about-game-media-wrapper {
    @apply max-w-4xl mx-auto;
}

.about-game-image-container {
    @apply relative rounded-2xl overflow-hidden shadow-2xl;
    aspect-ratio: 16 / 9;
}

.about-game-image {
    @apply w-full h-full object-cover transition-transform duration-500;
}

.about-game-image-overlay {
    @apply absolute inset-0 bg-black/40 flex items-center justify-center opacity-0 hover:opacity-100 transition-opacity duration-300 cursor-pointer;
}

.about-game-image-container:hover .about-game-image {
    @apply scale-110;
}

.about-game-video-container {
    @apply relative rounded-2xl overflow-hidden shadow-2xl;
    aspect-ratio: 16 / 9;
}

.about-game-video {
    @apply absolute top-0 left-0 w-full h-full border-0 rounded-2xl;
}

/* Content Typography */
.about-game-content h3 {
    @apply text-2xl font-bold text-white mt-8 mb-4;
}

.about-game-content h4 {
    @apply text-xl font-bold text-purple-300 mt-6 mb-3;
}

.about-game-content p {
    @apply mb-4 leading-relaxed;
}

.about-game-content ul {
    @apply list-disc list-inside mb-4 space-y-2;
}

.about-game-content li {
    @apply text-gray-300;
}

.about-game-content strong {
    @apply text-white font-semibold;
}

.about-game-content em {
    @apply text-purple-300 italic;
}

/* Welcome Display Animations */
@keyframes gradient-x {
    0%, 100% {
        background-position: 0% 50%;
    }
    50% {
        background-position: 100% 50%;
    }
}

@keyframes float-1 {
    0%, 100% { transform: translateY(0px) translateX(0px) rotate(0deg); }
    25% { transform: translateY(-10px) translateX(5px) rotate(90deg); }
    50% { transform: translateY(-5px) translateX(-5px) rotate(180deg); }
    75% { transform: translateY(-15px) translateX(3px) rotate(270deg); }
}

@keyframes float-2 {
    0%, 100% { transform: translateY(0px) translateX(0px) rotate(0deg); }
    33% { transform: translateY(-8px) translateX(-3px) rotate(120deg); }
    66% { transform: translateY(-12px) translateX(4px) rotate(240deg); }
}

@keyframes float-3 {
    0%, 100% { transform: translateY(0px) translateX(0px) rotate(0deg); }
    20% { transform: translateY(-6px) translateX(2px) rotate(72deg); }
    40% { transform: translateY(-10px) translateX(-4px) rotate(144deg); }
    60% { transform: translateY(-4px) translateX(6px) rotate(216deg); }
    80% { transform: translateY(-12px) translateX(-2px) rotate(288deg); }
}

@keyframes float-4 {
    0%, 100% { transform: translateY(0px) translateX(0px) scale(1); }
    50% { transform: translateY(-8px) translateX(3px) scale(1.2); }
}

@keyframes float-5 {
    0%, 100% { transform: translateY(0px) translateX(0px) rotate(0deg) scale(1); }
    25% { transform: translateY(-5px) translateX(-2px) rotate(90deg) scale(0.8); }
    75% { transform: translateY(-10px) translateX(4px) rotate(-90deg) scale(1.1); }
}

@keyframes float-6 {
    0%, 100% { transform: translateY(0px) translateX(0px) rotate(0deg); opacity: 0.7; }
    33% { transform: translateY(-7px) translateX(3px) rotate(120deg); opacity: 1; }
    66% { transform: translateY(-3px) translateX(-5px) rotate(240deg); opacity: 0.5; }
}

.animate-float-1 { animation: float-1 6s ease-in-out infinite; }
.animate-float-2 { animation: float-2 8s ease-in-out infinite; }
.animate-float-3 { animation: float-3 7s ease-in-out infinite; }
.animate-float-4 { animation: float-4 5s ease-in-out infinite; }
.animate-float-5 { animation: float-5 9s ease-in-out infinite; }
.animate-float-6 { animation: float-6 4s ease-in-out infinite; }

.animate-gradient-x {
    background-size: 400% 400%;
    animation: gradient-x 8s ease infinite;
}

/* Border Animation Classes */
.glow-1::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, #ff0080, #ff8c00, #40e0d0, #ff0080);
    background-size: 400% 400%;
    animation: gradient-x 3s ease infinite;
    border-radius: inherit;
    z-index: -1;
    filter: blur(2px);
}

.glow-2::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(90deg, #8b5cf6, #06b6d4, #10b981, #f59e0b, #8b5cf6);
    background-size: 300% 300%;
    animation: gradient-x 4s ease infinite;
    border-radius: inherit;
    z-index: -1;
    filter: blur(1px);
}

.glow-3::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: conic-gradient(from 0deg, #ff006e, #8338ec, #3a86ff, #06ffa5, #ffbe0b, #ff006e);
    animation: spin 6s linear infinite;
    border-radius: inherit;
    z-index: -1;
    filter: blur(3px);
}

@keyframes sweep-border {
    0% { background-position: -100% 0; }
    100% { background-position: 100% 0; }
}

.sweep-1::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(90deg, transparent, #ff0080, transparent);
    background-size: 50% 100%;
    animation: sweep-border 2s ease-in-out infinite;
    border-radius: inherit;
    z-index: -1;
}

.sweep-2::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, transparent, #40e0d0, transparent);
    background-size: 70% 100%;
    animation: sweep-border 3s ease-in-out infinite reverse;
    border-radius: inherit;
    z-index: -1;
}

@keyframes pulse-glow {
    0%, 100% {
        box-shadow: 0 0 20px rgba(139, 92, 246, 0.5);
        filter: brightness(1);
    }
    50% {
        box-shadow: 0 0 40px rgba(139, 92, 246, 0.8), 0 0 60px rgba(139, 92, 246, 0.4);
        filter: brightness(1.2);
    }
}

.pulse-1 {
    animation: pulse-glow 2s ease-in-out infinite;
    background: linear-gradient(45deg, #8b5cf6, #06b6d4);
}

.pulse-2 {
    animation: pulse-glow 3s ease-in-out infinite;
    background: linear-gradient(135deg, #ff0080, #ff8c00);
}

/* Gold gradient text */
.bg-gradient-to-r.from-gold-400.via-yellow-300.to-gold-400 {
    background: linear-gradient(to right, #fbbf24, #fde047, #fbbf24);
}

@keyframes shimmer {
    0% { background-position: -200% center; }
    100% { background-position: 200% center; }
}

.text-shimmer {
    background: linear-gradient(90deg, #fbbf24, #fde047, #ffffff, #fde047, #fbbf24);
    background-size: 200% 100%;
    -webkit-background-clip: text;
    background-clip: text;
    -webkit-text-fill-color: transparent;
    animation: shimmer 3s ease-in-out infinite;
}
</style>
</div>
