<?php

use Livewire\Volt\Component;
use Livewire\WithFileUploads;
use App\Models\FastEventsWelcome;
use App\Models\SRO_VT_SHARD\Char;
use App\Models\SRO_VT_PROXY\_Players;
use App\Models\SRO_VT_SHARD\_User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

new class extends Component {
    use WithFileUploads;

    public $selectedPlayerType = null;
    public $selectedCharacter = null;
    public $oldCharacterName = '';
    public $oldCharacterProofs = [];
    public $donationAmount = 0;
    public $selectedCurrency = 'USD';
    public $donationProofs = [];
    public $oldCharacterImages = [];
    public $calculatedBalance = 0;
    public $userApplication = null;
    public $userCharacters = [];
    public $isSubmitting = false;
    public $isClaiming = false;

    public function mount()
    {
        $this->loadUserData();
        $this->checkForRejectionNotification();
    }

    public function checkForRejectionNotification()
    {
        $recentRejection = FastEventsWelcome::where('JID', Auth::user()->JID)
            ->where('status', 'rejected')
            ->where('reviewed_at', '>=', now()->subMinutes(10))
            ->latest()
            ->first();

        if ($recentRejection) {
            $this->dispatch('show-rejection-notification', [
                'applicationId' => $recentRejection->id,
                'rejectionReason' => $recentRejection->rejection_reason ?? 'No specific reason provided',
                'reviewedAt' => $recentRejection->reviewed_at->format('M d, Y H:i'),
                'playerType' => ucfirst($recentRejection->player_type) . ' Player',
                'rewardAmount' => $recentRejection->reward_amount
            ]);
        }
    }

    public function loadUserData()
    {
        $user = Auth::user();

        $this->userApplication = FastEventsWelcome::where('JID', $user->JID)
            ->whereIn('status', ['pending', 'approved', 'claimed'])
            ->latest('created_at')
            ->first();

        $userCharIds = _User::where('UserJID', $user->JID)->pluck('CharID');
        $this->userCharacters = Char::whereIn('CharID', $userCharIds)
            ->select('CharName16', 'CharID', 'RefObjID', 'CurLevel')
            ->orderBy('CharName16')
            ->get()
            ->map(function($char) {
                $player = _Players::where('CharID', $char->CharID)->first();
                $isActive = $player && $player->cur_status == true;

                return [
                    'CharID' => $char->CharID,
                    'CharName16' => $char->CharName16,
                    'RefObjID' => $char->RefObjID,
                    'CurLevel' => $char->CurLevel,
                    'avatar' => Char::getCharAvatarGifByCharname($char->CharName16),
                    'isActive' => $isActive
                ];
            });

        $this->checkForRejectionNotification();
    }

    public function getRejectedApplications()
    {
        return FastEventsWelcome::where('JID', Auth::user()->JID)
            ->where('status', 'rejected')
            ->latest('reviewed_at')
            ->limit(3)
            ->get();
    }

    public function selectPlayerType($type)
    {
        $this->selectedPlayerType = $type;
        $this->calculateReward();
    }

    public function selectCharacter($characterId)
    {
        $userCharIds = _User::where('UserJID', Auth::user()->JID)->pluck('CharID');

        if ($userCharIds->contains($characterId)) {
            $player = _Players::where('CharID', $characterId)->first();
            if (!$player || !$player->cur_status) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character must be active (played at least once) for all applications.',
                    'duration' => 2500,
                ]);
                return;
            }

            $this->selectedCharacter = $characterId;
        }
    }

    public function updatedDonationAmount()
    {
        $this->calculateReward();
    }

    public function updatedSelectedCurrency()
    {
        $this->calculateReward();
    }

    public function calculateReward()
    {
        if ($this->selectedPlayerType === 'donate' && $this->donationAmount > 0) {
            $rate = $this->selectedCurrency === 'USD' ? 40 : 1;
            $this->calculatedBalance = floor(($this->donationAmount * $rate) * 0.5);
        } else {
            $this->calculatedBalance = 0;
        }
    }

    public function submitApplication()
    {
        $this->isSubmitting = true;

        try {
            $rules = [
                'selectedPlayerType' => 'required|in:new,old,donate',
                'selectedCharacter' => 'required|numeric',
            ];

            if ($this->selectedPlayerType === 'old') {
                $rules['oldCharacterName'] = 'required|string|max:255';
                $rules['oldCharacterProofs'] = 'required|array|min:1';
                $rules['oldCharacterProofs.*'] = 'image|mimes:jpeg,png,jpg,gif|max:5120';
            }

            if ($this->selectedPlayerType === 'donate') {
                $rules['donationAmount'] = 'required|numeric|min:1';
                $rules['selectedCurrency'] = 'required|in:USD,EGP';
                $rules['donationProofs'] = 'required|array|min:1';
                $rules['donationProofs.*'] = 'image|mimes:jpeg,png,jpg,gif|max:5120';
            }

            $this->validate($rules);

            $userCharIds = _User::where('UserJID', Auth::user()->JID)->pluck('CharID');

            if (!$userCharIds->contains($this->selectedCharacter)) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character not found or does not belong to you.',
                    'duration' => 2500,
                ]);
                return;
            }

            $character = Char::where('CharID', $this->selectedCharacter)->first();
            if (!$character) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character not found in database.',
                    'duration' => 2500,
                ]);
                return;
            }

            $player = _Players::where('CharID', $this->selectedCharacter)->first();
            if (!$player || !$player->cur_status) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character must be active (played at least once) for all applications.',
                    'duration' => 2500,
                ]);
                return;
            }

            $oldCharacterProofPaths = [];
            $donationProofPaths = [];
            $oldCharacterImagePaths = [];

            if ($this->selectedPlayerType === 'old' && !empty($this->oldCharacterProofs)) {
                foreach ($this->oldCharacterProofs as $file) {
                    $path = $file->store('welcome-event/old-character-proofs', 'public');
                    $oldCharacterProofPaths[] = $path;
                }
            }

            if ($this->selectedPlayerType === 'donate' && !empty($this->donationProofs)) {
                foreach ($this->donationProofs as $file) {
                    $path = $file->store('welcome-event/donation-proofs', 'public');
                    $donationProofPaths[] = $path;
                }
            }

            if (!empty($this->oldCharacterImages)) {
                foreach ($this->oldCharacterImages as $file) {
                    $path = $file->store('welcome-event/old-character-images', 'public');
                    $oldCharacterImagePaths[] = $path;
                }
            }

            $rewardAmount = $this->getRewardAmount();

            $player = _Players::where('CharID', $this->selectedCharacter)->first();
            $hwid = $player ? $player->Serial : '';

            FastEventsWelcome::create([
                'JID' => Auth::user()->JID,
                'HWID' => $hwid,
                'player_type' => $this->selectedPlayerType,
                'CharID' => $this->selectedCharacter,
                'old_character_name' => $this->selectedPlayerType === 'old' ? $this->oldCharacterName : null,
                'donation_amount' => $this->selectedPlayerType === 'donate' ? $this->donationAmount : null,
                'currency' => $this->selectedPlayerType === 'donate' ? $this->selectedCurrency : null,
                'reward_amount' => $rewardAmount,
                'old_character_proofs' => !empty($oldCharacterProofPaths) ? json_encode($oldCharacterProofPaths) : null,
                'donation_proofs' => !empty($donationProofPaths) ? json_encode($donationProofPaths) : null,
                'old_character_images' => !empty($oldCharacterImagePaths) ? json_encode($oldCharacterImagePaths) : null,
                'status' => 'pending',
                'submitted_at' => now(),
            ]);

            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'Application submitted successfully!',
                'duration' => 3000,
            ]);
            $this->loadUserData();

        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Validation failed: ' . implode(', ', $e->validator->errors()->all()),
                'duration' => 5000,
            ]);
        } catch (\Exception $e) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Failed to submit application: ' . $e->getMessage(),
                'duration' => 5000,
            ]);
        } finally {
            $this->isSubmitting = false;
        }
    }

    public function collectRewards()
    {
        $this->isClaiming = true;

        try {
            if (!$this->userApplication || !$this->userApplication->canBeClaimed()) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Unable to collect rewards. Please try again.',
                    'duration' => 2500,
                ]);
                $this->isClaiming = false;
                return;
            }

            $user = Auth::user();

            if ($user && $user->webUser) {
                $oldBalance = $user->webUser->balance;
                $user->webUser->balance += $this->userApplication->reward_amount;
                $user->webUser->save();
            } else {
                throw new \Exception('User account not properly configured');
            }

            $this->userApplication->update([
                'status' => 'claimed',
                'claimed_at' => now(),
            ]);

            $this->userApplication->refresh();

            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'Rewards collected successfully! Your balance has been updated.',
                'duration' => 3000,
            ]);

            $this->dispatch('updateBalances');
            $this->loadUserData();
            $this->dispatch('hide-congratulations');

        } catch (\Exception $e) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Error collecting rewards: ' . $e->getMessage(),
                'duration' => 2500,
            ]);
        } finally {
            $this->isClaiming = false;
        }
    }

    public function claimReward()
    {
        // Auto collect rewards directly without modal
        $this->collectRewards();
    }

    private function getRewardAmount()
    {
        switch ($this->selectedPlayerType) {
            case 'new':
                return 10;
            case 'old':
                return 40;
            case 'donate':
                return $this->calculatedBalance;
            default:
                return 0;
        }
    }

    public function resetApplication()
    {
        $this->selectedPlayerType = null;
        $this->selectedCharacter = null;
        $this->oldCharacterName = '';
        $this->oldCharacterProofs = [];
        $this->donationAmount = 0;
        $this->selectedCurrency = 'USD';
        $this->donationProofs = [];
        $this->oldCharacterImages = [];
        $this->calculatedBalance = 0;

        if ($this->userApplication && $this->userApplication->status === 'rejected') {
            $this->userApplication = null;

            $this->dispatch('show-notification', [
                'type' => 'info',
                'message' => 'Ready to create a new application. Your previous application history has been preserved.',
                'duration' => 3000,
            ]);
        }
    }

    public function with()
    {
        return [
            'hasApplication' => !is_null($this->userApplication),
        ];
    }
}; ?>
<div>
<div class="welcome2-container"
     x-data="{
         showNotification: false,
         notificationData: {},

         showAlert(data) {
             this.notificationData = data;
             this.showNotification = true;
             setTimeout(() => {
                 this.showNotification = false;
             }, data.duration || 3000);
         }
     }"
     @show-notification.window="showAlert($event.detail)">

    <!-- Notification -->
    <div x-show="showNotification"
         x-transition:enter="transform transition ease-out duration-300"
         x-transition:enter-start="translate-y-[-100%] opacity-0"
         x-transition:enter-end="translate-y-0 opacity-100"
         x-transition:leave="transform transition ease-in duration-200"
         x-transition:leave-start="translate-y-0 opacity-100"
         x-transition:leave-end="translate-y-[-100%] opacity-0"
         class="fixed top-4 right-4 z-50 max-w-sm bg-gray-900/90 backdrop-blur-lg rounded-2xl shadow-2xl border-l-4 overflow-hidden"
         :class="{
             'border-green-500': notificationData.type === 'success',
             'border-red-500': notificationData.type === 'error',
             'border-blue-500': notificationData.type === 'info'
         }">

        <!-- Glow effect -->
        <div class="absolute inset-0 opacity-20"
             :class="{
                 'bg-gradient-to-r from-green-500/20 to-emerald-500/20': notificationData.type === 'success',
                 'bg-gradient-to-r from-red-500/20 to-pink-500/20': notificationData.type === 'error',
                 'bg-gradient-to-r from-blue-500/20 to-cyan-500/20': notificationData.type === 'info'
             }"></div>

        <div class="relative p-4">
            <div class="flex items-start">
                <div class="flex-shrink-0">
                    <div x-show="notificationData.type === 'success'" class="text-green-400">
                        <svg class="w-6 h-6" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"/>
                        </svg>
                    </div>
                    <div x-show="notificationData.type === 'error'" class="text-red-400">
                        <svg class="w-6 h-6" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z"/>
                        </svg>
                    </div>
                    <div x-show="notificationData.type === 'info'" class="text-blue-400">
                        <svg class="w-6 h-6" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z"/>
                        </svg>
                    </div>
                </div>
                <div class="ml-3 flex-1">
                    <p class="text-sm font-medium text-gray-200" x-text="notificationData.message"></p>
                </div>
                <div class="ml-4 flex-shrink-0">
                    <button @click="showNotification = false" class="text-gray-400 hover:text-gray-200 transition-colors">
                        <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z"/>
                        </svg>
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Success Modal -->
    <!-- Removed - Auto collect rewards instead -->

    <!-- Header -->
    <div class="text-center mb-8 relative">
        <div class="absolute inset-0 bg-gradient-to-r from-blue-600/20 via-purple-600/20 to-green-600/20 rounded-2xl blur-xl"></div>
        <div class="relative bg-gray-900/80 backdrop-blur-lg rounded-2xl p-8 border border-gray-700/50">
            <h1 class="text-5xl font-bold bg-gradient-to-r from-blue-400 via-purple-400 to-green-400 bg-clip-text text-transparent mb-4">
                مرحباً بك في الحدث
            </h1>
            <p class="text-gray-300 text-lg">اختر نوع اللاعب الخاص بك واحصل على المكافآت الرائعة</p>
            <div class="mt-4 flex justify-center space-x-2">
                <div class="w-2 h-2 bg-blue-400 rounded-full animate-pulse"></div>
                <div class="w-2 h-2 bg-purple-400 rounded-full animate-pulse" style="animation-delay: 0.2s"></div>
                <div class="w-2 h-2 bg-green-400 rounded-full animate-pulse" style="animation-delay: 0.4s"></div>
            </div>
        </div>
    </div>

    @if (!$hasApplication)
        <!-- Player Type Selection -->
        @if (!$selectedPlayerType)
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                <!-- New Player -->
                <div wire:click="selectPlayerType('new')"
                     class="group relative bg-gradient-to-br from-gray-900 to-gray-800 p-8 rounded-2xl border border-blue-500/30 cursor-pointer hover:border-blue-400 transition-all duration-500 transform hover:scale-105 hover:shadow-2xl hover:shadow-blue-500/25">
                    <!-- Glow effect -->
                    <div class="absolute inset-0 bg-gradient-to-br from-blue-500/20 to-cyan-500/20 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                    <div class="relative text-center">
                        <div class="w-20 h-20 bg-gradient-to-br from-blue-500 to-cyan-400 rounded-full flex items-center justify-center mx-auto mb-6 group-hover:scale-110 transition-transform duration-300">
                            <svg class="w-10 h-10 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                            </svg>
                        </div>
                        <h3 class="text-2xl font-bold text-blue-400 mb-3">لاعب جديد</h3>
                        <p class="text-gray-300 text-sm mb-6 leading-relaxed">للاعبين الجدد الذين يبدؤون رحلتهم في اللعبة</p>
                        <div class="bg-gradient-to-r from-blue-600 to-cyan-500 text-white px-6 py-3 rounded-lg font-semibold shadow-lg">
                            مكافأة: 10 Balance ⭐
                        </div>
                    </div>
                </div>

                <!-- Old Player -->
                <div wire:click="selectPlayerType('old')"
                     class="group relative bg-gradient-to-br from-gray-900 to-gray-800 p-8 rounded-2xl border border-purple-500/30 cursor-pointer hover:border-purple-400 transition-all duration-500 transform hover:scale-105 hover:shadow-2xl hover:shadow-purple-500/25">
                    <!-- Glow effect -->
                    <div class="absolute inset-0 bg-gradient-to-br from-purple-500/20 to-pink-500/20 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                    <div class="relative text-center">
                        <div class="w-20 h-20 bg-gradient-to-br from-purple-500 to-pink-400 rounded-full flex items-center justify-center mx-auto mb-6 group-hover:scale-110 transition-transform duration-300">
                            <svg class="w-10 h-10 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                        </div>
                        <h3 class="text-2xl font-bold text-purple-400 mb-3">لاعب عائد</h3>
                        <p class="text-gray-300 text-sm mb-6 leading-relaxed">للاعبين القدامى العائدين إلى اللعبة</p>
                        <div class="bg-gradient-to-r from-purple-600 to-pink-500 text-white px-6 py-3 rounded-lg font-semibold shadow-lg">
                            مكافأة: 40 Balance 🎖️
                        </div>
                    </div>
                </div>

                <!-- Donate Player -->
                <div wire:click="selectPlayerType('donate')"
                     class="group relative bg-gradient-to-br from-gray-900 to-gray-800 p-8 rounded-2xl border border-green-500/30 cursor-pointer hover:border-green-400 transition-all duration-500 transform hover:scale-105 hover:shadow-2xl hover:shadow-green-500/25">
                    <!-- Glow effect -->
                    <div class="absolute inset-0 bg-gradient-to-br from-green-500/20 to-emerald-500/20 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                    <div class="relative text-center">
                        <div class="w-20 h-20 bg-gradient-to-br from-green-500 to-emerald-400 rounded-full flex items-center justify-center mx-auto mb-6 group-hover:scale-110 transition-transform duration-300">
                            <svg class="w-10 h-10 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"/>
                            </svg>
                        </div>
                        <h3 class="text-2xl font-bold text-green-400 mb-3">مكافأة التبرع</h3>
                        <p class="text-gray-300 text-sm mb-6 leading-relaxed">للاعبين الداعمين للخادم بالتبرعات</p>
                        <div class="bg-gradient-to-r from-green-600 to-emerald-500 text-white px-6 py-3 rounded-lg font-semibold shadow-lg">
                            مكافأة: متغيرة 💎
                        </div>
                    </div>
                </div>
            </div>
        @else
            <!-- Selected Player Type Card -->
            <div class="relative bg-gray-900/90 backdrop-blur-lg rounded-2xl shadow-2xl p-8 mb-8 border border-gray-700/50
                 {{ $selectedPlayerType === 'new' ? 'border-blue-500/50 shadow-blue-500/20' : '' }}
                 {{ $selectedPlayerType === 'old' ? 'border-purple-500/50 shadow-purple-500/20' : '' }}
                 {{ $selectedPlayerType === 'donate' ? 'border-green-500/50 shadow-green-500/20' : '' }}">

                <!-- Background glow -->
                <div class="absolute inset-0 rounded-2xl
                     {{ $selectedPlayerType === 'new' ? 'bg-gradient-to-br from-blue-500/10 to-cyan-500/5' : '' }}
                     {{ $selectedPlayerType === 'old' ? 'bg-gradient-to-br from-purple-500/10 to-pink-500/5' : '' }}
                     {{ $selectedPlayerType === 'donate' ? 'bg-gradient-to-br from-green-500/10 to-emerald-500/5' : '' }}">
                </div>

                <div class="relative flex items-center mb-8">
                    <div class="w-16 h-16 rounded-2xl flex items-center justify-center mr-6
                         {{ $selectedPlayerType === 'new' ? 'bg-gradient-to-br from-blue-500 to-cyan-400' : '' }}
                         {{ $selectedPlayerType === 'old' ? 'bg-gradient-to-br from-purple-500 to-pink-400' : '' }}
                         {{ $selectedPlayerType === 'donate' ? 'bg-gradient-to-br from-green-500 to-emerald-400' : '' }}">
                        @if($selectedPlayerType === 'new')
                            <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                            </svg>
                        @elseif($selectedPlayerType === 'old')
                            <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                        @elseif($selectedPlayerType === 'donate')
                            <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"/>
                            </svg>
                        @endif
                    </div>
                    <div class="flex-1">
                        <h2 class="text-3xl font-bold mb-2
                             {{ $selectedPlayerType === 'new' ? 'text-blue-400' : '' }}
                             {{ $selectedPlayerType === 'old' ? 'text-purple-400' : '' }}
                             {{ $selectedPlayerType === 'donate' ? 'text-green-400' : '' }}">
                            @if($selectedPlayerType === 'new')
                                لاعب جديد ⭐
                            @elseif($selectedPlayerType === 'old')
                                لاعب عائد 🎖️
                            @elseif($selectedPlayerType === 'donate')
                                مكافأة التبرع 💎
                            @endif
                        </h2>
                        <p class="text-gray-300">املأ البيانات أدناه لإرسال طلبك والحصول على المكافأة</p>
                    </div>
                    <div class="mr-auto">
                        <button wire:click="resetApplication"
                                class="text-gray-400 hover:text-red-400 transition-colors duration-300 p-2 rounded-lg hover:bg-gray-800/50">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                            </svg>
                        </button>
                    </div>
                </div>

                <!-- Character Selection -->
                <div class="relative mb-8">
                    <label class="block text-lg font-semibold text-gray-200 mb-4">🎮 اختر الشخصية</label>
                    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                        @foreach($userCharacters as $character)
                            <div wire:click="selectCharacter({{ $character['CharID'] }})"
                                 class="group relative p-4 rounded-xl border-2 cursor-pointer transition-all duration-300
                                        {{ $character['isActive'] ? 'hover:border-blue-400 bg-gray-800/50 border-gray-600' : 'opacity-50 cursor-not-allowed bg-gray-900/50 border-gray-700' }}
                                        {{ $selectedCharacter == $character['CharID'] ? 'border-blue-500 bg-gradient-to-br from-blue-500/10 to-cyan-500/5 shadow-lg shadow-blue-500/20' : '' }}">

                                <!-- Glow effect for selected -->
                                @if($selectedCharacter == $character['CharID'])
                                    <div class="absolute inset-0 bg-gradient-to-br from-blue-500/20 to-cyan-500/20 rounded-xl"></div>
                                @endif

                                <div class="relative flex items-center">
                                    <div class="relative">
                                        <img src="{{ $character['avatar'] }}"
                                             alt="{{ $character['CharName16'] }}"
                                             class="w-12 h-12 rounded-full mr-4 border-2 border-gray-600 group-hover:border-blue-400 transition-colors">
                                        @if($selectedCharacter == $character['CharID'])
                                            <div class="absolute -inset-1 bg-gradient-to-br from-blue-500 to-cyan-400 rounded-full opacity-50 blur-sm"></div>
                                        @endif
                                    </div>
                                    <div class="flex-1">
                                        <div class="font-semibold text-gray-200 group-hover:text-blue-300 transition-colors">
                                            {{ $character['CharName16'] }}
                                        </div>
                                        <div class="text-sm text-gray-400">Level {{ $character['CurLevel'] }}</div>
                                        @if(!$character['isActive'])
                                            <div class="text-xs text-red-400">❌ غير نشط</div>
                                        @else
                                            <div class="text-xs text-green-400">✅ نشط</div>
                                        @endif
                                    </div>
                                    @if($selectedCharacter == $character['CharID'])
                                        <div class="text-blue-400 animate-pulse">
                                            <svg class="w-6 h-6" fill="currentColor" viewBox="0 0 20 20">
                                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"/>
                                            </svg>
                                        </div>
                                    @endif
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>

                <!-- Old Player Fields -->
                @if($selectedPlayerType === 'old')
                    <div class="mb-6">
                        <label class="block text-lg font-semibold text-gray-200 mb-3">📝 اسم الشخصية القديمة</label>
                        <input type="text"
                               wire:model="oldCharacterName"
                               class="w-full px-4 py-3 bg-gray-800/50 border border-gray-600 rounded-xl text-gray-200 placeholder-gray-400 focus:ring-2 focus:ring-purple-500 focus:border-transparent transition-all duration-300"
                               placeholder="أدخل اسم شخصيتك القديمة">
                    </div>

                    <div class="mb-6">
                        <label class="block text-lg font-semibold text-gray-200 mb-3">📷 صور إثبات الشخصية القديمة</label>
                        <input type="file"
                               wire:model="oldCharacterProofs"
                               multiple
                               accept="image/*"
                               class="w-full px-4 py-3 bg-gray-800/50 border border-gray-600 rounded-xl text-gray-200 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:bg-purple-600 file:text-white hover:file:bg-purple-700 focus:ring-2 focus:ring-purple-500 focus:border-transparent transition-all duration-300">
                        <p class="text-sm text-gray-400 mt-2">📸 ارفع صور تثبت ملكيتك للشخصية القديمة</p>
                    </div>
                @endif

                <!-- Donate Player Fields -->
                @if($selectedPlayerType === 'donate')
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                        <div>
                            <label class="block text-lg font-semibold text-gray-200 mb-3">💰 مبلغ التبرع</label>
                            <input type="number"
                                   wire:model.live="donationAmount"
                                   min="1"
                                   class="w-full px-4 py-3 bg-gray-800/50 border border-gray-600 rounded-xl text-gray-200 placeholder-gray-400 focus:ring-2 focus:ring-green-500 focus:border-transparent transition-all duration-300"
                                   placeholder="أدخل المبلغ">
                        </div>
                        <div>
                            <label class="block text-lg font-semibold text-gray-200 mb-3">💱 العملة</label>
                            <select wire:model.live="selectedCurrency"
                                    class="w-full px-4 py-3 bg-gray-800/50 border border-gray-600 rounded-xl text-gray-200 focus:ring-2 focus:ring-green-500 focus:border-transparent transition-all duration-300">
                                <option value="USD">USD 💵</option>
                                <option value="EGP">EGP 🪙</option>
                            </select>
                        </div>
                    </div>

                    @if($calculatedBalance > 0)
                        <div class="relative bg-gradient-to-br from-green-500/20 to-emerald-500/20 border border-green-500/30 rounded-2xl p-6 mb-6 overflow-hidden">
                            <div class="absolute inset-0 bg-gradient-to-br from-green-500/10 to-emerald-500/10"></div>
                            <div class="relative text-center">
                                <div class="text-4xl font-bold bg-gradient-to-r from-green-400 to-emerald-400 bg-clip-text text-transparent mb-2">
                                    {{ $calculatedBalance }} Balance 💎
                                </div>
                                <div class="text-green-300 font-medium">المكافأة المحسوبة</div>
                                <div class="mt-2 flex justify-center space-x-1">
                                    <div class="w-2 h-2 bg-green-400 rounded-full animate-bounce"></div>
                                    <div class="w-2 h-2 bg-emerald-400 rounded-full animate-bounce" style="animation-delay: 0.1s"></div>
                                    <div class="w-2 h-2 bg-green-400 rounded-full animate-bounce" style="animation-delay: 0.2s"></div>
                                </div>
                            </div>
                        </div>
                    @endif

                    <div class="mb-6">
                        <label class="block text-lg font-semibold text-gray-200 mb-3">🧾 صور إثبات التبرع</label>
                        <input type="file"
                               wire:model="donationProofs"
                               multiple
                               accept="image/*"
                               class="w-full px-4 py-3 bg-gray-800/50 border border-gray-600 rounded-xl text-gray-200 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:bg-green-600 file:text-white hover:file:bg-green-700 focus:ring-2 focus:ring-green-500 focus:border-transparent transition-all duration-300">
                        <p class="text-sm text-gray-400 mt-2">📸 ارفع صور تثبت عملية التبرع</p>
                    </div>
                @endif

                <!-- Optional Character Images -->
                <div class="mb-8">
                    <label class="block text-lg font-semibold text-gray-200 mb-3">🖼️ صور الشخصية (اختيارية)</label>
                    <input type="file"
                           wire:model="oldCharacterImages"
                           multiple
                           accept="image/*"
                           class="w-full px-4 py-3 bg-gray-800/50 border border-gray-600 rounded-xl text-gray-200 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:bg-blue-600 file:text-white hover:file:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all duration-300">
                    <p class="text-sm text-gray-400 mt-2">📸 ارفع صور إضافية للشخصية إذا أردت</p>
                </div>

                <!-- Submit Button -->
                <div class="text-center">
                    <button wire:click="submitApplication"
                            @if(!$selectedCharacter || $isSubmitting) disabled @endif
                            class="group relative px-12 py-4 bg-gradient-to-r from-blue-600 via-purple-600 to-green-600 text-white font-bold text-lg rounded-2xl hover:from-blue-500 hover:via-purple-500 hover:to-green-500 transition-all duration-500 disabled:opacity-50 disabled:cursor-not-allowed transform hover:scale-105 shadow-2xl hover:shadow-blue-500/25">

                        <!-- Button glow effect -->
                        <div class="absolute inset-0 bg-gradient-to-r from-blue-600 via-purple-600 to-green-600 rounded-2xl blur-xl opacity-30 group-hover:opacity-50 transition-opacity duration-500"></div>

                        <span class="relative">
                            @if($isSubmitting)
                                <span class="flex items-center justify-center">
                                    <svg class="animate-spin -ml-1 mr-3 h-6 w-6 text-white" fill="none" viewBox="0 0 24 24">
                                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                                        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                    </svg>
                                    جاري الإرسال... ⏳
                                </span>
                            @else
                                إرسال الطلب 🚀
                            @endif
                        </span>
                    </button>
                </div>
            </div>
        @endif
    @else
        <!-- Application Status -->
        <div class="bg-white rounded-xl shadow-lg p-6">
            @if($userApplication->status === 'pending')
                <div class="text-center">
                    <div class="w-16 h-16 bg-yellow-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <svg class="w-8 h-8 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                    </div>
                    <h2 class="text-2xl font-bold text-yellow-700 mb-2">في انتظار المراجعة</h2>
                    <p class="text-yellow-600">طلبك قيد المراجعة من قبل الفريق</p>
                </div>
            @elseif($userApplication->status === 'approved')
                <div class="text-center">
                    <div class="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <svg class="w-8 h-8 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                    </div>
                    <h2 class="text-2xl font-bold text-green-700 mb-2">تم قبول طلبك!</h2>
                    <p class="text-green-600 mb-6">يمكنك الآن استلام المكافأة</p>

                    <div class="bg-green-50 rounded-lg p-4 mb-6">
                        <div class="text-3xl font-bold text-green-600">{{ $userApplication->reward_amount }} Balance</div>
                    </div>

                    <button wire:click="collectRewards"
                            @if($isClaiming) disabled @endif
                            class="bg-green-600 text-white px-8 py-3 rounded-lg font-medium hover:bg-green-700 transition-colors disabled:opacity-50">
                        @if($isClaiming)
                            جاري الاستلام...
                        @else
                            استلام المكافأة
                        @endif
                    </button>
                </div>
            @elseif($userApplication->status === 'rejected')
                <div class="text-center">
                    <div class="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <svg class="w-8 h-8 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                        </svg>
                    </div>
                    <h2 class="text-2xl font-bold text-red-700 mb-2">تم رفض الطلب</h2>
                    <p class="text-red-600 mb-6">للأسف تم رفض طلبك</p>

                    <button wire:click="resetApplication"
                            class="bg-blue-600 text-white px-8 py-3 rounded-lg font-medium hover:bg-blue-700 transition-colors">
                        إنشاء طلب جديد
                    </button>
                </div>
            @elseif($userApplication->status === 'claimed')
                <div class="text-center">
                    <div class="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <svg class="w-8 h-8 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                        </svg>
                    </div>
                    <h2 class="text-2xl font-bold text-blue-700 mb-2">تم استلام المكافأة</h2>
                    <p class="text-blue-600">تم إضافة {{ $userApplication->reward_amount }} Balance إلى حسابك</p>
                </div>
            @endif

            <!-- Application Details -->
            <div class="mt-8 border-t pt-6">
                <h3 class="text-lg font-medium text-gray-900 mb-4">تفاصيل الطلب</h3>
                <div class="grid grid-cols-1 sm:grid-cols-2 gap-4 text-sm">
                    <div>
                        <span class="text-gray-600">رقم الطلب:</span>
                        <span class="font-medium">#{{ $userApplication->id }}</span>
                    </div>
                    <div>
                        <span class="text-gray-600">نوع اللاعب:</span>
                        <span class="font-medium">{{ ucfirst($userApplication->player_type) }}</span>
                    </div>
                    <div>
                        <span class="text-gray-600">المكافأة:</span>
                        <span class="font-medium">{{ $userApplication->reward_amount }} Balance</span>
                    </div>
                    <div>
                        <span class="text-gray-600">تاريخ الإرسال:</span>
                        <span class="font-medium">{{ $userApplication->submitted_at->format('Y-m-d H:i') }}</span>
                    </div>
                </div>
            </div>
        </div>
    @endif
</div>
