<?php

use Livewire\Volt\Component;
use Livewire\WithFileUploads;
use App\Models\FastEventsWelcome;
use App\Models\SRO_VT_SHARD\Char;
use App\Models\SRO_VT_PROXY\_Players;
use App\Models\SRO_VT_SHARD\_User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

new class extends Component {
    use WithFileUploads;

    public $selectedPlayerType = null;
    public $selectedCharacter = null;
    public $oldCharacterName = '';
    public $oldCharacterProofs = [];
    public $donationAmount = 0;
    public $selectedCurrency = 'USD';
    public $donationProofs = [];
    public $oldCharacterImages = [];
    public $calculatedBalance = 0;
    public $userApplication = null;
    public $userCharacters = [];
    public $isSubmitting = false;
    public $isClaiming = false;

    public function mount()
    {
        $this->loadUserData();
        $this->checkForRejectionNotification();
    }

    public function checkForRejectionNotification()
    {
        $recentRejection = FastEventsWelcome::where('JID', Auth::user()->JID)
            ->where('status', 'rejected')
            ->where('reviewed_at', '>=', now()->subMinutes(10))
            ->latest()
            ->first();

        if ($recentRejection) {
            $this->dispatch('show-rejection-notification', [
                'applicationId' => $recentRejection->id,
                'rejectionReason' => $recentRejection->rejection_reason ?? 'No specific reason provided',
                'reviewedAt' => $recentRejection->reviewed_at->format('M d, Y H:i'),
                'playerType' => ucfirst($recentRejection->player_type) . ' Player',
                'rewardAmount' => $recentRejection->reward_amount
            ]);
        }
    }

    public function loadUserData()
    {
        $user = Auth::user();

        $this->userApplication = FastEventsWelcome::where('JID', $user->JID)
            ->whereIn('status', ['pending', 'approved', 'claimed'])
            ->latest('created_at')
            ->first();

        $userCharIds = _User::where('UserJID', $user->JID)->pluck('CharID');
        $this->userCharacters = Char::whereIn('CharID', $userCharIds)
            ->select('CharName16', 'CharID', 'RefObjID', 'CurLevel')
            ->orderBy('CharName16')
            ->get()
            ->map(function($char) {
                $player = _Players::where('CharID', $char->CharID)->first();
                $isActive = $player && $player->cur_status == true;

                return [
                    'CharID' => $char->CharID,
                    'CharName16' => $char->CharName16,
                    'RefObjID' => $char->RefObjID,
                    'CurLevel' => $char->CurLevel,
                    'avatar' => Char::getCharAvatarGifByCharname($char->CharName16),
                    'isActive' => $isActive
                ];
            });

        $this->checkForRejectionNotification();
    }

    public function getRejectedApplications()
    {
        return FastEventsWelcome::where('JID', Auth::user()->JID)
            ->where('status', 'rejected')
            ->latest('reviewed_at')
            ->limit(3)
            ->get();
    }

    public function selectPlayerType($type)
    {
        $this->selectedPlayerType = $type;
        $this->calculateReward();
    }

    public function selectCharacter($characterId)
    {
        $userCharIds = _User::where('UserJID', Auth::user()->JID)->pluck('CharID');

        if ($userCharIds->contains($characterId)) {
            $player = _Players::where('CharID', $characterId)->first();
            if (!$player || !$player->cur_status) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character must be active (played at least once) for all applications.',
                    'duration' => 2500,
                ]);
                return;
            }

            $this->selectedCharacter = $characterId;
        }
    }

    public function updatedDonationAmount()
    {
        $this->calculateReward();
    }

    public function updatedSelectedCurrency()
    {
        $this->calculateReward();
    }

    public function calculateReward()
    {
        if ($this->selectedPlayerType === 'donate' && $this->donationAmount > 0) {
            $rate = $this->selectedCurrency === 'USD' ? 40 : 1;
            $this->calculatedBalance = floor(($this->donationAmount * $rate) * 0.5);
        } else {
            $this->calculatedBalance = 0;
        }
    }

    public function submitApplication()
    {
        $this->isSubmitting = true;

        try {
            $rules = [
                'selectedPlayerType' => 'required|in:new,old,donate',
                'selectedCharacter' => 'required|numeric',
            ];

            if ($this->selectedPlayerType === 'old') {
                $rules['oldCharacterName'] = 'required|string|max:255';
                $rules['oldCharacterProofs'] = 'required|array|min:1';
                $rules['oldCharacterProofs.*'] = 'image|mimes:jpeg,png,jpg,gif|max:5120';
            }

            if ($this->selectedPlayerType === 'donate') {
                $rules['donationAmount'] = 'required|numeric|min:1';
                $rules['selectedCurrency'] = 'required|in:USD,EGP';
                $rules['donationProofs'] = 'required|array|min:1';
                $rules['donationProofs.*'] = 'image|mimes:jpeg,png,jpg,gif|max:5120';
            }

            $this->validate($rules);

            $userCharIds = _User::where('UserJID', Auth::user()->JID)->pluck('CharID');

            if (!$userCharIds->contains($this->selectedCharacter)) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character not found or does not belong to you.',
                    'duration' => 2500,
                ]);
                return;
            }

            $character = Char::where('CharID', $this->selectedCharacter)->first();
            if (!$character) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character not found in database.',
                    'duration' => 2500,
                ]);
                return;
            }

            $player = _Players::where('CharID', $this->selectedCharacter)->first();
            if (!$player || !$player->cur_status) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character must be active (played at least once) for all applications.',
                    'duration' => 2500,
                ]);
                return;
            }

            $oldCharacterProofPaths = [];
            $donationProofPaths = [];
            $oldCharacterImagePaths = [];

            if ($this->selectedPlayerType === 'old' && !empty($this->oldCharacterProofs)) {
                foreach ($this->oldCharacterProofs as $file) {
                    $path = $file->store('welcome-event/old-character-proofs', 'public');
                    $oldCharacterProofPaths[] = $path;
                }
            }

            if ($this->selectedPlayerType === 'donate' && !empty($this->donationProofs)) {
                foreach ($this->donationProofs as $file) {
                    $path = $file->store('welcome-event/donation-proofs', 'public');
                    $donationProofPaths[] = $path;
                }
            }

            if (!empty($this->oldCharacterImages)) {
                foreach ($this->oldCharacterImages as $file) {
                    $path = $file->store('welcome-event/old-character-images', 'public');
                    $oldCharacterImagePaths[] = $path;
                }
            }

            $rewardAmount = $this->getRewardAmount();

            $player = _Players::where('CharID', $this->selectedCharacter)->first();
            $hwid = $player ? $player->Serial : '';

            FastEventsWelcome::create([
                'JID' => Auth::user()->JID,
                'HWID' => $hwid,
                'player_type' => $this->selectedPlayerType,
                'CharID' => $this->selectedCharacter,
                'old_character_name' => $this->selectedPlayerType === 'old' ? $this->oldCharacterName : null,
                'donation_amount' => $this->selectedPlayerType === 'donate' ? $this->donationAmount : null,
                'currency' => $this->selectedPlayerType === 'donate' ? $this->selectedCurrency : null,
                'reward_amount' => $rewardAmount,
                'old_character_proofs' => !empty($oldCharacterProofPaths) ? json_encode($oldCharacterProofPaths) : null,
                'donation_proofs' => !empty($donationProofPaths) ? json_encode($donationProofPaths) : null,
                'old_character_images' => !empty($oldCharacterImagePaths) ? json_encode($oldCharacterImagePaths) : null,
                'status' => 'pending',
                'submitted_at' => now(),
            ]);

            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'Application submitted successfully!',
                'duration' => 3000,
            ]);
            $this->loadUserData();

        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Validation failed: ' . implode(', ', $e->validator->errors()->all()),
                'duration' => 5000,
            ]);
        } catch (\Exception $e) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Failed to submit application: ' . $e->getMessage(),
                'duration' => 5000,
            ]);
        } finally {
            $this->isSubmitting = false;
        }
    }

    public function collectRewards()
    {
        $this->isClaiming = true;

        try {
            if (!$this->userApplication || !$this->userApplication->canBeClaimed()) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Unable to collect rewards. Please try again.',
                    'duration' => 2500,
                ]);
                $this->isClaiming = false;
                return;
            }

            $user = Auth::user();

            if ($user && $user->webUser) {
                $oldBalance = $user->webUser->balance;
                $user->webUser->balance += $this->userApplication->reward_amount;
                $user->webUser->save();
            } else {
                throw new \Exception('User account not properly configured');
            }

            $this->userApplication->update([
                'status' => 'claimed',
                'claimed_at' => now(),
            ]);

            $this->userApplication->refresh();

            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'Rewards collected successfully! Your balance has been updated.',
                'duration' => 3000,
            ]);

            $this->dispatch('updateBalances');
            $this->loadUserData();
            $this->dispatch('hide-congratulations');

        } catch (\Exception $e) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Error collecting rewards: ' . $e->getMessage(),
                'duration' => 2500,
            ]);
        } finally {
            $this->isClaiming = false;
        }
    }

    public function claimReward()
    {
        // Auto collect rewards directly without modal
        $this->collectRewards();
    }

    private function getRewardAmount()
    {
        switch ($this->selectedPlayerType) {
            case 'new':
                return 10;
            case 'old':
                return 40;
            case 'donate':
                return $this->calculatedBalance;
            default:
                return 0;
        }
    }

    public function resetApplication()
    {
        $this->selectedPlayerType = null;
        $this->selectedCharacter = null;
        $this->oldCharacterName = '';
        $this->oldCharacterProofs = [];
        $this->donationAmount = 0;
        $this->selectedCurrency = 'USD';
        $this->donationProofs = [];
        $this->oldCharacterImages = [];
        $this->calculatedBalance = 0;

        if ($this->userApplication && $this->userApplication->status === 'rejected') {
            $this->userApplication = null;

            $this->dispatch('show-notification', [
                'type' => 'info',
                'message' => 'Ready to create a new application. Your previous application history has been preserved.',
                'duration' => 3000,
            ]);
        }
    }

    public function removeOldCharacterProof($index)
    {
        if (isset($this->oldCharacterProofs[$index])) {
            unset($this->oldCharacterProofs[$index]);
            $this->oldCharacterProofs = array_values($this->oldCharacterProofs);
        }
    }

    public function removeDonationProof($index)
    {
        if (isset($this->donationProofs[$index])) {
            unset($this->donationProofs[$index]);
            $this->donationProofs = array_values($this->donationProofs);
        }
    }

    public function removeOldCharacterImage($index)
    {
        if (isset($this->oldCharacterImages[$index])) {
            unset($this->oldCharacterImages[$index]);
            $this->oldCharacterImages = array_values($this->oldCharacterImages);
        }
    }

    public function with()
    {
        return [
            'hasApplication' => !is_null($this->userApplication),
        ];
    }
}; ?>

<div class="welcome2-container"
     x-data="{
         showNotification: false,
         notificationData: {},

         showAlert(data) {
             this.notificationData = data;
             this.showNotification = true;
             setTimeout(() => {
                 this.showNotification = false;
             }, data.duration || 3000);
         }
     }"
     @show-notification.window="showAlert($event.detail)">

    <!-- Notification -->
    <div x-show="showNotification"
         x-transition:enter="transform transition ease-out duration-300"
         x-transition:enter-start="translate-y-[-100%] opacity-0"
         x-transition:enter-end="translate-y-0 opacity-100"
         x-transition:leave="transform transition ease-in duration-200"
         x-transition:leave-start="translate-y-0 opacity-100"
         x-transition:leave-end="translate-y-[-100%] opacity-0"
         class="fixed top-4 right-4 z-50 max-w-sm bg-gray-900/90 backdrop-blur-lg rounded-2xl shadow-2xl border-l-4 overflow-hidden"
         :class="{
             'border-green-500': notificationData.type === 'success',
             'border-red-500': notificationData.type === 'error',
             'border-blue-500': notificationData.type === 'info'
         }">

        <!-- Glow effect -->
        <div class="absolute inset-0 opacity-20"
             :class="{
                 'bg-gradient-to-r from-green-500/20 to-emerald-500/20': notificationData.type === 'success',
                 'bg-gradient-to-r from-red-500/20 to-pink-500/20': notificationData.type === 'error',
                 'bg-gradient-to-r from-blue-500/20 to-cyan-500/20': notificationData.type === 'info'
             }"></div>

        <div class="relative p-4">
            <div class="flex items-start">
                <div class="flex-shrink-0">
                    <div x-show="notificationData.type === 'success'" class="text-green-400">
                        <svg class="w-6 h-6" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"/>
                        </svg>
                    </div>
                    <div x-show="notificationData.type === 'error'" class="text-red-400">
                        <svg class="w-6 h-6" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z"/>
                        </svg>
                    </div>
                    <div x-show="notificationData.type === 'info'" class="text-blue-400">
                        <svg class="w-6 h-6" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z"/>
                        </svg>
                    </div>
                </div>
                <div class="ml-3 flex-1">
                    <p class="text-sm font-medium text-gray-200" x-text="notificationData.message"></p>
                </div>
                <div class="ml-4 flex-shrink-0">
                    <button @click="showNotification = false" class="text-gray-400 hover:text-gray-200 transition-colors">
                        <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z"/>
                        </svg>
                    </button>
                </div>
            </div>
        </div>
    </div>

            <!-- Success Modal -->
    <!-- Removed - Auto collect rewards instead -->    <!-- Header -->
    <div class="text-center mb-8 relative">
        <div class="absolute inset-0 bg-gradient-to-r from-blue-600/20 via-purple-600/20 to-green-600/20 rounded-2xl blur-xl"></div>
        <div class="relative bg-gray-900/80 backdrop-blur-lg rounded-2xl p-8 border border-gray-700/50">
            <h1 class="text-5xl font-bold bg-gradient-to-r from-blue-400 via-purple-400 to-green-400 bg-clip-text text-transparent mb-4" data-translate="welcomeEvent.welcome_title">
                Welcome to the Event
            </h1>
            <p class="text-gray-300 text-lg" data-translate="welcomeEvent.choose_player_type">Choose your player type and get amazing rewards</p>
            <div class="mt-4 flex justify-center space-x-2">
                <div class="w-2 h-2 bg-blue-400 rounded-full animate-pulse"></div>
                <div class="w-2 h-2 bg-purple-400 rounded-full animate-pulse" style="animation-delay: 0.2s"></div>
                <div class="w-2 h-2 bg-green-400 rounded-full animate-pulse" style="animation-delay: 0.4s"></div>
            </div>
        </div>
    </div>

    <!-- Rejected Application Alert -->
    @php
        $rejectedApplications = $this->getRejectedApplications();
    @endphp

    @if($rejectedApplications->isNotEmpty())
        <div class="rejected-application-alert mb-8"
             x-data="{
                 show: true,
                 expandedId: null,
                 toggleDetails(id) {
                     this.expandedId = this.expandedId === id ? null : id;
                 }
             }"
             x-show="show"
             x-transition:enter="transform transition ease-out duration-300"
             x-transition:enter-start="translate-y-[-20px] opacity-0"
             x-transition:enter-end="translate-y-0 opacity-100">

            <div class="relative bg-gradient-to-br from-red-900/40 to-pink-900/40 backdrop-blur-lg rounded-2xl border border-red-500/30 p-6 shadow-2xl">
                <!-- Background Effects -->
                <div class="absolute inset-0 bg-gradient-to-br from-red-500/10 to-pink-500/10 rounded-2xl"></div>
                <div class="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-red-500 to-pink-500 rounded-t-2xl"></div>

                <!-- Alert Header -->
                <div class="relative flex items-start justify-between mb-4">
                    <div class="flex items-center">
                        <div class="w-12 h-12 bg-gradient-to-br from-red-500 to-pink-500 rounded-xl flex items-center justify-center mr-4 shadow-lg">
                            <svg class="w-6 h-6 text-white" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z"/>
                            </svg>
                        </div>
                        <div>
                            <h3 class="text-xl font-bold text-red-200" data-translate="welcomeEvent.rejectedAlert.title">
                                Your Application Has Been Rejected
                            </h3>
                            <p class="text-red-300/80 text-sm" data-translate="welcomeEvent.rejectedAlert.subtitle">
                                Please review the details below and submit a new application
                            </p>
                        </div>
                    </div>
                    <button @click="show = false"
                            class="text-red-300 hover:text-red-100 transition-colors p-2 rounded-lg hover:bg-red-500/20">
                        <svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z"/>
                        </svg>
                    </button>
                </div>

                <!-- Rejected Applications List -->
                <div class="relative space-y-3">
                    @foreach($rejectedApplications as $application)
                        <div class="rejection-details-card bg-black/20 border border-red-500/20 rounded-xl p-4 backdrop-blur-sm">
                            <!-- Application Summary -->
                            <div class="flex items-center justify-between cursor-pointer"
                                 @click="toggleDetails({{ $application->id }})">
                                <div class="flex items-center space-x-3">
                                    <div class="w-8 h-8 bg-red-500/20 rounded-lg flex items-center justify-center">
                                        <span class="text-red-300 font-bold text-sm">#{{ $application->id }}</span>
                                    </div>
                                    <div>
                                        <p class="text-red-200 font-medium">
                                            {{ ucfirst($application->player_type) }} Player Application
                                        </p>
                                        <p class="text-red-300/70 text-xs">
                                            <span data-translate="welcomeEvent.rejectedAlert.rejectedOn">Rejected on</span>
                                            {{ $application->reviewed_at?->format('M d, Y H:i') }}
                                        </p>
                                    </div>
                                </div>
                                <div class="flex items-center space-x-2">
                                    <span class="text-red-300 text-sm" data-translate="welcomeEvent.rejectedAlert.viewDetails">View Details</span>
                                    <svg class="w-4 h-4 text-red-300 transition-transform duration-200"
                                         :class="{ 'rotate-180': expandedId === {{ $application->id }} }"
                                         fill="currentColor" viewBox="0 0 20 20">
                                        <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z"/>
                                    </svg>
                                </div>
                            </div>

                            <!-- Expanded Details -->
                            <div x-show="expandedId === {{ $application->id }}"
                                 x-collapse
                                 class="mt-4 pt-4 border-t border-red-500/20">

                                <!-- Application Details Grid -->
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                                    <div class="detail-item">
                                        <span class="detail-label text-red-200/80" data-translate="welcomeEvent.rejectedAlert.applicationId">Application ID:</span>
                                        <span class="detail-value text-red-100">#{{ $application->id }}</span>
                                    </div>
                                    <div class="detail-item">
                                        <span class="detail-label text-red-200/80" data-translate="welcomeEvent.rejectedAlert.playerType">Player Type:</span>
                                        <span class="detail-value text-red-100">{{ ucfirst($application->player_type) }} Player</span>
                                    </div>
                                    <div class="detail-item">
                                        <span class="detail-label text-red-200/80" data-translate="welcomeEvent.rejectedAlert.character">Character:</span>
                                        <span class="detail-value text-red-100">{{ $application->character?->CharName16 ?? 'N/A' }}</span>
                                    </div>
                                    <div class="detail-item">
                                        <span class="detail-label text-red-200/80" data-translate="welcomeEvent.rejectedAlert.requestedReward">Requested Reward:</span>
                                        <span class="detail-value text-red-100">{{ $application->reward_amount }} Balance</span>
                                    </div>
                                </div>

                                <!-- Rejection Reason -->
                                <div class="rejection-reason bg-red-950/50 border border-red-500/30 rounded-lg p-4 mb-4">
                                    <h4 class="text-red-200 font-semibold mb-2" data-translate="welcomeEvent.rejectedAlert.rejectionReason">
                                        Rejection Reason:
                                    </h4>
                                    <p class="text-red-100/90 leading-relaxed">
                                        {{ $application->rejection_reason ?? 'No specific reason provided' }}
                                    </p>
                                </div>

                                <!-- Action Text -->
                                <div class="flex justify-end">
                                    <p class="text-blue-300 text-sm font-medium cursor-pointer hover:text-blue-200 transition-colors duration-200"
                                       wire:click="resetApplication">
                                        <span data-translate="welcomeEvent.rejectedAlert.createNewApplication">Create New Application</span>
                                    </p>
                                </div>
                            </div>
                        </div>
                    @endforeach
                </div>
            </div>
        </div>
    @endif

    @if (!$hasApplication)
        <!-- Player Type Selection -->
        @if (!$selectedPlayerType)
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                <!-- New Player -->
                <div wire:click="selectPlayerType('new')"
                     class="group relative bg-gradient-to-br from-gray-900 to-gray-800 p-8 rounded-2xl border border-blue-500/30 cursor-pointer hover:border-blue-400 transition-all duration-500 transform hover:scale-105 hover:shadow-2xl hover:shadow-blue-500/25">
                    <!-- Glow effect -->
                    <div class="absolute inset-0 bg-gradient-to-br from-blue-500/20 to-cyan-500/20 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                    <div class="relative text-center">
                        <div class="w-20 h-20 bg-gradient-to-br from-blue-500 to-cyan-400 rounded-full flex items-center justify-center mx-auto mb-6 group-hover:scale-110 transition-transform duration-300">
                            <svg class="w-10 h-10 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                            </svg>
                        </div>
                        <h3 class="text-2xl font-bold text-blue-400 mb-3" data-translate="welcomeEvent.new_player_title">New Player</h3>
                        <p class="text-gray-300 text-sm mb-6 leading-relaxed" data-translate="welcomeEvent.new_player_description">For new players starting their journey in the game</p>
                        <div class="bg-gradient-to-r from-blue-600 to-cyan-500 text-white px-6 py-3 rounded-lg font-semibold shadow-lg" data-translate="welcomeEvent.new_player_reward">
                            Reward: 10 Balance ⭐
                        </div>
                    </div>
                </div>

                <!-- Old Player -->
                <div wire:click="selectPlayerType('old')"
                     class="group relative bg-gradient-to-br from-gray-900 to-gray-800 p-8 rounded-2xl border border-purple-500/30 cursor-pointer hover:border-purple-400 transition-all duration-500 transform hover:scale-105 hover:shadow-2xl hover:shadow-purple-500/25">
                    <!-- Glow effect -->
                    <div class="absolute inset-0 bg-gradient-to-br from-purple-500/20 to-pink-500/20 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                    <div class="relative text-center">
                        <div class="w-20 h-20 bg-gradient-to-br from-purple-500 to-pink-400 rounded-full flex items-center justify-center mx-auto mb-6 group-hover:scale-110 transition-transform duration-300">
                            <svg class="w-10 h-10 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                        </div>
                        <h3 class="text-2xl font-bold text-purple-400 mb-3" data-translate="welcomeEvent.old_player_title">Returning Player</h3>
                        <p class="text-gray-300 text-sm mb-6 leading-relaxed" data-translate="welcomeEvent.old_player_description">For old players returning to the game</p>
                        <div class="bg-gradient-to-r from-purple-600 to-pink-500 text-white px-6 py-3 rounded-lg font-semibold shadow-lg" data-translate="welcomeEvent.old_player_reward">
                            Reward: 40 Balance 🎖️
                        </div>
                    </div>
                </div>

                <!-- Donate Player -->
                <div wire:click="selectPlayerType('donate')"
                     class="group relative bg-gradient-to-br from-gray-900 to-gray-800 p-8 rounded-2xl border border-green-500/30 cursor-pointer hover:border-green-400 transition-all duration-500 transform hover:scale-105 hover:shadow-2xl hover:shadow-green-500/25">
                    <!-- Glow effect -->
                    <div class="absolute inset-0 bg-gradient-to-br from-green-500/20 to-emerald-500/20 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                    <div class="relative text-center">
                        <div class="w-20 h-20 bg-gradient-to-br from-green-500 to-emerald-400 rounded-full flex items-center justify-center mx-auto mb-6 group-hover:scale-110 transition-transform duration-300">
                            <svg class="w-10 h-10 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"/>
                            </svg>
                        </div>
                        <h3 class="text-2xl font-bold text-green-400 mb-3" data-translate="welcomeEvent.donate_player_title">Donation Bonus</h3>
                        <p class="text-gray-300 text-sm mb-6 leading-relaxed" data-translate="welcomeEvent.donate_player_description">For players who supported the server with donations</p>
                        <div class="bg-gradient-to-r from-green-600 to-emerald-500 text-white px-6 py-3 rounded-lg font-semibold shadow-lg" data-translate="welcomeEvent.donate_player_reward">
                            Reward: Variable 💎
                        </div>
                    </div>
                </div>
            </div>
        @else
            <!-- Selected Player Type Card -->
            <div class="relative bg-gray-900/90 backdrop-blur-lg rounded-2xl shadow-2xl p-8 mb-8 border border-gray-700/50
                 {{ $selectedPlayerType === 'new' ? 'border-blue-500/50 shadow-blue-500/20' : '' }}
                 {{ $selectedPlayerType === 'old' ? 'border-purple-500/50 shadow-purple-500/20' : '' }}
                 {{ $selectedPlayerType === 'donate' ? 'border-green-500/50 shadow-green-500/20' : '' }}">

                <!-- Background glow -->
                <div class="absolute inset-0 rounded-2xl
                     {{ $selectedPlayerType === 'new' ? 'bg-gradient-to-br from-blue-500/10 to-cyan-500/5' : '' }}
                     {{ $selectedPlayerType === 'old' ? 'bg-gradient-to-br from-purple-500/10 to-pink-500/5' : '' }}
                     {{ $selectedPlayerType === 'donate' ? 'bg-gradient-to-br from-green-500/10 to-emerald-500/5' : '' }}">
                </div>

                <div class="relative flex items-center mb-8">
                    <div class="w-16 h-16 rounded-2xl flex items-center justify-center mr-6
                         {{ $selectedPlayerType === 'new' ? 'bg-gradient-to-br from-blue-500 to-cyan-400' : '' }}
                         {{ $selectedPlayerType === 'old' ? 'bg-gradient-to-br from-purple-500 to-pink-400' : '' }}
                         {{ $selectedPlayerType === 'donate' ? 'bg-gradient-to-br from-green-500 to-emerald-400' : '' }}">
                        @if($selectedPlayerType === 'new')
                            <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                            </svg>
                        @elseif($selectedPlayerType === 'old')
                            <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                        @elseif($selectedPlayerType === 'donate')
                            <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"/>
                            </svg>
                        @endif
                    </div>
                    <div class="flex-1">
                        <h2 class="text-3xl font-bold mb-2
                             {{ $selectedPlayerType === 'new' ? 'text-blue-400' : '' }}
                             {{ $selectedPlayerType === 'old' ? 'text-purple-400' : '' }}
                             {{ $selectedPlayerType === 'donate' ? 'text-green-400' : '' }}">
                            @if($selectedPlayerType === 'new')
                                <span data-translate="welcomeEvent.new_player_selected">New Player ⭐</span>
                            @elseif($selectedPlayerType === 'old')
                                <span data-translate="welcomeEvent.old_player_selected">Returning Player 🎖️</span>
                            @elseif($selectedPlayerType === 'donate')
                                <span data-translate="welcomeEvent.donate_player_selected">Donation Bonus 💎</span>
                            @endif
                        </h2>
                        <p class="text-gray-300" data-translate="welcomeEvent.fill_form_description">Fill out the data below to send your request and get the reward</p>
                    </div>
                    <div class="mr-auto">
                        <button wire:click="resetApplication"
                                class="text-gray-400 hover:text-red-400 transition-colors duration-300 p-2 rounded-lg hover:bg-gray-800/50">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                            </svg>
                        </button>
                    </div>
                </div>

                <!-- Character Selection -->
                <div class="relative mb-8">
                    <label class="block text-lg font-semibold text-gray-200 mb-4" data-translate="welcomeEvent.select_character">🎮 Select Character</label>
                    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                        @foreach($userCharacters as $character)
                            <div wire:click="selectCharacter({{ $character['CharID'] }})"
                                 class="group relative p-4 rounded-xl border-2 cursor-pointer transition-all duration-300
                                        {{ $character['isActive'] ? 'hover:border-blue-400 bg-gray-800/50 border-gray-600' : 'opacity-50 cursor-not-allowed bg-gray-900/50 border-gray-700' }}
                                        {{ $selectedCharacter == $character['CharID'] ? 'border-blue-500 bg-gradient-to-br from-blue-500/10 to-cyan-500/5 shadow-lg shadow-blue-500/20' : '' }}">

                                <!-- Glow effect for selected -->
                                @if($selectedCharacter == $character['CharID'])
                                    <div class="absolute inset-0 bg-gradient-to-br from-blue-500/20 to-cyan-500/20 rounded-xl"></div>
                                @endif

                                <div class="relative flex items-center">
                                    <div class="relative">
                                        <img src="{{ $character['avatar'] }}"
                                             alt="{{ $character['CharName16'] }}"
                                             class="w-12 h-12 rounded-full mr-4 border-2 border-gray-600 group-hover:border-blue-400 transition-colors">
                                        @if($selectedCharacter == $character['CharID'])
                                            <div class="absolute -inset-1 bg-gradient-to-br from-blue-500 to-cyan-400 rounded-full opacity-50 blur-sm"></div>
                                        @endif
                                    </div>
                                    <div class="flex-1">
                                        <div class="font-semibold text-gray-200 group-hover:text-blue-300 transition-colors">
                                            {{ $character['CharName16'] }}
                                        </div>
                                        <div class="text-sm text-gray-400">Level {{ $character['CurLevel'] }}</div>
                                        @if(!$character['isActive'])
                                            <div class="text-xs text-red-400" data-translate="welcomeEvent.character_inactive">❌ Inactive</div>
                                        @else
                                            <div class="text-xs text-green-400" data-translate="welcomeEvent.character_active">✅ Active</div>
                                        @endif
                                    </div>
                                    @if($selectedCharacter == $character['CharID'])
                                        <div class="text-blue-400 animate-pulse">
                                            <svg class="w-6 h-6" fill="currentColor" viewBox="0 0 20 20">
                                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"/>
                                            </svg>
                                        </div>
                                    @endif
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>

                <!-- Old Player Fields -->
                @if($selectedPlayerType === 'old')
                    <div class="mb-6">
                        <label class="block text-lg font-semibold text-gray-200 mb-3" data-translate="welcomeEvent.old_character_name">📝 Old Character Name</label>
                        <input type="text"
                               wire:model="oldCharacterName"
                               class="w-full px-4 py-3 bg-gray-800/50 border border-gray-600 rounded-xl text-gray-200 placeholder-gray-400 focus:ring-2 focus:ring-purple-500 focus:border-transparent transition-all duration-300"
                               placeholder="{{ __('welcomeEvent.old_character_name_placeholder', [], 'en') }}">
                    </div>

                    <div class="mb-6">
                        <label class="block text-lg font-semibold text-gray-200 mb-3" data-translate="welcomeEvent.old_character_proofs">📷 Old Character Proof Images</label>
                        <div class="relative"
                             x-data="{ dragover: false }"
                             @dragover.prevent="dragover = true"
                             @dragleave.prevent="dragover = false"
                             @drop.prevent="dragover = false; $refs.oldCharacterProofsInput.files = $event.dataTransfer.files; $refs.oldCharacterProofsInput.dispatchEvent(new Event('change', { bubbles: true }))">
                            <input type="file"
                                   wire:model="oldCharacterProofs"
                                   multiple
                                   accept="image/*"
                                   id="oldCharacterProofs"
                                   x-ref="oldCharacterProofsInput"
                                   class="hidden">
                            <div class="w-full px-6 py-8 bg-gray-800/50 border-2 border-dashed rounded-xl text-gray-200 transition-all duration-300 text-center cursor-pointer"
                                 :class="dragover ? 'border-purple-400 bg-purple-500/10' : 'border-gray-600 hover:border-purple-500'"
                                 @click="$refs.oldCharacterProofsInput.click()">
                                <div class="flex flex-col items-center space-y-3">
                                    <div class="w-16 h-16 bg-purple-500/20 rounded-2xl flex items-center justify-center">
                                        <svg class="w-8 h-8 text-purple-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"/>
                                        </svg>
                                    </div>
                                    <div>
                                        <span class="text-purple-400 font-medium text-lg" data-translate="welcomeEvent.choose_files">Choose Files</span>
                                        <p class="text-sm text-gray-400 mt-1" data-translate="welcomeEvent.or_drag_files">or drag and drop files here</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <p class="text-sm text-gray-400 mt-2" data-translate="welcomeEvent.old_character_proofs_desc">📸 Upload images that prove ownership of your old character</p>

                        <!-- Display uploaded files -->
                        @if($oldCharacterProofs)
                            <div class="mt-4 grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-3">
                                @foreach($oldCharacterProofs as $index => $file)
                                    <div class="relative group bg-gray-800/50 rounded-lg overflow-hidden border border-purple-500/30">
                                        <div class="aspect-square bg-gray-700 flex items-center justify-center">
                                            @if($file->isPreviewable())
                                                <img src="{{ $file->temporaryUrl() }}" alt="Preview" class="w-full h-full object-cover">
                                            @else
                                                <div class="text-center p-2">
                                                    <svg class="w-8 h-8 text-purple-400 mx-auto mb-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                                                    </svg>
                                                    <p class="text-xs text-purple-300">{{ Str::limit($file->getClientOriginalName(), 15) }}</p>
                                                </div>
                                            @endif
                                        </div>
                                        <div class="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-200">
                                            <div class="absolute bottom-2 left-2 right-2">
                                                <p class="text-white text-xs font-medium truncate">{{ $file->getClientOriginalName() }}</p>
                                                <p class="text-purple-300 text-xs">{{ round($file->getSize() / 1024, 1) }} KB</p>
                                            </div>
                                            <button type="button"
                                                    wire:click="removeOldCharacterProof({{ $index }})"
                                                    class="absolute top-2 right-2 w-6 h-6 bg-red-500 hover:bg-red-600 rounded-full flex items-center justify-center transition-colors">
                                                <svg class="w-3 h-3 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                                                </svg>
                                            </button>
                                        </div>
                                    </div>
                                @endforeach
                            </div>
                        @endif
                    </div>
                @endif

                <!-- Donate Player Fields -->
                @if($selectedPlayerType === 'donate')
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                        <div>
                            <label class="block text-lg font-semibold text-gray-200 mb-3" data-translate="welcomeEvent.donation_amount">💰 Donation Amount</label>
                            <input type="number"
                                   wire:model.live="donationAmount"
                                   min="1"
                                   class="w-full px-4 py-3 bg-gray-800/50 border border-gray-600 rounded-xl text-gray-200 placeholder-gray-400 focus:ring-2 focus:ring-green-500 focus:border-transparent transition-all duration-300"
                                   placeholder="{{ __('welcomeEvent.donation_amount_placeholder', [], 'en') }}">
                        </div>
                        <div>
                            <label class="block text-lg font-semibold text-gray-200 mb-3" data-translate="welcomeEvent.currency">💱 Currency</label>
                            <select wire:model.live="selectedCurrency"
                                    class="w-full px-4 py-3 bg-gray-800/50 border border-gray-600 rounded-xl text-gray-200 focus:ring-2 focus:ring-green-500 focus:border-transparent transition-all duration-300">
                                <option value="USD">USD 💵</option>
                                <option value="EGP">EGP 🪙</option>
                            </select>
                        </div>
                    </div>

                    @if($calculatedBalance > 0)
                        <div class="relative bg-gradient-to-br from-green-500/20 to-emerald-500/20 border border-green-500/30 rounded-2xl p-6 mb-6 overflow-hidden">
                            <div class="absolute inset-0 bg-gradient-to-br from-green-500/10 to-emerald-500/10"></div>
                            <div class="relative text-center">
                                <div class="text-4xl font-bold bg-gradient-to-r from-green-400 to-emerald-400 bg-clip-text text-transparent mb-2">
                                    {{ $calculatedBalance }} Balance 💎
                                </div>
                                <div class="text-green-300 font-medium" data-translate="welcomeEvent.calculated_reward">Calculated Reward</div>
                                <div class="mt-2 flex justify-center space-x-1">
                                    <div class="w-2 h-2 bg-green-400 rounded-full animate-bounce"></div>
                                    <div class="w-2 h-2 bg-emerald-400 rounded-full animate-bounce" style="animation-delay: 0.1s"></div>
                                    <div class="w-2 h-2 bg-green-400 rounded-full animate-bounce" style="animation-delay: 0.2s"></div>
                                </div>
                            </div>
                        </div>
                    @endif

                    <div class="mb-6">
                        <label class="block text-lg font-semibold text-gray-200 mb-3" data-translate="welcomeEvent.donation_proofs">🧾 Donation Proof Images</label>
                        <div class="relative"
                             x-data="{ dragover: false }"
                             @dragover.prevent="dragover = true"
                             @dragleave.prevent="dragover = false"
                             @drop.prevent="dragover = false; $refs.donationProofsInput.files = $event.dataTransfer.files; $refs.donationProofsInput.dispatchEvent(new Event('change', { bubbles: true }))">
                            <input type="file"
                                   wire:model="donationProofs"
                                   multiple
                                   accept="image/*"
                                   id="donationProofs"
                                   x-ref="donationProofsInput"
                                   class="hidden">
                            <div class="w-full px-6 py-8 bg-gray-800/50 border-2 border-dashed rounded-xl text-gray-200 transition-all duration-300 text-center cursor-pointer"
                                 :class="dragover ? 'border-green-400 bg-green-500/10' : 'border-gray-600 hover:border-green-500'"
                                 @click="$refs.donationProofsInput.click()">
                                <div class="flex flex-col items-center space-y-3">
                                    <div class="w-16 h-16 bg-green-500/20 rounded-2xl flex items-center justify-center">
                                        <svg class="w-8 h-8 text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"/>
                                        </svg>
                                    </div>
                                    <div>
                                        <span class="text-green-400 font-medium text-lg" data-translate="welcomeEvent.choose_files">Choose Files</span>
                                        <p class="text-sm text-gray-400 mt-1" data-translate="welcomeEvent.or_drag_files">or drag and drop files here</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <p class="text-sm text-gray-400 mt-2" data-translate="welcomeEvent.donation_proofs_desc">📸 Upload images that prove your donation process</p>

                        <!-- Display uploaded files -->
                        @if($donationProofs)
                            <div class="mt-4 grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-3">
                                @foreach($donationProofs as $index => $file)
                                    <div class="relative group bg-gray-800/50 rounded-lg overflow-hidden border border-green-500/30">
                                        <div class="aspect-square bg-gray-700 flex items-center justify-center">
                                            @if($file->isPreviewable())
                                                <img src="{{ $file->temporaryUrl() }}" alt="Preview" class="w-full h-full object-cover">
                                            @else
                                                <div class="text-center p-2">
                                                    <svg class="w-8 h-8 text-green-400 mx-auto mb-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                                                    </svg>
                                                    <p class="text-xs text-green-300">{{ Str::limit($file->getClientOriginalName(), 15) }}</p>
                                                </div>
                                            @endif
                                        </div>
                                        <div class="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-200">
                                            <div class="absolute bottom-2 left-2 right-2">
                                                <p class="text-white text-xs font-medium truncate">{{ $file->getClientOriginalName() }}</p>
                                                <p class="text-green-300 text-xs">{{ round($file->getSize() / 1024, 1) }} KB</p>
                                            </div>
                                            <button type="button"
                                                    wire:click="removeDonationProof({{ $index }})"
                                                    class="absolute top-2 right-2 w-6 h-6 bg-red-500 hover:bg-red-600 rounded-full flex items-center justify-center transition-colors">
                                                <svg class="w-3 h-3 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                                                </svg>
                                            </button>
                                        </div>
                                    </div>
                                @endforeach
                            </div>
                        @endif
                    </div>
                @endif

                <!-- Optional Character Images -->
                <div class="mb-8">
                    <label class="block text-lg font-semibold text-gray-200 mb-3" data-translate="welcomeEvent.character_images_optional">🖼️ Character Images (Optional)</label>
                    <div class="relative"
                         x-data="{ dragover: false }"
                         @dragover.prevent="dragover = true"
                         @dragleave.prevent="dragover = false"
                         @drop.prevent="dragover = false; $refs.oldCharacterImagesInput.files = $event.dataTransfer.files; $refs.oldCharacterImagesInput.dispatchEvent(new Event('change', { bubbles: true }))">
                        <input type="file"
                               wire:model="oldCharacterImages"
                               multiple
                               accept="image/*"
                               id="oldCharacterImages"
                               x-ref="oldCharacterImagesInput"
                               class="hidden">
                        <div class="w-full px-6 py-8 bg-gray-800/50 border-2 border-dashed rounded-xl text-gray-200 transition-all duration-300 text-center cursor-pointer"
                             :class="dragover ? 'border-blue-400 bg-blue-500/10' : 'border-gray-600 hover:border-blue-500'"
                             @click="$refs.oldCharacterImagesInput.click()">
                            <div class="flex flex-col items-center space-y-3">
                                <div class="w-16 h-16 bg-blue-500/20 rounded-2xl flex items-center justify-center">
                                    <svg class="w-8 h-8 text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"/>
                                    </svg>
                                </div>
                                <div>
                                    <span class="text-blue-400 font-medium text-lg" data-translate="welcomeEvent.choose_files">Choose Files</span>
                                    <p class="text-sm text-gray-400 mt-1" data-translate="welcomeEvent.or_drag_files">or drag and drop files here</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    <p class="text-sm text-gray-400 mt-2" data-translate="welcomeEvent.character_images_desc">📸 Upload additional character images if you want</p>

                    <!-- Display uploaded files -->
                    @if($oldCharacterImages)
                        <div class="mt-4 grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-3">
                            @foreach($oldCharacterImages as $index => $file)
                                <div class="relative group bg-gray-800/50 rounded-lg overflow-hidden border border-blue-500/30">
                                    <div class="aspect-square bg-gray-700 flex items-center justify-center">
                                        @if($file->isPreviewable())
                                            <img src="{{ $file->temporaryUrl() }}" alt="Preview" class="w-full h-full object-cover">
                                        @else
                                            <div class="text-center p-2">
                                                <svg class="w-8 h-8 text-blue-400 mx-auto mb-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                                                </svg>
                                                <p class="text-xs text-blue-300">{{ Str::limit($file->getClientOriginalName(), 15) }}</p>
                                            </div>
                                        @endif
                                    </div>
                                    <div class="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-200">
                                        <div class="absolute bottom-2 left-2 right-2">
                                            <p class="text-white text-xs font-medium truncate">{{ $file->getClientOriginalName() }}</p>
                                            <p class="text-blue-300 text-xs">{{ round($file->getSize() / 1024, 1) }} KB</p>
                                        </div>
                                        <button type="button"
                                                wire:click="removeOldCharacterImage({{ $index }})"
                                                class="absolute top-2 right-2 w-6 h-6 bg-red-500 hover:bg-red-600 rounded-full flex items-center justify-center transition-colors">
                                            <svg class="w-3 h-3 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                                            </svg>
                                        </button>
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    @endif
                </div>

                <!-- Submit Button -->
                <div class="text-center">
                    <button wire:click="submitApplication"
                            @if(!$selectedCharacter || $isSubmitting) disabled @endif
                            class="group relative px-12 py-4 bg-gradient-to-r from-blue-600 via-purple-600 to-green-600 text-white font-bold text-lg rounded-2xl hover:from-blue-500 hover:via-purple-500 hover:to-green-500 transition-all duration-500 disabled:opacity-50 disabled:cursor-not-allowed transform hover:scale-105 shadow-2xl hover:shadow-blue-500/25">

                        <!-- Button glow effect -->
                        <div class="absolute inset-0 bg-gradient-to-r from-blue-600 via-purple-600 to-green-600 rounded-2xl blur-xl opacity-30 group-hover:opacity-50 transition-opacity duration-500"></div>

                        <span class="relative">
                            @if($isSubmitting)
                                <span class="flex items-center justify-center">
                                    <svg class="animate-spin -ml-1 mr-3 h-6 w-6 text-white" fill="none" viewBox="0 0 24 24">
                                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                                        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                    </svg>
                                    <span data-translate="welcomeEvent.submitting">Submitting... ⏳</span>
                                </span>
                            @else
                                <span data-translate="welcomeEvent.submit_application">Submit Application 🚀</span>
                            @endif
                        </span>
                    </button>
                </div>
            </div>
        @endif
    @else
        <!-- Application Status -->
        <!-- <div class="bg-white rounded-xl shadow-lg p-6"> -->
        <div class="application-status" x-transition:enter="status-enter">
            @if($userApplication->status === 'pending')
                 <div class="status-card pending"
                     x-data="{
                         show: false,
                         timeRemaining: '',
                         interval: null,
                         updateCountdown() {
                             try {
                                 const deadlineStr = '{{ $userApplication->getReviewDeadline() ? $userApplication->getReviewDeadline()->format('Y-m-d H:i:s') : '' }}';
                                 if (!deadlineStr) {
                                     this.timeRemaining = 'Invalid deadline';
                                     return;
                                 }

                                 const deadline = new Date(deadlineStr.replace(' ', 'T'));
                                 const now = new Date();
                                 const diff = deadline - now;

                                 if (diff > 0) {
                                     const hours = Math.floor(diff / (1000 * 60 * 60));
                                     const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
                                     this.timeRemaining = hours + 'h ' + minutes + 'm';
                                 } else {
                                     this.timeRemaining = 'Review time expired';
                                 }
                             } catch (error) {
                                 console.error('Countdown error:', error);
                                 this.timeRemaining = 'Error calculating time';
                             }
                         },

                         init() {
                             this.show = true;
                             this.updateCountdown();
                             this.interval = setInterval(() => {
                                 this.updateCountdown();
                             }, 60000);
                         },

                         destroy() {
                             if (this.interval) {
                                 clearInterval(this.interval);
                             }
                         }
                     }"
                     x-init="init()"
                     @destroy.window="destroy()"
                     :class="{ 'show': show }"
                     x-cloak>

                    {{-- Floating Background Elements --}}
                    <div class="status-background-elements">
                        <div class="floating-element element-1"></div>
                        <div class="floating-element element-2"></div>
                        <div class="floating-element element-3"></div>
                    </div>

                    {{-- Header Section --}}
                    <div class="status-header-section">
                        <div class="status-icon pending">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12,20A8,8 0 0,0 20,12A8,8 0 0,0 12,4A8,8 0 0,0 4,12A8,8 0 0,0 12,20M12,2A10,10 0 0,1 22,12A10,10 0 0,1 12,22C6.47,22 2,17.5 2,12A10,10 0 0,1 12,2M12.5,7V12.25L17,14.92L16.25,16.15L11,13V7H12.5Z"/>
                            </svg>
                        </div>
                        <div class="status-content">
                            <h2 data-translate="welcomeEvent.pending.title">Application Under Review</h2>
                            <p data-translate="welcomeEvent.pending.subtitle">Your application is being reviewed by our team</p>
                        </div>
                    </div>

                    {{-- Countdown Timer Section --}}
                    <div class="countdown-section">
                        <h3 data-translate="welcomeEvent.pending.reviewDeadline">Review Deadline</h3>
                        <div class="countdown-display" x-text="timeRemaining"></div>
                        <p data-translate="welcomeEvent.pending.reviewNote">Applications are typically reviewed within 24 hours</p>
                    </div>

                    {{-- Status Information Cards --}}
                    <div class="status-info-grid">
                        <div class="info-card">
                            <div class="info-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M14,2H6A2,2 0 0,0 4,4V20A2,2 0 0,0 6,22H18A2,2 0 0,0 20,20V8L14,2M18,20H6V4H13V9H18V20Z"/>
                                </svg>
                            </div>
                            <div class="info-content">
                                <h4 data-translate="welcomeEvent.pending.applicationId">Application ID</h4>
                                <p>#{{ $userApplication->id }}</p>
                            </div>
                        </div>

                        <div class="info-card">
                            <div class="info-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M7,15H9C9,16.08 10.37,17 12,17C13.63,17 15,16.08 15,15C15,13.9 13.96,13.5 11.76,12.97C9.64,12.44 7,11.78 7,9C7,7.21 8.47,5.69 10.5,5.18V3H13.5V5.18C15.53,5.69 17,7.21 17,9H15C15,7.92 13.63,7 12,7C10.37,7 9,7.92 9,9C9,10.1 10.04,10.5 12.24,11.03C14.36,11.56 17,12.22 17,15C17,16.79 15.53,18.31 13.5,18.82V21H10.5V18.82C8.47,18.31 7,16.79 7,15Z"/>
                                </svg>
                            </div>
                            <div class="info-content">
                                <h4 data-translate="welcomeEvent.pending.expectedReward">Expected Reward</h4>
                                <p>{{ $userApplication->reward_amount }} Balance</p>
                            </div>
                        </div>

                        <div class="info-card">
                            <div class="info-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,2A10,10 0 0,1 22,12A10,10 0 0,1 12,22A10,10 0 0,1 2,12A10,10 0 0,1 12,2M12,4A8,8 0 0,0 4,12A8,8 0 0,0 12,20A8,8 0 0,0 20,12A8,8 0 0,0 12,4M12,6A6,6 0 0,1 18,12A6,6 0 0,1 12,18A6,6 0 0,1 6,12A6,6 0 0,1 12,6M12,8A4,4 0 0,0 8,12A4,4 0 0,0 12,16A4,4 0 0,0 16,12A4,4 0 0,0 12,8Z"/>
                                </svg>
                            </div>
                            <div class="info-content">
                                <h4 data-translate="welcomeEvent.pending.reviewInProgress">Review in Progress</h4>
                                <p data-translate="welcomeEvent.pending.pleaseWait">Please wait while our team reviews</p>
                            </div>
                        </div>

                        <div class="info-card">
                            <div class="info-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M16,17V19H2V17S2,13 9,13 16,17 16,17M12.5,7.5A3.5,3.5 0 0,1 9,11A3.5,3.5 0 0,1 5.5,7.5A3.5,3.5 0 0,1 9,4A3.5,3.5 0 0,1 12.5,7.5M15.94,13A5.32,5.32 0 0,1 18,17V19H22V17S22,13.37 15.94,13M15,4A3.39,3.39 0 0,1 16.07,6.5C16.07,8.14 15.07,9.5 14,9.5C13.65,9.5 13.34,9.43 13.05,9.31C13.6,8.75 14,7.9 14,6.9C14,5.9 13.6,5.05 13.05,4.69C13.34,4.57 13.65,4.5 14,4.5L15,4Z"/>
                                </svg>
                            </div>
                            <div class="info-content">
                                <h4 data-translate="welcomeEvent.pending.reviewTeam">Review Team</h4>
                                <p data-translate="welcomeEvent.pending.reviewProcess">Professional review process</p>
                            </div>
                        </div>
                    </div>

                    {{-- What Happens Next --}}
                    <div class="next-steps-section">
                        <h3 data-translate="welcomeEvent.pending.whatNext">What Happens Next?</h3>
                        <ul>
                            <li data-translate="welcomeEvent.pending.step1">Our team will review your application and supporting documents</li>
                            <li data-translate="welcomeEvent.pending.step2">You will be notified of the decision via this page</li>
                            <li data-translate="welcomeEvent.pending.step3">If approved, you can claim your reward immediately</li>
                        </ul>

                        {{-- Additional helpful information --}}
                        <div class="review-info-section">
                            <div class="review-info-card">
                                <div class="review-info-icon">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M11,9H13V7H11M12,20C7.59,20 4,16.41 4,12C4,7.59 7.59,4 12,4C16.41,4 20,7.59 20,12C20,16.41 16.41,20 12,20M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M11,17H13V11H11V17Z"/>
                                    </svg>
                                </div>
                                <div class="review-info-content">
                                    <h4 data-translate="welcomeEvent.pending.submittedSuccessfully">Application Submitted Successfully</h4>
                                    <p data-translate="welcomeEvent.pending.underReview">Your application is now under review by our expert team</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            @elseif($userApplication->status === 'approved')
                <div class="text-center">
                    <div class="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <svg class="w-8 h-8 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                    </div>
                    <h2 class="text-2xl font-bold text-green-700 mb-2" data-translate="welcomeEvent.status.approved.title">Your Request is Approved!</h2>
                    <p class="text-green-600 mb-6" data-translate="welcomeEvent.status.approved.description">You can now collect your reward</p>

                    <div class="bg-green-50 rounded-lg p-4 mb-6">
                        <div class="text-3xl font-bold text-green-600">{{ $userApplication->reward_amount }} Balance</div>
                    </div>

                    <button wire:click="collectRewards"
                            @if($isClaiming) disabled @endif
                            class="bg-green-600 text-white px-8 py-3 rounded-lg font-medium hover:bg-green-700 transition-colors disabled:opacity-50">
                        @if($isClaiming)
                            <span data-translate="welcomeEvent.collecting_rewards">Collecting...</span>
                        @else
                            <span data-translate="welcomeEvent.collect_reward">Collect Reward</span>
                        @endif
                    </button>
                </div>
            @elseif($userApplication->status === 'claimed')
                <div class="text-center">
                    <div class="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <svg class="w-8 h-8 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                        </svg>
                    </div>
                    <h2 class="text-2xl font-bold text-blue-700 mb-2" data-translate="welcomeEvent.status.claimed.title">Reward Collected</h2>
                    <p class="text-blue-600" data-translate="welcomeEvent.status.claimed.description">{{ $userApplication->reward_amount }} Balance has been added to your account</p>
                </div>
            @endif

            <!-- Application Details -->
           {{-- Application Details --}}
            <div class="application-details">
                <h3 data-translate="welcomeEvent.applicationDetails.title">Application Details</h3>
                <div class="details-grid">
                    <div class="detail-item">
                        <span class="detail-label" data-translate="welcomeEvent.applicationDetails.id">Application ID:</span>
                        <span class="detail-value">#{{ $userApplication->id }}</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label" data-translate="welcomeEvent.applicationDetails.type">Player Type:</span>
                        <span class="detail-value">{{ ucfirst($userApplication->player_type) }} Player</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label" data-translate="welcomeEvent.applicationDetails.character">Character:</span>
                        <span class="detail-value">{{ $userApplication->character?->CharName16 ?? 'N/A' }}</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label" data-translate="welcomeEvent.applicationDetails.submitted">Submitted:</span>
                        <span class="detail-value">{{ $userApplication->submitted_at?->format('M d, Y H:i') }}</span>
                    </div>
                    @if($userApplication->reviewed_at)
                        <div class="detail-item">
                            <span class="detail-label" data-translate="welcomeEvent.applicationDetails.reviewed">Reviewed:</span>
                            <span class="detail-value">{{ $userApplication->reviewed_at->format('M d, Y H:i') }}</span>
                        </div>
                    @endif
                    @if($userApplication->donation_amount)
                        <div class="detail-item">
                            <span class="detail-label" data-translate="welcomeEvent.applicationDetails.donation">Donation Amount:</span>
                            <span class="detail-value">{{ $userApplication->donation_amount }} {{ $userApplication->currency }}</span>
                        </div>
                    @endif
                    @if($userApplication->old_character_name)
                        <div class="detail-item">
                            <span class="detail-label" data-translate="welcomeEvent.applicationDetails.oldCharacter">Old Character:</span>
                            <span class="detail-value">{{ $userApplication->old_character_name }}</span>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    @endif
</div>

