<?php

use Livewire\Volt\Component;
use Livewire\WithFileUploads;
use App\Models\FastEventsWelcome;
use App\Models\SRO_VT_SHARD\Char;
use App\Models\SRO_VT_PROXY\_Players;
use App\Models\SRO_VT_SHARD\_User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

new class extends Component {
    use WithFileUploads;

    public $selectedPlayerType = null;
    public $selectedCharacter = null;
    public $oldCharacterName = '';
    public $oldCharacterProofs = [];
    public $donationAmount = 0;
    public $selectedCurrency = 'USD';
    public $donationProofs = [];
    public $oldCharacterImages = [];
    public $calculatedBalance = 0;
    public $userApplication = null;
    public $userCharacters = [];
    public $isSubmitting = false;
    public $isClaiming = false;

    public function mount()
    {
        $this->loadUserData();
        $this->checkForRejectionNotification();
    }

    public function checkForRejectionNotification()
    {
        // Check for any recently rejected application (within last 10 minutes)
        $recentRejection = FastEventsWelcome::where('JID', Auth::user()->JID)
            ->where('status', 'rejected')
            ->where('reviewed_at', '>=', now()->subMinutes(10))
            ->latest()
            ->first();

        if ($recentRejection) {
            // Show rejection notification if recently rejected
            $this->dispatch('show-rejection-notification', [
                'applicationId' => $recentRejection->id,
                'rejectionReason' => $recentRejection->rejection_reason ?? 'No specific reason provided',
                'reviewedAt' => $recentRejection->reviewed_at->format('M d, Y H:i'),
                'playerType' => ucfirst($recentRejection->player_type) . ' Player',
                'rewardAmount' => $recentRejection->reward_amount
            ]);
        }
    }

    public function loadUserData()
    {
        $user = Auth::user();

        // Get the latest application for the user (to show current status)
        // This allows users to have multiple applications if previous ones were rejected
        $this->userApplication = FastEventsWelcome::where('JID', $user->JID)
            ->whereIn('status', ['pending', 'approved', 'claimed']) // Don't include rejected ones
            ->latest('created_at')
            ->first();

        // Load user characters: Get CharIDs from _User table first, then get character details
        $userCharIds = _User::where('UserJID', $user->JID)->pluck('CharID');
        $this->userCharacters = Char::whereIn('CharID', $userCharIds)
            ->select('CharName16', 'CharID', 'RefObjID', 'CurLevel')
            ->orderBy('CharName16')
            ->get()
            ->map(function($char) {
                // Check if character is active in _Players table
                $player = _Players::where('CharID', $char->CharID)->first();
                $isActive = $player && $player->cur_status == true;

                return [
                    'CharID' => $char->CharID,
                    'CharName16' => $char->CharName16,
                    'RefObjID' => $char->RefObjID,
                    'CurLevel' => $char->CurLevel,
                    'avatar' => Char::getCharAvatarGifByCharname($char->CharName16),
                    'isActive' => $isActive
                ];
            });

        // Check for rejection notification after loading data
        $this->checkForRejectionNotification();
    }

    public function getRejectedApplications()
    {
        return FastEventsWelcome::where('JID', Auth::user()->JID)
            ->where('status', 'rejected')
            ->latest('reviewed_at')
            ->limit(3) // Show only last 3 rejected applications
            ->get();
    }

    public function selectPlayerType($type)
    {
        $this->selectedPlayerType = $type;
        $this->calculateReward();
    }

    public function selectCharacter($characterId)
    {
        // Check if this character belongs to the user by checking through _User table
        $userCharIds = _User::where('UserJID', Auth::user()->JID)->pluck('CharID');

        if ($userCharIds->contains($characterId)) {
            // For all player types, check if character has cur_status = true in _Players table
            $player = _Players::where('CharID', $characterId)->first();
            if (!$player || !$player->cur_status) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character must be active (played at least once) for all applications.',
                    'duration' => 2500,
                ]);
                return;
            }

            $this->selectedCharacter = $characterId;
        }
    }

    public function updatedDonationAmount()
    {
        $this->calculateReward();
    }

    public function updatedSelectedCurrency()
    {
        $this->calculateReward();
    }

    public function calculateReward()
    {
        if ($this->selectedPlayerType === 'donate' && $this->donationAmount > 0) {
            $rate = $this->selectedCurrency === 'USD' ? 40 : 1;
            $this->calculatedBalance = floor(($this->donationAmount * $rate) * 0.5);
        } else {
            $this->calculatedBalance = 0;
        }
    }

    public function submitApplication()
    {
        $this->isSubmitting = true;

        try {
            // Validate based on player type
            $rules = [
                'selectedPlayerType' => 'required|in:new,old,donate',
                'selectedCharacter' => 'required|numeric',
            ];

            // Only validate old character specific fields for old players
            if ($this->selectedPlayerType === 'old') {
                $rules['oldCharacterName'] = 'required|string|max:255';
                $rules['oldCharacterProofs'] = 'required|array|min:1';
                $rules['oldCharacterProofs.*'] = 'image|mimes:jpeg,png,jpg,gif|max:5120';
            }

            // Only validate donation specific fields for donates
            if ($this->selectedPlayerType === 'donate') {
                $rules['donationAmount'] = 'required|numeric|min:1';
                $rules['selectedCurrency'] = 'required|in:USD,EGP';
                $rules['donationProofs'] = 'required|array|min:1';
                $rules['donationProofs.*'] = 'image|mimes:jpeg,png,jpg,gif|max:5120';
            }

            $this->validate($rules);

            // Check character ownership through _User table
            $userCharIds = _User::where('UserJID', Auth::user()->JID)->pluck('CharID');

            if (!$userCharIds->contains($this->selectedCharacter)) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character not found or does not belong to you.',
                    'duration' => 2500,
                ]);
                return;
            }

            // Get character details
            $character = Char::where('CharID', $this->selectedCharacter)->first();
            if (!$character) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character not found in database.',
                    'duration' => 2500,
                ]);
                return;
            }

            // For all player types, check if character has been played (cur_status = true)
            $player = _Players::where('CharID', $this->selectedCharacter)->first();
            if (!$player || !$player->cur_status) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character must be active (played at least once) for all applications.',
                    'duration' => 2500,
                ]);
                return;
            }

            // Store uploaded files
            $oldCharacterProofPaths = [];
            $donationProofPaths = [];
            $oldCharacterImagePaths = [];

            if ($this->selectedPlayerType === 'old' && !empty($this->oldCharacterProofs)) {
                foreach ($this->oldCharacterProofs as $file) {
                    $path = $file->store('welcome-event/old-character-proofs', 'public');
                    $oldCharacterProofPaths[] = $path;
                }
            }

            if ($this->selectedPlayerType === 'donate' && !empty($this->donationProofs)) {
                foreach ($this->donationProofs as $file) {
                    $path = $file->store('welcome-event/donation-proofs', 'public');
                    $donationProofPaths[] = $path;
                }
            }

            if (!empty($this->oldCharacterImages)) {
                foreach ($this->oldCharacterImages as $file) {
                    $path = $file->store('welcome-event/old-character-images', 'public');
                    $oldCharacterImagePaths[] = $path;
                }
            }

            // Calculate reward amount
            $rewardAmount = $this->getRewardAmount();

            // Get HWID from _Players table
            $player = _Players::where('CharID', $this->selectedCharacter)->first();
            $hwid = $player ? $player->Serial : '';

            // Create application
            FastEventsWelcome::create([
                'JID' => Auth::user()->JID,
                'HWID' => $hwid,
                'player_type' => $this->selectedPlayerType,
                'CharID' => $this->selectedCharacter,
                'old_character_name' => $this->selectedPlayerType === 'old' ? $this->oldCharacterName : null,
                'donation_amount' => $this->selectedPlayerType === 'donate' ? $this->donationAmount : null,
                'currency' => $this->selectedPlayerType === 'donate' ? $this->selectedCurrency : null,
                'reward_amount' => $rewardAmount,
                'old_character_proofs' => !empty($oldCharacterProofPaths) ? json_encode($oldCharacterProofPaths) : null,
                'donation_proofs' => !empty($donationProofPaths) ? json_encode($donationProofPaths) : null,
                'old_character_images' => !empty($oldCharacterImagePaths) ? json_encode($oldCharacterImagePaths) : null,
                'status' => 'pending',
                'submitted_at' => now(),
            ]);

            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'Application submitted successfully!',
                'duration' => 3000,
            ]);
            $this->loadUserData();

        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Validation failed: ' . implode(', ', $e->validator->errors()->all()),
                'duration' => 5000,
            ]);
        } catch (\Exception $e) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Failed to submit application: ' . $e->getMessage(),
                'duration' => 5000,
            ]);
        } finally {
            $this->isSubmitting = false;
        }
    }

    public function collectRewards()
    {
        $this->isClaiming = true;

        try {
            if (!$this->userApplication || !$this->userApplication->canBeClaimed()) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Unable to collect rewards. Please try again.',
                    'duration' => 2500,
                ]);
                $this->isClaiming = false;
                return;
            }

            $user = Auth::user();

            if ($user && $user->webUser) {
                $oldBalance = $user->webUser->balance;
                $user->webUser->balance += $this->userApplication->reward_amount;
                $user->webUser->save();
            } else {
                throw new \Exception('User account not properly configured');
            }

            $this->userApplication->update([
                'status' => 'claimed',
                'claimed_at' => now(),
            ]);

            $this->userApplication->refresh();

            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'Rewards collected successfully! Your balance has been updated.',
                'duration' => 3000,
            ]);

            $this->dispatch('updateBalances');
            $this->loadUserData();
            $this->dispatch('hide-congratulations');

        } catch (\Exception $e) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Error collecting rewards: ' . $e->getMessage(),
                'duration' => 2500,
            ]);
        } finally {
            $this->isClaiming = false;
        }
    }

    public function claimReward()
    {
        // This function is now used only to show the congratulations modal
        // The actual claiming is done by collectRewards()
        $this->dispatch('show-congratulations', [
            'rewardAmount' => $this->userApplication->reward_amount
        ]);
    }

    private function getRewardAmount()
    {
        switch ($this->selectedPlayerType) {
            case 'new':
                return 10;
            case 'old':
                return 40;
            case 'donate':
                return $this->calculatedBalance;
            default:
                return 0;
        }
    }

    public function resetApplication()
    {
        // Don't delete the existing application, just reset form fields
        $this->selectedPlayerType = null;
        $this->selectedCharacter = null;
        $this->oldCharacterName = '';
        $this->oldCharacterProofs = [];
        $this->donationAmount = 0;
        $this->selectedCurrency = 'USD';
        $this->donationProofs = [];
        $this->oldCharacterImages = [];
        $this->calculatedBalance = 0;

        // If user has a rejected application, allow them to create a new one
        if ($this->userApplication && $this->userApplication->status === 'rejected') {
            // Create a new application entry while keeping the old one for history
            $this->userApplication = null;

            $this->dispatch('show-notification', [
                'type' => 'info',
                'message' => 'Ready to create a new application. Your previous application history has been preserved.',
                'duration' => 3000,
            ]);
        }
    }

    public function with()
    {
        return [
            'hasApplication' => !is_null($this->userApplication),
        ];
    }
}; ?>

<div class="welcome-event-container"
     x-data="{
         // Livewire entangled properties
         selectedPlayer: @entangle('selectedPlayerType'),
         selectedCharacter: @entangle('selectedCharacter'),
         donationAmount: @entangle('donationAmount'),
         currency: @entangle('selectedCurrency'),
         oldCharacterName: @entangle('oldCharacterName'),

         // Rejection notification data
         showRejectionNotification: false,
         rejectionData: {},

         // Congratulations modal data
         showCongratulationsModal: false,
         congratulationsData: {},

         // Helper methods
         selectPlayerType(type) {
             this.selectedPlayer = type;
             $wire.selectPlayerType(type);
         },

         selectCharacterFromList(charId) {
             this.selectedCharacter = charId;
             $wire.selectCharacter(charId);
         },

         // Handle rejection notification
         showRejectionAlert(data) {
             this.rejectionData = data;
             this.showRejectionNotification = true;

             // Auto hide after 10 seconds
             setTimeout(() => {
                 this.hideRejectionNotification();
             }, 10000);
         },

         hideRejectionNotification() {
             this.showRejectionNotification = false;
             setTimeout(() => {
                 this.rejectionData = {};
             }, 500);
         },

         // Handle congratulations modal
         showCongratulations(data) {
             this.congratulationsData = data;
             this.showCongratulationsModal = true;

             // Play celebration sound effect (optional)
             try {
                 const audio = new Audio('/sounds/celebration.mp3');
                 audio.volume = 0.3;
                 audio.play().catch(() => {
                     // Sound failed to play, continue silently
                 });
             } catch (e) {
                 // Audio not available, continue silently
             }
         },

         hideCongratulations() {
             this.showCongratulationsModal = false;
             setTimeout(() => {
                 this.congratulationsData = {};
             }, 500);
         }
     }"
     @show-rejection-notification.window="showRejectionAlert($event.detail)"
     @show-congratulations.window="showCongratulations($event.detail)"
     @hide-congratulations.window="hideCongratulations()">

    {{-- Rejection Notification Overlay --}}
    <div class="rejection-notification-overlay"
         x-show="showRejectionNotification"
         x-transition:enter="notification-enter"
         x-transition:enter-start="notification-enter-start"
         x-transition:enter-end="notification-enter-end"
         x-transition:leave="notification-leave"
         x-transition:leave-start="notification-leave-start"
         x-transition:leave-end="notification-leave-end"
         @click="hideRejectionNotification()"
         x-cloak>

        <div class="rejection-notification-modal" @click.stop>
            {{-- Close Button --}}
            <button class="notification-close-btn" @click="hideRejectionNotification()">
                <svg viewBox="0 0 24 24" fill="currentColor">
                    <path d="M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z"/>
                </svg>
            </button>

            {{-- Notification Header --}}
            <div class="notification-header">
                <div class="notification-icon-container">
                    <div class="notification-icon rejected">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z"/>
                        </svg>
                    </div>
                    <div class="notification-pulse-rings">
                        <div class="pulse-ring ring-1"></div>
                        <div class="pulse-ring ring-2"></div>
                        <div class="pulse-ring ring-3"></div>
                    </div>
                </div>

                <div class="notification-content">
                    <h2 class="notification-title" data-translate="page.4t-app.rejectionNotification.title">Application Rejected</h2>
                    <p class="notification-subtitle" data-translate="page.4t-app.rejectionNotification.subtitle">Your application has been reviewed and rejected</p>
                </div>
            </div>

            {{-- Application Details --}}
            <div class="notification-details">
                <div class="detail-row">
                    <div class="detail-label">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M14,2H6A2,2 0 0,0 4,4V20A2,2 0 0,0 6,22H18A2,2 0 0,0 20,20V8L14,2M18,20H6V4H13V9H18V20Z"/>
                        </svg>
                        <span data-translate="page.4t-app.rejectionNotification.applicationId">Application ID</span>
                    </div>
                    <div class="detail-value" x-text="'#' + rejectionData.applicationId"></div>
                </div>

                <div class="detail-row">
                    <div class="detail-label">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12,4A4,4 0 0,1 16,8A4,4 0 0,1 12,12A4,4 0 0,1 8,8A4,4 0 0,1 12,4M12,14C16.42,14 20,15.79 20,18V20H4V18C4,15.79 7.58,14 12,14Z"/>
                        </svg>
                        <span data-translate="page.4t-app.rejectionNotification.playerType">Player Type</span>
                    </div>
                    <div class="detail-value" x-text="rejectionData.playerType"></div>
                </div>

                <div class="detail-row">
                    <div class="detail-label">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M7,15H9C9,16.08 10.37,17 12,17C13.63,17 15,16.08 15,15C15,13.9 13.96,13.5 11.76,12.97C9.64,12.44 7,11.78 7,9C7,7.21 8.47,5.69 10.5,5.18V3H13.5V5.18C15.53,5.69 17,7.21 17,9H15C15,7.92 13.63,7 12,7C10.37,7 9,7.92 9,9C9,10.1 10.04,10.5 12.24,11.03C14.36,11.56 17,12.22 17,15C17,16.79 15.53,18.31 13.5,18.82V21H10.5V18.82C8.47,18.31 7,16.79 7,15Z"/>
                        </svg>
                        <span data-translate="page.4t-app.rejectionNotification.requestedReward">Requested Reward</span>
                    </div>
                    <div class="detail-value" x-text="rejectionData.rewardAmount + ' Balance'"></div>
                </div>

                <div class="detail-row">
                    <div class="detail-label">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12,20A8,8 0 0,0 20,12A8,8 0 0,0 12,4A8,8 0 0,0 4,12A8,8 0 0,0 12,20M12,2A10,10 0 0,1 22,12A10,10 0 0,1 12,22C6.47,22 2,17.5 2,12A10,10 0 0,1 12,2M12.5,7V12.25L17,14.92L16.25,16.15L11,13V7H12.5Z"/>
                        </svg>
                        <span data-translate="page.4t-app.rejectionNotification.reviewedAt">Reviewed At</span>
                    </div>
                    <div class="detail-value" x-text="rejectionData.reviewedAt"></div>
                </div>
            </div>

            {{-- Rejection Reason --}}
            <div class="rejection-reason-section" x-show="rejectionData.rejectionReason">
                <h3 class="reason-title" data-translate="page.4t-app.rejectionNotification.rejectionReason">Rejection Reason</h3>
                <div class="reason-content">
                    <div class="reason-icon">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M11,15H13V17H11V15M11,7H13V13H11V7M12,2C6.47,2 2,6.5 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M12,20A8,8 0 0,1 4,12A8,8 0 0,1 12,4A8,8 0 0,1 20,12A8,8 0 0,1 12,20Z"/>
                        </svg>
                    </div>
                    <p x-text="rejectionData.rejectionReason"></p>
                </div>
            </div>

            {{-- Action Buttons --}}
            <div class="notification-actions">
                <button class="notification-btn secondary" @click="hideRejectionNotification()" data-translate="page.4t-app.rejectionNotification.dismiss">Dismiss</button>
                <button class="notification-btn primary" @click="hideRejectionNotification(); $wire.resetApplication()" data-translate="page.4t-app.rejectionNotification.tryAgain">Try Again</button>
            </div>
        </div>
    </div>

    {{-- Congratulations Modal --}}
    <div class="congratulations-modal-overlay"
         x-show="showCongratulationsModal"
         x-transition:enter="transition ease-out duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         x-transition:leave="transition ease-in duration-200"
         x-transition:leave-start="opacity-100"
         x-transition:leave-end="opacity-0"
         @click="hideCongratulations()"
         x-cloak>

        <div class="congratulations-modal" @click.stop>
            {{-- Celebration Effects --}}
            <div class="celebration-effects">
                <div class="fireworks">
                    @for($i = 0; $i < 15; $i++)
                        <div class="firework firework-{{ $i + 1 }}"></div>
                    @endfor
                </div>
                <div class="confetti-rain">
                    @for($i = 0; $i < 25; $i++)
                        <div class="confetti-piece confetti-{{ $i + 1 }}"></div>
                    @endfor
                </div>
            </div>

            {{-- Modal Content --}}
            <div class="congratulations-content">
                {{-- Trophy Icon --}}
                <div class="congratulations-icon">
                    <div class="trophy-container">
                        <svg viewBox="0 0 24 24" fill="currentColor" class="trophy-icon">
                            <path d="M12,15.39L8.24,17.66L9.23,13.38L5.91,10.5L10.29,10.13L12,6.09L13.71,10.13L18.09,10.5L14.77,13.38L15.76,17.66M22,9.24L14.81,8.63L12,2L9.19,8.63L2,9.24L7.45,13.97L5.82,21L12,17.27L18.18,21L16.54,13.97L22,9.24Z"/>
                        </svg>
                        <div class="trophy-glow"></div>
                    </div>
                </div>

                {{-- Congratulations Text --}}
                <div class="congratulations-header">
                    <h1 class="congratulations-title">🎉 Congratulations! 🎉</h1>
                    <p class="congratulations-subtitle">Your application has been approved!</p>
                </div>

                {{-- Reward Display --}}
                <div class="reward-display">
                    <div class="reward-box">
                        <div class="reward-amount-large" x-text="congratulationsData.rewardAmount"></div>
                        <div class="reward-currency-large">Balance</div>
                    </div>
                    <p class="reward-description">Your reward is ready to be collected!</p>
                </div>

                {{-- Success Message --}}
                <div class="success-message">
                    <div class="success-icon">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                        </svg>
                    </div>
                    <p>Your application has been reviewed and approved by our team. You can now collect your welcome bonus!</p>
                </div>

                {{-- Action Buttons --}}
                <div class="congratulations-actions">
                    <button class="modal-btn secondary" @click="hideCongratulations()">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z"/>
                        </svg>
                        Close
                    </button>
                    <button class="collect-rewards-btn"
                            wire:click="collectRewards"
                            :disabled="$wire.isClaiming"
                            :class="{ 'collecting': $wire.isClaiming }"
                            @click="if (!$wire.isClaiming) hideCongratulations()">
                        <div class="btn-content">
                            <svg class="btn-icon" viewBox="0 0 24 24" fill="currentColor" x-show="!$wire.isClaiming">
                                <path d="M5,9V21H1V9H5M9,21A2,2 0 0,1 7,19V9C7,8.45 7.22,7.95 7.59,7.59L14.17,1L15.23,2.06C15.5,2.33 15.67,2.7 15.67,3.11L15.64,3.43L14.69,8H21C21.5,8 22,8.4 22,9C22,9.13 22,9.26 21.93,9.4L19.92,17.5C19.73,18.4 18.88,19 18,19H9M9,17H17.5L19,10H12.5L13.8,3.74L9,8.5V17Z"/>
                            </svg>
                            <div class="btn-spinner" x-show="$wire.isClaiming">
                                <div class="spinner"></div>
                            </div>
                            <span x-show="!$wire.isClaiming">Collect Rewards</span>
                            <span x-show="$wire.isClaiming">Collecting...</span>
                        </div>
                        <div class="btn-glow" x-show="!$wire.isClaiming"></div>
                    </button>
                </div>
            </div>
        </div>
    </div>

    {{-- Header Section --}}
    <div class="welcome-event-header">
        <h1 class="welcome-title" data-translate="welcomeEvent.title">Welcome Event</h1>
        <p class="welcome-description" data-translate="welcomeEvent.description">Join our welcome event and get rewards!</p>
    </div>

    {{-- Rejected Applications Section --}}
    @php
        $rejectedApplications = $this->getRejectedApplications();
    @endphp

    @if($rejectedApplications->count() > 0)
        <div class="rejected-applications-section"
             x-data="{
                 showRejected: true,
                 selectedRejection: null,

                 showRejectionDetails(application) {
                     this.selectedRejection = application;
                 },

                 hideRejectionDetails() {
                     this.selectedRejection = null;
                 }
             }">

            <div class="rejected-section-header">
                <div class="rejected-icon">
                    <svg viewBox="0 0 24 24" fill="currentColor">
                        <path d="M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z"/>
                    </svg>
                </div>
                <div class="rejected-header-content">
                    <h2 class="rejected-title" data-translate="page.4t-app.rejectedApplicationsSection.title">Rejected Applications</h2>
                    <p class="rejected-subtitle" data-translate="page.4t-app.rejectedApplicationsSection.subtitle">Your previous applications that were rejected</p>
                </div>
                <button class="toggle-rejected-btn" @click="showRejected = !showRejected">
                    <svg x-show="!showRejected" viewBox="0 0 24 24" fill="currentColor">
                        <path d="M7.41,8.58L12,13.17L16.59,8.58L18,10L12,16L6,10L7.41,8.58Z"/>
                    </svg>
                    <svg x-show="showRejected" viewBox="0 0 24 24" fill="currentColor">
                        <path d="M7.41,15.41L12,10.83L16.59,15.41L18,14L12,8L6,14L7.41,15.41Z"/>
                    </svg>
                </button>
            </div>

            <div class="rejected-applications-list"
                 x-show="showRejected"
                 x-transition:enter="transition ease-out duration-300"
                 x-transition:enter-start="opacity-0 transform -translate-y-4"
                 x-transition:enter-end="opacity-100 transform translate-y-0">

                @foreach($rejectedApplications as $rejection)
                    <div class="rejected-application-card"
                         @click="showRejectionDetails({
                             id: {{ $rejection->id }},
                             playerType: '{{ ucfirst($rejection->player_type) }} Player',
                             characterName: '{{ $rejection->character?->CharName16 ?? 'N/A' }}',
                             rewardAmount: {{ $rejection->reward_amount }},
                             rejectionReason: '{{ addslashes($rejection->rejection_reason) }}',
                             reviewedAt: '{{ $rejection->reviewed_at?->format('M d, Y H:i') }}'
                         })">

                        <div class="rejection-card-header">
                            <div class="rejection-card-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z"/>
                                </svg>
                            </div>
                            <div class="rejection-card-info">
                                <h3>{{ ucfirst($rejection->player_type) }} Player Application</h3>
                                <p>Character: {{ $rejection->character?->CharName16 ?? 'N/A' }}</p>
                                <p>Reviewed: {{ $rejection->reviewed_at?->format('M d, Y H:i') }}</p>
                            </div>
                            <div class="rejection-card-reward">
                                <span class="reward-amount">{{ $rejection->reward_amount }}</span>
                                <span class="reward-currency">Balance</span>
                            </div>
                        </div>

                        <div class="rejection-reason-preview">
                            <p>{{ Str::limit($rejection->rejection_reason, 100) }}</p>
                        </div>

                        <div class="rejection-card-actions">
                            <button class="view-details-btn">View Details</button>
                        </div>
                    </div>
                @endforeach
            </div>

            {{-- Rejection Details Modal --}}
            <div class="rejection-details-overlay"
                 x-show="selectedRejection"
                 x-transition:enter="transition ease-out duration-300"
                 x-transition:enter-start="opacity-0"
                 x-transition:enter-end="opacity-100"
                 @click="hideRejectionDetails()"
                 x-cloak>

                <div class="rejection-details-modal" @click.stop>
                    <button class="modal-close-btn" @click="hideRejectionDetails()">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z"/>
                        </svg>
                    </button>

                    <div class="modal-header">
                        <div class="modal-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z"/>
                            </svg>
                        </div>
                        <div class="modal-title-section">
                            <h2 data-translate="page.4t-app.rejectedApplicationsSection.detailsTitle">Application Details</h2>
                            <p x-text="selectedRejection?.playerType"></p>
                        </div>
                    </div>

                    <div class="modal-details">
                        <div class="detail-item">
                            <span class="detail-label" data-translate="page.4t-app.rejectedApplicationsSection.applicationId">Application ID:</span>
                            <span class="detail-value" x-text="'#' + selectedRejection?.id"></span>
                        </div>
                        <div class="detail-item">
                            <span class="detail-label" data-translate="page.4t-app.rejectedApplicationsSection.character">Character:</span>
                            <span class="detail-value" x-text="selectedRejection?.characterName"></span>
                        </div>
                        <div class="detail-item">
                            <span class="detail-label" data-translate="page.4t-app.rejectedApplicationsSection.requestedReward">Requested Reward:</span>
                            <span class="detail-value" x-text="selectedRejection?.rewardAmount + ' Balance'"></span>
                        </div>
                        <div class="detail-item">
                            <span class="detail-label" data-translate="page.4t-app.rejectedApplicationsSection.reviewedAt">Reviewed At:</span>
                            <span class="detail-value" x-text="selectedRejection?.reviewedAt"></span>
                        </div>
                    </div>

                    <div class="modal-reason">
                        <h4 data-translate="page.4t-app.rejectedApplicationsSection.rejectionReason">Rejection Reason</h4>
                        <p x-text="selectedRejection?.rejectionReason"></p>
                    </div>

                    <div class="modal-actions">
                        <button class="modal-btn secondary" @click="hideRejectionDetails()" data-translate="page.4t-app.rejectedApplicationsSection.close">Close</button>
                        <button class="modal-btn primary" @click="hideRejectionDetails(); $wire.resetApplication()" data-translate="page.4t-app.rejectedApplicationsSection.createNew">Create New Application</button>
                    </div>
                </div>
            </div>
        </div>
    @endif

    {{-- Main Content --}}
    @if (!$hasApplication)
        {{-- Welcome Card Based on Player Type --}}
        @if ($selectedPlayerType)
            <div class="welcome-celebration-card {{ $selectedPlayerType }}-player"
                 x-data="{ show: false }"
                 x-init="setTimeout(() => show = true, 100)"
                 :class="{ 'show': show }">

                {{-- Floating Particles --}}
                <div class="welcome-particles">
                    @for($i = 0; $i < 8; $i++)
                        <div class="particle particle-{{ $i + 1 }}"></div>
                    @endfor
                </div>

                {{-- Card Header --}}
                <div class="welcome-card-header">
                    <div class="welcome-card-icon {{ $selectedPlayerType }}-player">
                        @if($selectedPlayerType === 'new')
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12 2L13.09 8.26L18 9L13.09 9.74L12 16L10.91 9.74L6 9L10.91 8.26L12 2Z"/>
                            </svg>
                        @elseif($selectedPlayerType === 'old')
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M11.5 14L8.5 17L7.91 16.41L10.5 13.83L7.91 11.24L8.5 10.65L11.5 13.65L14.5 10.65L15.09 11.24L12.5 13.83L15.09 16.41L14.5 17L11.5 14ZM12 2A10 10 0 0 1 22 12A10 10 0 0 1 12 22A10 10 0 0 1 2 12A10 10 0 0 1 12 2Z"/>
                            </svg>
                        @elseif($selectedPlayerType === 'donate')
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12 2C13.1 2 14 2.9 14 4C14 5.1 13.1 6 12 6C10.9 6 10 5.1 10 4C10 2.9 10.9 2 12 2ZM13 7H11V22H13V7ZM7 11H9V22H7V11ZM15 15H17V22H15V15Z"/>
                            </svg>
                        @endif
                    </div>

                    <div class="welcome-card-title-section">
                        <h2 class="welcome-card-title" data-translate="welcomeEvent.welcomeCards.{{ $selectedPlayerType }}Player.title">
                            @if($selectedPlayerType === 'new')
                                Welcome New Player!
                            @elseif($selectedPlayerType === 'old')
                                Welcome Back!
                            @elseif($selectedPlayerType === 'donate')
                                Thank You for Donating!
                            @endif
                        </h2>
                        <p class="welcome-card-subtitle" data-translate="welcomeEvent.welcomeCards.{{ $selectedPlayerType }}Player.subtitle">
                            @if($selectedPlayerType === 'new')
                                Get your new player bonus
                            @elseif($selectedPlayerType === 'old')
                                Claim your returning player reward
                            @elseif($selectedPlayerType === 'donate')
                                Receive your donation bonus
                            @endif
                        </p>
                    </div>
                </div>

                {{-- Card Content --}}
                <div class="welcome-card-content">
                    <p class="welcome-card-description" data-translate="welcomeEvent.welcomeCards.{{ $selectedPlayerType }}Player.description">
                        @if($selectedPlayerType === 'new')
                            Welcome to our server! As a new player, you're eligible for special rewards.
                        @elseif($selectedPlayerType === 'old')
                            Welcome back! We're excited to have you return to our server.
                        @elseif($selectedPlayerType === 'donate')
                            Thank you for supporting our server with your donation!
                        @endif
                    </p>

                    {{-- Features Grid --}}
                    <div class="welcome-card-features">
                        @for($i = 0; $i < 4; $i++)
                            <div class="welcome-feature-item">
                                <div class="welcome-feature-icon {{ $selectedPlayerType }}-player">
                                    @if($i === 0)
                                        <svg viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M12 2C13.1 2 14 2.9 14 4C14 5.1 13.1 6 12 6C10.9 6 10 5.1 10 4C10 2.9 10.9 2 12 2ZM21 9V7L15 1H5C3.89 1 3 1.89 3 3V21C3 22.11 3.89 23 5 23H19C20.11 23 21 22.11 21 21V9H21Z"/>
                                        </svg>
                                    @elseif($i === 1)
                                        <svg viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M12 2L3.09 8.26L4 9L11 14L20 9L21.09 8.26L12 2Z"/>
                                        </svg>
                                    @elseif($i === 2)
                                        <svg viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M16 4C18.2 4 20 5.8 20 8C20 10.2 18.2 12 16 12C13.8 12 12 10.2 12 8C12 5.8 13.8 4 16 4ZM18 14.54C19.24 15.19 20 16.45 20 18V22H22V18C22 16.81 20.7 15.6 18.64 15.16L18 14.54Z"/>
                                        </svg>
                                    @else
                                        <svg viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M19 3H5C3.9 3 3 3.9 3 5V19C3 20.1 3.9 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.9 20.1 3 19 3ZM19 19H5V5H19V19ZM17 12H7V10H17V12Z"/>
                                        </svg>
                                    @endif
                                </div>
                                <div class="welcome-feature-text">
                                    <h4 data-translate="welcomeEvent.welcomeCards.{{ $selectedPlayerType }}Player.features.{{ $i + 1 }}.title">Feature {{ $i + 1 }}</h4>
                                    <p data-translate="welcomeEvent.welcomeCards.{{ $selectedPlayerType }}Player.features.{{ $i + 1 }}.description">Feature description</p>
                                </div>
                            </div>
                        @endfor
                    </div>
                </div>
            </div>
        @endif

        {{-- Player Type Selection --}}
        @if (!$selectedPlayerType)
            <div class="player-type-selection">
                <h2 class="section-title" data-translate="welcomeEvent.selectPlayerType">Select Your Player Type</h2>
                <div class="player-types-grid">
                    {{-- New Player Card --}}
                    <div class="player-type-card new-player"
                         @click="selectPlayerType('new')"
                         :class="{ 'selected': selectedPlayer === 'new' }">
                        <div class="card-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12 2L13.09 8.26L18 9L13.09 9.74L12 16L10.91 9.74L6 9L10.91 8.26L12 2Z"/>
                            </svg>
                        </div>
                        <div class="card-content">
                            <h3 data-translate="welcomeEvent.newPlayer.title">New Player</h3>
                            <p data-translate="welcomeEvent.newPlayer.description">Join us for the first time and get 10 Balance!</p>
                            <div class="card-reward">
                                <span class="reward-amount">10</span>
                                <span class="reward-currency">Balance</span>
                            </div>
                        </div>
                    </div>

                    {{-- Old Player Card --}}
                    <div class="player-type-card old-player"
                         @click="selectPlayerType('old')"
                         :class="{ 'selected': selectedPlayer === 'old' }">
                        <div class="card-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12,4A4,4 0 0,1 16,8A4,4 0 0,1 12,12A4,4 0 0,1 8,8A4,4 0 0,1 12,4M12,14C16.42,14 20,15.79 20,18V20H4V18C4,15.79 7.58,14 12,14Z"/>
                            </svg>
                        </div>
                        <div class="card-content">
                            <h3 data-translate="welcomeEvent.oldPlayer.title">Returning Player</h3>
                            <p data-translate="welcomeEvent.oldPlayer.description">Welcome back! Prove your previous character and get 40 Balance!</p>
                            <div class="card-reward">
                                <span class="reward-amount">40</span>
                                <span class="reward-currency">Balance</span>
                            </div>
                        </div>
                    </div>

                    {{-- Donate Player Card --}}
                    <div class="player-type-card donate-player"
                         @click="selectPlayerType('donate')"
                         :class="{ 'selected': selectedPlayer === 'donate' }">
                        <div class="card-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M7,15H9C9,16.08 10.37,17 12,17C13.63,17 15,16.08 15,15C15,13.9 13.96,13.5 11.76,12.97C9.64,12.44 7,11.78 7,9C7,7.21 8.47,5.69 10.5,5.18V3H13.5V5.18C15.53,5.69 17,7.21 17,9H15C15,7.92 13.63,7 12,7C10.37,7 9,7.92 9,9C9,10.1 10.04,10.5 12.24,11.03C14.36,11.56 17,12.22 17,15C17,16.79 15.53,18.31 13.5,18.82V21H10.5V18.82C8.47,18.31 7,16.79 7,15Z"/>
                            </svg>
                        </div>
                        <div class="card-content">
                            <h3 data-translate="welcomeEvent.donatePlayer.title">Donation Bonus</h3>
                            <p data-translate="welcomeEvent.donatePlayer.description">Show your donation receipt and get 50% bonus Balance!</p>
                            <div class="card-reward">
                                <span class="reward-amount">50%</span>
                                <span class="reward-currency">Bonus</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        @else
            {{-- Application Form --}}
            <div class="application-form" x-transition:enter="form-enter">
                <h2 class="section-title">
                    @if($selectedPlayerType === 'new')
                        <span data-translate="welcomeEvent.newPlayer.title">New Player</span>
                    @elseif($selectedPlayerType === 'old')
                        <span data-translate="welcomeEvent.oldPlayer.title">Returning Player</span>
                    @elseif($selectedPlayerType === 'donate')
                        <span data-translate="welcomeEvent.donatePlayer.title">Donation Bonus</span>
                    @endif
                    <span data-translate="welcomeEvent.forms.characterSelection"> - Character Selection</span>
                </h2>

                {{-- Character Selection --}}
                <div class="form-section">
                    <label class="form-label" data-translate="welcomeEvent.forms.selectCharacter">Select Your Character</label>
                    <p class="form-help" data-translate="welcomeEvent.forms.selectCharacterDesc">Choose the character you want to use for this application</p>

                    <div class="character-selection">
                        @foreach($userCharacters as $character)
                            <div class="character-card {{ !$character['isActive'] ? 'inactive' : '' }}"
                                 @click="{{ $character['isActive'] ? 'selectCharacterFromList(' . $character['CharID'] . ')' : '' }}"
                                 :class="{ 'selected': selectedCharacter == {{ $character['CharID'] }} }">

                                <div class="character-avatar">
                                    <img src="{{ $character['avatar'] }}" alt="{{ $character['CharName16'] }}" loading="lazy">
                                    @if(!$character['isActive'])
                                        <div class="inactive-overlay">
                                            <svg viewBox="0 0 24 24" fill="currentColor">
                                                <path d="M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z"/>
                                            </svg>
                                        </div>
                                    @endif
                                </div>

                                <div class="character-info">
                                    <h4 class="character-name">{{ $character['CharName16'] }}</h4>
                                    <p class="character-level">Level {{ $character['CurLevel'] }}</p>
                                    <p class="character-status {{ $character['isActive'] ? 'active' : 'inactive' }}">
                                        {{ $character['isActive'] ? 'Active' : 'Inactive' }}
                                    </p>
                                </div>
                            </div>
                        @endforeach

                        @if(empty($userCharacters))
                            <div class="no-characters">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,4A4,4 0 0,1 16,8A4,4 0 0,1 12,12A4,4 0 0,1 8,8A4,4 0 0,1 12,4M12,14C16.42,14 20,15.79 20,18V20H4V18C4,15.79 7.58,14 12,14Z"/>
                                </svg>
                                <p>No characters found. Please create a character in game first.</p>
                            </div>
                        @endif
                    </div>
                </div>

                {{-- Old Player Specific Fields --}}
                @if($selectedPlayerType === 'old')
                    <div class="form-section">
                        <label class="form-label" data-translate="welcomeEvent.forms.oldCharacterName">Old Character Name *</label>
                        <p class="form-help" data-translate="welcomeEvent.forms.oldCharacterNameDesc">Enter the name of your previous character from another server</p>

                        <input type="text"
                               wire:model="oldCharacterName"
                               class="form-input"
                               placeholder="Enter your old character name"
                               maxlength="255">

                        @error('oldCharacterName')
                            <div class="form-error">{{ $message }}</div>
                        @enderror
                    </div>

                    <div class="form-section">
                        <label class="form-label" data-translate="welcomeEvent.forms.oldCharacterProofs">Character Proof Images *</label>
                        <p class="form-help" data-translate="welcomeEvent.forms.oldCharacterProofsDesc">Upload screenshots showing your old character (inventory, level, etc.)</p>

                        <div class="file-upload-container" x-data="{
                            isDragging: false,
                            handleFiles(files) {
                                for (let file of files) {
                                    if (file.type.startsWith('image/')) {
                                        $wire.upload('oldCharacterProofs', file, (uploaded) => {
                                            // File uploaded successfully
                                        }, (error) => {
                                            console.error('Upload error:', error);
                                        });
                                    }
                                }
                            }
                        }">
                            <div class="file-upload-area"
                                 @dragover.prevent="isDragging = true"
                                 @dragleave.prevent="isDragging = false"
                                 @drop.prevent="isDragging = false; handleFiles($event.dataTransfer.files)"
                                 :class="{ 'dragging': isDragging }">

                                <input type="file"
                                       wire:model="oldCharacterProofs"
                                       multiple
                                       accept="image/*"
                                       class="file-input"
                                       id="oldCharacterProofs"
                                       @change="handleFiles($event.target.files)">

                                <label for="oldCharacterProofs" class="file-upload-label">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M14,2H6A2,2 0 0,0 4,4V20A2,2 0 0,0 6,22H18A2,2 0 0,0 20,20V8L14,2M18,20H6V4H13V9H18V20Z"/>
                                    </svg>
                                    <span>Click to upload or drag and drop</span>
                                    <small>PNG, JPG, GIF up to 5MB each</small>
                                </label>
                            </div>

                            @if($oldCharacterProofs)
                                <div class="uploaded-files">
                                    @foreach($oldCharacterProofs as $index => $file)
                                        <div class="uploaded-file">
                                            <img src="{{ $file->temporaryUrl() }}" alt="Proof {{ $index + 1 }}" class="file-preview">
                                            <button type="button" wire:click="removeUploadedFile('oldCharacterProofs', {{ $index }})" class="remove-file">
                                                <svg viewBox="0 0 24 24" fill="currentColor">
                                                    <path d="M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z"/>
                                                </svg>
                                            </button>
                                        </div>
                                    @endforeach
                                </div>
                            @endif
                        </div>

                        @error('oldCharacterProofs.*')
                            <div class="form-error">{{ $message }}</div>
                        @enderror
                    </div>
                @endif

                {{-- Donate Player Specific Fields --}}
                @if($selectedPlayerType === 'donate')
                    <div class="form-section" x-data="{
                        updateCalculation() {
                            $wire.call('calculateReward');
                        }
                    }">
                        <label class="form-label" data-translate="welcomeEvent.forms.donationAmount">Donation Amount *</label>
                        <p class="form-help" data-translate="welcomeEvent.forms.donationAmountDesc">Enter the amount you donated</p>

                        <div class="donation-input-group">
                            <input type="number"
                                   wire:model.live="donationAmount"
                                   class="form-input donation-amount"
                                   placeholder="0"
                                   min="1"
                                   step="0.01"
                                   @input="updateCalculation()">

                            <select wire:model.live="selectedCurrency"
                                    class="form-select currency-select"
                                    @change="updateCalculation()">
                                <option value="USD">USD</option>
                                <option value="EGP">EGP</option>
                            </select>
                        </div>

                        @if($calculatedBalance > 0)
                            <div class="reward-calculation">
                                <p>You will receive: <strong>{{ $calculatedBalance }} Balance</strong></p>
                                <small>
                                    @if($selectedCurrency === 'USD')
                                        ({{ $donationAmount }} USD × 40 × 50% = {{ $calculatedBalance }} Balance)
                                    @else
                                        ({{ $donationAmount }} EGP × 1 × 50% = {{ $calculatedBalance }} Balance)
                                    @endif
                                </small>
                            </div>
                        @endif

                        @error('donationAmount')
                            <div class="form-error">{{ $message }}</div>
                        @enderror
                    </div>

                    <div class="form-section">
                        <label class="form-label" data-translate="welcomeEvent.forms.donationProofs">Donation Proof Images *</label>
                        <p class="form-help" data-translate="welcomeEvent.forms.donationProofsDesc">Upload screenshots of your donation receipts</p>

                        <div class="file-upload-container" x-data="{
                            isDragging: false,
                            handleFiles(files) {
                                for (let file of files) {
                                    if (file.type.startsWith('image/')) {
                                        $wire.upload('donationProofs', file);
                                    }
                                }
                            }
                        }">
                            <div class="file-upload-area"
                                 @dragover.prevent="isDragging = true"
                                 @dragleave.prevent="isDragging = false"
                                 @drop.prevent="isDragging = false; handleFiles($event.dataTransfer.files)"
                                 :class="{ 'dragging': isDragging }">

                                <input type="file"
                                       wire:model="donationProofs"
                                       multiple
                                       accept="image/*"
                                       class="file-input"
                                       id="donationProofs"
                                       @change="handleFiles($event.target.files)">

                                <label for="donationProofs" class="file-upload-label">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M14,2H6A2,2 0 0,0 4,4V20A2,2 0 0,0 6,22H18A2,2 0 0,0 20,20V8L14,2M18,20H6V4H13V9H18V20Z"/>
                                    </svg>
                                    <span>Click to upload or drag and drop</span>
                                    <small>PNG, JPG, GIF up to 5MB each</small>
                                </label>
                            </div>

                            @if($donationProofs)
                                <div class="uploaded-files">
                                    @foreach($donationProofs as $index => $file)
                                        <div class="uploaded-file">
                                            <img src="{{ $file->temporaryUrl() }}" alt="Donation Proof {{ $index + 1 }}" class="file-preview">
                                            <button type="button" wire:click="removeUploadedFile('donationProofs', {{ $index }})" class="remove-file">
                                                <svg viewBox="0 0 24 24" fill="currentColor">
                                                    <path d="M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z"/>
                                                </svg>
                                            </button>
                                        </div>
                                    @endforeach
                                </div>
                            @endif
                        </div>

                        @error('donationProofs.*')
                            <div class="form-error">{{ $message }}</div>
                        @enderror
                    </div>
                @endif

                {{-- Optional Character Images --}}
                <div class="form-section">
                    <label class="form-label" data-translate="welcomeEvent.forms.characterImages">Character Images (Optional)</label>
                    @if($selectedPlayerType === 'new')
                        <p class="form-help" data-translate="welcomeEvent.forms.characterIntroImagesDesc">Upload screenshots of your new character to introduce yourself</p>
                    @else
                        <p class="form-help" data-translate="welcomeEvent.forms.oldCharacterImagesDesc">Upload additional character screenshots if you want</p>
                    @endif

                    <div class="file-upload-container" x-data="{
                        isDragging: false,
                        handleFiles(files) {
                            for (let file of files) {
                                if (file.type.startsWith('image/')) {
                                    $wire.upload('oldCharacterImages', file);
                                }
                            }
                        }
                    }">
                        <div class="file-upload-area"
                             @dragover.prevent="isDragging = true"
                             @dragleave.prevent="isDragging = false"
                             @drop.prevent="isDragging = false; handleFiles($event.dataTransfer.files)"
                             :class="{ 'dragging': isDragging }">

                            <input type="file"
                                   wire:model="oldCharacterImages"
                                   multiple
                                   accept="image/*"
                                   class="file-input"
                                   id="oldCharacterImages"
                                   @change="handleFiles($event.target.files)">

                            <label for="oldCharacterImages" class="file-upload-label">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M14,2H6A2,2 0 0,0 4,4V20A2,2 0 0,0 6,22H18A2,2 0 0,0 20,20V8L14,2M18,20H6V4H13V9H18V20Z"/>
                                </svg>
                                <span>Click to upload or drag and drop</span>
                                <small>PNG, JPG, GIF up to 5MB each</small>
                            </label>
                        </div>

                        @if($oldCharacterImages)
                            <div class="uploaded-files">
                                @foreach($oldCharacterImages as $index => $file)
                                    <div class="uploaded-file">
                                        <img src="{{ $file->temporaryUrl() }}" alt="Character Image {{ $index + 1 }}" class="file-preview">
                                        <button type="button" wire:click="removeUploadedFile('oldCharacterImages', {{ $index }})" class="remove-file">
                                            <svg viewBox="0 0 24 24" fill="currentColor">
                                                <path d="M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z"/>
                                            </svg>
                                        </button>
                                    </div>
                                @endforeach
                            </div>
                        @endif
                    </div>
                </div>

                {{-- Submit Button --}}
                <div class="form-actions">
                    <button wire:click="resetApplication" class="btn btn-secondary" data-translate="welcomeEvent.buttons.back">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M20,11V13H8L13.5,18.5L12.08,19.92L4.16,12L12.08,4.08L13.5,5.5L8,11H20Z"/>
                        </svg>
                        Back
                    </button>
                    <button wire:click="submitApplication"
                            class="btn btn-primary"
                            :disabled="$wire.isSubmitting || !selectedCharacter"
                            :class="{ 'loading': $wire.isSubmitting }"
                            data-translate="welcomeEvent.buttons.submit">
                        <svg x-show="!$wire.isSubmitting" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                        </svg>
                        <div x-show="$wire.isSubmitting" class="btn-spinner">
                            <div class="spinner"></div>
                        </div>
                        <span x-show="!$wire.isSubmitting">Submit Application</span>
                        <span x-show="$wire.isSubmitting">Submitting...</span>
                    </button>
                </div>
            </div>
        @endif
    @else
        {{-- Application Status Display --}}
        <div class="application-status" x-transition:enter="status-enter">
            @if($userApplication->status === 'pending')
                <div class="status-card pending"
                     x-data="{
                         show: false,
                         timeRemaining: '',
                         updateCountdown() {
                             const deadline = new Date('{{ $userApplication->getReviewDeadline() }}');
                             const now = new Date();
                             const diff = deadline - now;

                             if (diff > 0) {
                                 const hours = Math.floor(diff / (1000 * 60 * 60));
                                 const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
                                 this.timeRemaining = hours + 'h ' + minutes + 'm';
                             } else {
                                 this.timeRemaining = 'Review time expired';
                             }
                         }
                     }"
                     x-init="show = true; updateCountdown(); setInterval(updateCountdown, 60000)"
                     :class="{ 'show': show }"
                     x-cloak>

                    {{-- Floating Background Elements --}}
                    <div class="status-background-elements">
                        <div class="floating-element element-1"></div>
                        <div class="floating-element element-2"></div>
                        <div class="floating-element element-3"></div>
                    </div>

                    {{-- Header Section --}}
                    <div class="status-header-section">
                        <div class="status-icon pending">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12,20A8,8 0 0,0 20,12A8,8 0 0,0 12,4A8,8 0 0,0 4,12A8,8 0 0,0 12,20M12,2A10,10 0 0,1 22,12A10,10 0 0,1 12,22C6.47,22 2,17.5 2,12A10,10 0 0,1 12,2M12.5,7V12.25L17,14.92L16.25,16.15L11,13V7H12.5Z"/>
                            </svg>
                        </div>
                        <div class="status-content">
                            <h2 data-translate="page.4t-app.pending.title">Application Under Review</h2>
                            <p data-translate="page.4t-app.pending.subtitle">Your application is being reviewed by our team</p>
                        </div>
                    </div>

                    {{-- Countdown Timer Section --}}
                    <div class="countdown-section">
                        <h3 data-translate="page.4t-app.pending.reviewDeadline">Review Deadline</h3>
                        <div class="countdown-display" x-text="timeRemaining"></div>
                        <p data-translate="page.4t-app.pending.reviewNote">Applications are typically reviewed within 24 hours</p>
                    </div>

                    {{-- Status Information Cards --}}
                    <div class="status-info-grid">
                        <div class="info-card">
                            <div class="info-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M14,2H6A2,2 0 0,0 4,4V20A2,2 0 0,0 6,22H18A2,2 0 0,0 20,20V8L14,2M18,20H6V4H13V9H18V20Z"/>
                                </svg>
                            </div>
                            <div class="info-content">
                                <h4 data-translate="page.4t-app.pending.applicationId">Application ID</h4>
                                <p>#{{ $userApplication->id }}</p>
                            </div>
                        </div>

                        <div class="info-card">
                            <div class="info-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M7,15H9C9,16.08 10.37,17 12,17C13.63,17 15,16.08 15,15C15,13.9 13.96,13.5 11.76,12.97C9.64,12.44 7,11.78 7,9C7,7.21 8.47,5.69 10.5,5.18V3H13.5V5.18C15.53,5.69 17,7.21 17,9H15C15,7.92 13.63,7 12,7C10.37,7 9,7.92 9,9C9,10.1 10.04,10.5 12.24,11.03C14.36,11.56 17,12.22 17,15C17,16.79 15.53,18.31 13.5,18.82V21H10.5V18.82C8.47,18.31 7,16.79 7,15Z"/>
                                </svg>
                            </div>
                            <div class="info-content">
                                <h4 data-translate="page.4t-app.pending.expectedReward">Expected Reward</h4>
                                <p>{{ $userApplication->reward_amount }} Balance</p>
                            </div>
                        </div>
                    </div>

                    {{-- What Happens Next --}}
                    <div class="next-steps-section">
                        <h3 data-translate="page.4t-app.pending.whatNext">What Happens Next?</h3>
                        <ul>
                            <li data-translate="page.4t-app.pending.step1">Our team will review your application and supporting documents</li>
                            <li data-translate="page.4t-app.pending.step2">You will be notified of the decision via this page</li>
                            <li data-translate="page.4t-app.pending.step3">If approved, you can claim your reward immediately</li>
                        </ul>
                    </div>
                </div>

            @elseif($userApplication->status === 'approved')
                <div class="status-card approved"
                     x-data="{
                         show: false,
                         celebrating: false,
                         startCelebration() {
                             this.celebrating = true;
                             setTimeout(() => this.celebrating = false, 3000);
                         }
                     }"
                     x-init="show = true; setTimeout(() => startCelebration(), 500)"
                     :class="{ 'show': show, 'celebrating': celebrating }"
                     x-cloak>

                    {{-- Celebration Particles --}}
                    <div class="celebration-particles">
                        @for($i = 0; $i < 12; $i++)
                            <div class="particle particle-{{ $i + 1 }}"></div>
                        @endfor
                    </div>

                    {{-- Header Section --}}
                    <div class="status-header-section">
                        <div class="status-icon approved">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                            </svg>
                        </div>
                        <div class="status-content">
                            <h2 data-translate="page.4t-app.approved.title">Congratulations!</h2>
                            <p data-translate="page.4t-app.approved.subtitle">Your application has been approved</p>
                        </div>
                    </div>

                    {{-- Reward Display --}}
                    <div class="reward-showcase">
                        <div class="reward-amount-display">
                            <span class="reward-amount">{{ $userApplication->reward_amount }}</span>
                            <span class="reward-currency">Balance</span>
                        </div>
                        <p data-translate="page.4t-app.approved.rewardReady">Your reward is ready to claim!</p>
                    </div>

                    {{-- Claim Button --}}
                    @if($userApplication->canBeClaimed())
                        <div class="claim-section" x-data="{ isHovering: false }">
                            <button @click="showCongratulations({ rewardAmount: {{ $userApplication->reward_amount }} })"
                                    class="claim-reward-btn"
                                    :class="{ 'hover': isHovering }"
                                    @mouseenter="isHovering = true"
                                    @mouseleave="isHovering = false">
                                <div class="btn-content">
                                    <svg class="btn-icon" viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M5,9V21H1V9H5M9,21A2,2 0 0,1 7,19V9C7,8.45 7.22,7.95 7.59,7.59L14.17,1L15.23,2.06C15.5,2.33 15.67,2.7 15.67,3.11L15.64,3.43L14.69,8H21C21.5,8 22,8.4 22,9C22,9.13 22,9.26 21.93,9.4L19.92,17.5C19.73,18.4 18.88,19 18,19H9M9,17H17.5L19,10H12.5L13.8,3.74L9,8.5V17Z"/>
                                    </svg>
                                    <span data-translate="page.4t-app.claimReward">Claim Reward</span>
                                </div>
                                <div class="btn-glow"></div>
                            </button>
                        </div>
                    @else
                        <div class="already-claimed-section">
                            <div class="claimed-message">
                                <svg class="claimed-icon" viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                                </svg>
                                <h3 data-translate="page.4t-app.alreadyClaimed">Reward Already Claimed</h3>
                                <p data-translate="page.4t-app.alreadyClaimedDesc">You have already claimed this reward</p>
                            </div>
                        </div>
                    @endif
                </div>

            @elseif($userApplication->status === 'rejected')
                <div class="status-card rejected"
                     x-data="{
                         show: false
                     }"
                     x-init="show = true"
                     :class="{ 'show': show }"
                     x-cloak>

                    {{-- Header Section --}}
                    <div class="status-header-section">
                        <div class="status-icon rejected">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z"/>
                            </svg>
                        </div>
                        <div class="status-content">
                            <h2 data-translate="page.4t-app.rejected.title">Application Rejected</h2>
                            <p data-translate="page.4t-app.rejected.subtitle">Unfortunately, your application was not approved</p>
                        </div>
                    </div>

                    {{-- Rejection Reason --}}
                    @if($userApplication->rejection_reason)
                        <div class="rejection-reason">
                            <h3 data-translate="page.4t-app.rejected.reason">Rejection Reason</h3>
                            <p>{{ $userApplication->rejection_reason }}</p>
                        </div>
                    @endif

                    {{-- New Application Option --}}
                    <div class="new-application-section">
                        <p data-translate="page.4t-app.rejected.canReapply">You can submit a new application with corrected information</p>
                        <button wire:click="resetApplication" class="btn btn-primary" data-translate="page.4t-app.rejected.newApplication">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M17,3H5C3.89,3 3,3.9 3,5V19C3,20.1 3.89,21 5,21H19C20.1,21 21,20.1 21,19V7L17,3M19,19H5V5H16V8H19V19Z"/>
                            </svg>
                            Create New Application
                        </button>
                    </div>
                </div>

            @elseif($userApplication->status === 'claimed')
                <div class="status-card claimed"
                     x-data="{
                         show: false,
                         confetti: false,
                         startConfetti() {
                             this.confetti = true;
                             setTimeout(() => this.confetti = false, 5000);
                         }
                     }"
                     x-init="show = true; setTimeout(() => startConfetti(), 1000)"
                     :class="{ 'show': show, 'confetti': confetti }"
                     x-cloak>

                    {{-- Confetti Animation --}}
                    <div class="confetti-container" x-show="confetti">
                        @for($i = 0; $i < 20; $i++)
                            <div class="confetti confetti-{{ $i + 1 }}"></div>
                        @endfor
                    </div>

                    {{-- Header Section --}}
                    <div class="status-header-section">
                        <div class="status-icon claimed">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                            </svg>
                        </div>
                        <div class="status-content">
                            <h2 data-translate="page.4t-app.claimed.title">Reward Claimed!</h2>
                            <p data-translate="page.4t-app.claimed.subtitle">You have successfully claimed your reward</p>
                        </div>
                    </div>

                    {{-- Claimed Amount Display --}}
                    <div class="claimed-amount-section">
                        <div class="claimed-amount-card">
                            <h3 data-translate="page.4t-app.claimed.amountReceived">Amount Received</h3>
                            <div class="claimed-amount">{{ $userApplication->reward_amount }} Balance</div>
                            <p data-translate="page.4t-app.claimed.addedToAccount">This amount has been added to your account</p>
                        </div>
                    </div>
                </div>
            @endif

            {{-- Application Details --}}
            <div class="application-details">
                <h3 data-translate="page.4t-app.applicationDetails.title">Application Details</h3>
                <div class="details-grid">
                    <div class="detail-item">
                        <span class="detail-label" data-translate="page.4t-app.applicationDetails.id">Application ID:</span>
                        <span class="detail-value">#{{ $userApplication->id }}</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label" data-translate="page.4t-app.applicationDetails.type">Player Type:</span>
                        <span class="detail-value">{{ ucfirst($userApplication->player_type) }} Player</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label" data-translate="page.4t-app.applicationDetails.character">Character:</span>
                        <span class="detail-value">{{ $userApplication->character?->CharName16 ?? 'N/A' }}</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label" data-translate="page.4t-app.applicationDetails.submitted">Submitted:</span>
                        <span class="detail-value">{{ $userApplication->submitted_at?->format('M d, Y H:i') }}</span>
                    </div>
                    @if($userApplication->reviewed_at)
                        <div class="detail-item">
                            <span class="detail-label" data-translate="page.4t-app.applicationDetails.reviewed">Reviewed:</span>
                            <span class="detail-value">{{ $userApplication->reviewed_at->format('M d, Y H:i') }}</span>
                        </div>
                    @endif
                    @if($userApplication->donation_amount)
                        <div class="detail-item">
                            <span class="detail-label" data-translate="page.4t-app.applicationDetails.donation">Donation Amount:</span>
                            <span class="detail-value">{{ $userApplication->donation_amount }} {{ $userApplication->currency }}</span>
                        </div>
                    @endif
                    @if($userApplication->old_character_name)
                        <div class="detail-item">
                            <span class="detail-label" data-translate="page.4t-app.applicationDetails.oldCharacter">Old Character:</span>
                            <span class="detail-value">{{ $userApplication->old_character_name }}</span>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    @endif
</div>
