<?php
use Livewire\Volt\Component;
use App\Models\FastEvent;
use Carbon\Carbon;

new class extends Component {
    public $activeEvents = [];
    public $currentPage = 'events'; // للتنقل بين الصفحات
    public $selectedEvent = null; // الحدث المختار

    /**
     * تهيئة البيانات عند تحميل الكومبوننت
     */
    public function mount()
    {
        $this->loadActiveEvents();
    }

    /**
     * تحميل الأحداث النشطة فقط
     */
    public function loadActiveEvents()
    {
        try {
            // جلب الأحداث النشطة فقط والتي لم تنته صلاحيتها
            $events = FastEvent::where('is_active', true)
                ->where(function($query) {
                    $query->whereNull('expired_at')
                          ->orWhere('expired_at', '>', Carbon::now());
                })
                ->orderBy('created_at', 'desc')
                ->get();

            $this->activeEvents = $events->toArray();
        } catch (\Exception $e) {
            $this->activeEvents = [];
            session()->flash('error', 'Error loading events: ' . $e->getMessage());
        }
    }

    /**
     * تحديث البيانات
     */
    public function refreshEvents()
    {
        $this->loadActiveEvents();
        session()->flash('success', 'Events refreshed successfully!');
    }

    /**
     * عرض FastEvent عند اختيار حدث معين
     */
    public function showEventDetails($eventId)
    {
        $event = collect($this->activeEvents)->firstWhere('id', $eventId);

        if ($event) {
            $this->selectedEvent = $event;
            $this->currentPage = 'event-details';
        }
    }

    /**
     * العودة لصفحة الأحداث الرئيسية
     */
    public function backToEvents()
    {
        $this->currentPage = 'events';
        $this->selectedEvent = null;
    }
};
?>
<div>
    <div class="fast-events-container">
        {{-- Main Header --}}
        <header class="fast-events-header">
            <div class="header-content">
                <div class="header-icon">
                    <i class="fas fa-calendar-star"></i>
                </div>
                <div class="header-text">
                    <h1 class="header-title" data-translate="fastEvents.title">Active Events</h1>
                    <p class="header-subtitle" data-translate="fastEvents.subtitle">Live Gaming Events & Rewards</p>
                </div>
                <button
                    wire:click="refreshEvents"
                    class="refresh-btn"
                    wire:loading.class="loading"
                    wire:loading.attr="disabled"
                >
                    <i class="fas fa-sync-alt" wire:loading.class="fa-spinner fa-spin"></i>
                    <span wire:loading.remove data-translate="fastEvents.refresh">Refresh Events</span>
                    <span wire:loading data-translate="fastEvents.refreshing">Refreshing...</span>
                </button>
            </div>
        </header>

        {{-- Success Messages --}}
        @if (session()->has('success'))
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <span>{{ session('success') }}</span>
            </div>
        @endif

        {{-- Error Alert --}}
        @if (session()->has('error'))
            <div class="alert alert-error">
                <i class="fas fa-exclamation-triangle"></i>
                <span>{{ session('error') }}</span>
            </div>
        @endif

        {{-- Events Grid --}}
        <div x-show="$wire.currentPage === 'events'" class="events-grid">
            @if($activeEvents && count($activeEvents) > 0)
                @foreach($activeEvents as $index => $event)
                    <div class="event-card" @click="$wire.showEventDetails({{ $event['id'] }})" style="cursor: pointer;">
                        <div class="card-glow"></div>

                        {{-- Card Header --}}
                        <div class="event-card-header">
                            <div class="event-status">
                                <div class="status-dot active"></div>
                                <span data-translate="fastEvents.eventCard.active">Active</span>
                            </div>
                            @if($event['expired_at'])
                                <div class="event-timer">
                                    <i class="fas fa-clock"></i>
                                    <span>{{ \Carbon\Carbon::parse($event['expired_at'])->diffForHumans() }}</span>
                                </div>
                            @endif
                        </div>

                        {{-- Card Body --}}
                        <div class="event-card-body">
                            <div class="event-icon">
                                <i class="fas fa-star"></i>
                            </div>
                            <h3 class="event-title">{{ $event['event_name'] }}</h3>
                            @if($event['blade_name'])
                                <div class="event-type">
                                    <i class="fas fa-bolt"></i>
                                    <span>{{ ucfirst(str_replace('-', ' ', $event['blade_name'])) }}</span>
                                </div>
                            @else
                                <div class="event-type">
                                    <i class="fas fa-gift"></i>
                                    <span>Fast Event</span>
                                </div>
                            @endif
                        </div>
                    </div>
                @endforeach
            @endif
        </div>

        {{-- Event Details Page --}}
        <div x-show="$wire.currentPage === 'event-details'" x-transition>
            {{-- Back Button --}}
            <div class="event-details-header">
                <button @click="$wire.backToEvents()" class="back-btn-elegant">
                    <div class="back-btn-content">
                        <svg class="back-icon" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M19 12H5M12 19L5 12L12 5" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                        </svg>
                        <span class="back-text" data-translate="fastEvents.back">Back to Events</span>
                    </div>
                    <div class="back-btn-glow"></div>
                </button>
                @if($selectedEvent)
                    <div class="event-details-title-section">
                        <h2 class="event-details-title">{{ $selectedEvent['event_name'] ?? 'Event Details' }}</h2>
                        <div class="event-details-subtitle">
                            @if($selectedEvent['blade_name'])
                                <span class="event-type-badge">{{ ucfirst(str_replace('-', ' ', $selectedEvent['blade_name'])) }}</span>
                            @endif
                            @if($selectedEvent['expired_at'])
                                <span class="event-expiry">
                                    <svg class="clock-icon" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="2"/>
                                        <polyline points="12,6 12,12 16,14" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                    </svg>
                                    Expires {{ \Carbon\Carbon::parse($selectedEvent['expired_at'])->diffForHumans() }}
                                </span>
                            @endif
                        </div>
                    </div>
                @endif
            </div>

            {{-- FastEvent Content --}}
            <div class="event-content-container">
                @if($selectedEvent && $selectedEvent['blade_name'])
                    @if($selectedEvent['blade_name'] === 'treasure-event')
                        <livewire:fast-events.treasure-event />
                    @elseif($selectedEvent['blade_name'] === 'welcome')
                        <livewire:fast-events.welcome2 />
                    @else
                        <div class="no-content-message">
                            <i class="fas fa-info-circle"></i>
                            <p>Event content is not available at the moment.</p>
                        </div>
                    @endif
                @else
                    <div class="no-content-message">
                        <i class="fas fa-exclamation-triangle"></i>
                        <p>No event selected or event data is missing.</p>
                    </div>
                @endif
            </div>
        </div>

        {{-- No Events State --}}
        @if(!$activeEvents || count($activeEvents) == 0)
            <div class="no-events-state">
                <div class="no-events-icon">
                    <i class="fas fa-calendar-times"></i>
                </div>
                <h2 class="no-events-title" data-translate="fastEvents.noEvents">No Active Events</h2>
                <p class="no-events-description" data-translate="fastEvents.noEventsMessage">
                    There are currently no active events. Check back later for new adventures!
                </p>
            </div>
        @endif
    </div>
</div>
