<?php
use Livewire\Volt\Component;
use App\Models\Z_Website\Market\MarketPackage;
use App\Models\Z_Website\Market\MarketItem;
use App\Models\Z_Website\Market\MarketTransaction;
use App\Models\SRO_VT_SHARD\RefObjCommon;
use App\Models\SRO_VT_SHARD\RefObjItem;
use App\Models\SRO_VT_SHARD\Chest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\WithPagination;

new class extends Component {
    use WithPagination;

    public $searchTerm = '';
    public $filterType = 'all'; // all, available, expiring, limited, unlimited
    public $priceRange = 'all'; // all, low, medium, high
    public $sortType = 'all'; // all, + dynamic sort types
    public $balanceType = 'all'; // all, balance, gift_balance, silk, donate
    public $sortBy = 'sort_order'; // sort_order, price_asc, price_desc, name_asc, name_desc
    public $perPage = 12; // عدد العناصر في كل صفحة

    public function mount()
    {
        // لا نحتاج لتحميل البيانات في mount مع pagination
    }

    //load data with pagination - استخدام render بدلاً من خاصية
    public function render() : mixed
    {
        $query = MarketPackage::where('price', '>', 0)
            ->where('stock', '!=', 0) // Only exclude out of stock packages
            ->where(function ($query) {
                $query->whereNull('expiry_date')->orWhere('expiry_date', '>', now()); // Not expired
            });

        // تطبيق المرشحات على الاستعلام
        $query = $this->applyFiltersToQuery($query);

        // الحصول على العناصر مع pagination
        $filteredPackages = $query->orderBy($this->getSortColumn(), $this->getSortDirection())
            ->paginate($this->perPage)
            ->withQueryString(); // للحفاظ على معاملات الاستعلام في URL

        return view('livewire.backend.pages.packages-shop', [
            'filteredpackages' => $filteredPackages,
            'totalPackages' => $query->count()
        ]);
    }

    private function applyFiltersToQuery($query)
    {
        // تطبيق مرشح البحث
        if (!empty($this->searchTerm)) {
            $query->where(function ($q) {
                $q->where('package_name', 'like', '%' . $this->searchTerm . '%')
                  ->orWhere('description', 'like', '%' . $this->searchTerm . '%');
            });
        }

        // تطبيق مرشح النوع (حالة المخزون)
        if ($this->filterType !== 'all') {
            switch ($this->filterType) {
                case 'available':
                    $query->where(function ($q) {
                        $q->where('stock', '>', 0)->orWhere('stock', -1);
                    })->where(function ($q) {
                        $q->whereNull('expiry_date')->orWhere('expiry_date', '>', now());
                    });
                    break;
                case 'expiring':
                    $query->whereNotNull('expiry_date')
                          ->where('expiry_date', '>', now())
                          ->where('expiry_date', '<', now()->addDays(7));
                    break;
                case 'limited':
                    $query->whereBetween('stock', [1, 10]);
                    break;
                case 'unlimited':
                    $query->where('stock', -1);
                    break;
            }
        }

        // تطبيق مرشح نوع الرصيد
        if ($this->balanceType !== 'all') {
            $query->where('balance_type', $this->balanceType);
        }

        // تطبيق مرشح نطاق السعر
        if ($this->priceRange !== 'all') {
            switch ($this->priceRange) {
                case 'low':
                    $query->where('price', '<=', 1000);
                    break;
                case 'medium':
                    $query->whereBetween('price', [1001, 10000]);
                    break;
                case 'high':
                    $query->where('price', '>', 10000);
                    break;
            }
        }

        // تطبيق مرشح نوع الفرز (الفئة)
        if ($this->sortType !== 'all' && !empty($this->sortType)) {
            $query->where('sort_type', $this->sortType);
        }

        return $query;
    }

    private function getSortColumn()
    {
        switch ($this->sortBy) {
            case 'price_asc':
            case 'price_desc':
                return 'price';
            case 'name_asc':
            case 'name_desc':
                return 'package_name';
            case 'sort_order':
            default:
                return 'sort_order';
        }
    }

    private function getSortDirection()
    {
        switch ($this->sortBy) {
            case 'price_desc':
            case 'name_desc':
                return 'desc';
            case 'price_asc':
            case 'name_asc':
            case 'sort_order':
            default:
                return 'asc';
        }
    }

    // دالة مبسطة للفلاتر التي لا تحتاج pagination كامل
    public function applyFilters()
    {
        $this->resetPage(); // إعادة تعيين الصفحة عند تطبيق المرشحات
    }

    private function isExpired($package)
    {
        if (!$package->expiry_date) {
            return false;
        }

        $expiryDate = is_string($package->expiry_date) ? \Carbon\Carbon::parse($package->expiry_date) : $package->expiry_date;

        return $expiryDate->isPast();
    }

    //Get Items from package
    public function getPackageItems($packageId)
    {
        $items = MarketPackage::getItems($packageId);
        return $items
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'name' => $item->name,
                    'description' => $item->description,
                    'image' => $this->getImageUrl($item->image),
                    'Count' => $item->Count ?? 1,
                    'price' => $item->price ?? 0,
                    'RefItemID' => $item->RefItemID ?? null,
                    'CodeName128' => $item->CodeName128 ?? null,
                ];
            })
            ->toArray();
    }

    //Get Items from package for internal use
    private function getPackageItemsCollection($packageId)
    {
        return MarketPackage::getItems($packageId);
    }

    public function updatedSearchTerm()
    {
        $this->applyFilters();
    }

    public function updatedPriceRange()
    {
        $this->applyFilters();
    }

    public function updatedSortType()
    {
        $this->applyFilters();
    }

    public function updatedBalanceType()
    {
        $this->applyFilters();
    }

    public function updatedSortBy()
    {
        $this->applyFilters();
    }

    public function setFilter($type)
    {
        $this->filterType = $type;
        $this->applyFilters();
    }

    public function setPriceRange($range)
    {
        $this->priceRange = $range;
        $this->applyFilters();
    }

    public function setSortType($type)
    {
        $this->sortType = $type;
        $this->applyFilters();
    }

    public function setBalanceType($type)
    {
        $this->balanceType = $type;
        $this->applyFilters();
    }

    public function setSortBy($sortBy)
    {
        $this->sortBy = $sortBy;
        $this->applyFilters();
    }

    public function resetFilters()
    {
        $this->searchTerm = '';
        $this->filterType = 'all';
        $this->priceRange = 'all';
        $this->sortType = 'all';
        $this->balanceType = 'all';
        $this->sortBy = 'sort_order';
        $this->applyFilters();
    }

    // إضافة دالة لتغيير عدد العناصر في الصفحة
    public function changePerPage($perPage)
    {
        $this->perPage = $perPage;
        $this->resetPage();
    }

    private function getImageUrl($imagePath)
    {
        if (empty($imagePath)) {
            return asset('assets/images/default-item.svg');
        }

        // إذا كان الرابط يحتوي على http أو https فهو رابط خارجي
        if (str_starts_with($imagePath, 'http')) {
            return $imagePath;
        }

        // إذا كان يبدأ بـ storage فهو ملف محلي من storage
        if (str_starts_with($imagePath, 'storage/')) {
            return asset($imagePath);
        }

        // إذا كان ملف من storage بدون البادئة
        if (str_contains($imagePath, '.')) {
            return asset('storage/' . $imagePath);
        }

        // افتراضي
        return asset('assets/images/default-item.svg');
    }

    public function getAvailableSortTypesProperty()
    {
        return MarketPackage::where('price', '>', 0)
            ->where('stock', '!=', 0)
            ->where(function ($query) {
                $query->whereNull('expiry_date')->orWhere('expiry_date', '>', now());
            })
            ->pluck('sort_type')
            ->filter()
            ->unique()
            ->sort()
            ->values()
            ->toArray();
    }

    public function getAvailableBalanceTypesProperty()
    {
        return MarketPackage::where('price', '>', 0)
            ->where('stock', '!=', 0)
            ->where(function ($query) {
                $query->whereNull('expiry_date')->orWhere('expiry_date', '>', now());
            })
            ->pluck('balance_type')
            ->filter()
            ->unique()
            ->values()
            ->toArray();
    }

    //buy Package
    public function buyPackage($PackgeID, $quantity = 1)
    {
        // التحقق من صحة الكمية
        if ($quantity <= 0) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:packagesShop.notifications.invalidQuantity',
                'duration' => 5000,
            ]);
            return;
        }

        //Get Package Information by PackageID
        $package = MarketPackage::find($PackgeID);
        if (!$package) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:packagesShop.notifications.packageNotFound',
                'duration' => 5000,
            ]);
            return;
        }

        //check user balance
        $user = Auth::user();
        if (!$user || !$user->webUser) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:packagesShop.notifications.loginRequired',
                'duration' => 5000,
            ]);
            return;
        }

        // التحقق من وجود JID
        if (!$user->JID) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:packagesShop.notifications.loginRequired',
                'duration' => 5000,
            ]);
            return;
        }

        // إضافة debugging للـ user data
        \Log::info('User data for package purchase:', [
            'auth_id' => Auth::id(),
            'user_jid' => $user->JID,
            'balance_type' => $package->balance_type,
            'package_price' => $package->price,
            'package_id' => $package->id,
            'quantity' => $quantity,
        ]);

        // التحقق من الرصيد بناء على نوع العملة المطلوبة
        $totalCost = $package->price * $quantity;
        $hasEnoughBalance = false;

        switch ($package->balance_type) {
            case 'balance':
                $hasEnoughBalance = $user->webUser->balance >= $totalCost;
                $currentBalance = $user->webUser->balance;
                break;
            case 'gift_balance':
                $hasEnoughBalance = $user->webUser->gift_balance >= $totalCost;
                $currentBalance = $user->webUser->gift_balance;
                break;
            case 'silk':
                $hasEnoughBalance = $user->skSilk->silk_own >= $totalCost;
                $currentBalance = $user->skSilk->silk_own;
                break;
            case 'donate':
                $hasEnoughBalance = $user->skSilk->silk_gift >= $totalCost;
                $currentBalance = $user->skSilk->silk_gift;
                break;
            default:
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'trans:packagesShop.notifications.invalidCurrency',
                    'duration' => 5000,
                ]);
                return;
        }

        if (!$hasEnoughBalance) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:packagesShop.notifications.insufficientBalance',
                'duration' => 5000,
                'params' => [
                    'currency' => ucfirst(str_replace('_', ' ', $package->balance_type)),
                    'required' => number_format($totalCost, 2),
                    'available' => number_format($currentBalance, 2)
                ]
            ]);
            return;
        }

        //check stock if out of stock
        if ($package->stock !== -1 && $package->stock <= 0) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:packagesShop.notifications.packageOutOfStock',
                'duration' => 5000,
            ]);
            return;
        }

        //check request user from stock
        if ($package->stock !== -1 && $package->stock < $quantity) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:packagesShop.notifications.notEnoughStock',
                'duration' => 5000,
                'params' => [
                    'available' => $package->stock
                ]
            ]);
            return;
        }

        //check expiry date
        if ($package->expiry_date) {
            $expiryDate = is_string($package->expiry_date) ? \Carbon\Carbon::parse($package->expiry_date) : $package->expiry_date;

            if ($expiryDate < now()) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'trans:packagesShop.notifications.packageExpired',
                    'duration' => 5000,
                ]);
                return;
            }
        }

        //Check User Slots in Chest
        $packageItems = $this->getPackageItemsCollection($package->id);
        //Get Total Sum from RefobjCommon by ids
        $totalSum = RefObjCommon::getSumMaxStack($packageItems->pluck('id')->toArray()) * $quantity ?? 1;
        //Get Free Slots in Chest
        $freeSlots = Chest::getFreeSlots($user->JID);
        if ($freeSlots < $totalSum) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:packagesShop.notifications.inventoryFull',
                'duration' => 5000,
                'params' => [
                    'required' => $totalSum,
                    'available' => $freeSlots,
                    'needed' => ($totalSum - $freeSlots)
                ]
            ]);
            return;
        }

        //if package has items
        if ($packageItems->isEmpty()) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Error: Package has no items!',
                'duration' => 5000,
            ]);
            return;
        }




        //Start transaction
        DB::beginTransaction();
        try {
            // استخدام البروسيدور الجديد لشراء الباكج

            // تسجيل البيانات المرسلة للـ stored procedure
            \Log::info('Calling stored procedure for package purchase:', [
                'JID' => $user->JID,
                'type' => 'package',
                'package_id' => $package->id,
                'quantity' => $quantity,
                'balance_type' => $package->balance_type,
                'package_price' => $package->price,
                'total_cost' => $totalCost,
                'current_balance' => $currentBalance,
            ]);

            $results = DB::connection('sqlsrv')->select('EXEC [market_buy] ?, ?, ?, ?, ?', [
                $user->JID, // JID
                'package', // type
                $package->id, // package id
                $quantity, // quantity
                $package->balance_type, // balance type
            ]);

            // تسجيل النتيجة المرجعة
            \Log::info('Package stored procedure result:', ['results' => $results]);

            // التحقق من نتيجة الإجراء المخزن
            if (!$results || !isset($results[0])) {
                throw new \Exception('No response from stored procedure');
            }

            $result = $results[0];

            // التحقق من وجود status
            if (!isset($result->status)) {
                throw new \Exception('Invalid stored procedure response - missing status field');
            }

            // التحقق من حالة النجاح (status = 1 يعني نجاح)
            if ($result->status != 1) {
                $errorMessage = isset($result->message) ? $result->message : 'Unknown error';
                throw new \Exception($errorMessage . ' (Status: ' . $result->status . ')');
            }

            // تسجيل النجاح
            \Log::info('Package purchase successful', [
                'package_id' => $package->id,
                'user_jid' => $user->JID,
                'quantity' => $quantity,
                'message' => $result->message ?? 'Success',
            ]);

            // تحديث المخزون إذا لم يكن لا نهائي
            if ($package->stock !== -1) {
                $package->stock -= $quantity;
                $package->save();
            }

            //خصم مبلغ من رصيد المستخدم
            $currentBalance = $user->webUser->balance;

            if ($currentBalance < $totalCost) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Error: Insufficient balance!',
                    'duration' => 5000
                ]);
                return;
            }

            if ($package->balance_type === 'balance') {
                $user->webUser->balance -= $totalCost;
                $user->webUser->save();
            } elseif ($package->balance_type === 'gift_balance') {
                $user->webUser->gift_balance -= $totalCost;
                $user->webUser->save();
            } elseif ($package->balance_type === 'silk') {
                $user->skSilk->silk_own -= $totalCost;
                $user->skSilk->save();
            } elseif ($package->balance_type === 'donate') {
                $user->skSilk->silk_gift -= $totalCost;
                $user->skSilk->save();
            }

            DB::commit();

            // إرسال رسالة نجاح
            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'trans:packagesShop.notifications.purchaseSuccess',
                'duration' => 5000,
            ]);

            // تسجيل المعاملة بعد النجاح
            // نحتاج نجيب الرصيد الجديد بعد الخصم من الـ stored procedure
            $user->refresh(); // تحديث بيانات المستخدم
            $newBalance = 0;
            switch ($package->balance_type) {
                case 'balance':
                    $newBalance = $user->webUser->balance ?? 0;
                    break;
                case 'gift_balance':
                    $newBalance = $user->webUser->gift_balance ?? 0;
                    break;
                case 'silk':
                    $newBalance = $user->skSilk->silk_own ?? 0;
                    break;
                case 'donate':
                    $newBalance = $user->skSilk->silk_gift ?? 0;
                    break;
            }

            // استخدام createLog بدلاً من logTransaction لتجنب الخصم المضاعف
            MarketTransaction::createLog([
                'user_jid' => $user->JID,
                'username' => $user->username ?? $user->UserID ?? 'Unknown',
                'transaction_type' => 'package',
                'item_id' => $package->id,
                'item_name' => $package->package_name,
                'quantity' => $quantity,
                'unit_price' => $package->price,
                'total_price' => $totalCost,
                'currency_type' => $package->balance_type,
                'balance_before' => $currentBalance,
                'balance_after' => $newBalance,
                'status' => 'success',
                'transaction_details' => [
                    'package_name' => $package->package_name,
                    'purchase_method' => 'stored_procedure',
                    'items_count' => $this->getPackageItemsCollection($package->id)->count(),
                ]
            ]);

            // تحديث الأرصدة في الواجهة الأمامية
            $this->dispatch('updateBalances');
        } catch (\Exception $e) {
            DB::rollBack();

            // تسجيل المعاملة الفاشلة
            try {
                MarketTransaction::createLog([
                    'user_jid' => $user->JID,
                    'username' => $user->username ?? $user->UserID ?? 'Unknown',
                    'transaction_type' => 'package',
                    'item_id' => $package->id,
                    'item_name' => $package->package_name,
                    'quantity' => $quantity,
                    'unit_price' => $package->price,
                    'total_price' => $totalCost,
                    'currency_type' => $package->balance_type,
                    'balance_before' => $currentBalance,
                    'balance_after' => $currentBalance, // لم يتغير الرصيد بسبب الفشل
                    'status' => 'failed',
                    'error_message' => $e->getMessage(),
                    'transaction_details' => [
                        'package_name' => $package->package_name,
                        'failure_reason' => $e->getMessage(),
                        'attempted_method' => 'stored_procedure',
                    ]
                ]);
            } catch (\Exception $logException) {
                // إذا فشل تسجيل الـ log، لا نريد أن يتوقف التطبيق
                \Log::error('Failed to log failed transaction: ' . $logException->getMessage());
            }

            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:packagesShop.notifications.purchaseFailed',
                'duration' => 5000,
                'params' => [
                    'error' => $e->getMessage()
                ]
            ]);
        }
    }

    // Helper methods for package status
    public function getPackageStatus($package)
    {
        if ($package->stock === -1) {
            return 'unlimited';
        } elseif ($package->stock <= 5) {
            return 'limited';
        } else {
            return 'in_stock';
        }
    }

    public function isExpiringSoon($package)
    {
        if (!$package->expiry_date) {
            return false;
        }

        $expiryDate = is_string($package->expiry_date) ? \Carbon\Carbon::parse($package->expiry_date) : $package->expiry_date;

        return $expiryDate->isAfter(now()) && $expiryDate->isBefore(now()->addDays(7));
    }

    public function getPackageBadges($package)
    {
        $badges = [];

        if ($package->sort_type === 'bestseller') {
            $badges[] = ['type' => 'hot', 'text' => 'HOT'];
        }

        if ($this->isExpiringSoon($package)) {
            $badges[] = ['type' => 'expiring', 'text' => 'EXPIRING'];
        }

        if ($package->stock <= 5 && $package->stock > 0) {
            $badges[] = ['type' => 'limited', 'text' => 'LIMITED'];
        }

        return $badges;
    }
};
?>

<div>
    <div class="space-y-8" x-data="{
        showModal: false,
        isLoading: false,
        selectedPackage: null,
        packageItems: [],
        showItemsModal: false,
        loadingItems: false,
        packageStats: {
            total_packages: {{ $totalPackages }},
            filtered_packages: {{ $filteredpackages->total() }},
            current_page: {{ $filteredpackages->currentPage() }},
            last_page: {{ $filteredpackages->lastPage() }},
            per_page: {{ $filteredpackages->perPage() }}
        }
    }" x-init="isLoading = true;
    setTimeout(() => isLoading = false, 800)">

        <!-- Loading State -->
        <div x-show="isLoading" class="packages-loading">
            <div class="loading-spinner"></div>
            <p class="text-gray-400" wire:ignore  data-translate="packagesShop.labels.loading">Loading packages...</p>
        </div>

        <div x-show="!isLoading" x-transition class="slide-in-up">

            <!-- Page Header -->
            <div class="text-center mb-8">
                <div
                    class="w-20 h-20 mx-auto mb-6 bg-gradient-to-r from-purple-600 to-cyan-500 rounded-3xl flex items-center justify-center gaming-glow">
                    <i class="fas fa-cube text-white text-3xl"></i>
                </div>
                <h1 class="text-4xl font-bold text-white mb-3 dashboard-gaming-text"
                    wire:ignore  data-translate="packagesShop.title">
                    Premium Packages
                </h1>
                <p class="text-xl text-purple-300 mb-6" wire:ignore  data-translate="packagesShop.subtitle">
                    Discover Epic Game Packages
                </p>
                <p class="text-gray-400 max-w-2xl mx-auto" wire:ignore  data-translate="packagesShop.description">
                    Choose from our exclusive collection of premium game packages
                </p>
            </div>

            <!-- Stats Cards -->
            <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-8">
                <div
                    class="bg-gradient-to-br from-purple-900/50 to-purple-800/30 rounded-xl p-4 border border-purple-500/30">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-purple-300 text-sm" wire:ignore  data-translate="packagesShop.labels.totalPackages">Total
                                Packages</p>
                            <p class="text-white text-2xl font-bold" x-text="packageStats.total_packages">
                                {{ $totalPackages }}</p>
                        </div>
                        <i class="fas fa-boxes text-purple-400 text-2xl"></i>
                    </div>
                </div>

                <div class="bg-gradient-to-br from-cyan-900/50 to-cyan-800/30 rounded-xl p-4 border border-cyan-500/30">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-cyan-300 text-sm">Current Page</p>
                            <p class="text-white text-2xl font-bold" x-text="packageStats.current_page + ' / ' + packageStats.last_page">
                                {{ $filteredpackages->currentPage() }} / {{ $filteredpackages->lastPage() }}</p>
                        </div>
                        <i class="fas fa-file-alt text-cyan-400 text-2xl"></i>
                    </div>
                </div>

                <div
                    class="bg-gradient-to-br from-orange-900/50 to-orange-800/30 rounded-xl p-4 border border-orange-500/30">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-orange-300 text-sm">Per Page</p>
                            <p class="text-white text-2xl font-bold" x-text="packageStats.per_page">
                                {{ $filteredpackages->perPage() }}</p>
                        </div>
                        <i class="fas fa-list text-orange-400 text-2xl"></i>
                    </div>
                </div>

                <div
                    class="bg-gradient-to-br from-green-900/50 to-green-800/30 rounded-xl p-4 border border-green-500/30">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-green-300 text-sm" wire:ignore data-translate="packagesShop.status.filtered">Filtered</p>
                            <p class="text-white text-2xl font-bold" x-text="packageStats.filtered_packages">
                                {{ $filteredpackages->total() }}</p>
                        </div>
                        <i class="fas fa-filter text-green-400 text-2xl"></i>
                    </div>
                </div>
            </div>

            <!-- Filters Section -->
            <div class="packages-filters">
                <div class="grid grid-cols-1 lg:grid-cols-12 gap-6">

                    <!-- Search -->
                    <div class="lg:col-span-6">
                        <label class="filter-label" wire:ignore  data-translate="common.search">Search</label>
                        <input type="text" wire:model.live="searchTerm" class="search-input"
                            placeholder="{{ __('Search packages...') }}"
                            wire:ignore  data-translate-placeholder="packagesShop.searchPlaceholder">
                    </div>

                    <!-- Sort By -->
                    <div class="lg:col-span-3">
                        <label class="filter-label" wire:ignore data-translate="packagesShop.labels.sortBy">Sort By</label>
                        <select wire:model.live="sortBy" class="search-input">
                            <option value="sort_order" wire:ignore data-translate="packagesShop.sorting.defaultOrder">Default Order</option>
                            <option value="name_asc" wire:ignore data-translate="packagesShop.sorting.nameAZ">Name A-Z</option>
                            <option value="name_desc" wire:ignore data-translate="packagesShop.sorting.nameZA">Name Z-A</option>
                            <option value="price_asc" wire:ignore data-translate="packagesShop.sorting.priceLowHigh">Price Low to High</option>
                            <option value="price_desc" wire:ignore data-translate="packagesShop.sorting.priceHighLow">Price High to Low</option>
                        </select>
                    </div>

                    <!-- Quick Filters Indicator -->
                    <div class="lg:col-span-3 flex items-end">
                        <div class="flex flex-wrap gap-2">
                            @if ($searchTerm)
                                <span class="px-3 py-1 bg-purple-600 text-white text-sm rounded-full">
                                    Search: "{{ $searchTerm }}"
                                </span>
                            @endif
                            @if ($filterType !== 'all')
                                <span class="px-3 py-1 bg-cyan-600 text-white text-sm rounded-full">
                                    {{ ucfirst($filterType) }}
                                </span>
                            @endif
                            @if ($priceRange !== 'all')
                                <span class="px-3 py-1 bg-orange-600 text-white text-sm rounded-full">
                                    {{ ucfirst($priceRange) }} Price
                                </span>
                            @endif
                        </div>
                    </div>
                </div>

                <!-- Advanced Filters Row -->
                <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mt-6">

                    <!-- Status Filter -->
                    <div>
                        <label class="filter-label" wire:ignore data-translate="packagesShop.labels.status">Status</label>
                        <div class="filter-buttons">
                            <button wire:click="setFilter('all')"
                                class="filter-btn {{ $filterType === 'all' ? 'active' : '' }}">
                                All
                            </button>
                            <button wire:click="setFilter('available')"
                                class="filter-btn {{ $filterType === 'available' ? 'active' : '' }}"
                                wire:ignore  data-translate="packagesShop.filters.available">
                                Available
                            </button>
                            <button wire:click="setFilter('expiring')"
                                class="filter-btn {{ $filterType === 'expiring' ? 'active' : '' }}"
                                wire:ignore  data-translate="packagesShop.filters.expiring">
                                Expiring
                            </button>
                            <button wire:click="setFilter('limited')"
                                class="filter-btn {{ $filterType === 'limited' ? 'active' : '' }}">
                                Limited
                            </button>
                            <button wire:click="setFilter('unlimited')"
                                class="filter-btn {{ $filterType === 'unlimited' ? 'active' : '' }}">
                                Unlimited
                            </button>
                        </div>
                    </div>

                    <!-- Price Range -->
                    <div>
                        <label class="filter-label" wire:ignore data-translate="packagesShop.labels.priceRange">Price Range</label>
                        <div class="filter-buttons">
                            <button wire:click="setPriceRange('all')"
                                class="filter-btn {{ $priceRange === 'all' ? 'active' : '' }}">
                                All
                            </button>
                            <button wire:click="setPriceRange('low')"
                                class="filter-btn {{ $priceRange === 'low' ? 'active' : '' }}"
                                wire:ignore  data-translate="packagesShop.filters.lowPrice">
                                ≤ 1K
                            </button>
                            <button wire:click="setPriceRange('medium')"
                                class="filter-btn {{ $priceRange === 'medium' ? 'active' : '' }}"
                                wire:ignore  data-translate="packagesShop.filters.mediumPrice">
                                1K-10K
                            </button>
                            <button wire:click="setPriceRange('high')"
                                class="filter-btn {{ $priceRange === 'high' ? 'active' : '' }}"
                                wire:ignore  data-translate="packagesShop.filters.highPrice">
                                >10K
                            </button>
                        </div>
                    </div>

                    <!-- Balance Type -->
                    <div>
                        <label class="filter-label">Currency Type</label>
                        <div class="filter-buttons">
                            <button wire:click="setBalanceType('all')"
                                class="filter-btn {{ $balanceType === 'all' ? 'active' : '' }}">
                                All
                            </button>
                            @foreach ($this->availableBalanceTypes as $type)
                                <button wire:click="setBalanceType('{{ $type }}')"
                                    class="filter-btn {{ $balanceType === $type ? 'active' : '' }}">
                                    @if ($type === 'balance')
                                        💎
                                    @elseif($type === 'gift_balance')
                                        <div class="relative inline-block mr-1">
                                            <svg class="balance-icon-svg" width="24" height="24"
                                                viewBox="0 0 24 24" fill="none"
                                                xmlns="http://www.w3.org/2000/svg">
                                                <defs>
                                                    <linearGradient id="starGradient" x1="0%" y1="0%"
                                                        x2="100%" y2="100%">
                                                        <stop offset="0%" style="stop-color:#ffcc02" />
                                                        <stop offset="50%" style="stop-color:#ff9800" />
                                                        <stop offset="100%" style="stop-color:#f57c00" />
                                                    </linearGradient>
                                                    <filter id="starGlow">
                                                        <feGaussianBlur stdDeviation="2" result="coloredBlur" />
                                                        <feMerge>
                                                            <feMergeNode in="coloredBlur" />
                                                            <feMergeNode in="SourceGraphic" />
                                                        </feMerge>
                                                    </filter>
                                                </defs>
                                                <polygon
                                                    points="12,2 15.09,8.26 22,9.27 17,14.14 18.18,21.02 12,17.77 5.82,21.02 7,14.14 2,9.27 8.91,8.26 12,2"
                                                    fill="url(#starGradient)" stroke="#f57c00" stroke-width="1"
                                                    filter="url(#starGlow)" />
                                                <circle cx="12" cy="12" r="2" fill="#fff" />
                                                <path d="M10 12l1-1 1 1-1 1-1-1z" fill="#f57c00" />
                                            </svg>
                                        </div>
                                    @elseif($type === 'silk')
                                        <div class="relative inline-block mr-1">
                                            <svg class="balance-icon-svg" width="24" height="24"
                                                viewBox="0 0 24 24" fill="none"
                                                xmlns="http://www.w3.org/2000/svg">
                                                <defs>
                                                    <linearGradient id="silkGradient" x1="0%" y1="0%"
                                                        x2="100%" y2="100%">
                                                        <stop offset="0%" style="stop-color:#fff9c4" />
                                                        <stop offset="50%" style="stop-color:#ffeb3b" />
                                                        <stop offset="100%" style="stop-color:#f57f17" />
                                                    </linearGradient>
                                                    <filter id="sparkle">
                                                        <feGaussianBlur stdDeviation="1" result="coloredBlur" />
                                                        <feMerge>
                                                            <feMergeNode in="coloredBlur" />
                                                            <feMergeNode in="SourceGraphic" />
                                                        </feMerge>
                                                    </filter>
                                                </defs>
                                                <path
                                                    d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"
                                                    fill="url(#silkGradient)" stroke="#fbc02d" stroke-width="1.5"
                                                    filter="url(#sparkle)" />
                                                <circle cx="8" cy="6" r="1.5" fill="#fbc02d"
                                                    opacity="0.8" />
                                                <circle cx="16" cy="6" r="1.5" fill="#fbc02d"
                                                    opacity="0.8" />
                                                <circle cx="6" cy="18" r="1" fill="#fbc02d"
                                                    opacity="0.6" />
                                                <circle cx="18" cy="18" r="1" fill="#fbc02d"
                                                    opacity="0.6" />
                                                <path d="M10 12l1-1 1 1-1 1-1-1z" fill="#f57f17" />
                                                <circle cx="12" cy="12" r="0.5" fill="#fff" />
                                            </svg>
                                        </div>
                                    @elseif($type === 'donate')
                                        <div class="relative inline-block mr-1">
                                            <svg class="balance-icon-svg mr-1" width="18" height="18"
                                                viewBox="0 0 24 24" fill="none"
                                                xmlns="http://www.w3.org/2000/svg">
                                                <defs>
                                                    <linearGradient id="heartGradient" x1="0%" y1="0%"
                                                        x2="100%" y2="100%">
                                                        <stop offset="0%" style="stop-color:#f8bbd9" />
                                                        <stop offset="50%" style="stop-color:#e91e63" />
                                                        <stop offset="100%" style="stop-color:#ad1457" />
                                                    </linearGradient>
                                                </defs>
                                                <path
                                                    d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"
                                                    fill="url(#heartGradient)" stroke="#c2185b" stroke-width="1.5" />
                                                <path d="M12 7v6M9 10h6" stroke="#fff" stroke-width="2"
                                                    stroke-linecap="round" /><text x="12" y="16" text-anchor="middle"
                                                    fill="#fff" font-size="6" font-weight="bold">4T</text>
                                                <circle cx="8" cy="6" r="1" fill="#fff"
                                                    opacity="0.7" />
                                                <circle cx="16" cy="6" r="1" fill="#fff"
                                                    opacity="0.7" />
                                            </svg>
                                        </div>
                                    @else
                                        💎
                                    @endif
                                    {{ ucfirst(str_replace('_', ' ', $type)) }}
                                </button>
                            @endforeach
                        </div>
                    </div>

                    <!-- Package Category -->
                    <div>
                        <label class="filter-label">Category</label>
                        <div class="filter-buttons">
                            <button wire:click="setSortType('all')"
                                class="filter-btn {{ $sortType === 'all' ? 'active' : '' }}">
                                All
                            </button>
                            @foreach ($this->availableSortTypes as $type)
                                <button wire:click="setSortType('{{ $type }}')"
                                    class="filter-btn {{ $sortType === $type ? 'active' : '' }}">
                                    {{ ucfirst($type) }}
                                </button>
                            @endforeach
                        </div>
                    </div>
                </div>

                <!-- Filter Actions -->
                <div class="flex justify-between items-center mt-6">
                    <div class="text-purple-300 text-sm">
                        Showing {{ $filteredpackages->count() }} of {{ $totalPackages }} packages
                        (Page {{ $filteredpackages->currentPage() }} of {{ $filteredpackages->lastPage() }})
                    </div>
                    <button wire:click="resetFilters" class="btn-reset-filters"
                        wire:ignore  data-translate="packagesShop.labels.resetFilters">
                        <i class="fas fa-undo mr-2"></i>
                        Reset All Filters
                    </button>
                </div>
            </div>

            <!-- Pagination Controls and Per Page Selector -->
            <div class="flex flex-col lg:flex-row justify-between items-center gap-4 mb-6 p-4 bg-gray-800/30 rounded-xl">
                <!-- Per Page Selector -->
                <div class="flex items-center gap-3">
                    <span class="text-purple-200 text-sm">Packages per page:</span>
                    <div class="flex gap-2">
                        @foreach([6, 12, 24, 48] as $pageSize)
                            <button @click="$wire.changePerPage({{ $pageSize }})"
                                    class="px-3 py-1 rounded-lg text-sm font-medium transition-all
                                           {{ $perPage === $pageSize
                                              ? 'bg-purple-600 text-white shadow-lg'
                                              : 'bg-gray-700 text-gray-300 hover:bg-gray-600' }}">
                                {{ $pageSize }}
                            </button>
                        @endforeach
                    </div>
                </div>

                <!-- Navigation Info -->
                @if($totalPackages > 0)
                    <div class="text-sm text-purple-200">
                        Showing {{ $filteredpackages->firstItem() }} to {{ $filteredpackages->lastItem() }}
                        of {{ $totalPackages }} results
                    </div>
                @endif

                <!-- Quick Page Navigation -->
                @if($filteredpackages->hasPages())
                    <div class="flex items-center gap-2">
                        <!-- Previous Page -->
                        @if($filteredpackages->onFirstPage())
                            <span class="px-3 py-1 rounded-lg bg-gray-700 text-gray-500 cursor-not-allowed">
                                Previous
                            </span>
                        @else
                            <button wire:click="previousPage"
                                    class="px-3 py-1 rounded-lg bg-purple-600 text-white hover:bg-purple-700 transition-colors">
                                Previous
                            </button>
                        @endif

                        <!-- Page Numbers (show only a few) -->
                        @php
                            $start = max(1, $filteredpackages->currentPage() - 2);
                            $end = min($filteredpackages->lastPage(), $filteredpackages->currentPage() + 2);
                        @endphp

                        @if($start > 1)
                            <button wire:click="gotoPage(1)" class="px-3 py-1 rounded-lg bg-gray-700 text-gray-300 hover:bg-gray-600">1</button>
                            @if($start > 2)
                                <span class="text-gray-500">...</span>
                            @endif
                        @endif

                        @for($i = $start; $i <= $end; $i++)
                            <button wire:click="gotoPage({{ $i }})"
                                    class="px-3 py-1 rounded-lg {{ $i === $filteredpackages->currentPage()
                                           ? 'bg-purple-600 text-white'
                                           : 'bg-gray-700 text-gray-300 hover:bg-gray-600' }}">
                                {{ $i }}
                            </button>
                        @endfor

                        @if($end < $filteredpackages->lastPage())
                            @if($end < $filteredpackages->lastPage() - 1)
                                <span class="text-gray-500">...</span>
                            @endif
                            <button wire:click="gotoPage({{ $filteredpackages->lastPage() }})"
                                    class="px-3 py-1 rounded-lg bg-gray-700 text-gray-300 hover:bg-gray-600">
                                {{ $filteredpackages->lastPage() }}
                            </button>
                        @endif

                        <!-- Next Page -->
                        @if($filteredpackages->hasMorePages())
                            <button wire:click="nextPage"
                                    class="px-3 py-1 rounded-lg bg-purple-600 text-white hover:bg-purple-700 transition-colors">
                                Next
                            </button>
                        @else
                            <span class="px-3 py-1 rounded-lg bg-gray-700 text-gray-500 cursor-not-allowed">
                                Next
                            </span>
                        @endif
                    </div>
                @endif
            </div>

            <!-- Packages Grid -->
            @if (count($filteredpackages) > 0)
                <div class="packages-container">
                    @foreach ($filteredpackages as $index => $package)
                        <div class="package-card gaming-glow" x-data="{
                            packageId: {{ $package->id }},
                            animationDelay: {{ $index * 100 }}
                        }" x-init="setTimeout(() => $el.style.animationDelay = animationDelay + 'ms', 100)"
                            style="animation-delay: {{ $index * 100 }}ms;">

                            <!-- Package Status Indicator -->
                            <div class="package-status-indicator"></div>

                            <!-- Package Header -->
                            <div class="package-header">
                                <div class="package-badges">
                                    @foreach ($this->getPackageBadges($package) as $badge)
                                        <span class="package-badge badge-{{ $badge['type'] }}">
                                            {{ $badge['text'] }}
                                        </span>
                                    @endforeach
                                </div>

                                <h3 class="package-title">{{ $package->package_name }}</h3>
                                <p class="package-description">
                                    {{ $package->description ?: 'Premium gaming package with exclusive items' }}</p>
                            </div>

                            <!-- Package Content -->
                            <div class="package-content">
                                <!-- Price -->
                                <div class="package-price">
                                    <span class="price-value">{{ number_format($package->price) }}</span>
                                    <span class="price-currency">
                                        @if ($package->balance_type === 'balance')
                                            💎
                                        @elseif ($package->balance_type === 'gift_balance')
                                            <svg class="balance-icon-svg" width="24" height="24"
                                                viewBox="0 0 24 24" fill="none"
                                                xmlns="http://www.w3.org/2000/svg">
                                                <defs>
                                                    <linearGradient id="starGradient" x1="0%" y1="0%"
                                                        x2="100%" y2="100%">
                                                        <stop offset="0%" style="stop-color:#ffcc02" />
                                                        <stop offset="50%" style="stop-color:#ff9800" />
                                                        <stop offset="100%" style="stop-color:#f57c00" />
                                                    </linearGradient>
                                                    <filter id="starGlow">
                                                        <feGaussianBlur stdDeviation="2" result="coloredBlur" />
                                                        <feMerge>
                                                            <feMergeNode in="coloredBlur" />
                                                            <feMergeNode in="SourceGraphic" />
                                                        </feMerge>
                                                    </filter>
                                                </defs>
                                                <polygon
                                                    points="12,2 15.09,8.26 22,9.27 17,14.14 18.18,21.02 12,17.77 5.82,21.02 7,14.14 2,9.27 8.91,8.26 12,2"
                                                    fill="url(#starGradient)" stroke="#f57c00" stroke-width="1"
                                                    filter="url(#starGlow)" />
                                                <circle cx="12" cy="12" r="2" fill="#fff" />
                                                <path d="M10 12l1-1 1 1-1 1-1-1z" fill="#f57c00" />
                                            </svg>
                                        @elseif ($package->balance_type === 'silk')
                                            <svg class="balance-icon-svg" width="20" height="20"
                                                viewBox="0 0 24 24" fill="none"
                                                xmlns="http://www.w3.org/2000/svg">
                                                <path
                                                    d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"
                                                    fill="#ffeb3b" stroke="#fbc02d" stroke-width="1" />
                                                <path d="M8 12l2-2 2 2-2 2-2-2z" fill="#fbc02d" />
                                                <circle cx="6" cy="7" r="1" fill="#fbc02d" />
                                                <circle cx="18" cy="7" r="1" fill="#fbc02d" /><text
                                                    x="12" y="22" text-anchor="middle" fill="#fbc02d" font-size="6"
                                                    font-weight="bold">✨</text>
                                            </svg>
                                        @elseif ($package->balance_type === 'donate')
                                            <svg class="balance-icon-svg mr-1" width="18" height="18"
                                                viewBox="0 0 24 24" fill="none"
                                                xmlns="http://www.w3.org/2000/svg">
                                                <defs>
                                                    <linearGradient id="heartGradient" x1="0%" y1="0%"
                                                        x2="100%" y2="100%">
                                                        <stop offset="0%" style="stop-color:#f8bbd9" />
                                                        <stop offset="50%" style="stop-color:#e91e63" />
                                                        <stop offset="100%" style="stop-color:#ad1457" />
                                                    </linearGradient>
                                                </defs>
                                                <path
                                                    d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"
                                                    fill="url(#heartGradient)" stroke="#c2185b" stroke-width="1.5" />
                                                <path d="M12 7v6M9 10h6" stroke="#fff" stroke-width="2"
                                                    stroke-linecap="round" /><text x="12" y="16" text-anchor="middle"
                                                    fill="#fff" font-size="6" font-weight="bold">4T</text>
                                                <circle cx="8" cy="6" r="1" fill="#fff"
                                                    opacity="0.7" />
                                                <circle cx="16" cy="6" r="1" fill="#fff"
                                                    opacity="0.7" />
                                            </svg>
                                        @else
                                            ⭐
                                        @endif
                                    </span>
                                </div>

                                <!-- Package Stats -->
                                <div class="package-stats">
                                    <div class="stat-item">
                                        <div class="stat-label" wire:ignore  data-translate="packagesShop.labels.stock">Stock</div>
                                        <div class="stat-value stat-stock {{ $this->getPackageStatus($package) }}">
                                            @if ($package->stock === -1)
                                                <span wire:ignore  data-translate="packagesShop.status.unlimited">Unlimited</span>
                                            @else
                                                {{ $package->stock }}
                                            @endif
                                        </div>
                                    </div>

                                    @if ($package->expiry_date)
                                        <div class="stat-item">
                                            <div class="stat-label" wire:ignore  data-translate="packagesShop.labels.expiresOn">
                                                Expires</div>
                                            <div
                                                class="stat-value stat-expiry {{ $this->isExpiringSoon($package) ? 'expiring' : '' }}">
                                                {{ \Carbon\Carbon::parse($package->expiry_date)->format('M d') }}
                                            </div>
                                        </div>
                                    @endif
                                </div>

                                <!-- Package Items Preview -->
                                <div class="package-items-preview">
                                    <div class="items-header">
                                        <span wire:ignore  data-translate="packagesShop.labels.items">Package Items</span>
                                        <span class="items-count">
                                            {{ $this->getPackageItemsCollection($package->id)->count() }} <span
                                                wire:ignore  data-translate="packagesShop.labels.items">items</span>
                                        </span>
                                    </div>

                                    <div class="items-grid">
                                        @foreach ($this->getPackageItemsCollection($package->id)->take(6) as $item)
                                            <div class="item-preview" title="{{ $item->name }}">
                                                <img src="{{ $this->getImageUrl($item->image) }}"
                                                    alt="{{ $item->name }}" loading="lazy">
                                            </div>
                                        @endforeach

                                        @if ($this->getPackageItemsCollection($package->id)->count() > 6)
                                            <div
                                                class="item-preview flex items-center justify-center bg-purple-900/50">
                                                <span class="text-white text-sm font-bold">
                                                    +{{ $this->getPackageItemsCollection($package->id)->count() - 6 }}
                                                </span>
                                            </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- Package Actions -->
                                <div class="package-actions">
                                    <button wire:click="buyPackage({{ $package->id }})"
                                        class="btn-buy-package gaming-button-hover"
                                        wire:ignore  data-translate="packagesShop.labels.buyNow">
                                        Buy Now
                                    </button>

                                    <button
                                        @click="
                                            selectedPackage = {{ $package->id }};
                                            loadingItems = true;
                                            showItemsModal = true;
                                            $wire.getPackageItems({{ $package->id }}).then(items => {
                                                packageItems = items;
                                                loadingItems = false;
                                            }).catch(() => {
                                                loadingItems = false;
                                            });
                                        "
                                        class="btn-view-items" wire:ignore  data-translate="packagesShop.labels.viewItems">
                                        View Items
                                    </button>
                                </div>
                            </div>
                        </div>
                    @endforeach
                </div>

                <!-- Bottom Pagination -->
                @if($filteredpackages->hasPages())
                    <div class="bottom-pagination">
                        <div class="pagination-wrapper">
                            {{ $filteredpackages->links('pagination.custom') }}
                        </div>
                    </div>
                @endif
            @else
                <!-- Empty State -->
                <div class="packages-empty">
                    <div class="empty-icon">📦</div>
                    <h3 class="empty-title" wire:ignore  data-translate="packagesShop.labels.noPackages">No packages found</h3>
                    <p class="empty-description">Try adjusting your search criteria or filters</p>
                </div>
            @endif
        </div>

        <!-- Items Modal -->
        <div x-show="showItemsModal" x-transition:enter="transition ease-out duration-300"
            x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100"
            x-transition:leave="transition ease-in duration-200" x-transition:leave-start="opacity-100"
            x-transition:leave-end="opacity-0"
            class="fixed inset-0 bg-black/80 backdrop-blur-sm z-50 flex items-center justify-center p-4"
            @click.self="showItemsModal = false">

            <div class="bg-gradient-to-br from-gray-900 to-gray-800 rounded-2xl max-w-4xl w-full max-h-[80vh] overflow-hidden border border-purple-500/30"
                x-transition:enter="transition ease-out duration-300 transform"
                x-transition:enter-start="scale-95 opacity-0" x-transition:enter-end="scale-100 opacity-100"
                x-transition:leave="transition ease-in duration-200 transform"
                x-transition:leave-start="scale-100 opacity-100" x-transition:leave-end="scale-95 opacity-0">

                <div class="p-6 border-b border-purple-500/30">
                    <div class="flex items-center justify-between">
                        <h3 class="text-2xl font-bold text-white" wire:ignore  data-translate="packagesShop.labels.items">Package
                            Items</h3>
                        <button @click="showItemsModal = false"
                            class="text-gray-400 hover:text-white transition-colors">
                            <i class="fas fa-times text-xl"></i>
                        </button>
                    </div>
                </div>

                <div class="p-6 overflow-y-auto max-h-[60vh]">
                    <!-- Loading State -->
                    <div x-show="loadingItems" class="text-center py-12">
                        <div class="loading-spinner mx-auto mb-4"></div>
                        <p class="text-gray-400">Loading package items...</p>
                    </div>

                    <!-- Items Grid -->
                    <div x-show="!loadingItems" class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                        <template x-for="item in packageItems" :key="item.id">
                            <div
                                class="bg-gradient-to-br from-purple-900/30 to-purple-800/20 rounded-lg p-4 border border-purple-500/20 hover:border-cyan-500/40 transition-all">
                                <div class="aspect-square bg-gray-800/50 rounded-lg mb-3 overflow-hidden">
                                    <img :src="item.image || '/assets/images/default-item.svg'" :alt="item.name"
                                        class="w-full h-full object-cover">
                                </div>
                                <h4 class="text-white font-semibold mb-1 truncate" x-text="item.name"></h4>
                                <p class="text-purple-300 text-sm truncate"
                                    x-text="item.description || 'Premium item'"></p>
                                <div class="flex items-center justify-between mt-2">
                                    <span class="text-cyan-400 text-sm">Count:</span>
                                    <span class="text-white font-bold" x-text="item.Count || 1"></span>
                                </div>
                            </div>
                        </template>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
