<?php
use Livewire\Volt\Component;
use App\Models\Z_Website\Market\MarketItem;
use App\Models\Z_Website\Market\MarketTransaction;
use App\Models\SRO_VT_SHARD\RefObjCommon;
use App\Models\SRO_VT_SHARD\Chest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\WithPagination;

new class extends Component {
    use WithPagination;

    public $searchTerm = '';
    public $filterType = 'all'; // all, available, expiring
    public $priceRange = 'all'; // all, low, medium, high
    public $sortType = 'all'; // all, premium, normal, event
    public $selectedImageUrl = '';
    public $selectedImageAlt = '';
    public $perPage = 12; // عدد العناصر في كل صفحة

    public function mount()
    {
        // لا نحتاج لتحميل البيانات في mount مع pagination
    }

    //load data with pagination - استخدام render بدلاً من خاصية
    public function render() : mixed
    {
        $query = MarketItem::where('balance_type', 'balance')
            ->where('stock', '!=', 0) // Exclude out of stock items
            ->where('price', '>', 0) // Exclude free items
            ->where('sort_type', '!=', '') // Exclude items without a sort type
            //check expiry_date if not null Check if expiry_date is after now
            ->where(function ($query) {
                $query->whereNull('expiry_date')
                      ->orWhere('expiry_date', '>', now());
            })
            ->where('package_id', 0); // Exclude package items

        // تطبيق المرشحات على الاستعلام
        $query = $this->applyFiltersToQuery($query);

        // الحصول على العناصر مع pagination
        $filteredItems = $query->orderBy('sort_order')
            ->paginate($this->perPage)
            ->withQueryString(); // للحفاظ على معاملات الاستعلام في URL

        return view('livewire.backend.pages.item-shop', [
            'filteredItems' => $filteredItems,
            'totalItems' => $query->count()
        ]);
    }

    private function applyFiltersToQuery($query)
    {
        // تطبيق مرشح البحث
        if (!empty($this->searchTerm)) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . $this->searchTerm . '%')
                  ->orWhere('description', 'like', '%' . $this->searchTerm . '%');
            });
        }

        // تطبيق مرشح النوع (حالة المخزون)
        if ($this->filterType === 'available') {
            $query->where(function ($q) {
                $q->where('stock', '>', 0)->orWhere('stock', -1);
            });
        } elseif ($this->filterType === 'expiring') {
            $query->whereNotNull('expiry_date')
                  ->where('expiry_date', '>', now())
                  ->where('expiry_date', '<', now()->addDays(7));
        }

        // تطبيق مرشح نطاق السعر
        if ($this->priceRange !== 'all') {
            switch ($this->priceRange) {
                case 'low':
                    $query->where('price', '<=', 1000);
                    break;
                case 'medium':
                    $query->whereBetween('price', [1001, 10000]);
                    break;
                case 'high':
                    $query->where('price', '>', 10000);
                    break;
            }
        }

        // تطبيق مرشح نوع الفرز
        if ($this->sortType !== 'all' && !empty($this->sortType)) {
            $query->where('sort_type', $this->sortType);
        }

        return $query;
    }

    // دالة مبسطة للفلاتر التي لا تحتاج pagination كامل
    public function applyFilters()
    {
        $this->resetPage(); // إعادة تعيين الصفحة عند تطبيق المرشحات
    }

    public function updatedSearchTerm()
    {
        $this->applyFilters();
    }

    public function updatedPriceRange()
    {
        $this->applyFilters();
    }

    public function updatedSortType()
    {
        $this->applyFilters();
    }

    public function setFilter($type)
    {
        $this->filterType = $type;
        $this->applyFilters();
    }

    public function setPriceRange($range)
    {
        $this->priceRange = $range;
        $this->applyFilters();
    }

    public function setSortType($type)
    {
        $this->sortType = $type;
        $this->applyFilters();
    }

    public function resetFilters()
    {
        $this->searchTerm = '';
        $this->filterType = 'all';
        $this->priceRange = 'all';
        $this->sortType = 'all';
        $this->applyFilters();
    }

    // إضافة دالة لتغيير عدد العناصر في الصفحة
    public function changePerPage($perPage)
    {
        $this->perPage = $perPage;
        $this->resetPage();
    }


    private function getImageUrl($imagePath)
    {
        if (empty($imagePath)) {
            return asset('assets/images/default-item.svg');
        }

        // إذا كان الرابط يحتوي على http أو https فهو رابط خارجي
        if (str_starts_with($imagePath, 'http')) {
            return $imagePath;
        }

        // إذا كان يبدأ بـ storage فهو ملف محلي من storage
        if (str_starts_with($imagePath, 'storage/')) {
            return asset($imagePath);
        }

        // إذا كان ملف من storage بدون البادئة
        if (str_contains($imagePath, '.')) {
            return asset('storage/' . $imagePath);
        }

        // افتراضي
        return asset('assets/images/default-item.svg');
    }

    public function getAvailableSortTypesProperty()
    {
        return MarketItem::where('balance_type', 'balance')
            ->where('stock', '!=', 0)
            ->where('price', '>', 0)
            ->where('sort_type', '!=', '')
            ->where(function ($query) {
                $query->whereNull('expiry_date')
                      ->orWhere('expiry_date', '>', now());
            })
            ->where('package_id', 0)
            ->pluck('sort_type')
            ->filter()
            ->unique()
            ->sort() // ترتيب أبجدي من A إلى Z
            ->values()
            ->toArray();
    }

    //buy item
    public function buyItem($itemId, $quantity = 1)
    {
        // التحقق من صحة الكمية
        if ($quantity <= 0) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:itemShop.notifications.invalidQuantity',
                'duration' => 5000
            ]);
            return;
        }

        //Get Item Information by itemId
        $item = MarketItem::find($itemId);
        if (!$item) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:itemShop.notifications.itemNotFound',
                'duration' => 5000
            ]);
            return;
        }

        //check user balance
        $user = Auth::user();
        if (!$user || !$user->webUser) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:itemShop.notifications.loginRequired',
                'duration' => 5000
            ]);
            return;
        }

        // التحقق من وجود JID
        if (!$user->JID) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:itemShop.notifications.loginRequired',
                'duration' => 5000
            ]);
            return;
        }

        // إضافة debugging للـ user data
        \Log::info('User data for purchase:', [
            'auth_id' => Auth::id(),
            'user_jid' => $user->JID,
            'balance' => $user->webUser->balance
        ]);

        // التحقق من الرصيد بناء على نوع العملة المطلوبة
        $totalCost = $item->price * $quantity;
        $hasEnoughBalance = false;
        $currentBalance = 0;

        switch ($item->balance_type) {
            case 'balance':
                $hasEnoughBalance = $user->webUser->balance >= $totalCost;
                $currentBalance = $user->webUser->balance;
                break;
            case 'gift_balance':
                $hasEnoughBalance = $user->webUser->gift_balance >= $totalCost;
                $currentBalance = $user->webUser->gift_balance;
                break;
            case 'silk':
                $hasEnoughBalance = $user->skSilk->silk_own >= $totalCost;
                $currentBalance = $user->skSilk->silk_own;
                break;
            case 'donate':
                $hasEnoughBalance = $user->skSilk->silk_gift >= $totalCost;
                $currentBalance = $user->skSilk->silk_gift;
                break;
            default:
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'trans:itemShop.notifications.invalidCurrency',
                    'duration' => 5000,
                ]);
                return;
        }

        if (!$hasEnoughBalance) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:itemShop.notifications.insufficientBalance',
                'duration' => 5000,
                'params' => [
                    'currency' => ucfirst(str_replace('_', ' ', $item->balance_type)),
                    'required' => number_format($totalCost, 2),
                    'available' => number_format($currentBalance, 2)
                ]
            ]);
            return;
        }

        //check stock if out of stock
        if ($item->stock != -1 && $item->stock <= 0) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:itemShop.notifications.itemOutOfStock',
                'duration' => 5000
            ]);
            return;
        }

        //check request user from stock
        if ($item->stock != -1 && $item->stock < $quantity) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:itemShop.notifications.notEnoughStock',
                'duration' => 5000,
                'params' => [
                    'available' => $item->stock
                ]
            ]);
            return;
        }

        //check expiry date
        if ($item->expiry_date) {
            $expiryDate = is_string($item->expiry_date)
                ? \Carbon\Carbon::parse($item->expiry_date)
                : $item->expiry_date;

            if ($expiryDate < now()) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'trans:itemShop.notifications.itemExpired',
                    'duration' => 5000
                ]);
                return;
            }
        }

        //Before add Item to user inventory, check if user has enough space in inventory
        if ($item->item_type === 'item') {
            try {
                // جلب MaxStack من RefObjCommon + RefObjItem
                $refObjCommon = RefObjCommon::where('ID', $item->RefItemID)->with('getRefObjItem')->first();

                if (!$refObjCommon || !$refObjCommon->getRefObjItem) {
                    $this->dispatch('show-notification', [
                        'type' => 'error',
                        'message' => 'trans:itemShop.notifications.itemNotFound',
                        'duration' => 5000
                    ]);
                    return;
                }

                $maxStack = $refObjCommon->getRefObjItem->MaxStack ?? 1;
                if ($maxStack <= 0) {
                    $maxStack = 1;
                }

                $totalNeeded = $item->Count * $quantity;
                $newSlotsNeeded = ceil($totalNeeded / $maxStack);

                $availableSlots = Chest::where('UserJID', $user->JID)
                    ->where(function ($query) {
                        $query->whereNull('ItemID')->orWhere('ItemID', 0);
                    })
                    ->where('slot', '<=', 149)
                    ->count();

                // إذا عدد الخانات المطلوبة أكبر من المتاحة => رفض العملية
                if ($newSlotsNeeded > $availableSlots) {
                    $missingSlots = $newSlotsNeeded - $availableSlots;
                    $this->dispatch('show-notification', [
                        'type' => 'error',
                        'message' => 'trans:itemShop.notifications.inventoryFull',
                        'duration' => 5000,
                        'params' => [
                            'needed' => $missingSlots
                        ]
                    ]);
                    return;
                }
            } catch (\Exception $e) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Error checking inventory space: ' . $e->getMessage(),
                    'duration' => 5000
                ]);
                return;
            }
        }

        //Start transaction
        DB::beginTransaction();
        try {
            if($item->item_type === 'item'){
                // البروسيدور يقوم بخصم الرصيد بنفسه للعناصر

                // تسجيل البيانات المرسلة للـ stored procedure
                \Log::info('Calling stored procedure with data:', [
                    'JID' => $user->JID,
                    'type' => 'item',
                    'id' => $item->id,
                    'quantity' => $quantity,
                    'balance_type' => $item->balance_type,
                    'user_balance' => $user->webUser->balance,
                    'item_price' => $item->price
                ]);

                $results = DB::connection('sqlsrv')->select('EXEC [market_buy] ?, ?, ?, ?, ?', [
                    $user->JID,        // استخدام JID بدلاً من Auth::id()
                    'item',
                    $item->id,
                    $quantity,         // إرسال الكمية المطلوبة بدلاً من total_needed
                    $item->balance_type
                ]);

                // تسجيل النتيجة المرجعة
                \Log::info('Stored procedure result:', ['results' => $results]);

            }else if ($item->item_type === 'Donate'){
                // خصم الرصيد يدوياً للأنواع الأخرى
                $user->webUser->balance -= $item->price * $quantity;
                $user->webUser->save();

                $user->skSilk->silk_gift += $item->Count * $quantity;
                $user->skSilk->save();
            }else if ($item->item_type === 'Silk'){
                // خصم الرصيد يدوياً للأنواع الأخرى
                $user->webUser->balance -= $item->price * $quantity;
                $user->webUser->save();

                $user->skSilk->silk_own += $item->Count * $quantity;
                $user->skSilk->save();
            }

            // التحقق من نتيجة الإجراء المخزن للعناصر فقط
            if($item->item_type === 'item'){
                if (!$results || !isset($results[0])) {
                    throw new \Exception('No response from stored procedure');
                }

                $result = $results[0];

                // التحقق من وجود status بدلاً من Result
                if (!isset($result->status)) {
                    throw new \Exception('Invalid stored procedure response - missing status field');
                }

                // التحقق من حالة النجاح (status = 1 يعني نجاح)
                if ($result->status != 1) {
                    $errorMessage = isset($result->message) ? $result->message : 'Unknown error';
                    throw new \Exception($errorMessage . ' (Status: ' . $result->status . ')');
                }

                // تسجيل النجاح
                \Log::info('Item purchase successful', [
                    'item_id' => $item->id,
                    'user_jid' => $user->JID,
                    'quantity' => $quantity,
                    'message' => $result->message ?? 'Success'
                ]);
            }

            // تحديث المخزون إذا لم يكن لا نهائي
            if ($item->stock != -1) {
                $item->stock -= $quantity;
                $item->save();
            }

            DB::commit();

            // إرسال رسالة نجاح
            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'trans:itemShop.notifications.purchaseSuccess',
                'duration' => 5000
            ]);

            // تسجيل المعاملة بعد النجاح
            // نحتاج نجيب الرصيد الجديد بعد الخصم
            $user->refresh(); // تحديث بيانات المستخدم
            $newBalance = 0;
            switch ($item->balance_type) {
                case 'balance':
                    $newBalance = $user->webUser->balance ?? 0;
                    break;
                case 'gift_balance':
                    $newBalance = $user->webUser->gift_balance ?? 0;
                    break;
                case 'silk':
                    $newBalance = $user->skSilk->silk_own ?? 0;
                    break;
                case 'donate':
                    $newBalance = $user->skSilk->silk_gift ?? 0;
                    break;
            }

            // تسجيل المعاملة
            MarketTransaction::createLog([
                'user_jid' => $user->JID,
                'username' => $user->username ?? $user->UserID ?? 'Unknown',
                'transaction_type' => 'item',
                'item_id' => $item->id,
                'item_name' => $item->name,
                'quantity' => $quantity,
                'unit_price' => $item->price,
                'total_price' => $totalCost,
                'currency_type' => $item->balance_type,
                'balance_before' => $currentBalance,
                'balance_after' => $newBalance,
                'status' => 'success',
                'transaction_details' => [
                    'item_name' => $item->name,
                    'item_type' => $item->item_type,
                    'purchase_method' => $item->item_type === 'item' ? 'stored_procedure' : 'manual',
                    'item_count' => $item->Count ?? 1,
                ]
            ]);

            // تحديث الأرصدة في الواجهة الأمامية
            $this->dispatch('updateBalances');

        } catch (\Exception $e) {
            DB::rollBack();

            // تسجيل المعاملة الفاشلة
            try {
                MarketTransaction::createLog([
                    'user_jid' => $user->JID,
                    'username' => $user->username ?? $user->UserID ?? 'Unknown',
                    'transaction_type' => 'item',
                    'item_id' => $item->id,
                    'item_name' => $item->name,
                    'quantity' => $quantity,
                    'unit_price' => $item->price,
                    'total_price' => $totalCost,
                    'currency_type' => $item->balance_type,
                    'balance_before' => $currentBalance,
                    'balance_after' => $currentBalance, // لم يتغير الرصيد بسبب الفشل
                    'status' => 'failed',
                    'error_message' => $e->getMessage(),
                    'transaction_details' => [
                        'item_name' => $item->name,
                        'item_type' => $item->item_type,
                        'failure_reason' => $e->getMessage(),
                        'attempted_method' => $item->item_type === 'item' ? 'stored_procedure' : 'manual',
                    ]
                ]);
            } catch (\Exception $logException) {
                // إذا فشل تسجيل الـ log، لا نريد أن يتوقف التطبيق
                \Log::error('Failed to log failed item transaction: ' . $logException->getMessage());
            }

            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:itemShop.notifications.purchaseFailed',
                'duration' => 5000,
                'params' => [
                    'error' => $e->getMessage()
                ]
            ]);
        }
    }
};
?>
<div>


    <div class="min-h-screen bg-gradient-to-br from-gray-900 via-purple-900 to-violet-900 p-4"
         x-data="{
            selectedImageUrl: @entangle('selectedImageUrl'),
            selectedImageAlt: @entangle('selectedImageAlt'),
            isLoading: false
         }"
         x-init="isLoading = true; setTimeout(() => isLoading = false, 1000)">

        <!-- Loading State -->
        <div x-show="isLoading" class="flex items-center justify-center py-20">
            <div class="text-center">
                <div class="animate-spin rounded-full h-16 w-16 border-b-4 border-cyan-400 mx-auto mb-4"></div>
                <p class="text-purple-200" wire:ignore  data-translate="itemShop.status.loading">Loading items...</p>
            </div>
        </div>

        <div x-show="!isLoading" x-transition class="max-w-7xl mx-auto">

            <!-- Page Header -->
            <div class="text-center mb-8">
                <div class="w-20 h-20 mx-auto mb-6 bg-gradient-to-r from-cyan-500 to-purple-600 rounded-3xl flex items-center justify-center shadow-lg shadow-purple-500/25">
                    <!-- SVG Gem Icon -->
                    <svg class='w-6 h-6' fill='none' stroke='currentColor' viewBox='0 0 24 24'><path stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z'/></svg>
                </div>
                <h1 class="text-4xl font-bold text-white mb-4" wire:ignore  data-translate="itemShop.title">Premium Item Shop</h1>
                <p class="text-purple-200 text-lg max-w-2xl mx-auto mb-6" wire:ignore  data-translate="itemShop.subtitle">Discover legendary items and enhance your gaming experience</p>

                <!-- Filter Stats -->
                <div class="flex justify-center gap-4 text-sm text-purple-300">
                    <span>Total Items: <strong class="text-cyan-400">{{ $totalItems }}</strong></span>
                    <span>•</span>
                    <span>Page: <strong class="text-cyan-400">{{ $filteredItems->currentPage() }}</strong> of <strong class="text-cyan-400">{{ $filteredItems->lastPage() }}</strong></span>
                    <span>•</span>
                    <span>Showing: <strong class="text-cyan-400">{{ $filteredItems->count() }}</strong> items per page</span>
                </div>
            </div>

            <!-- Filters Section -->
            <div class="item-shop-filters p-6 mb-8">
                <div class="space-y-6">
                    <!-- Row 1: Stock Status Filters -->
                    <div class="flex flex-col lg:flex-row gap-6 items-start lg:items-center">
                        <!-- Stock Status Filter Buttons -->
                        <div class="flex flex-wrap gap-3">
                            <button @click="$wire.setFilter('all')"
                                    class="filter-button px-6 py-3 rounded-xl font-medium {{ $filterType === 'all' ? 'active' : '' }}"
                                    wire:ignore  data-translate="itemShop.filters.showAll">
                                All Items
                            </button>
                            <button @click="$wire.setFilter('available')"
                                    class="filter-button px-6 py-3 rounded-xl font-medium {{ $filterType === 'available' ? 'active' : '' }}"
                                    wire:ignore  data-translate="itemShop.filters.showAvailable">
                                In Stock
                            </button>
                            <button @click="$wire.setFilter('expiring')"
                                    class="filter-button px-6 py-3 rounded-xl font-medium {{ $filterType === 'expiring' ? 'active' : '' }}"
                                    wire:ignore  data-translate="itemShop.filters.showExpiring">
                                Expiring Soon
                            </button>
                        </div>

                        <!-- Search Input -->
                        <div class="flex-1 max-w-md">
                            <input type="text"
                                   wire:model.live="searchTerm"
                                   class="search-input w-full px-4 py-3 text-white placeholder-gray-400"
                                   placeholder="Search items by name..."
                                   wire:ignore  data-translate-placeholder="itemShop.filters.searchPlaceholder">
                        </div>

                        <!-- Reset Button -->
                        <button @click="$wire.resetFilters()"
                                class="reset-button px-6 py-3 rounded-xl font-medium"
                                wire:ignore  data-translate="itemShop.filters.resetFilters">
                            Reset Filters
                        </button>
                    </div>

                    <!-- Row 2: Price and Sort Type Filters -->
                    <div class="flex flex-col lg:flex-row gap-6 items-start lg:items-center">
                        <!-- Price Range Filter -->
                        <div class="flex flex-col gap-2">
                            <label class="text-sm font-medium text-purple-200" wire:ignore  data-translate="itemShop.filters.priceRange">Price Range:</label>
                            <div class="flex flex-wrap gap-2">
                                <button @click="$wire.setPriceRange('all')"
                                        class="filter-button px-4 py-2 rounded-lg text-sm {{ $priceRange === 'all' ? 'active' : '' }}"
                                        wire:ignore  data-translate="itemShop.filters.priceRanges.all">
                                    All Prices
                                </button>
                                <button @click="$wire.setPriceRange('low')"
                                        class="filter-button px-4 py-2 rounded-lg text-sm {{ $priceRange === 'low' ? 'active' : '' }}"
                                        wire:ignore  data-translate="itemShop.filters.priceRanges.low">
                                    Low (≤ 1,000)
                                </button>
                                <button @click="$wire.setPriceRange('medium')"
                                        class="filter-button px-4 py-2 rounded-lg text-sm {{ $priceRange === 'medium' ? 'active' : '' }}"
                                        wire:ignore  data-translate="itemShop.filters.priceRanges.medium">
                                    Medium (1,001 - 10,000)
                                </button>
                                <button @click="$wire.setPriceRange('high')"
                                        class="filter-button px-4 py-2 rounded-lg text-sm {{ $priceRange === 'high' ? 'active' : '' }}"
                                        wire:ignore  data-translate="itemShop.filters.priceRanges.high">
                                    High (> 10,000)
                                </button>
                            </div>
                        </div>

                        <!-- Sort Type Filter -->
                        <div class="flex flex-col gap-2">
                            <label class="text-sm font-medium text-purple-200" wire:ignore  data-translate="itemShop.filters.sortType">Sort Type:</label>
                            <div class="flex flex-wrap gap-2">
                                <button @click="$wire.setSortType('all')"
                                        class="filter-button px-4 py-2 rounded-lg text-sm {{ $sortType === 'all' ? 'active' : '' }}"
                                        wire:ignore  data-translate="itemShop.filters.sortTypes.all">
                                    All Types
                                </button>
                                @foreach($this->availableSortTypes as $type)
                                    @if(!empty($type))
                                        <button @click="$wire.setSortType('{{ $type }}')"
                                                class="filter-button px-4 py-2 rounded-lg text-sm {{ $sortType === $type ? 'active' : '' }}">
                                            {{ ucfirst($type) }}
                                        </button>
                                    @endif
                                @endforeach
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Pagination Controls and Per Page Selector -->
            <div class="flex flex-col lg:flex-row justify-between items-center gap-4 mb-6 p-4 bg-gray-800/30 rounded-xl">
                <!-- Per Page Selector -->
                <div class="flex items-center gap-3">
                    <span class="text-purple-200 text-sm">Items per page:</span>
                    <div class="flex gap-2">
                        @foreach([6, 12, 24, 48] as $pageSize)
                            <button @click="$wire.changePerPage({{ $pageSize }})"
                                    class="px-3 py-1 rounded-lg text-sm font-medium transition-all
                                           {{ $perPage === $pageSize
                                              ? 'bg-purple-600 text-white shadow-lg'
                                              : 'bg-gray-700 text-gray-300 hover:bg-gray-600' }}">
                                {{ $pageSize }}
                            </button>
                        @endforeach
                    </div>
                </div>

                <!-- Navigation Info -->
                @if($totalItems > 0)
                    <div class="text-sm text-purple-200">
                        Showing {{ $filteredItems->firstItem() }} to {{ $filteredItems->lastItem() }}
                        of {{ $totalItems }} results
                    </div>
                @endif

                <!-- Quick Page Navigation -->
                @if($filteredItems->hasPages())
                    <div class="flex items-center gap-2">
                        <!-- Previous Page -->
                        @if($filteredItems->onFirstPage())
                            <span class="px-3 py-1 rounded-lg bg-gray-700 text-gray-500 cursor-not-allowed">
                                Previous
                            </span>
                        @else
                            <button wire:click="previousPage"
                                    class="px-3 py-1 rounded-lg bg-purple-600 text-white hover:bg-purple-700 transition-colors">
                                Previous
                            </button>
                        @endif

                        <!-- Page Numbers (show only a few) -->
                        @php
                            $start = max(1, $filteredItems->currentPage() - 2);
                            $end = min($filteredItems->lastPage(), $filteredItems->currentPage() + 2);
                        @endphp

                        @if($start > 1)
                            <button wire:click="gotoPage(1)" class="px-3 py-1 rounded-lg bg-gray-700 text-gray-300 hover:bg-gray-600">1</button>
                            @if($start > 2)
                                <span class="text-gray-500">...</span>
                            @endif
                        @endif

                        @for($i = $start; $i <= $end; $i++)
                            <button wire:click="gotoPage({{ $i }})"
                                    class="px-3 py-1 rounded-lg {{ $i === $filteredItems->currentPage()
                                           ? 'bg-purple-600 text-white'
                                           : 'bg-gray-700 text-gray-300 hover:bg-gray-600' }}">
                                {{ $i }}
                            </button>
                        @endfor

                        @if($end < $filteredItems->lastPage())
                            @if($end < $filteredItems->lastPage() - 1)
                                <span class="text-gray-500">...</span>
                            @endif
                            <button wire:click="gotoPage({{ $filteredItems->lastPage() }})"
                                    class="px-3 py-1 rounded-lg bg-gray-700 text-gray-300 hover:bg-gray-600">
                                {{ $filteredItems->lastPage() }}
                            </button>
                        @endif

                        <!-- Next Page -->
                        @if($filteredItems->hasMorePages())
                            <button wire:click="nextPage"
                                    class="px-3 py-1 rounded-lg bg-purple-600 text-white hover:bg-purple-700 transition-colors">
                                Next
                            </button>
                        @else
                            <span class="px-3 py-1 rounded-lg bg-gray-700 text-gray-500 cursor-not-allowed">
                                Next
                            </span>
                        @endif
                    </div>
                @endif
            </div>

            <!-- Items Grid -->
            @if($filteredItems && $filteredItems->count() > 0)
                <div class="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-8">
                    @foreach($filteredItems as $item)
                        @php
                            $hasStock = $item->stock == -1 || $item->stock > 0;
                            $hasExpiry = $item->expiry_date !== null;
                            $cardClass = '';

                            if ($hasStock && $hasExpiry) {
                                $cardClass = 'has-both';
                            } elseif ($hasStock) {
                                $cardClass = 'has-stock';
                            } elseif ($hasExpiry) {
                                $cardClass = 'has-expiry';
                            }

                            // Calculate expiry information
                            $expiryData = null;
                            if ($item->expiry_date) {
                                $expiryDate = is_string($item->expiry_date)
                                    ? \Carbon\Carbon::parse($item->expiry_date)
                                    : $item->expiry_date;
                                $daysLeft = now()->diffInDays($expiryDate, false);
                                $hoursLeft = now()->diffInHours($expiryDate, false);
                                $expiryData = [
                                    'days' => $daysLeft,
                                    'hours' => $hoursLeft,
                                    'isUrgent' => $daysLeft <= 1
                                ];
                            }
                        @endphp

                        <div class="item-card {{ $cardClass }} p-6" x-data="{
                            showDetails: false,
                            quantity: 1,

                         }">

                            <!-- Stock Counter Badge -->
                            @if($item->stock != -1)
                                <div class="counter-badge stock-counter {{ $item->stock <= 10 ? 'low' : '' }}">
                                    @if($item->stock <= 10)
                                        <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.728-.833-2.5 0L4.268 18.5c-.77.833.192 2.5 1.732 2.5z"></path>
                                        </svg>
                                    @else
                                        <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"></path>
                                        </svg>
                                    @endif
                                    {{ number_format($item->stock) }}
                                </div>
                            @else
                                <div class="counter-badge stock-counter unlimited">
                                    <svg class="w-3 h-3" fill="currentColor" viewBox="0 0 24 24">
                                        <path d="M18.6 6.62c-1.44 0-2.8.56-3.77 1.53L12 10.66 10.48 9.14c-.64-.64-1.49-.99-2.4-.99-1.87 0-3.39 1.51-3.39 3.38S6.21 14.91 8.08 14.91c.91 0 1.76-.35 2.4-.99L12 12.41l1.52 1.51c.64.64 1.49.99 2.4.99 1.87 0 3.39-1.51 3.39-3.38s-1.52-3.37-3.39-3.37z"/>
                                    </svg>
                                    ∞
                                </div>
                            @endif

                            <!-- Expiry Counter Badge -->
                            @if($expiryData)
                                <div class="counter-badge expiry-counter {{ $expiryData['isUrgent'] ? 'urgent' : '' }}">
                                    <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                    </svg>
                                    @if($expiryData['days'] > 0)
                                        {{ floor($expiryData['days']) }} <span wire:ignore  data-translate="itemShop.counter.days">Days</span>
                                    @else
                                        {{ floor($expiryData['hours']) }} <span wire:ignore  data-translate="itemShop.counter.hours">Hours</span>
                                    @endif
                                </div>
                            @endif
                            <!-- Item Image -->
                            <div class="item-image-container mb-6 h-48 flex items-center justify-center"
                                 ">
                                <div class="item-rarity-glow"></div>
                                <img src="{{ $this->getImageUrl($item->image) }}"
                                     alt="{{ $item->name }}"
                                     class="item-image max-h-full max-w-full object-contain"
                                     onerror="this.src='{{ asset('assets/images/default-item.svg') }}'">
                                <div class="absolute top-3 right-3">
                                    <!-- SVG Search Plus Icon -->
                                    <svg class="w-5 h-5 text-white opacity-75 hover:opacity-100 transition-opacity" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0zM10 7v3m0 0v3m0-3h3m-3 0H7"></path>
                                    </svg>
                                </div>
                            </div>

                            <!-- Item Info -->
                            <div class="space-y-4">
                                <!-- Name and Type -->
                                <div>
                                    <h3 class="item-name text-xl font-bold mb-2">{{ $item->name }}</h3>
                                    <span class="item-type-badge">{{ ucfirst($item->item_type) }}</span>
                                </div>

                                <!-- Description -->
                                @if($item->description)
                                    <p class="item-description text-sm">{{ Str::limit($item->description, 100) }}</p>
                                @endif

                                <!-- Stock Status -->
                                <div class="flex items-center justify-between">
                                    @if($item->stock == -1)
                                        <span class="stock-status stock-available">
                                            <!-- SVG Infinity Icon -->
                                            <svg class="w-4 h-4" fill="currentColor" viewBox="0 0 24 24">
                                                <path d="M18.6 6.62c-1.44 0-2.8.56-3.77 1.53L12 10.66 10.48 9.14c-.64-.64-1.49-.99-2.4-.99-1.87 0-3.39 1.51-3.39 3.38S6.21 14.91 8.08 14.91c.91 0 1.76-.35 2.4-.99L12 12.41l1.52 1.51c.64.64 1.49.99 2.4.99 1.87 0 3.39-1.51 3.39-3.38s-1.52-3.37-3.39-3.37z"/>
                                            </svg>
                                            <span wire:ignore  data-translate="itemShop.details.unlimited">Unlimited</span>
                                        </span>
                                    @elseif($item->stock > 10)
                                        <span class="stock-status stock-available">
                                            <!-- SVG Check Icon -->
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                                            </svg>
                                            <span wire:ignore  data-translate="itemShop.card.stockAvailable">In Stock</span>
                                        </span>
                                    @elseif($item->stock > 0)
                                        <span class="stock-status stock-low">
                                            <!-- SVG Warning Icon -->
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.728-.833-2.5 0L4.268 18.5c-.77.833.192 2.5 1.732 2.5z"></path>
                                            </svg>
                                            <span wire:ignore  data-translate="itemShop.card.stockLow">Low Stock</span>
                                        </span>
                                    @else
                                        <span class="stock-status stock-out">
                                            <!-- SVG X Icon -->
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                            </svg>
                                            <span wire:ignore  data-translate="itemShop.card.stockOut">Out of Stock</span>
                                        </span>
                                    @endif

                                    <!-- Details Toggle -->
                                    <button @click="showDetails = !showDetails"
                                            class="details-toggle">
                                        <span x-show="!showDetails" wire:ignore  data-translate="itemShop.card.showDetails">Show Details</span>
                                        <span x-show="showDetails" wire:ignore  data-translate="itemShop.card.hideDetails">Hide Details</span>
                                        <!-- SVG Chevron Down Icon -->
                                        <svg class="w-4 h-4 ml-1 transition-transform" :class="showDetails ? 'rotate-180' : ''" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                                        </svg>
                                    </button>
                                </div>

                                <!-- Item Details (Expandable) -->
                                <div x-show="showDetails" x-transition class="item-details">
                                    <div class="space-y-2">
                                        @if($item->Count)
                                            <div class="detail-item">
                                                <span class="detail-label" wire:ignore  data-translate="itemShop.details.count">Count per Purchase:</span>
                                                <span class="detail-value">{{ $item->Count }}</span>
                                            </div>
                                        @endif
                                        <div class="detail-item">
                                            <span class="detail-label" wire:ignore  data-translate="itemShop.details.balanceType">Balance Type:</span>
                                            <span class="detail-value">{{ ucfirst($item->balance_type) }}</span>
                                        </div>
                                        @if($item->sort_type)
                                            <div class="detail-item">
                                                <span class="detail-label" wire:ignore  data-translate="itemShop.details.sortType">Sort Type:</span>
                                                <span class="detail-value">
                                                    @php
                                                        $sortTypeClass = 'bg-blue-500/20 text-blue-300'; // default
                                                        if(strtolower($item->sort_type) === 'premium') {
                                                            $sortTypeClass = 'bg-purple-500/20 text-purple-300';
                                                        } elseif(strtolower($item->sort_type) === 'event') {
                                                            $sortTypeClass = 'bg-orange-500/20 text-orange-300';
                                                        }
                                                    @endphp
                                                    <span class="px-2 py-1 text-xs rounded-full {{ $sortTypeClass }}">
                                                        {{ ucfirst($item->sort_type) }}
                                                    </span>
                                                </span>
                                            </div>
                                        @endif
                                        @if($item->expiry_date)
                                            <div class="detail-item">
                                                <span class="detail-label" wire:ignore  data-translate="itemShop.details.expiryDate">Expiry Date:</span>
                                                <span class="detail-value">
                                                    @if(is_string($item->expiry_date))
                                                        {{ \Carbon\Carbon::parse($item->expiry_date)->format('Y-m-d H:i') }}
                                                    @else
                                                        {{ $item->expiry_date->format('Y-m-d H:i') }}
                                                    @endif
                                                </span>
                                            </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- Expiry Warning -->
                                @if($item->expiry_date)
                                    @php
                                        $expiryDate = is_string($item->expiry_date)
                                            ? \Carbon\Carbon::parse($item->expiry_date)
                                            : $item->expiry_date;
                                        $isExpiring = $expiryDate->isBefore(now()->addDays(7)) && $expiryDate->isAfter(now());
                                    @endphp
                                    @if($isExpiring)
                                        <div class="expiry-warning">
                                            <!-- SVG Clock Icon -->
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                            </svg>
                                            <span wire:ignore  data-translate="itemShop.card.itemExpiring">Expires Soon</span>
                                        </div>
                                    @endif
                                @endif

                                <!-- Price Section -->
                                <div class="price-section">
                                    <div class="flex items-center justify-between mb-4">
                                        <div>
                                            <p class="text-sm text-gray-400 mb-1" wire:ignore  data-translate="itemShop.card.price">Price:</p>
                                            <p class="price-display">💎 {{ number_format($item->price, 2) }}</p>
                                        </div>

                                        <!-- Quantity Controls -->
                                        <div class="quantity-controls">
                                            <button @click="quantity = Math.max(1, quantity - 1)"
                                                    class="quantity-btn"
                                                    :disabled="quantity <= 1">
                                                <!-- SVG Minus Icon -->
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4"></path>
                                                </svg>
                                            </button>
                                            <input type="number"
                                                   x-model.number="quantity"
                                                   class="quantity-input"
                                                   min="1" max="999">
                                            <button @click="quantity = Math.min(999, quantity + 1)"
                                                    class="quantity-btn"
                                                    :disabled="quantity >= 999">
                                                <!-- SVG Plus Icon -->
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                                                </svg>
                                            </button>
                                        </div>
                                    </div>

                                    <!-- Total Price -->
                                    <div class="text-center mb-4">
                                        <p class="text-sm text-gray-400 mb-1" wire:ignore  data-translate="itemShop.card.totalPrice">Total Price:</p>
                                        💎 <b class="total-price" x-text="({{ $item->price }} * quantity).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',')">
                                           {{ number_format($item->price, 2) }}
                                        </b>
                                    </div>

                                    <!-- Buy Button -->
                                    @php
                                        $isExpired = false;
                                        if ($item->expiry_date) {
                                            $expiryDate = is_string($item->expiry_date)
                                                ? \Carbon\Carbon::parse($item->expiry_date)
                                                : $item->expiry_date;
                                            $isExpired = $expiryDate->isPast();
                                        }
                                    @endphp
                                    <button @click="$wire.buyItem({{ $item->id }}, quantity)"
                                            class="buy-button w-full"
                                            :disabled="isLoading || {{ $item->stock == 0 ? 'true' : 'false' }} || {{ $isExpired ? 'true' : 'false' }}">
                                        <span x-show="!isLoading">
                                            @if($item->stock == 0)
                                                <span wire:ignore  data-translate="itemShop.card.outOfStock">Out of Stock</span>
                                            @elseif($isExpired)
                                                <span wire:ignore  data-translate="itemShop.card.expired">Expired</span>
                                            @else
                                                <!-- SVG Shopping Cart Icon -->
                                                <svg class="w-4 h-4 mr-2 inline-block" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4m0 0L7 13m0 0l-2.5-5M7 13l-2.5 5M17 17a2 2 0 100 4 2 2 0 000-4zM9 17a2 2 0 100 4 2 2 0 000-4z"></path>
                                                </svg>
                                                <span wire:ignore  data-translate="itemShop.card.buyNow">Buy Now</span>
                                            @endif
                                        </span>
                                        <span x-show="isLoading">
                                            <!-- SVG Spinner Icon -->
                                            <svg class="w-4 h-4 mr-2 inline-block animate-spin" fill="none" viewBox="0 0 24 24">
                                                <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                                                <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                            </svg>
                                            Processing...
                                        </span>
                                    </button>
                                </div>
                            </div>
                        </div>
                    @endforeach
                </div>

                <!-- Bottom Pagination -->
                @if($filteredItems->hasPages())
                    <div class="bottom-pagination">
                        <div class="pagination-wrapper">
                            {{ $filteredItems->links('pagination.custom') }}
                        </div>
                    </div>
                @endif
            @else
                <!-- No Items State -->
                <div class="text-center py-20">
                    <div class="w-24 h-24 mx-auto mb-6 bg-gradient-to-r from-gray-600 to-gray-800 rounded-3xl flex items-center justify-center">
                        <!-- SVG Box Open Icon -->
                        <svg class="w-12 h-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"></path>
                        </svg>
                    </div>
                    <h3 class="text-2xl font-bold text-white mb-4" wire:ignore  data-translate="itemShop.status.noItems">No items found</h3>
                    <p class="text-gray-400 mb-8" wire:ignore  data-translate="itemShop.status.noItemsDesc">Try adjusting your search or filter criteria</p>
                    <button @click="$wire.resetFilters()"
                            class="reset-button px-8 py-3"
                            wire:ignore  data-translate="itemShop.filters.resetFilters">
                        Reset Filters
                    </button>
                </div>
            @endif
        </div>


    </div>
</div>
