<?php

use Livewire\Volt\Component;
use Illuminate\Support\Facades\Auth;

new class extends Component {
    public $user;
    public $currentPage = '4t-app';
    public $sidebarOpen = false;
    public $dashboardConfig = [];

    public function mount($user = null, $currentPage = null, $sidebarOpen = null, $dashboardConfig = null)
    {
        // Initialize all values independently
        $this->user = Auth::user();

        // Check if there's a saved page in session, otherwise default to '4t-app'
        $savedPage = session('dashboard_current_page', '4t-app');
        $this->currentPage = $savedPage;

        $this->sidebarOpen = false;
        $this->dashboardConfig = $this->loadDashboardConfig();
    }

    public function loadDashboardConfig()
    {
        $configPath = resource_path('data/dashboard-config.json');
        if (file_exists($configPath)) {
            return json_decode(file_get_contents($configPath), true);
        }

        // Fallback configuration
        return $this->getDefaultConfig();
    }

    public function getDefaultConfig()
    {
        return [
            'navigation' => [
                'menu_items' => [
                    [
                        'key' => '4t-app',
                        'title' => '4T-App',
                        'subtitle' => 'Mobile Application',
                        'icon' => '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="5" y="2" width="14" height="20" rx="2" ry="2"/>
                            <path d="M12 18h.01"/>
                        </svg>',
                        'section' => 'main',
                        'include' => 'livewire.backend.pages.4t-app'
                    ]
                ]
            ]
        ];
    }

    public function toggleSidebar()
    {
        $this->sidebarOpen = !$this->sidebarOpen;
        $this->dispatch('sidebarToggled', $this->sidebarOpen);
    }

    public function setCurrentPage($page)
    {
        $this->currentPage = $page;

        // Save the current page in session
        session(['dashboard_current_page' => $page]);
    }

    public function getPageTitle()
    {
        $item = collect($this->dashboardConfig['navigation']['menu_items'])->firstWhere('key', $this->currentPage);
        return $item ? $item['title'] . ' Management' : 'Dashboard';
    }

    public function getMenuItemsBySection($section)
    {
        return collect($this->dashboardConfig['navigation']['menu_items'])->where('section', $section);
    }

    public function getCurrentPageData()
    {
        return collect($this->dashboardConfig['navigation']['menu_items'])->firstWhere('key', $this->currentPage);
    }

    public function getDashboardTitle()
    {
        return $this->dashboardConfig['dashboard']['title'] ?? 'Dashboard';
    }

    public function getDashboardSubtitle()
    {
        return $this->dashboardConfig['dashboard']['subtitle'] ?? 'Control Panel';
    }

    public function getSectionTitle($section)
    {
        $sectionData = collect($this->dashboardConfig['navigation']['sections'])->firstWhere('key', $section);
        return $sectionData['title'] ?? ucfirst($section);
    }

    public function renderMenuItem($item)
    {
        $isActive = $this->currentPage === $item['key'];
        return [
            'isActive' => $isActive,
            'iconClass' => $isActive ? 'active' : 'inactive',
            'arrowClass' => $isActive ? 'active' : 'inactive'
        ];
    }

    public function renderSectionDivider($section)
    {
        return [
            'title' => $this->getSectionTitle($section),
            'translateKey' => "section.{$section}.title"
        ];
    }

    public function getUserAvatarData($size = 'default')
    {
        $sizes = [
            'small' => 'w-4 h-4',
            'default' => 'w-6 h-6',
            'large' => 'w-8 h-8'
        ];

        return [
            'iconSvg' => '<svg class="text-white ' . ($sizes[$size] ?? $sizes['default']) . '" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                <circle cx="12" cy="7" r="4"/>
            </svg>',
            'name' => $this->user->name ?? $this->user->username ?? 'User',
            'email' => $this->user->email ?? 'Player'
        ];
    }

    // Event Listeners for parent-child communication
    protected $listeners = [
        'sidebarToggled' => 'handleSidebarToggle',
        'pageChanged' => 'handlePageChange',
        'forceRefresh' => '$refresh'
    ];

    public function handleSidebarToggle($isOpen)
    {
        $this->sidebarOpen = $isOpen;
    }

    public function handlePageChange($page)
    {
        $this->currentPage = $page;
    }

    // Add refresh capability for real-time updates
    public function refreshComponent()
    {
        $this->dispatch('$refresh');
    }
};
?>
<div x-data="{
    activeTab: @entangle('currentPage'),
    tabTransition: false,

    init() {
        // Load saved page from sessionStorage on init
        const savedPage = sessionStorage.getItem('dashboard_current_page');
        if (savedPage && savedPage !== this.activeTab) {
            this.activeTab = savedPage;
            $wire.setCurrentPage(savedPage);
        }
    },

    switchTab(tabKey) {
        if (this.activeTab !== tabKey) {
            this.tabTransition = true;
            setTimeout(() => {
                this.activeTab = tabKey;
                $wire.setCurrentPage(tabKey);
                sessionStorage.setItem('dashboard_current_page', tabKey);
                setTimeout(() => this.tabTransition = false, 50);
            }, 150);
        }
    }
}" x-cloak class="tablet-dashboard-wrapper">

    <!-- Tablet Dashboard Container -->
    <div class="tablet-dashboard-container tablet-fade-in">



        <!-- Tablet Navigation Tabs -->
        <div class="tablet-dashboard-tabs tablet-slide-up">
            <div class="tablet-dashboard-tabs-container">

                <!-- All Tabs in One Container -->
                <div class="tablet-dashboard-tabs-nav">
                    {{-- Main Section Items --}}
                    @foreach($this->getMenuItemsBySection('main') as $item)
                        <div class="tablet-dashboard-tab {{ $currentPage === $item['key'] ? 'active' : '' }}"
                             x-on:click="switchTab('{{ $item['key'] }}')">

                            <div class="tablet-dashboard-tab-icon">
                                {!! $item['icon'] !!}
                            </div>

                            <div class="tablet-dashboard-tab-content">
                                <h3 wire:ignore  data-translate="{{ $item['translate']['title'] ?? 'menu.' . $item['key'] . '.title' }}">
                                    {{ $item['title'] }}
                                </h3>
                            </div>
                        </div>
                    @endforeach

                    {{-- Shop Section Items --}}
                    @foreach($this->getMenuItemsBySection('shop') as $item)
                        <div class="tablet-dashboard-tab {{ $currentPage === $item['key'] ? 'active' : '' }}"
                             x-on:click="switchTab('{{ $item['key'] }}')">

                            <div class="tablet-dashboard-tab-icon">
                                {!! $item['icon'] !!}
                            </div>

                            <div class="tablet-dashboard-tab-content">
                                <h3 wire:ignore  data-translate="{{ $item['translate']['title'] ?? 'menu.' . $item['key'] . '.title' }}">
                                    {{ $item['title'] }}
                                </h3>
                            </div>
                        </div>
                    @endforeach

                    {{-- Inventory Section Items --}}
                    @foreach($this->getMenuItemsBySection('inventory') as $item)
                        <div class="tablet-dashboard-tab {{ $currentPage === $item['key'] ? 'active' : '' }}"
                             x-on:click="switchTab('{{ $item['key'] }}')">

                            <div class="tablet-dashboard-tab-icon">
                                {!! $item['icon'] !!}
                            </div>

                            <div class="tablet-dashboard-tab-content">
                                <h3 wire:ignore  data-translate="{{ $item['translate']['title'] ?? 'menu.' . $item['key'] . '.title' }}">
                                    {{ $item['title'] }}
                                </h3>
                            </div>
                        </div>
                    @endforeach

                    <!-- Logs Section -->
                    @foreach($this->getMenuItemsBySection('logs') as $item)
                        <div class="tablet-dashboard-tab {{ $currentPage === $item['key'] ? 'active' : '' }}"
                             x-on:click="switchTab('{{ $item['key'] }}')">

                            <div class="tablet-dashboard-tab-icon">
                                {!! $item['icon'] !!}
                            </div>

                            <div class="tablet-dashboard-tab-content">
                                <h3 wire:ignore  data-translate="{{ $item['translate']['title'] ?? 'menu.' . $item['key'] . '.title' }}">
                                    {{ $item['title'] }}
                                </h3>
                            </div>
                        </div>
                    @endforeach
                </div>

            </div>
        </div>

        <!-- Tablet Content Area -->
        <div class="tablet-dashboard-content">
            <div class="tablet-dashboard-content-inner"
                 x-show="!tabTransition"
                 wire:key="tablet-content-{{ $currentPage }}">

                @php $currentPageData = $this->getCurrentPageData(); @endphp

                @if($currentPageData)
                    {{-- Dynamic content loading based on page --}}
                    @if($currentPage === '4t-app')
                        <livewire:backend.pages.4t-app wire:key="tablet-4t-app-component" />
                    @elseif($currentPage === 'quests')
                        <livewire:backend.pages.quests wire:key="tablet-quests-component" />
                    @elseif($currentPage === 'char-list')
                        <livewire:backend.pages.char-list wire:key="tablet-char-list-component" />
                    @elseif($currentPage === 'shipping')
                        <livewire:backend.pages.shipping wire:key="tablet-shipping-component" />
                    @elseif($currentPage === 'packages-shop')
                        <livewire:backend.pages.packages-shop wire:key="tablet-packages-shop-component" />
                    @elseif($currentPage === 'item-shop')
                        <livewire:backend.pages.item-shop wire:key="tablet-item-shop-component" />
                    @elseif($currentPage === 'gift-item-shop')
                        <livewire:backend.pages.gift-item-shop wire:key="tablet-gift-item-shop-component" />
                    @elseif($currentPage === 'players-shop')
                        <livewire:backend.pages.players-shop wire:key="tablet-players-shop-component" />
                    @elseif($currentPage === 'web-inventory')
                        <livewire:backend.pages.web-inventory wire:key="tablet-web-inventory-component" />
                    @elseif($currentPage === 'market-inventory')
                        <livewire:backend.pages.market-inventory wire:key="tablet-market-inventory-component" />
                    @elseif($currentPage === 'market-logs')
                        <livewire:backend.pages.market-logs wire:key="tablet-market-logs-component" />
                    @elseif($currentPage === 'rewards-systems')
                        <livewire:backend.pages.rewards-systems wire:key="tablet-rewards-systems-component" />
                    @else
                        {{-- Fallback include --}}
                        @try
                            @include($currentPageData['include'])
                        @catch(Exception $e)
                            <div class="tablet-dashboard-error">
                                <div class="tablet-dashboard-error-icon">
                                    <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-white">
                                        <path d="M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/>
                                        <line x1="12" y1="9" x2="12" y2="13"/>
                                        <line x1="12" y1="17" x2="12.01" y2="17"/>
                                    </svg>
                                </div>
                                <h3 class="tablet-dashboard-error-title" wire:ignore  data-translate="error.pageNotFound.title">
                                    Include Error
                                </h3>
                                <p class="tablet-dashboard-error-description">
                                    {{ $currentPage }} template could not be loaded.
                                </p>
                            </div>
                        @endtry
                    @endif
                @else
                    {{-- Fallback default content --}}
                    <div class="tablet-dashboard-error">
                        <div class="tablet-dashboard-error-icon">
                            <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-white">
                                <path d="M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/>
                                <line x1="12" y1="9" x2="12" y2="13"/>
                                <line x1="12" y1="17" x2="12.01" y2="17"/>
                            </svg>
                        </div>
                        <h3 class="tablet-dashboard-error-title" wire:ignore  data-translate="error.pageNotFound.title">
                            Page Not Found
                        </h3>
                        <p class="tablet-dashboard-error-description" wire:ignore  data-translate="error.pageNotFound.description">
                            The requested page could not be loaded. Please try again.
                        </p>
                    </div>
                @endif
            </div>

            {{-- Loading State --}}
            <div x-show="tabTransition"
                 class="tablet-dashboard-content-inner">
                <div class="tablet-dashboard-error">
                    <div class="tablet-dashboard-error-icon">
                        <svg class="animate-spin" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-purple-400">
                            <path d="M21 12a9 9 0 11-6.219-8.56"/>
                        </svg>
                    </div>
                    <h3 class="tablet-dashboard-error-title" wire:ignore  data-translate="common.loading">
                        Loading...
                    </h3>
                    <p class="tablet-dashboard-error-description">
                        Switching to {{ $currentPageData['title'] ?? 'Page' }}
                    </p>
                </div>
            </div>
        </div>

    </div>
</div>
