<?php

use Livewire\Volt\Component;
use Illuminate\Support\Facades\Auth;

new class extends Component {
    public $user;
    public $currentPage = '4t-app';
    public $sidebarOpen = false;
    public $dashboardConfig = [];

    public function mount($user = null, $currentPage = null, $sidebarOpen = null, $dashboardConfig = null)
    {
        // Initialize all values independently
        $this->user = Auth::user();

        // Check if there's a saved page in session, otherwise default to '4t-app'
        $savedPage = session('dashboard_current_page', '4t-app');
        $this->currentPage = $savedPage;

        $this->sidebarOpen = false;
        $this->dashboardConfig = $this->loadDashboardConfig();
    }

    public function loadDashboardConfig()
    {
        $configPath = resource_path('data/dashboard-config.json');
        if (file_exists($configPath)) {
            return json_decode(file_get_contents($configPath), true);
        }

        // Fallback configuration
        return $this->getDefaultConfig();
    }

    public function getDefaultConfig()
    {
        return [
            'navigation' => [
                'menu_items' => [
                    [
                        'key' => '4t-app',
                        'title' => '4T-App',
                        'subtitle' => 'Mobile Application',
                        'icon' => '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="5" y="2" width="14" height="20" rx="2" ry="2"/>
                            <path d="M12 18h.01"/>
                        </svg>',
                        'section' => 'main',
                        'include' => 'livewire.backend.pages.4t-app'
                    ]
                ]
            ]
        ];
    }

    public function toggleSidebar()
    {
        $this->sidebarOpen = !$this->sidebarOpen;
        $this->dispatch('sidebarToggled', $this->sidebarOpen);
    }

    public function setCurrentPage($page)
    {
        $this->currentPage = $page;

        // Save the current page in session
        session(['dashboard_current_page' => $page]);
    }

    public function getPageTitle()
    {
        $item = collect($this->dashboardConfig['navigation']['menu_items'])->firstWhere('key', $this->currentPage);
        return $item ? $item['title'] . ' Management' : 'Dashboard';
    }

    public function getMenuItemsBySection($section)
    {
        return collect($this->dashboardConfig['navigation']['menu_items'])->where('section', $section);
    }

    public function getCurrentPageData()
    {
        return collect($this->dashboardConfig['navigation']['menu_items'])->firstWhere('key', $this->currentPage);
    }

    public function getDashboardTitle()
    {
        return $this->dashboardConfig['dashboard']['title'] ?? 'Dashboard';
    }

    public function getDashboardSubtitle()
    {
        return $this->dashboardConfig['dashboard']['subtitle'] ?? 'Control Panel';
    }

    public function getSectionTitle($section)
    {
        $sectionData = collect($this->dashboardConfig['navigation']['sections'])->firstWhere('key', $section);
        return $sectionData['title'] ?? ucfirst($section);
    }

    public function renderMenuItem($item)
    {
        $isActive = $this->currentPage === $item['key'];
        return [
            'isActive' => $isActive,
            'iconClass' => $isActive ? 'active' : 'inactive',
            'arrowClass' => $isActive ? 'active' : 'inactive'
        ];
    }

    public function renderSectionDivider($section)
    {
        return [
            'title' => $this->getSectionTitle($section),
            'translateKey' => "section.{$section}.title"
        ];
    }

    public function getUserAvatarData($size = 'default')
    {
        $sizes = [
            'small' => 'w-4 h-4',
            'default' => 'w-6 h-6',
            'large' => 'w-8 h-8'
        ];

        return [
            'iconSvg' => '<svg class="text-white ' . ($sizes[$size] ?? $sizes['default']) . '" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                <circle cx="12" cy="7" r="4"/>
            </svg>',
            'name' => $this->user->name ?? $this->user->username ?? 'User',
            'email' => $this->user->email ?? 'Player'
        ];
    }

    // Event Listeners for parent-child communication
    protected $listeners = [
        'sidebarToggled' => 'handleSidebarToggle',
        'pageChanged' => 'handlePageChange',
        'forceRefresh' => '$refresh'
    ];

    public function handleSidebarToggle($isOpen)
    {
        $this->sidebarOpen = $isOpen;
    }

    public function handlePageChange($page)
    {
        $this->currentPage = $page;
    }

    // Add refresh capability for real-time updates
    public function refreshComponent()
    {
        $this->dispatch('$refresh');
    }
};
?>
<div x-data="{
    activeTab: @entangle('currentPage'),
    menuOpen: false,
    slideTransition: false,
    animationLoaded: false,

    init() {
        this.animationLoaded = true;
        // Load saved page from sessionStorage on init
        const savedPage = sessionStorage.getItem('dashboard_current_page');
        if (savedPage && savedPage !== this.activeTab) {
            this.activeTab = savedPage;
            $wire.setCurrentPage(savedPage);
        }
    },

    switchTab(tabKey) {
        if (this.activeTab !== tabKey) {
            this.slideTransition = true;
            setTimeout(() => {
                this.activeTab = tabKey;
                $wire.setCurrentPage(tabKey);
                sessionStorage.setItem('dashboard_current_page', tabKey);
                this.menuOpen = false;
                setTimeout(() => this.slideTransition = false, 100);
            }, 200);
        }
    },

    toggleMenu() {
        this.menuOpen = !this.menuOpen;
    }
}"
x-cloak
class="mobile-dashboard-wrapper"
:class="{ 'mobile-menu-open': menuOpen }">

    <!-- Mobile Dashboard Container -->
    <div class="mobile-dashboard-container" :class="{ 'mobile-fade-in': animationLoaded }">

        <!-- Mobile Header -->
        <div class="mobile-dashboard-header mobile-slide-down" x-data="{ headerAnimating: false }">
            <div class="mobile-dashboard-header-content">
                <div class="mobile-dashboard-header-left"
                     x-effect="if (slideTransition) { headerAnimating = true; setTimeout(() => headerAnimating = false, 300); }"
                     :class="{ 'mobile-header-fade': headerAnimating }">
                    @php $currentPageData = $this->getCurrentPageData(); @endphp
                    <h1 wire:ignore  data-translate="{{ $currentPageData['translate']['title'] ?? 'menu.' . $currentPage . '.title' }}">
                        {{ $currentPageData['title'] ?? 'Current Page' }}
                    </h1>
                    <p wire:ignore  data-translate="{{ $currentPageData['translate']['description'] ?? 'menu.' . $currentPage . '.description' }}">
                        {{ $currentPageData['subtitle'] ?? $this->getPageTitle() }}
                    </p>
                </div>

                <div class="mobile-dashboard-header-right">


                    <!-- Mobile Menu Toggle -->
                    <button @click="toggleMenu()" class="mobile-menu-toggle">
                        <span class="mobile-menu-hamburger">
                            <span></span>
                            <span></span>
                            <span></span>
                        </span>
                    </button>
                </div>
            </div>
        </div>

        <!-- Mobile Navigation Menu Overlay -->
        <div class="mobile-dashboard-menu-overlay"
             x-show="menuOpen"
             x-transition:enter="mobile-overlay-enter"
             x-transition:enter-end="mobile-overlay-enter-end"
             x-transition:leave="mobile-overlay-leave-end"
             @click="menuOpen = false">
        </div>

        <!-- Mobile Navigation Menu -->
        <div class="mobile-dashboard-menu"
             x-show="menuOpen"
             x-transition:enter="mobile-menu-enter"
             x-transition:enter-end="mobile-menu-enter-end"
             x-transition:leave="mobile-menu-leave-end">

            <div class="mobile-dashboard-menu-header">
                <h3 wire:ignore  data-translate="menu.navigation.title">Navigation</h3>
                <button @click="menuOpen = false" class="mobile-menu-close">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <line x1="18" y1="6" x2="6" y2="18"/>
                        <line x1="6" y1="6" x2="18" y2="18"/>
                    </svg>
                </button>
            </div>

            <div class="mobile-dashboard-menu-content">
                {{-- Main Section Items --}}
                <div class="mobile-menu-section">
                    <h4 wire:ignore  data-translate="section.main.title" class="mobile-menu-section-title">Main Management</h4>
                    <div class="mobile-menu-items">
                        @foreach($this->getMenuItemsBySection('main') as $item)
                            <div class="mobile-menu-item {{ $currentPage === $item['key'] ? 'active' : '' }}"
                                 @click="switchTab('{{ $item['key'] }}')">
                                <div class="mobile-menu-item-icon">
                                    {!! $item['icon'] !!}
                                </div>
                                <div class="mobile-menu-item-content">
                                    <h5 wire:ignore  data-translate="{{ $item['translate']['title'] ?? 'menu.' . $item['key'] . '.title' }}">
                                        {{ $item['title'] }}
                                    </h5>
                                    <p wire:ignore  data-translate="{{ $item['translate']['description'] ?? 'menu.' . $item['key'] . '.description' }}">
                                        {{ $item['subtitle'] ?? 'Mobile Application' }}
                                    </p>
                                </div>
                                <div class="mobile-menu-item-arrow">
                                    <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M9 18l6-6-6-6"/>
                                    </svg>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>

                {{-- Shop Section Items --}}
                <div class="mobile-menu-section">
                    <h4 wire:ignore  data-translate="section.shop.title" class="mobile-menu-section-title">Shop Management</h4>
                    <div class="mobile-menu-items">
                        @foreach($this->getMenuItemsBySection('shop') as $item)
                            <div class="mobile-menu-item {{ $currentPage === $item['key'] ? 'active' : '' }}"
                                 @click="switchTab('{{ $item['key'] }}')">
                                <div class="mobile-menu-item-icon">
                                    {!! $item['icon'] !!}
                                </div>
                                <div class="mobile-menu-item-content">
                                    <h5 wire:ignore  data-translate="{{ $item['translate']['title'] ?? 'menu.' . $item['key'] . '.title' }}">
                                        {{ $item['title'] }}
                                    </h5>
                                    <p wire:ignore  data-translate="{{ $item['translate']['description'] ?? 'menu.' . $item['key'] . '.description' }}">
                                        {{ $item['subtitle'] ?? 'Mobile Application' }}
                                    </p>
                                </div>
                                <div class="mobile-menu-item-arrow">
                                    <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M9 18l6-6-6-6"/>
                                    </svg>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>

                {{-- Inventory Section Items --}}
                <div class="mobile-menu-section">
                    <h4 wire:ignore  data-translate="section.inventory.title" class="mobile-menu-section-title">Inventory</h4>
                    <div class="mobile-menu-items">
                        @foreach($this->getMenuItemsBySection('inventory') as $item)
                            <div class="mobile-menu-item {{ $currentPage === $item['key'] ? 'active' : '' }}"
                                 @click="switchTab('{{ $item['key'] }}')">
                                <div class="mobile-menu-item-icon">
                                    {!! $item['icon'] !!}
                                </div>
                                <div class="mobile-menu-item-content">
                                    <h5 wire:ignore  data-translate="{{ $item['translate']['title'] ?? 'menu.' . $item['key'] . '.title' }}">
                                        {{ $item['title'] }}
                                    </h5>
                                    <p wire:ignore  data-translate="{{ $item['translate']['description'] ?? 'menu.' . $item['key'] . '.description' }}">
                                        {{ $item['subtitle'] ?? 'Mobile Application' }}
                                    </p>
                                </div>
                                <div class="mobile-menu-item-arrow">
                                    <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M9 18l6-6-6-6"/>
                                    </svg>
                                </div>
                            </div>
                        @endforeach

                        <!-- Logs Section -->
                        <div class="mobile-menu-section">
                            <h4 class="mobile-menu-section-title">
                                <span wire:ignore  data-translate="section.logs.title">Logs & Analytics</span>
                            </h4>
                        </div>

                        @foreach($this->getMenuItemsBySection('logs') as $item)
                            <div class="mobile-menu-item {{ $currentPage === $item['key'] ? 'active' : '' }}"
                                 @click="switchTab('{{ $item['key'] }}')">
                                <div class="mobile-menu-item-icon">
                                    {!! $item['icon'] !!}
                                </div>
                                <div class="mobile-menu-item-content">
                                    <h5 wire:ignore  data-translate="{{ $item['translate']['title'] ?? 'menu.' . $item['key'] . '.title' }}">
                                        {{ $item['title'] }}
                                    </h5>
                                    <p wire:ignore  data-translate="{{ $item['translate']['description'] ?? 'menu.' . $item['key'] . '.description' }}">
                                        {{ $item['subtitle'] ?? 'Mobile Application' }}
                                    </p>
                                </div>
                                <div class="mobile-menu-item-arrow">
                                    <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M9 18l6-6-6-6"/>
                                    </svg>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>
            </div>
        </div>

        <!-- Mobile Content Area -->
        <div class="mobile-dashboard-content">
            <div class="mobile-dashboard-content-inner"
                 x-show="!slideTransition"
                 wire:key="mobile-content-{{ $currentPage }}">

                @php $currentPageData = $this->getCurrentPageData(); @endphp

                @if($currentPageData)
                    {{-- Dynamic content loading based on page --}}
                    @if($currentPage === '4t-app')
                        <livewire:backend.pages.4t-app wire:key="mobile-4t-app-component" />
                    @elseif($currentPage === 'quests')
                        <livewire:backend.pages.quests wire:key="mobile-quests-component" />
                    @elseif($currentPage === 'char-list')
                        <livewire:backend.pages.char-list wire:key="mobile-char-list-component" />
                    @elseif($currentPage === 'shipping')
                        <livewire:backend.pages.shipping wire:key="mobile-shipping-component" />
                    @elseif($currentPage === 'packages-shop')
                        <livewire:backend.pages.packages-shop wire:key="mobile-packages-shop-component" />
                    @elseif($currentPage === 'item-shop')
                        <livewire:backend.pages.item-shop wire:key="mobile-item-shop-component" />
                    @elseif($currentPage === 'gift-item-shop')
                        <livewire:backend.pages.gift-item-shop wire:key="mobile-gift-item-shop-component" />
                    @elseif($currentPage === 'players-shop')
                        <livewire:backend.pages.players-shop wire:key="mobile-players-shop-component" />
                    @elseif($currentPage === 'web-inventory')
                        <livewire:backend.pages.web-inventory wire:key="mobile-web-inventory-component" />
                    @elseif($currentPage === 'market-inventory')
                        <livewire:backend.pages.market-inventory wire:key="mobile-market-inventory-component" />
                    @elseif($currentPage === 'market-logs')
                        <livewire:backend.pages.market-logs wire:key="mobile-market-logs-component" />
                    @elseif($currentPage === 'rewards-systems')
                        <livewire:backend.pages.rewards-systems wire:key="mobile-rewards-systems-component" />
                    @else
                        {{-- Fallback include --}}
                        @try
                            @include($currentPageData['include'])
                        @catch(Exception $e)
                            <div class="mobile-dashboard-error">
                                <div class="mobile-dashboard-error-icon">
                                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/>
                                        <line x1="12" y1="9" x2="12" y2="13"/>
                                        <line x1="12" y1="17" x2="12.01" y2="17"/>
                                    </svg>
                                </div>
                                <h3 class="mobile-dashboard-error-title" wire:ignore  data-translate="error.pageNotFound.title">
                                    Include Error
                                </h3>
                                <p class="mobile-dashboard-error-description" wire:ignore  data-translate="error.include.message">
                                    {{ $currentPage }} template could not be loaded.
                                </p>
                            </div>
                        @endtry
                    @endif
                @else
                    {{-- Fallback default content --}}
                    <div class="mobile-dashboard-error">
                        <div class="mobile-dashboard-error-icon">
                            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/>
                                <line x1="12" y1="9" x2="12" y2="13"/>
                                <line x1="12" y1="17" x2="12.01" y2="17"/>
                            </svg>
                        </div>
                        <h3 class="mobile-dashboard-error-title" wire:ignore  data-translate="error.pageNotFound.title">
                            Page Not Found
                        </h3>
                        <p class="mobile-dashboard-error-description" wire:ignore  data-translate="error.pageNotFound.description">
                            The requested page could not be loaded. Please try again.
                        </p>
                    </div>
                @endif
            </div>

            {{-- Loading State --}}
            <div x-show="slideTransition" class="mobile-dashboard-content-loading">
                <div class="mobile-loading-container">
                    <div class="mobile-loading-icon">
                        <svg class="mobile-bounce" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="2" y="5" width="20" height="14" rx="2" ry="2"/>
                            <path d="M2 17h20"/>
                            <path d="M6 21h12"/>
                            <path d="M12 17v4"/>
                        </svg>
                    </div>
                    <h3 class="mobile-loading-title" wire:ignore  data-translate="common.loading">
                        Loading...
                    </h3>
                    <p class="mobile-loading-description" wire:ignore  data-translate="mobile.switching.page">
                        Switching to {{ $currentPageData['title'] ?? 'Page' }}
                    </p>
                    <div class="mobile-loading-dots">
                        <span></span>
                        <span></span>
                        <span></span>
                    </div>
                </div>
            </div>
        </div>

    </div>
</div>
