<?php

use Livewire\Volt\Component;
use Illuminate\Support\Facades\Auth;

new class extends Component {
    public $user;
    public $currentPage = '4t-app';
    public $sidebarOpen = false;
    public $dashboardConfig = [];

    public function mount($user = null, $currentPage = null, $sidebarOpen = null, $dashboardConfig = null)
    {
        // Initialize all values independently
        $this->user = Auth::user();

        // Check if there's a saved page in session, otherwise default to '4t-app'
        $savedPage = session('dashboard_current_page', '4t-app');
        $this->currentPage = $savedPage;

        $this->sidebarOpen = false;
        $this->dashboardConfig = $this->loadDashboardConfig();
    }

    public function loadDashboardConfig()
    {
        $configPath = resource_path('data/dashboard-config.json');
        if (file_exists($configPath)) {
            return json_decode(file_get_contents($configPath), true);
        }

        // Fallback configuration
        return $this->getDefaultConfig();
    }

    public function getDefaultConfig()
    {
        return [
            'navigation' => [
                'menu_items' => [
                    [
                        'key' => '4t-app',
                        'title' => '4T-App',
                        'subtitle' => 'Mobile Application',
                        'icon' => '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="5" y="2" width="14" height="20" rx="2" ry="2"/>
                            <path d="M12 18h.01"/>
                        </svg>',
                        'section' => 'main',
                        'include' => 'livewire.backend.pages.4t-app'
                    ]
                ]
            ]
        ];
    }

    public function toggleSidebar()
    {
        $this->sidebarOpen = !$this->sidebarOpen;
        $this->dispatch('sidebarToggled', $this->sidebarOpen);
    }

    public function setCurrentPage($page)
    {
        $this->currentPage = $page;

        // Save the current page in session
        session(['dashboard_current_page' => $page]);
    }

    public function getPageTitle()
    {
        $item = collect($this->dashboardConfig['navigation']['menu_items'])->firstWhere('key', $this->currentPage);
        return $item ? $item['title'] . ' Management' : 'Dashboard';
    }

    public function getMenuItemsBySection($section)
    {
        return collect($this->dashboardConfig['navigation']['menu_items'])->where('section', $section);
    }

    public function getCurrentPageData()
    {
        return collect($this->dashboardConfig['navigation']['menu_items'])->firstWhere('key', $this->currentPage);
    }

    public function getDashboardTitle()
    {
        return $this->dashboardConfig['dashboard']['title'] ?? 'Dashboard';
    }

    public function getDashboardSubtitle()
    {
        return $this->dashboardConfig['dashboard']['subtitle'] ?? 'Control Panel';
    }

    public function getSectionTitle($section)
    {
        $sectionData = collect($this->dashboardConfig['navigation']['sections'])->firstWhere('key', $section);
        return $sectionData['title'] ?? ucfirst($section);
    }

    public function renderMenuItem($item)
    {
        $isActive = $this->currentPage === $item['key'];
        return [
            'isActive' => $isActive,
            'iconClass' => $isActive ? 'active' : 'inactive',
            'arrowClass' => $isActive ? 'active' : 'inactive'
        ];
    }

    public function renderSectionDivider($section)
    {
        return [
            'title' => $this->getSectionTitle($section),
            'translateKey' => "section.{$section}.title"
        ];
    }

    public function getUserAvatarData($size = 'default')
    {
        $sizes = [
            'small' => 'w-4 h-4',
            'default' => 'w-6 h-6',
            'large' => 'w-8 h-8'
        ];

        return [
            'iconSvg' => '<svg class="text-white ' . ($sizes[$size] ?? $sizes['default']) . '" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                <circle cx="12" cy="7" r="4"/>
            </svg>',
            'name' => $this->user->name ?? $this->user->username ?? 'User',
            'email' => $this->user->email ?? 'Player'
        ];
    }

    // Event Listeners for parent-child communication
    protected $listeners = [
        'sidebarToggled' => 'handleSidebarToggle',
        'pageChanged' => 'handlePageChange',
        'forceRefresh' => '$refresh'
    ];

    public function handleSidebarToggle($isOpen)
    {
        $this->sidebarOpen = $isOpen;
    }

    public function handlePageChange($page)
    {
        $this->currentPage = $page;
    }

    // Add refresh capability for real-time updates
    public function refreshComponent()
    {
        $this->dispatch('$refresh');
    }
};
?>
<div x-data="{
    sidebarOpen: @entangle('sidebarOpen'),
    currentPage: @entangle('currentPage'),

    init() {
        // Load saved page from sessionStorage on init
        const savedPage = sessionStorage.getItem('dashboard_current_page');
        if (savedPage && savedPage !== this.currentPage) {
            this.currentPage = savedPage;
            $wire.setCurrentPage(savedPage);
        }

        // Watch for currentPage changes and save to sessionStorage
        this.$watch('currentPage', (newPage) => {
            sessionStorage.setItem('dashboard_current_page', newPage);
        });
    },

    setPage(page) {
        this.currentPage = page;
        $wire.setCurrentPage(page);
        sessionStorage.setItem('dashboard_current_page', page);
    }
}" x-cloak>
    <div class="desktop-dashboard-wrapper">

    <!-- Desktop Dashboard Container -->
    <div class="desktop-dashboard-container">
        <!-- Desktop Sidebar -->
        <div class="desktop-dashboard-sidebar desktop-dashboard-sidebar-glow lg:block hidden"
             x-show="true">


            <!-- Desktop Navigation Menu -->
            <nav class="desktop-dashboard-nav">
                <!-- Main Section -->
                @foreach($this->getMenuItemsBySection('main') as $item)
                    @php $menuData = $this->renderMenuItem($item); @endphp
                    <button x-on:click="setPage('{{ $item['key'] }}')"
                            class="desktop-dashboard-menu-item {{ $menuData['isActive'] ? 'active' : '' }} desktop-tooltip desktop-glow-effect"
                            data-tooltip="{{ $item['subtitle'] }}">
                        <div class="desktop-dashboard-menu-content">
                            <div class="desktop-dashboard-menu-icon {{ $menuData['iconClass'] }}">
                                {!! $item['icon'] !!}
                            </div>
                            <div class="desktop-dashboard-menu-text">
                                <span class="desktop-dashboard-menu-title" wire:ignore  data-translate="{{ $item['translate']['title'] ?? 'menu.' . $item['key'] . '.title' }}">{{ $item['title'] }}</span>
                                <p class="desktop-dashboard-menu-subtitle" wire:ignore  data-translate="{{ $item['translate']['subtitle'] ?? 'menu.' . $item['key'] . '.subtitle' }}">{{ $item['subtitle'] }}</p>
                            </div>
                            <svg class="desktop-dashboard-menu-arrow {{ $menuData['arrowClass'] }}" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M9 18l6-6-6-6"/>
                            </svg>
                        </div>
                    </button>
                @endforeach

                @foreach(['shop', 'inventory', 'logs'] as $section)
                    @if($this->getMenuItemsBySection($section)->count() > 0)
                        @php $sectionData = $this->renderSectionDivider($section); @endphp
                        <!-- Desktop {{ ucfirst($section) }} Section Divider -->
                        <div class="desktop-dashboard-divider desktop-slide-up">
                            <div class="desktop-dashboard-divider-content">
                                <div class="desktop-dashboard-divider-line"></div>
                                <span class="desktop-dashboard-divider-text" wire:ignore  data-translate="{{ $sectionData['translateKey'] }}">{{ $sectionData['title'] }}</span>
                                <div class="desktop-dashboard-divider-line"></div>
                            </div>
                        </div>

                        <!-- Desktop {{ ucfirst($section) }} Section -->
                        @foreach($this->getMenuItemsBySection($section) as $item)
                            @php $menuData = $this->renderMenuItem($item); @endphp
                            <button x-on:click="setPage('{{ $item['key'] }}')"
                                    class="desktop-dashboard-menu-item {{ $menuData['isActive'] ? 'active' : '' }} desktop-tooltip desktop-glow-effect"
                                    data-tooltip="{{ $item['subtitle'] }}">
                                <div class="desktop-dashboard-menu-content">
                                    <div class="desktop-dashboard-menu-icon {{ $menuData['iconClass'] }}">
                                        {!! $item['icon'] !!}
                                    </div>
                                    <div class="desktop-dashboard-menu-text">
                                        <span class="desktop-dashboard-menu-title" wire:ignore  data-translate="{{ $item['translate']['title'] ?? 'menu.' . $item['key'] . '.title' }}">{{ $item['title'] }}</span>
                                        <p class="desktop-dashboard-menu-subtitle" wire:ignore  data-translate="{{ $item['translate']['subtitle'] ?? 'menu.' . $item['key'] . '.subtitle' }}">{{ $item['subtitle'] }}</p>
                                    </div>
                                    <svg class="desktop-dashboard-menu-arrow {{ $menuData['arrowClass'] }}" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M9 18l6-6-6-6"/>
                                    </svg>
                                </div>
                            </button>
                        @endforeach
                    @endif
                @endforeach
            </nav>
        </div>


        <!-- Desktop Content Area -->
        <div class="desktop-dashboard-content dashboard-content-responsive">
            <!-- Desktop Top Bar -->
            <div class="hidden lg:block desktop-dashboard-topbar">
                <div class="desktop-dashboard-topbar-content">
                    <div class="desktop-dashboard-topbar-left">
                        <h1   data-translate="page.{{ $this->currentPage }}.title">{{ $this->getPageTitle() }}</h1>
                        <p  data-translate="page.{{ $this->currentPage }}.description">{{ $this->getPageTitle() }} Management System</p>

                        {{-- Debug info - Current Page --}}
                        <small class="text-gray-400 text-xs">Current Page: {{ $currentPage }} | Time: {{ now()->format('H:i:s') }}</small>
                    </div>
                </div>
            </div>

            <!-- Desktop Page Content -->
            <div class="desktop-dashboard-page">
                <div class="max-w-7xl mx-auto">
                    <!-- Content will be loaded here based on selected page -->
                    <div class="desktop-dashboard-main-card"
                         wire:key="page-{{ $currentPage }}"
                         x-show="true">
                        @php
                            $currentPageData = $this->getCurrentPageData();
                        @endphp

                        @if($currentPageData)
                            {{-- Dynamic content loading based on component or include --}}
                            <div wire:key="content-{{ $currentPage }}">
                                @if($currentPage === '4t-app')
                                    <livewire:backend.pages.4t-app wire:key="4t-app-component" />
                                @elseif($currentPage === 'quests')
                                    <livewire:backend.pages.quests wire:key="quests-component" />
                                @elseif($currentPage === 'char-list')
                                    <livewire:backend.pages.char-list wire:key="char-list-component" />
                                @elseif($currentPage === 'shipping')
                                    <livewire:backend.pages.shipping wire:key="shipping-component" />
                                @elseif($currentPage === 'packages-shop')
                                    <livewire:backend.pages.packages-shop wire:key="packages-shop-component" />
                                @elseif($currentPage === 'item-shop')
                                    <livewire:backend.pages.item-shop wire:key="item-shop-component" />
                                @elseif($currentPage === 'gift-item-shop')
                                    <livewire:backend.pages.gift-item-shop wire:key="gift-item-shop-component" />
                                @elseif($currentPage === 'players-shop')
                                    <livewire:backend.pages.players-shop wire:key="players-shop-component" />
                                @elseif($currentPage === 'web-inventory')
                                    <livewire:backend.pages.web-inventory wire:key="web-inventory-component" />
                                @elseif($currentPage === 'market-inventory')
                                    <livewire:backend.pages.market-inventory wire:key="market-inventory-component" />
                                @elseif($currentPage === 'market-logs')
                                    <livewire:backend.pages.market-logs wire:key="market-logs-component" />
                                @elseif($currentPage === 'rewards-systems')
                                    <livewire:backend.pages.rewards-systems wire:key="rewards-systems-component" />
                                @else
                                    {{-- Fallback include --}}
                                    @try
                                        @include($currentPageData['include'])
                                    @catch(Exception $e)
                                        <div class="desktop-dashboard-error">
                                            <div class="desktop-dashboard-error-icon">
                                                <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-white">
                                                    <path d="M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/>
                                                    <line x1="12" y1="9" x2="12" y2="13"/>
                                                    <line x1="12" y1="17" x2="12.01" y2="17"/>
                                                </svg>
                                            </div>
                                            <h3 class="desktop-dashboard-error-title">Include Error</h3>
                                            <p class="desktop-dashboard-error-description">{{ $currentPage }} template could not be loaded.</p>
                                        </div>
                                    @endtry
                                @endif
                            </div>
                        @else
                            {{-- Fallback default content --}}
                            <div class="desktop-dashboard-error">
                                <div class="desktop-dashboard-error-icon">
                                    <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-white">
                                        <path d="M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/>
                                        <line x1="12" y1="9" x2="12" y2="13"/>
                                        <line x1="12" y1="17" x2="12.01" y2="17"/>
                                    </svg>
                                </div>
                                <h3 class="desktop-dashboard-error-title" wire:ignore  data-translate="error.pageNotFound.title">Page Not Found</h3>
                                <p class="desktop-dashboard-error-description" wire:ignore  data-translate="error.pageNotFound.description">The requested page could not be loaded. Please try again.</p>
                            </div>
                        @endif
                    </div>
                </div>
            </div>
        </div>
    </div>
    </div>
</div>
