<?php

use Livewire\Volt\Component;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Hash;
use Illuminate\Auth\Events\Registered;
use App\Models\SRO_VT_ACCOUNT\SkSilk;
use App\Models\SRO_VT_ACCOUNT\TbUser;
use App\Models\Z_Website\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Helpers\LanguageHelper;

new class extends Component {
    #[Validate('required|string|alpha_num|min:6|max:16')]
    public string $StrUserID = '';

    #[Validate('required|string|email|max:255')]
    public string $email = '';

    #[Validate('required|string|min:6|confirmed')]
    public string $sro_password = '';

    #[Validate('required|string|min:6')]
    public string $sro_password_confirmation = '';

    public bool $terms = false;

    #[Validate('required|string')]
    public string $lang = '';

    public $activeModal = null;

    public function mount()
    {
        $this->activeModal = null;
    // Initialize lang from server-side detection so registration uses current page language by default
    $this->lang = \App\Helpers\LanguageHelper::determineUserLanguage();
    }

    public function openRegisterModal()
    {
        $this->activeModal = 'register';
        \Log::info('Register Modal Opened', ['activeModal' => $this->activeModal]);
    }

    public function openModal()
    {
        $this->openRegisterModal();
    }

    #[\Livewire\Attributes\On('modal-switch')]
    public function handleModalSwitch($data)
    {
        if (isset($data['target']) && $data['target'] === 'register' &&
            isset($data['key']) && isset($data['source']) && $data['source'] === 'login') {

            $sessionData = session()->get($data['key']);
            if ($sessionData && $sessionData['action'] === 'open_register') {
                session()->forget($data['key']);
                $this->openRegisterModal();
            }
        }
    }

    public function switchToLogin()
    {
        $switchKey = 'modal_switch_' . uniqid();
        session()->put($switchKey, [
            'action' => 'open_login',
            'from' => 'register',
            'timestamp' => now()->timestamp
        ]);

        $this->activeModal = null;
        $this->reset(['StrUserID', 'email', 'sro_password', 'sro_password_confirmation', 'terms']);
        $this->resetErrorBag();

        $this->dispatch('modal-switch', [
            'target' => 'login',
            'key' => $switchKey,
            'source' => 'register'
        ])->to('auth.login');
    }

    public function closeModal()
    {
        $this->activeModal = null;
        $this->reset(['StrUserID', 'email', 'sro_password', 'sro_password_confirmation', 'terms']);
        $this->resetErrorBag();
        Log::info('Register Modal Closed');
    }

    public function register()
    {
        try {
            // Validate the input
            $validatedData = $this->validate([
                'StrUserID' => 'required|string|alpha_num|min:6|max:16|unique:sqlsrv_account.SRO_VT_ACCOUNT.dbo.TB_User,StrUserID',
                'email' => 'required|string|email|max:255',
                'sro_password' => 'required|string|min:6|confirmed',
                'terms' => 'accepted',
            ], [
                'StrUserID.required' => 'authentication.username_required',
                'StrUserID.alpha_num' => 'authentication.username_alphanumeric',
                'StrUserID.min' => 'authentication.username_min_length',
                'StrUserID.max' => 'authentication.username_max_length',
                'StrUserID.unique' => 'authentication.username_exists',
                'email.required' => 'authentication.email_required',
                'email.email' => 'authentication.email_invalid',
                'sro_password.required' => 'authentication.password_required',
                'sro_password.min' => 'authentication.password_min_length',
                'sro_password.confirmed' => 'authentication.password_mismatch',
                'terms.accepted' => 'authentication.terms_required',
            ]);

            // Start database transaction
            DB::beginTransaction();

            // Create the game account user
            $tbUser = TbUser::create([
                'StrUserID' => strtolower($this->StrUserID),
                'Name' => $this->StrUserID,
                'password' => md5($this->sro_password),
                'certificate_num' => $this->sro_password,
                'Status' => 1,
                'GMrank' => 0,
                'Email' => $this->email,
                'regtime' => Carbon::now()->format('Y-m-d H:i:s'),
                'reg_ip' => request()->ip(),
                'sec_primary' => 3,
                'sec_content' => 3
            ]);

            if (!$tbUser || !$tbUser->JID) {
                throw new \Exception(trans('auth.account_creation_failed'));
            }

            // Create silk record
            $skSilk = SkSilk::create([
                'JID' => $tbUser->JID,
                'silk_own' => 0,
                'silk_gift' => 0,
                'silk_point' => 0
            ]);

            if (!$skSilk) {
                throw new \Exception(trans('auth.silk_record_failed'));
            }

            // Create website user
            $langFromRequest = trim((string) request()->input('lang', ''));
            $finalLang = $langFromRequest !== '' ? $langFromRequest : ($this->lang ?: LanguageHelper::determineUserLanguage());

            $webUser = User::create([
                'JID' => $tbUser->JID,
                'username' => $this->StrUserID,
                'balance' => 0,
                'gift_balance' => 0,
                'Active' => 1,
                'is_dark' => false,
                // Use the language selected on the page (bound to the Livewire property)
                'lang' => $finalLang,
                'role_id' => 2,
            ]);

            if (!$webUser) {
                throw new \Exception(trans('auth.website_account_failed'));
            }

            // Commit transaction
            DB::commit();

            // Login the user using the game account (TbUser) since that's what our auth system uses
            Auth::login($tbUser);

            // Ensure session is regenerated and mirror the login component's session value
            // so the session is preserved consistently after registration.
            session()->regenerate();

            // Show success notification
            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'trans:auth.register_success',
                'duration' => 5000,
                'params' => ['username' => $this->StrUserID]
            ]);

            // Dispatch event to preserve language after success
            $this->dispatch('preserve-language-state');

            $this->closeModal();

            //$this->redirectIntended(default: route('home', absolute: false), navigate: true);
            // Just refresh the page instead of redirecting
            $this->js('window.location.reload()');

        } catch (\Illuminate\Validation\ValidationException $e) {
            // Handle validation errors
            foreach ($e->errors() as $field => $messages) {
                foreach ($messages as $message) {
                    $this->dispatch('show-notification', [
                        'type' => 'error',
                        'message' => $message,
                        'duration' => 5000
                    ]);
                }
            }
            throw $e;
        } catch (\Exception $e) {
            // Rollback transaction
            DB::rollBack();

            Log::error('Registration failed', [
                'error' => $e->getMessage(),
                'username' => $this->StrUserID,
                'email' => $this->email,
                'ip' => request()->ip()
            ]);

            // Show error notification
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'trans:auth.register_error',
                'duration' => 5000
            ]);

            // Dispatch event to preserve language state after error
            $this->dispatch('preserve-language-state');
        }
    }
};

?>

<div>
    <!-- Register Modal Trigger Button -->
    <button class="auth-btn auth-btn-register hover:scale-105 transform transition-all duration-300 w-full lg:w-auto"
        wire:click="openRegisterModal">
        <i class="fas fa-user-plus mr-2 rtl:mr-0 rtl:ml-2"></i>
        <span wire:ignore data-translate="authentication.register">Register</span>
    </button>

    <!-- Register Modal -->
    @if($activeModal === 'register')
        <div x-data="{
            show: true,
            switching: false,
            preventDoubleSwitch() {
                if (this.switching) return false;
                this.switching = true;
                setTimeout(() => { this.switching = false; }, 1000);
                return true;
            },
            init() {
                // Preserve language after Alpine component initialization
                this.$nextTick(() => {
                    if (window.languageManager) {
                        setTimeout(() => {
                            window.languageManager.applyTranslations();
                        }, 100);
                    }
                });
            }
        }" x-show="show"
             x-transition:enter="transition ease-out duration-500"
             x-transition:enter-start="opacity-0"
             x-transition:enter-end="opacity-100"
             x-transition:leave="transition ease-in duration-300"
             x-transition:leave-start="opacity-100"
             x-transition:leave-end="opacity-0"
             @click.away="$wire.closeModal()"
             class="modal-overlay register-modal"
             wire:ignore.self>

            <!-- Animated Glass Background -->
            <div class="absolute inset-0 overflow-hidden pointer-events-none">
                <div class="glass-orb glass-orb-1"></div>
                <div class="glass-orb glass-orb-2"></div>
                <div class="glass-orb glass-orb-3"></div>
            </div>

            <div x-transition:enter="transition ease-out duration-500 delay-150"
                 x-transition:enter-start="opacity-0 scale-75 rotate-3"
                 x-transition:enter-end="opacity-100 scale-100 rotate-0"
                 x-transition:leave="transition ease-in duration-200"
                 x-transition:leave-start="opacity-100 scale-100 rotate-0"
                 x-transition:leave-end="opacity-0 scale-90 rotate-1"
                 class="modal-container"
                 onclick="event.stopPropagation()">

                <!-- Close Button -->
                <button @click="show = false; $wire.closeModal()"
                    class="modal-close">
                    <i class="fas fa-times"></i>
                    <!-- Fallback if FontAwesome doesn't load -->
                    <span class="close-fallback">×</span>
                </button>


                <!-- Modal Header -->
                <div class="modal-header">
                    <div class="modal-icon">
                        <i class="fas fa-user-plus"></i>
                    </div>
                    <h2 class="modal-title" wire:ignore data-translate="authentication.joinAdventure">
                        Join the Adventure!
                    </h2>
                    <p class="modal-subtitle" wire:ignore data-translate="authentication.createAccountLegend">
                        Create your account and become a legend
                    </p>
                </div>

                <!-- Registration Form -->
               <form wire:submit="register" class="space-y-6"
                     x-data="{
                         init() {
                             // Listen for Livewire updates and reapply translations
                             this.$wire.on('*', () => {
                                 this.$nextTick(() => {
                                     if (window.languageManager) {
                                         setTimeout(() => {
                                             window.languageManager.applyTranslations();
                                         }, 50);
                                     }
                                 });
                             });

                             // Force language preservation on form interactions
                             this.$el.addEventListener('input', () => {
                                 if (window.languageManager) {
                                     window.languageManager.debounceApplyTranslations();
                                 }
                             });
                         }
                     }">
                    <!-- Hidden field to capture current page language -->
                    <input type="hidden" wire:model.defer="lang" id="livewire-register-lang" />

                    <script>
                        (function() {
                            // savedLang priority: localStorage preferred-language -> server-provided languageConfig -> fallback 'en'
                            function setInitialLang() {
                                const savedLang = localStorage.getItem('preferred-language') || (window.languageConfig && window.languageConfig.currentLanguage) || 'en';
                                const el = document.getElementById('livewire-register-lang');
                                if (el) el.value = savedLang;

                                try {
                                    if (window.Livewire && el) {
                                        const compEl = el.closest('[wire\\:id]');
                                        if (compEl) {
                                            const lw = Livewire.find(compEl.getAttribute('wire:id'));
                                            lw && lw.set && lw.set('lang', savedLang);
                                        }
                                    }
                                } catch (err) {
                                    // ignore
                                }
                            }

                            document.addEventListener('DOMContentLoaded', () => setInitialLang());
                        })();
                    </script>
                    <!-- Username Field -->
                    <div class="form-group">
                        <label class="form-label" wire:ignore data-translate="authentication.gameUsername">Game Username</label>
                        <div class="relative">
                            <div class="absolute inset-y-0 ltr:left-0 rtl:right-0 ltr:pl-3 rtl:pr-3 flex items-center pointer-events-none">
                                <i class="fas fa-gamepad text-gray-400"></i>
                            </div>
                            <input wire:model="StrUserID" type="text" autocomplete="username"
                                class="glass-input w-full ltr:pl-10 rtl:pr-10 ltr:pr-4 rtl:pl-4 @error('StrUserID') border-red-500 @enderror"
                                wire:ignore data-translate="authentication.enterGameUsername"
                                placeholder="Enter your game username (6-16 characters)"
                                required />
                        </div>
                        @error('StrUserID')
                            <span class="text-red-500 text-sm mt-1">{{ $message }}</span>
                        @enderror
                    </div>

                    <!-- Email Field -->
                    <div class="form-group">
                        <label class="form-label" wire:ignore data-translate="authentication.emailAddress">Email Address</label>
                        <div class="relative">
                            <div class="absolute inset-y-0 ltr:left-0 rtl:right-0 ltr:pl-3 rtl:pr-3 flex items-center pointer-events-none">
                                <i class="fas fa-envelope text-gray-400"></i>
                            </div>
                            <input wire:model="email" type="email" autocomplete="email"
                                class="glass-input w-full ltr:pl-10 rtl:pr-10 ltr:pr-4 rtl:pl-4 @error('email') border-red-500 @enderror"
                                wire:ignore data-translate="authentication.enterEmailAddress"
                                placeholder="Enter your email address"
                                required />
                        </div>
                        @error('email')
                            <span class="text-red-500 text-sm mt-1">{{ $message }}</span>
                        @enderror
                    </div>

                    <!-- Password Field -->
                    <div class="form-group">
                        <label class="form-label" wire:ignore data-translate="authentication.gamePassword">Game Password</label>
                        <div class="relative">
                            <div class="absolute inset-y-0 ltr:left-0 rtl:right-0 ltr:pl-3 rtl:pr-3 flex items-center pointer-events-none">
                                <i class="fas fa-lock text-gray-400"></i>
                            </div>
                            <input wire:model="sro_password" type="password" autocomplete="new-password"
                                class="glass-input w-full ltr:pl-10 rtl:pr-10 ltr:pr-4 rtl:pl-4 @error('sro_password') border-red-500 @enderror"
                                wire:ignore data-translate="authentication.enterGamePassword"
                                placeholder="Enter your game password (minimum 6 characters)"
                                required />
                        </div>
                        @error('sro_password')
                            <span class="text-red-500 text-sm mt-1">{{ $message }}</span>
                        @enderror
                    </div>

                    <!-- Confirm Password Field -->
                    <div class="form-group">
                        <label class="form-label" wire:ignore data-translate="authentication.confirmPassword">Confirm Password</label>
                        <div class="relative">
                            <div class="absolute inset-y-0 ltr:left-0 rtl:right-0 ltr:pl-3 rtl:pr-3 flex items-center pointer-events-none">
                                <i class="fas fa-lock text-gray-400"></i>
                            </div>
                            <input wire:model="sro_password_confirmation" type="password" autocomplete="new-password"
                                class="glass-input w-full ltr:pl-10 rtl:pr-10 ltr:pr-4 rtl:pl-4 @error('sro_password_confirmation') border-red-500 @enderror"
                                wire:ignore data-translate="authentication.confirmYourPassword"
                                placeholder="Confirm your password"
                                required />
                        </div>
                        @error('sro_password_confirmation')
                            <span class="text-red-500 text-sm mt-1">{{ $message }}</span>
                        @enderror
                    </div>

                    <!-- Terms and Conditions -->
                    <div class="form-group">
                        <div class="flex items-center text-sm">
                            <input wire:model="terms" type="checkbox" class="mr-2 rtl:mr-0 rtl:ml-2" required />
                            <span class="text-gray-300">
                                <span wire:ignore data-translate="authentication.agreeToTerms">I agree to the</span>
                                <a href="#" class="text-purple-400 hover:text-blue-400 underline" wire:ignore data-translate="authentication.termsAndConditions">Terms and Conditions</a>
                            </span>
                        </div>
                        @error('terms')
                            <span class="text-red-500 text-sm mt-1">{{ $message }}</span>
                        @enderror
                    </div>

                    <!-- Submit Button -->
                    <button type="submit" wire:loading.attr="disabled"
                        class="glass-button">
                        <span wire:loading.remove class="flex items-center space-x-2 rtl:space-x-reverse">
                            <i class="fas fa-user-plus"></i>
                            <span wire:ignore data-translate="authentication.createAccount">Create Account</span>
                        </span>
                        <span wire:loading class="flex items-center justify-center space-x-2 rtl:space-x-reverse">
                            <i class="fas fa-spinner fa-spin"></i>
                            <span wire:ignore data-translate="authentication.creating">Creating...</span>
                        </span>
                    </button>
                </form>



                <!-- Footer -->
                <div class="form-footer">
                    <span class="inline-flex items-center space-x-1 rtl:space-x-reverse">
                        <span wire:ignore data-translate="authentication.alreadyHaveAccount">
                            Already have an account?
                        </span>
                        <button type="button"
                            @click="if(preventDoubleSwitch()) $wire.switchToLogin()"
                            :disabled="switching"
                            class="text-purple-400 hover:text-blue-400 ltr:ml-1 rtl:mr-1 transition-all duration-200 bg-transparent border-none cursor-pointer disabled:opacity-50 font-medium hover:underline transform hover:scale-105"
                            data-en="Sign in" data-ar="تسجيل الدخول" data-translate="authentication.signIn">
                            <span x-text="switching ? (window.getTranslation ? window.getTranslation('authentication.switching') : 'Switching...') : (window.getTranslation ? window.getTranslation('authentication.signIn') : 'Sign in')"
                                x-bind:data-switching-text="switching ? (window.getTranslation ? window.getTranslation('authentication.switching') : 'Switching...') : ''"
                                data-translate="authentication.switching"></span>
                        </button>
                    </span>
                </div>
            </div>
        </div>
    @endif
</div>

@push('scripts')
<script>
document.addEventListener('livewire:init', () => {
    // Before submit ensure Livewire lang is set from savedLang (localStorage or server-provided)
    document.addEventListener('submit', (e) => {
        const form = e.target.closest('form[wire\\:submit]');
        if (form) {
            const savedLang = localStorage.getItem('preferred-language') || (window.languageConfig && window.languageConfig.currentLanguage) || 'en';
            try {
                const el = form.querySelector('#livewire-register-lang');
                if (el) el.value = savedLang;
                if (window.Livewire && el) {
                    const compEl = el.closest('[wire\\:id]');
                    if (compEl) {
                        const lw = Livewire.find(compEl.getAttribute('wire:id'));
                        lw && lw.set && lw.set('lang', savedLang);
                    }
                }
            } catch (err) {}
        }
    });

    // Handle notification translation using languageManager as fallback; keep UI notifications
    Livewire.on('show-notification', (event) => {
        const data = event.detail || event;
        let message = data.message || '';
        if (message && window.languageManager) {
            try {
                if (message.includes('.')) {
                    message = window.languageManager.getTranslation(message);
                    if (data.params) {
                        Object.entries(data.params).forEach(([k, v]) => {
                            message = message.replace(`[${k}]`, v);
                        });
                    }
                }
            } catch (err) {}
        }

        showNotification(data.type, message, data.duration || 5000);
    });
});

// Notification function (unchanged)
function showNotification(type, message, duration = 5000) {
    const notification = document.createElement('div');
    notification.className = `fixed top-4 right-4 p-4 rounded-lg shadow-lg z-50 max-w-sm ${
        type === 'error' ? 'bg-red-600' : 'bg-green-600'
    } text-white transform translate-x-full transition-transform duration-300`;

    notification.innerHTML = `
        <div class="flex justify-between items-start">
            <div>
                <p class="text-sm">${message}</p>
            </div>
            <button onclick="this.parentElement.parentElement.remove()" class="ml-2 text-white hover:text-gray-200">
                <svg class="w-4 h-4" fill="currentColor" viewBox="0 0 20 20">
                    <path fill-rule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clip-rule="evenodd"></path>
                </svg>
            </button>
        </div>
    `;

    document.body.appendChild(notification);

    // Animate in
    setTimeout(() => {
        notification.classList.remove('translate-x-full');
    }, 100);

    // Auto remove
    setTimeout(() => {
        notification.classList.add('translate-x-full');
        setTimeout(() => {
            if (notification.parentElement) {
                notification.remove();
            }
        }, 300);
    }, duration);
}

// Backup: nothing relying on languageManager in this file anymore
</script>
@endpush
